package com.engisis.sysphs.util;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.Hashtable;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.log4j.Logger;

/**
 * Class to post-process the XMI serialization.
 * 
 * @author barbau
 *
 */
public class XMIPostWriter implements Runnable
{
    private static final Logger log = Logger.getLogger(XMIPostWriter.class);
    /**
     * stream from which the content is read
     */
    private InputStream is;
    /**
     * file where the content is written into
     */
    private OutputStreamWriter osw;
    /**
     * ID replacement map
     */
    private Hashtable<String, String> idmaps;
    /**
     * namespace replacement map
     */
    private Hashtable<String, String> nsmaps;
    
    /**
     * Creates a post-processor
     * 
     * @param is
     *            input stream
     * @param file
     *            output stream
     * @param idmaps
     *            ID map
     * @param nsmaps
     *            namespace map
     * @throws FileNotFoundException
     */
    public XMIPostWriter(InputStream is, String file, Hashtable<String, String> idmaps, Hashtable<String, String> nsmaps)
            throws FileNotFoundException
    {
        this.is = is;
        osw = new OutputStreamWriter( new FileOutputStream(file), StandardCharsets.UTF_8);
        this.idmaps = idmaps;
        this.nsmaps = nsmaps;
    }
    
    /**
     * Essentially rewrites the input stream into the file stream, but performs
     * ID/namespace replacements
     */
    @Override
    public void run()
    {
        int eventType;
        
        // check if there are more events
        // in the input stream
        int skip = 0;
        try
        {
            XMLInputFactory xmlif = XMLInputFactory.newFactory();
            xmlif.setProperty(XMLInputFactory.IS_REPLACING_ENTITY_REFERENCES, Boolean.valueOf(false));
            XMLStreamReader xmlr = xmlif.createXMLStreamReader(is);
            while (xmlr.hasNext())
            {
                eventType = xmlr.next();
                switch (eventType)
                {
                case XMLStreamConstants.START_ELEMENT:
                    // manage skip
                    if (skip == 0)
                    {
                        if ("eAnnotations".equals(xmlr.getLocalName()))
                            skip++;
                    }
                    else
                        skip++;
                    if (skip > 0)
                        continue;
                    
                    write("<");
                    String prefix = xmlr.getPrefix();
                    if (prefix != null && !prefix.isEmpty())
                        write(prefix + ":");
                    write(xmlr.getLocalName());
                    
                    for (int i = 0; i < xmlr.getAttributeCount(); i++)
                    {
                        write(" ");
                        prefix = xmlr.getAttributePrefix(i);
                        String name = xmlr.getAttributeLocalName(i);
                        String value = xmlr.getAttributeValue(i);
                        if ("href".equals(name) && idmaps != null)
                        {
                            String rep = idmaps.get(value);
                            if (rep != null)
                                value = rep;
                        }
                        if ("xmi".equals(prefix) && "version".equals(name))
                            continue;
                        if (prefix != null && !prefix.isEmpty())
                            write(prefix + ":");
                        write(name + "=\"" + escape(value) + "\"");
                    }
                    for (int i = 0; i < xmlr.getNamespaceCount(); i++)
                    {
                        // just because of the wrong URI used by MD
                        // TODO: define that in EMFUtil
                        String namespace = xmlr.getNamespaceURI(i);
                        if (nsmaps != null)
                        {
                            String rep = nsmaps.get(namespace);
                            if (rep != null)
                                namespace = rep;
                        }
                        
                        // if
                        // (namespace.startsWith("http://www.eclipse.org/uml2/"))
                        // namespace = "http://www.omg.org/spec/UML/20131001";
                        // else if
                        // (namespace.equals("http://www.omg.org/spec/SysML/20150709/SysML"))
                        // namespace =
                        // "http://www.omg.org/spec/SysML/20131201/SysML";
                        write(" xmlns:" + xmlr.getNamespacePrefix(i) + "=\"" + namespace + "\"");
                    }
                    
                    write(">");
                    break;
                case XMLStreamConstants.END_ELEMENT:
                    if (skip > 0)
                    {
                        skip--;
                        continue;
                    }
                    write("</");
                    if (xmlr.getPrefix() != null && !xmlr.getPrefix().isEmpty())
                        write(xmlr.getPrefix() + ":");
                    write(xmlr.getLocalName() + ">");
                    break;
                
                case XMLStreamConstants.PROCESSING_INSTRUCTION:
                    if (skip > 0)
                        continue;
                    write("<?" + xmlr.getText() + "?>");
                    break;
                case XMLStreamConstants.CHARACTERS:
                    if (skip > 0)
                        continue;
                    write(escape(new String(xmlr.getTextCharacters(), xmlr.getTextStart(), xmlr.getTextLength())));
                    break;
                case XMLStreamConstants.COMMENT:
                    break;
                case XMLStreamConstants.START_DOCUMENT:
                    write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
                    break;
                case XMLStreamConstants.END_DOCUMENT:
                    osw.flush();
                    osw.close();
                    break;
                case XMLStreamConstants.ENTITY_REFERENCE:
                    if (skip > 0)
                        continue;
                    write("&" + xmlr.getLocalName() + ";");
                    break;
                case XMLStreamConstants.ATTRIBUTE:
                    break;
                case XMLStreamConstants.DTD:
                    break;
                case XMLStreamConstants.CDATA:
                    if (skip > 0)
                        continue;
                    write("<![CDATA[" + new String(xmlr.getTextCharacters(), xmlr.getTextStart(), xmlr.getTextLength())
                            + "]]>");
                    break;
                case XMLStreamConstants.SPACE:
                    break;
                default:
                    break;
                }
            }
        }
        catch (XMLStreamException e)
        {
            log.error("Error while parsing the file", e);
        }
        catch (IOException e)
        {
            log.error("Error while parsing the file", e);
        }
        
    }
    
    /**
     * Writes the specified string in the stream
     * 
     * @param str
     *            string to write
     * @throws IOException
     */
    private void write(String str) throws IOException
    {
        osw.write(str);
    }
    
    /**
     * Excepes the specified string
     * 
     * @param str
     *            string to escape
     * @return
     */
    private static String escape(String str)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < str.length(); i++)
        {
            char ch = str.charAt(i);
            switch (ch)
            {
            case '<':
                sb.append("&lt;");
                break;
            case '>':
                sb.append("&gt;");
                break;
            case '&':
                sb.append("&amp;");
                break;
            case '"':
                sb.append("&quot;");
                break;
            default:
                sb.append(ch);
            }
        }
        return sb.toString();
    }
    
}
