package com.engisis.sysphs.util;

import java.util.Hashtable;

import com.engisis.sysphs.util.ExpressionLanguageBaseListener;
import com.engisis.sysphs.util.ExpressionLanguageParser;
import com.engisis.sysphs.util.ExpressionLanguageParser.Add_opContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Arithmetic_expressionContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Array_subscriptsContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Component_referenceContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.EquationContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.ExpressionContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Expression_listContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.FactorContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.For_indexContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.For_indicesContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.For_statementContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Function_argumentContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Function_argumentsContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Function_call_argsContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.If_equationContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.If_statementContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Logical_expressionContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Logical_factorContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Logical_termContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Mul_opContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Name_pathContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Named_argumentContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Named_argumentsContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Output_expression_listContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.PrimaryContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Rel_opContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.RelationContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.Simple_expressionContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.StatementContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.StatementsContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.SubscriptContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.TermContext;
import com.engisis.sysphs.util.ExpressionLanguageParser.While_statementContext;

import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeProperty;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.apache.log4j.Logger;

/**
 * Abstract translator from the SysPhS expression language to simulation
 * languages. By default, copies the content.
 * 
 * @author barbau
 *
 */
public abstract class ExpressionLanguageToSimulation extends ExpressionLanguageBaseListener
{
    private static final Logger log = Logger.getLogger(ExpressionLanguageToSimulation.class);
    /**
     * substitutions to perform
     */
    private Hashtable<String, String> substitutions;
    /**
     * Associates string values with syntax tree nodes
     */
    private ParseTreeProperty<String> ptp;
    /**
     * root syntax tree node
     */
    private ParseTree root = null;
    
    /**
     * Sets the substitution table
     * 
     * @param substitutions
     *            substitution table
     */
    public void setSubstitutions(Hashtable<String, String> substitutions)
    {
        this.substitutions = substitutions;
    }
    
    /**
     * Things to execute before a parsing is realized
     */
    protected void beforeParsing()
    {
        ptp = new ParseTreeProperty<String>();
    }
    
    /**
     * Things to execute after a parsing is realized
     */
    protected void afterParsing()
    {
    }
    
    /**
     * Returns the string value associated with a syntax tree node
     * 
     * @param pt
     *            syntax tree node
     * @return associated value
     */
    public String getValue(ParseTree pt)
    {
        String v = ptp.get(pt);
        if (v == null)
            log.error("The value of " + pt.getClass() + " is null");
        return v;
    }
    
    /**
     * Associates a value to a syntax tree node
     * 
     * @param pt
     *            syntax tree node
     * @param str
     *            string value
     */
    public void setValue(ParseTree pt, String str)
    {
        ptp.put(pt, str);
    }
    
    @Override
    public void enterEquation(EquationContext ctx)
    {
        if (root == null)
        {
            root = ctx;
            beforeParsing();
        }
        super.enterEquation(ctx);
    }
    
    @Override
    public void exitEquation(EquationContext ctx)
    {
        if (ctx == root)
        {
            afterParsing();
            root = null;
        }
        processStraight(ctx);
    }
    
    @Override
    public void enterStatements(StatementsContext ctx)
    {
        if (root == null)
        {
            root = ctx;
            beforeParsing();
        }
        super.enterStatements(ctx);
    }
    
    @Override
    public void exitStatements(StatementsContext ctx)
    {
        if (ctx == root)
        {
            afterParsing();
            root = null;
        }
        processStraight(ctx);
    }
    
    @Override
    public void exitStatement(StatementContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitIf_equation(If_equationContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.getChildCount(); i++)
        {
            ParseTree pt = ctx.getChild(i);
            if (pt instanceof TerminalNode)
            {
                int t = ((TerminalNode) pt).getSymbol().getType();
                if (t == ExpressionLanguageParser.IF)
                    sb.append("if");
                else if (t == ExpressionLanguageParser.THEN)
                    sb.append("then\n");
                else if (t == ExpressionLanguageParser.ELSEIF)
                    sb.append("elseif");
                else if (t == ExpressionLanguageParser.ELSE)
                    sb.append("else\n");
                else if (t == ExpressionLanguageParser.END)
                    sb.append("end ");
            }
            else if (pt instanceof ExpressionContext)
                sb.append(" " + getValue(pt) + " ");
            else if (pt instanceof EquationContext)
                sb.append(getValue(pt) + ";\n");
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitIf_statement(If_statementContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.getChildCount(); i++)
        {
            ParseTree pt = ctx.getChild(i);
            if (pt instanceof TerminalNode)
            {
                int t = ((TerminalNode) pt).getSymbol().getType();
                if (t == ExpressionLanguageParser.IF)
                    sb.append("if");
                else if (t == ExpressionLanguageParser.THEN)
                    sb.append("then\n");
                else if (t == ExpressionLanguageParser.ELSEIF)
                    sb.append("elseif");
                else if (t == ExpressionLanguageParser.ELSE)
                    sb.append("else\n");
                else if (t == ExpressionLanguageParser.END)
                    sb.append("end ");
            }
            else if (pt instanceof ExpressionContext)
                sb.append(" " + getValue(pt) + " ");
            else if (pt instanceof StatementContext)
                sb.append(getValue(pt) + ";\n");
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitFor_statement(For_statementContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        sb.append("for " + getValue(ctx.for_indices()) + " loop\n");
        for (StatementContext sc : ctx.statement())
            sb.append(getValue(sc) + ";\n");
        sb.append("end for");
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitFor_indices(For_indicesContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitFor_index(For_indexContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitWhile_statement(While_statementContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        sb.append("while " + getValue(ctx.expression()) + " loop");
        for (StatementContext sc : ctx.statement())
            sb.append(getValue(sc) + ";\n");
        sb.append("end while");
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void enterExpression(ExpressionContext ctx)
    {
        if (root == null)
        {
            root = ctx;
            beforeParsing();
        }
        super.enterExpression(ctx);
    }

    @Override
    public void exitExpression(ExpressionContext ctx)
    {
        if (ctx == root)
        {
            afterParsing();
            root = null;
        }
        processWithSpaces(ctx);
    }
    
    @Override
    public void exitSimple_expression(Simple_expressionContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitLogical_expression(Logical_expressionContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (Logical_termContext ltc : ctx.logical_term())
        {
            if (ltc != ctx.logical_term(0))
                sb.append(" " + getValue(ctx.OR(0)) + " ");
            sb.append(getValue(ltc));
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitLogical_term(Logical_termContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (Logical_factorContext lfc : ctx.logical_factor())
        {
            if (lfc != ctx.logical_factor(0))
                sb.append(" " + getValue(ctx.AND(0)) + " ");
            sb.append(getValue(lfc));
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitLogical_factor(Logical_factorContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        if (ctx.NOT() != null)
            sb.append(getValue(ctx.NOT()) + " ");
        sb.append(getValue(ctx.relation()));
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitRelation(RelationContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitRel_op(Rel_opContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitArithmetic_expression(Arithmetic_expressionContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitAdd_op(Add_opContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitTerm(TermContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitMul_op(Mul_opContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitFactor(FactorContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitPrimary(PrimaryContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        if (ctx.UNSIGNED_NUMBER() != null)
            sb.append(ctx.getText());
        else if (ctx.STRING() != null)
            sb.append(ctx.STRING());
        else if (ctx.TRUE() != null)
            sb.append("true");
        else if (ctx.FALSE() != null)
            sb.append("false");
        else if (ctx.function_call_args() != null)
        {
            if (ctx.name_path() != null)
                sb.append(getValue(ctx.name_path()));
            else if (ctx.DER() != null)
                sb.append("der");
            sb.append(getValue(ctx.function_call_args()));
        }
        else if (ctx.component_reference() != null)
            sb.append(getValue(ctx.component_reference()));
        else if (ctx.output_expression_list() != null)
            sb.append("(" + getValue(ctx.output_expression_list()) + ")");
        else if (ctx.expression_list().size() != 0)
        {
            sb.append("[");
            for (Expression_listContext elc : ctx.expression_list())
            {
                if (elc != ctx.expression_list(0))
                    sb.append(";");
                sb.append(getValue(elc));
            }
            sb.append("]");
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitFunction_call_args(Function_call_argsContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitName_path(Name_pathContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitComponent_reference(Component_referenceContext ctx)
    {
        
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.getChildCount(); i++)
            sb.append(getValue(ctx.getChild(i)));
        String s = sb.toString();
        if (substitutions != null)
        {
            String v = substitutions.get(s);
            if (v != null)
            {
                setValue(ctx, v);
                return;
            }
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitFunction_arguments(Function_argumentsContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        if (ctx.named_arguments() != null)
            sb.append(getValue(ctx.named_arguments()));
        else
        {
            sb.append(getValue(ctx.function_argument()));
            if (ctx.function_arguments() != null)
                sb.append("," + getValue(ctx.function_arguments()));
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitNamed_arguments(Named_argumentsContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitNamed_argument(Named_argumentContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitFunction_argument(Function_argumentContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        if (ctx.expression() != null)
            sb.append(getValue(ctx.expression()));
        else
        {
            sb.append("function " + getValue(ctx.IDENT()) + "(");
            sb.append(getValue(ctx.named_arguments()));
            sb.append(")");
        }
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitOutput_expression_list(Output_expression_listContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpression_list(Expression_listContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitArray_subscripts(Array_subscriptsContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitSubscript(SubscriptContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void visitTerminal(TerminalNode node)
    {
        setValue(node, node.getText());
    }
    
    protected void processStraight(ParseTree ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.getChildCount(); i++)
        {
            sb.append(getValue(ctx.getChild(i)));
        }
        setValue(ctx, sb.toString());
    }
    
    protected void processWithSpaces(ParseTree ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.getChildCount(); i++)
        {
            if (i != 0)
                sb.append(" ");
            sb.append(getValue(ctx.getChild(i)));
        }
        setValue(ctx, sb.toString());
    }
}
