package com.engisis.sysphs.translation.simulink;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Stack;

import com.engisis.sysphs.deserialization.simulink.SimulinkDeserializer;
import com.engisis.sysphs.language.simscape.SBranch;
import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SComponentReference;
import com.engisis.sysphs.language.simscape.SConnection;
import com.engisis.sysphs.language.simscape.SConnectionPortBlock;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SEquation;
import com.engisis.sysphs.language.simscape.SInput;
import com.engisis.sysphs.language.simscape.SMember;
import com.engisis.sysphs.language.simscape.SMemberAssignment;
import com.engisis.sysphs.language.simscape.SMemberPath;
import com.engisis.sysphs.language.simscape.SNode;
import com.engisis.sysphs.language.simscape.SOutput;
import com.engisis.sysphs.language.simscape.SParameter;
import com.engisis.sysphs.language.simscape.SPhysicalBlock;
import com.engisis.sysphs.language.simscape.SPhysicalConnectionPoint;
import com.engisis.sysphs.language.simscape.SPhysicalLine;
import com.engisis.sysphs.language.simscape.SVariable;
import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SConnectionPoint;
import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.simulink.SFContinuousStateVariable;
import com.engisis.sysphs.language.simulink.SFDiscreteStateVariable;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFParameter;
import com.engisis.sysphs.language.simulink.SFVariable;
import com.engisis.sysphs.language.simulink.SFVariableAssignment;
import com.engisis.sysphs.language.simulink.SFunction;
import com.engisis.sysphs.language.simulink.SFunction2;
import com.engisis.sysphs.language.simulink.SFunction2Block;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SInterface;
import com.engisis.sysphs.language.simulink.SLine;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SNamedElement;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SReference;
import com.engisis.sysphs.language.simulink.SSubsystem;
import com.engisis.sysphs.language.simulink.SSystem;
import com.engisis.sysphs.language.simulink.SSystemParameterAssignment;
import com.engisis.sysphs.language.stateflow.SChart;
import com.engisis.sysphs.language.stateflow.SChartBlock;
import com.engisis.sysphs.language.stateflow.SChartSystem;
import com.engisis.sysphs.language.stateflow.SConnectNode;
import com.engisis.sysphs.language.stateflow.SData;
import com.engisis.sysphs.language.stateflow.SDataScope;
import com.engisis.sysphs.language.stateflow.SLinkNode;
import com.engisis.sysphs.language.stateflow.SState;
import com.engisis.sysphs.language.stateflow.SStateflow;
import com.engisis.sysphs.language.stateflow.STransition;
import com.engisis.sysphs.language.stateflow.STreeNode;
import com.engisis.sysphs.util.ANTLRErrorListener;
import com.engisis.sysphs.util.EMFUtil;
import com.engisis.sysphs.util.SimulationToSysMLTranslator;
import com.engisis.sysphs.util.SysMLUtil;
import com.engisis.sysphs.util.UMLModelErrorException;
import com.engisis.sysphs.util.SysMLUtil.LibraryComponent;
import com.engisis.sysphs.util.SysMLUtil.LibraryParameter;
import com.engisis.sysphs.util.SysMLUtil.LibraryPort;
import com.engisis.sysphs.generation.simulink.MATLABLexer;
import com.engisis.sysphs.generation.simulink.MATLABParser;

import org.antlr.v4.runtime.ANTLRInputStream;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeWalker;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.ChangeEvent;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.InstanceSpecification;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.LiteralReal;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.PseudostateKind;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.TimeEvent;
import org.eclipse.uml2.uml.TimeExpression;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.ValueSpecification;

/**
 * Translator from Simulink to SysML.
 * 
 * @author barbau
 *
 */
public class SimulinkToSysMLTranslator extends SimulationToSysMLTranslator
{
    
    private static final Logger log = Logger.getLogger(SimulinkToSysMLTranslator.class);
    
    /**
     * Root Simulink model
     */
    private SModel smodel;
    /**
     * List of Stateflow objects
     */
    private List<SStateflow> lstateflow;
    
    /**
     * SysML utility
     */
    private SysMLUtil sysmlutil;
    /**
     * root Simscape package
     */
    private Package uroot;
    
    /**
     * map with string as keys, and units as values
     */
    private Hashtable<String, DataType> units = new Hashtable<String, DataType>();
    /**
     * map with value types as key, and port types as values
     */
    private Hashtable<DataType, Class> porttypes = new Hashtable<DataType, Class>();
    /**
     * reference table
     */
    private Hashtable<ReferenceKey, NamedElement> refs = new Hashtable<ReferenceKey, NamedElement>();
    /**
     * substitition table by class
     */
    private Hashtable<Class, Hashtable<String, String>> subs = new Hashtable<Class, Hashtable<String, String>>();
    
    /**
     * map that gives the syntax tree node associated with Simulink objects
     */
    private Hashtable<SElement, ParseTree> expressions;
    
    /**
     * first-pass processing stack
     */
    private Stack<ReferenceKey> toProcess = new Stack<ReferenceKey>();
    /**
     * second-pass processing stack, needed because all properties need to be
     * typed
     */
    private Stack<ReferenceKey> toProcess2 = new Stack<ReferenceKey>();
    
    /**
     * expression extractor
     */
    private SimscapeExpressionExtractor see = new SimscapeExpressionExtractor();
    
    protected static class ReferenceKey
    {
        private SNamedElement[] key;
        
        public ReferenceKey(SNamedElement[] key)
        {
            this.key = key.clone();
        }
        
        public SNamedElement[] getKey()
        {
            return key.clone();
        }
        
        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof ReferenceKey)
            {
                ReferenceKey rk = (ReferenceKey) obj;
                if (rk.key.length != key.length)
                    return false;
                for (int i = 0; i < key.length; i++)
                    if (rk.key[i] != key[i])
                        return false;
                return true;
            }
            return false;
        }
        
        @Override
        public int hashCode()
        {
            return toString().hashCode();
        }
        
        @Override
        public String toString()
        {
            StringBuilder sb = new StringBuilder();
            for (SNamedElement melem : key)
                sb.append(melem.getName() + " ");
            return sb.toString();
        }
    }
    
    private static ReferenceKey getKey(SNamedElement... elements)
    {
        return new ReferenceKey(elements);
    }
    
    @Override
    public void execute(String model, ResourceSet rs, File outputdirectory, String target, String paths[])
            throws IOException, UMLModelErrorException
    {
        if (model == null)
            throw new IllegalArgumentException("The model can't be null");
        if (rs == null)
            throw new IllegalArgumentException("The ResourceSet can't be null");
        File file = new File(model);
        if (!file.isFile() || !file.canRead())
            throw new IllegalArgumentException("The file must be readable");
        SimulinkDeserializer sd = new SimulinkDeserializer();
        sd.deserialize(model);
        expressions = sd.getExpressions();
        smodel = sd.getModel();
        lstateflow = sd.getStateflow();
        
        String[] paths2 = null;
        
        if (paths == null)
            paths2 = new String[1];
        else
        {
            int len = paths.length;
            paths2 = new String[len + 1];
            for (int i = 0; i < len; i++)
                paths2[i] = paths[i];
        }
        
        paths2[paths2.length - 1] = file.getParent();
        
        URI uri = null;
        if (outputdirectory == null)
        {
            uri = URI.createFileURI(model).trimFileExtension();
            String fn = uri.lastSegment() + getFileNameSuffix();
            uri = uri.trimSegments(1).appendSegment(fn).appendFileExtension("xmi");
        }
        else
        {
            uri = URI.createFileURI(outputdirectory.getAbsolutePath());
            String fn = URI.createFileURI(model).trimFileExtension().lastSegment() + getFileNameSuffix();
            uri = uri.appendSegment(fn).appendFileExtension("xmi");
        }
        
        Resource r = EMFUtil.createSysMLModel(rs, uri, paths2);
        
        EMFUtil.getSimulationLibrary(rs, URI.createFileURI(model), paths2);
        sysmlutil = new SysMLUtil(r.getResourceSet());
        // get or create root model
        for (EObject eo : r.getContents())
            if (eo instanceof Package)
                uroot = (Package) eo;
            
        if (uroot == null)
        {
            log.info("Creating root model...");
            uroot = UMLFactory.eINSTANCE.createModel();
            SysMLUtil.setName(uroot, "Data");
            r.getContents().add(uroot);
        }
        else
        {
            log.info("Reusing root model " + uroot.getName());
        }
        // make sure the name of the system is the name of the model
        if (smodel == null || smodel.getSystem() == null)
        {
            log.error("The model is null or has no system");
            return;
        }
        smodel.getSystem().setName(smodel.getName());
        Class uclass = createBlock(smodel.getSystem());
        
        while (toProcess.size() != 0)
        {
            ReferenceKey rk = toProcess.pop();
            if (rk.getKey().length == 1)
            {
                SNamedElement snm0 = rk.getKey()[0];
                // Remember to put specializations before generalizations
                if (snm0 instanceof SBlock)
                {
                    processProperty((SBlock) snm0);
                    if (snm0 instanceof SPhysicalBlock)
                        toProcess2.add(rk);
                }
                else if (snm0 instanceof SMember)
                    processProperty((SMember) snm0);
                else if (snm0 instanceof SDomain)
                    processBlock((SDomain) snm0);
                else if (snm0 instanceof SComponent)
                {
                    Class res = processBlock((SComponent) snm0);
                    if (uclass.getName() != null && res.getName().equals(uclass.getName()))
                        SysMLUtil.setName(uclass, uclass.getName() + "Container");
                    toProcess2.add(rk);
                }
                else if (snm0 instanceof SFunction2)
                    processBlock((SFunction2) snm0);
                else if (snm0 instanceof SChartSystem)
                {
                    processBlock((SChartSystem) snm0);
                    toProcess2.add(rk);
                }
                else if (snm0 instanceof SSystem)
                {
                    processBlock((SSystem) snm0);
                    toProcess2.add(rk);
                }
                else if (snm0 instanceof SState)
                    processState((SState) snm0);
                else if (snm0 instanceof STransition)
                    processTransition((STransition) snm0);
                else
                    log.error("Element not processed: " + snm0);
            }
        }
        while (toProcess2.size() != 0)
        {
            ReferenceKey rk = toProcess2.pop();
            if (rk.getKey().length == 1)
            {
                SNamedElement snm0 = rk.getKey()[0];
                if (snm0 instanceof SComponent)
                    processBlock2((SComponent) snm0);
                else if (snm0 instanceof SChartSystem)
                    processBlock2((SChartSystem) snm0);
                else if (snm0 instanceof SSystem)
                    processBlock2((SSystem) snm0);
                else if (snm0 instanceof SPhysicalBlock)
                    processProperty2((SPhysicalBlock) snm0);
                else if (snm0 instanceof SInterface)
                    processProperty2((SInterface) snm0);
                else
                    log.error("Element not processed a second time: " + snm0);
            }
        }
        
        log.info("Saving " + r.getURI());
        // look for better root class
        
        outputrootname = uclass.getQualifiedName();
        outputfilename = r.getURI().toFileString();
        EMFUtil.saveResource(r, target);
    }
    
    private Class createBlock(SSystem ssystem)
    {
        if (ssystem == null)
        {
            log.error("Null system found");
            return null;
        }
        ReferenceKey key = getKey(ssystem);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Class)
                return (Class) uelement;
            throw new IllegalStateException("The element is not a class: " + ssystem.getName());
        }
        Class ublock = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(ublock);
        ublock.applyStereotype(sysmlutil.getBlock());
        
        log.info("Creating class " + ssystem.getName());
        
        if (ssystem instanceof SChartSystem)
        {
            StateMachine ustatemachine = UMLFactory.eINSTANCE.createStateMachine();
            ublock.setClassifierBehavior(ustatemachine);
            log.info("Creating state machine " + ssystem.getName());
        }
        
        refs.put(key, ublock);
        toProcess.push(key);
        return ublock;
    }
    
    private Class createBlock(SFunction2 sfunction2)
    {
        if (sfunction2 == null)
        {
            log.error("Null SFunction found");
            return null;
        }
        ReferenceKey key = getKey(sfunction2);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Class)
                return (Class) uelement;
            throw new IllegalStateException("The element is not a class: " + sfunction2.getName());
        }
        Class ublock = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(ublock);
        ublock.applyStereotype(sysmlutil.getConstraintBlock());
        
        log.info("Creating constraint block " + sfunction2.getName());
        
        SysMLUtil.setName(ublock, sfunction2.getName());
        
        for (SFVariable sfvariable : sfunction2.getVariables())
        {
            Property uproperty = UMLFactory.eINSTANCE.createProperty();
            ublock.getOwnedAttributes().add(uproperty);
            SysMLUtil.setName(uproperty, sfvariable.getName());
            refs.put(getKey(sfvariable), uproperty);
            uproperty.setType(sysmlutil.getSysMLDouble());
        }
        
        for (SFVariableAssignment sfvariableassignment : sfunction2.getAssignments())
        {
            Constraint uconstraint = UMLFactory.eINSTANCE.createConstraint();
            OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
            // need to translate
            oe.getBodies().add(sfvariableassignment.getExpression());
            oe.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
            uconstraint.setSpecification(oe);
            ublock.getOwnedRules().add(uconstraint);
        }
        
        refs.put(key, ublock);
        toProcess.push(key);
        return ublock;
    }
    
    private Class createBlock(SDomain sdomain)
    {
        if (sdomain == null)
            throw new IllegalArgumentException("The domain should not be null");
        ReferenceKey key = getKey(sdomain);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Class)
                return (Class) uelement;
            throw new IllegalStateException("The element is not a class: " + sdomain.getName());
        }
        Class ublock = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(ublock);
        ublock.applyStereotype(sysmlutil.getBlock());
        
        log.info("Creating class " + sdomain.getName());
        
        refs.put(key, ublock);
        toProcess.push(key);
        return ublock;
    }
    
    private Class createBlock(SComponent scomponent)
    {
        if (scomponent == null)
            throw new IllegalArgumentException("The component should not be null");
        ReferenceKey key = getKey(scomponent);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Class)
                return (Class) uelement;
            throw new IllegalStateException("The element is not a class: " + scomponent.getName());
        }
        Class ublock = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(ublock);
        ublock.applyStereotype(sysmlutil.getBlock());
        
        log.info("Creating class " + scomponent.getName());
        
        refs.put(key, ublock);
        toProcess.push(key);
        return ublock;
    }
    
    private Property createProperty(SBlock sblock)
    {
        if (sblock == null)
            throw new IllegalArgumentException("The block should not be null");
        ReferenceKey key = getKey(sblock);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Property)
                return (Property) uelement;
            throw new IllegalStateException("The element is not a port: " + sblock.getName());
        }
        Property uproperty = null;
        if (sblock instanceof SInport || sblock instanceof SOutport || sblock instanceof SConnectionPortBlock)
        {
            uproperty = UMLFactory.eINSTANCE.createPort();
            log.info("Creating port " + sblock.getName());
        }
        else
        {
            uproperty = UMLFactory.eINSTANCE.createProperty();
            log.info("Creating property " + sblock.getName());
        }
        
        refs.put(key, uproperty);
        toProcess.push(key);
        return uproperty;
    }
    
    private Property createProperty(SMember smember)
    {
        if (smember == null)
            throw new IllegalArgumentException("The member should not be null");
        ReferenceKey key = getKey(smember);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Property)
                return (Property) uelement;
            throw new IllegalStateException("The element is not a property: " + smember.getName());
        }
        
        Property uproperty = null;
        if (smember instanceof SInput || smember instanceof SOutput || smember instanceof SNode)
        {
            uproperty = UMLFactory.eINSTANCE.createPort();
            log.info("Creating port " + smember.getName());
        }
        else
        {
            uproperty = UMLFactory.eINSTANCE.createProperty();
            log.info("Creating property " + smember.getName());
        }
        
        refs.put(key, uproperty);
        if (smember.getConnectionPortBlock() != null)
            refs.put(getKey(smember.getConnectionPortBlock()), uproperty);
        toProcess.push(key);
        return uproperty;
    }
    
    private Class processBlock(SSystem ssystem)
    {
        Class ublock = createBlock(ssystem);
        if (ublock == null)
        {
            log.error("Can't retrieve block");
            return null;
        }
        
        log.info("Processing block " + ssystem.getName());
        SysMLUtil.setName(ublock, ssystem.getName());
        // owned blocks
        for (SBlock sblock : ssystem.getOwnedBlocks())
        {
            if (sblock instanceof SInport || sblock instanceof SOutport || sblock instanceof SSubsystem
                    || sblock instanceof SReference || sblock instanceof SPhysicalBlock || sblock instanceof SChartBlock
                    || sblock instanceof SFunction2Block || sblock instanceof SInterface)
            {
                Property uproperty = createProperty(sblock);
                if (uproperty instanceof Port)
                    ublock.getOwnedPorts().add((Port) uproperty);
                else
                    ublock.getOwnedAttributes().add(uproperty);
            }
            else
            {
                log.error("Unsupported block type: " + sblock.getClass());
            }
        }
        // owned lines
        return ublock;
    }
    
    private Class processBlock(SChartSystem schartsystem)
    {
        Class ublock = createBlock((SSystem) schartsystem);
        if (ublock == null)
        {
            log.error("Can't retrieve block");
            return null;
        }
        
        log.info("Processing chart " + schartsystem.getName());
        SysMLUtil.setName(ublock, schartsystem.getName());
        
        for (SBlock sblock : schartsystem.getOwnedBlocks())
        {
            if (sblock instanceof SInport || sblock instanceof SOutport)
            {
                Property uproperty = createProperty(sblock);
                if (uproperty instanceof Port)
                    ublock.getOwnedPorts().add((Port) uproperty);
                else
                    ublock.getOwnedAttributes().add(uproperty);
            }
            else
            {
                log.error("Unsupported block type: " + sblock.getClass());
            }
        }
        
        if (schartsystem.getOwningSubsystem() instanceof SChartBlock)
        {
            SChart schart = ((SChartBlock) schartsystem.getOwningSubsystem()).getChart();
            if (schart == null)
            {
                log.error("Chart not found for block " + schartsystem.getName());
                return ublock;
            }
            
            StateMachine ustatemachine = (StateMachine) ublock.getClassifierBehavior();
            SysMLUtil.setName(ustatemachine, ublock.getName() + "SB");
            
            Hashtable<String, String> sub = getClassSubstitutions(ublock);
            
            Region uregion = UMLFactory.eINSTANCE.createRegion();
            
            ustatemachine.getRegions().add(uregion);
            
            for (STreeNode stn : schart.getTreeNode())
            {
                if (stn instanceof SState)
                {
                    State ustate = createState((SState) stn);
                    uregion.getSubvertices().add(ustate);
                }
            }
            for (SLinkNode sln : schart.getLinkNode())
            {
                if (sln instanceof STransition)
                {
                    STransition stransition = (STransition) sln;
                    Transition utransition = createTransition(stransition);
                    uregion.getTransitions().add(utransition);
                }
                else if (sln instanceof SData)
                {
                    SData sdata = (SData) sln;
                    if (sdata.getScope() == SDataScope.INPUT)
                    {
                        // Port uport = UMLFactory.eINSTANCE.createPort();
                        // ublock.getOwnedAttributes().add(uport);
                        // uport.setName(sdata.getName());
                        // uport.setType(getSignalPortType());
                        // sub.put(sdata.getName(), sdata.getName() + "." +
                        // usignalportproperty.getName());
                        // log.info("Adding input " + sdata.getName());
                    }
                    else if (sdata.getScope() == SDataScope.OUTPUT)
                    {
                        // Port uport = UMLFactory.eINSTANCE.createPort();
                        // ublock.getOwnedAttributes().add(uport);
                        // uport.setName(sdata.getName());
                        // uport.setType(getSignalPortType());
                        // uport.setIsConjugated(true);
                        // sub.put(sdata.getName(), sdata.getName() + "." +
                        // usignalportproperty.getName());
                        // log.info("Adding output " + sdata.getName());
                    }
                    else if (sdata.getScope() == SDataScope.CONSTANT)
                    {
                        Property uproperty = UMLFactory.eINSTANCE.createProperty();
                        ublock.getOwnedAttributes().add(uproperty);
                        uproperty.applyStereotype(sysmlutil.getPhSConstant());
                        SysMLUtil.setName(uproperty, sdata.getName());
                        uproperty.setType(sysmlutil.getSysMLDouble());
                        log.info("Adding constant " + sdata.getName());
                        
                        if (sdata.getValue() != null)
                        {
                            uproperty.setDefaultValue(getValueSpecification(sdata.getValue()));
                            log.info("Setting default value to " + uproperty.getDefaultValue());
                        }
                    }
                    sdata.getDataType();
                }
            }
        }
        
        return ublock;
    }
    
    private State createState(SState sstate)
    {
        if (sstate == null)
            throw new IllegalArgumentException("The state should not be null");
        ReferenceKey key = getKey(sstate);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof State)
                return (State) uelement;
            throw new IllegalStateException("The element is not a state: " + sstate.getName());
        }
        
        State ustate = UMLFactory.eINSTANCE.createState();
        log.info("Creating state " + sstate.getName());
        
        refs.put(key, ustate);
        toProcess.push(key);
        return ustate;
    }
    
    private State processState(SState sstate)
    {
        State ustate = createState(sstate);
        log.info("Processing state " + sstate.getName());
        SysMLUtil.setName(ustate, sstate.getName());
        
        Class ublock = (Class) ustate.getContainer().getStateMachine().getContext();
        Hashtable<String, String> sub = subs.get(ublock);
        
        String label = sstate.getLabel();
        if (label != null && !label.isEmpty())
        {
            log.info("Parsing label " + label);
            
            MatlabExpressionExtractor mee = new MatlabExpressionExtractor();
            mee.prepareNextStateLabelParsing(sub);
            MATLABLexer ml = new MATLABLexer(new ANTLRInputStream(label));
            CommonTokenStream cts = new CommonTokenStream(ml);
            MATLABParser mp = new MATLABParser(cts);
            mp.addErrorListener(new ANTLRErrorListener(log));
            
            ParserRuleContext prc = mp.state_label();
            new ParseTreeWalker().walk(mee, prc);
            
            if (mee.getEntry() != null)
            {
                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < mee.getEntry().size(); i++)
                {
                    if (i != 0)
                        sb.append('\n');
                    sb.append(mee.getEntry().get(i));
                }
                if (sb.length() != 0)
                {
                    OpaqueBehavior ob = UMLFactory.eINSTANCE.createOpaqueBehavior();
                    ob.getBodies().add(sb.toString());
                    ob.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                    ustate.setEntry(ob);
                }
            }
            if (mee.getDo() != null)
            {
                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < mee.getDo().size(); i++)
                {
                    if (i != 0)
                        sb.append('\n');
                    sb.append(mee.getDo().get(i));
                }
                if (sb.length() != 0)
                {
                    OpaqueBehavior ob = UMLFactory.eINSTANCE.createOpaqueBehavior();
                    ob.getBodies().add(sb.toString());
                    ob.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                    ustate.setEntry(ob);
                }
            }
            if (mee.getExit() != null)
            {
                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < mee.getExit().size(); i++)
                {
                    if (i != 0)
                        sb.append('\n');
                    sb.append(mee.getExit().get(i));
                }
                if (sb.length() != 0)
                {
                    OpaqueBehavior ob = UMLFactory.eINSTANCE.createOpaqueBehavior();
                    ob.getBodies().add(sb.toString());
                    ob.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                    ustate.setExit(ob);
                }
            }
        }
        
        return ustate;
    }
    
    private Transition createTransition(STransition stransition)
    {
        if (stransition == null)
            throw new IllegalArgumentException("The transition should not be null");
        ReferenceKey key = getKey(stransition);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Transition)
                return (Transition) uelement;
            throw new IllegalStateException("The element is not a transition: " + stransition.getName());
        }
        
        Transition utransition = UMLFactory.eINSTANCE.createTransition();
        log.info("Creating transition " + stransition.getName());
        
        refs.put(key, utransition);
        toProcess.push(key);
        return utransition;
    }
    
    private Transition processTransition(STransition stransition)
    {
        Transition utransition = createTransition(stransition);
        log.info("Processing transition " + stransition.getName());
        SConnectNode src = stransition.getSrc();
        if (src == null)
        {
            Pseudostate upseudostate = UMLFactory.eINSTANCE.createPseudostate();
            upseudostate.setKind(PseudostateKind.INITIAL_LITERAL);
            utransition.getContainer().getSubvertices().add(upseudostate);
            utransition.setSource(upseudostate);
        }
        else if (src instanceof SState)
        {
            State ustate = createState((SState) src);
            utransition.setSource(ustate);
        }
        else
            log.error("Unsupported transition source type: " + src.getClass());
        
        SConnectNode dst = stransition.getDst();
        if (dst == null)
            log.error("Null transition target for transition " + stransition);
        else if (dst instanceof SState)
        {
            State ustate = createState((SState) dst);
            utransition.setTarget(ustate);
        }
        else
            log.error("Unsupported transition target type: " + dst.getClass());
        
        Class ublock = (Class) utransition.getContainer().getStateMachine().getContext();
        Hashtable<String, String> sub = subs.get(ublock);
        
        String label = stransition.getLabel();
        log.info("Parsing label " + label);
        if (label != null && !label.isEmpty())
        {
            MatlabExpressionExtractor mee = new MatlabExpressionExtractor();
            mee.prepareNextStateLabelParsing(sub);
            MATLABLexer ml = new MATLABLexer(new ANTLRInputStream(label));
            CommonTokenStream cts = new CommonTokenStream(ml);
            MATLABParser mp = new MATLABParser(cts);
            mp.addErrorListener(new ANTLRErrorListener(log));
            
            ParserRuleContext prc = mp.transition_label();
            new ParseTreeWalker().walk(mee, prc);
            
            if (mee.getTransitionCondition() != null)
            {
                Trigger t = UMLFactory.eINSTANCE.createTrigger();
                ChangeEvent ce = UMLFactory.eINSTANCE.createChangeEvent();
                ublock.getNearestPackage().getPackagedElements().add(ce);
                t.setEvent(ce);
                OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
                ce.setChangeExpression(oe);
                oe.getBodies().add(mee.getTransitionCondition());
                oe.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                utransition.getTriggers().add(t);
            }
            else if (mee.getTransitionTime() != null)
            {
                Trigger t = UMLFactory.eINSTANCE.createTrigger();
                TimeEvent te = UMLFactory.eINSTANCE.createTimeEvent();
                ublock.getNearestPackage().getPackagedElements().add(te);
                t.setEvent(te);
                TimeExpression tex = UMLFactory.eINSTANCE.createTimeExpression();
                OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
                te.setIsRelative(true);
                te.setWhen(tex);
                tex.setExpr(oe);
                oe.getBodies().add(mee.getTransitionTime());
                oe.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                utransition.getTriggers().add(t);
            }
            else
                log.error("Unable to parse the transition label: " + label);
        }
        
        return utransition;
    }
    
    private Class processBlock2(SSystem ssystem)
    {
        Class ublock = createBlock(ssystem);
        if (ublock == null)
        {
            log.error("Can't retrieve block");
            return null;
        }
        
        log.info("Processing block " + ssystem.getName() + " a second time");
        
        // create connectors
        for (SLine sline : ssystem.getOwnedLines())
        {
            if (sline instanceof SPhysicalLine)
            {
                List<SPhysicalConnectionPoint> spcps = ((SPhysicalLine) sline).getPoints();
                if (spcps.size() < 2)
                {
                    log.error("The physical line doesn't have enough points");
                    continue;
                }
                SPhysicalConnectionPoint spcp0 = spcps.get(0);
                Property p00 = (Property) refs.get(getKey(spcp0.getBlock()));
                Property p01 = (Property) refs.get(getKey(spcp0.getPort()));
                for (int i = 1; i < spcps.size(); i++)
                {
                    SPhysicalConnectionPoint spcp1 = spcps.get(i);
                    Property p10 = (Property) refs.get(getKey(spcp1.getBlock()));
                    Property p11 = (Property) refs.get(getKey(spcp1.getPort()));
                    createConnector(ublock, p00, p01, p10, p11);
                    log.info("Creating physical line between " + p00.getName() + "." + p01 + " and " + p10.getName()
                            + "." + p11);
                }
            }
            else
            {
                SConnectionPoint src = sline.getSource();
                Property pr0 = (Property) refs.get(getKey(src.getBlock()));
                Port po0 = src.getPort() != null ? (Port) refs.get(getKey(src.getPort())) : null;
                for (SConnectionPoint dst : sline.getDestinations())
                {
                    Property pr1 = (Property) refs.get(getKey(dst.getBlock()));
                    Port po1 = dst.getPort() != null ? (Port) refs.get(getKey(dst.getPort())) : null;
                    
                    if (pr0 == null || pr1 == null)
                        continue;
                    createConnector(ublock, pr0, po0, pr1, po1);
                    log.info(
                            "Creating line between " + pr0.getName() + "." + po0 + " and " + pr1.getName() + "." + po1);
                }
            }
        }
        
        // owned blocks
        for (SBlock sblock : ssystem.getOwnedBlocks())
        {
            if (sblock instanceof SFunction2Block)
            {
                processSFunction2(ssystem, sblock);
            }
        }
        // owned lines
        return ublock;
    }
    
    private Class processBlock(SFunction2 sfunction2)
    {
        return createBlock(sfunction2);
    }
    
    private Class processBlock(SDomain sdomain)
    {
        Class ublock = createBlock(sdomain);
        SysMLUtil.setName(ublock, sdomain.getName());
        
        log.info("Processing domain " + sdomain.getName());
        
        // add sim property/flow property
        
        Property usp = UMLFactory.eINSTANCE.createProperty();
        ublock.getOwnedAttributes().add(usp);
        SysMLUtil.setName(usp, "phs" + sdomain.getName());
        usp.applyStereotype(sysmlutil.getFlowProperty());
        usp.setValue(sysmlutil.getFlowProperty(), "direction", sysmlutil.getFlowPropertyInout());
        usp.applyStereotype(sysmlutil.getPhSVariable());
        
        Class usb = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(usb);
        Generalization ugeneralization = UMLFactory.eINSTANCE.createGeneralization();
        ugeneralization.setGeneral(sysmlutil.getConservedSubstance());
        ugeneralization.setSpecific(usb);
        usb.getGeneralizations().add(ugeneralization);
        SysMLUtil.setName(usb, "PhS" + sdomain.getName());
        usp.setType(usb);
        
        // members
        for (SMember smember : sdomain.getVariables())
        {
            Property uproperty = createProperty(smember);
            usb.getOwnedAttributes().add(uproperty);
        }
        return ublock;
    }
    
    private Class processBlock(SComponent scomponent)
    {
        Class ublock = createBlock(scomponent);
        // clean branches/components
        getComponentSubstitutions(scomponent);
        
        log.info("Processing component " + scomponent.getName());
        SysMLUtil.setName(ublock, scomponent.getName());
        
        // make sure constraint properties are created
        Property uconstraintproperty = getOrCreateConstraintProperty(ublock);
        Class uconstraint = (Class) uconstraintproperty.getType();
        SysMLUtil.setName(uconstraint, ublock.getName() + "Constraint");
        SysMLUtil.setName(uconstraintproperty, "_" + uconstraint.getName());
        
        // base
        if (scomponent.getBaseComponent() != null)
        {
            Class ubase = createBlock(scomponent.getBaseComponent());
            
            Generalization ugen = UMLFactory.eINSTANCE.createGeneralization();
            ublock.getGeneralizations().add(ugen);
            ugen.setGeneral(ubase);
            ugen.setSpecific(ublock);
            /*
             * Generalization ugen2 =
             * UMLFactory.eINSTANCE.createGeneralization();
             * uconstraint.getGeneralizations().add(ugen2);
             * ugen2.setSpecific(uconstraint); ugen2.setGeneral((Classifier)
             * getOrCreateConstraintProperty(ubase).getType());
             */
        }
        // members
        for (SMember smember : scomponent.getOwnedMembers())
        {
            Property uproperty = createProperty(smember);
            if (uproperty instanceof Port)
                ublock.getOwnedPorts().add((Port) uproperty);
            else
                ublock.getOwnedAttributes().add(uproperty);
        }
        
        return ublock;
    }
    
    private Class processBlock2(SComponent scomponent)
    {
        Class ublock = createBlock(scomponent);
        
        log.info("Processing component " + scomponent.getName() + " a second time");
        
        // substitutions
        Hashtable<String, String> substitutions = new Hashtable<String, String>();
        
        // create associated constraint
        Property uconstraintproperty = getOrCreateConstraintProperty(ublock);
        Class uconstraint = (Class) uconstraintproperty.getType();
        
        // setup
        // connections
        for (SConnection sconn : scomponent.getOwnedConnections())
        {
            Iterator<SMemberPath> iter = sconn.getPoints().iterator();
            if (iter.hasNext())
            {
                List<Property> src = convertList(iter.next().getPath());
                while (iter.hasNext())
                {
                    List<Property> dst = convertList(iter.next().getPath());
                    createConnector(ublock, src, dst);
                    log.info("Adding connection");
                }
            }
        }
        // assignments
        loop: for (SMemberAssignment sma : scomponent.getOwnedAssignments())
        {
            log.info("Translating assignment");
            if (sma == null)
                continue;
            InstanceSpecification is = null;
            for (SMember sm : sma.getMemberPath())
            {
                Property prop = createProperty(sm);
                // is null if first member
                if (is == null)
                {
                    // create or
                    if (prop.getDefaultValue() instanceof InstanceValue)
                    {
                        log.info("Retrieving existing instance specification for " + sm.getName());
                        InstanceValue iv = (InstanceValue) prop.getDefaultValue();
                        is = iv.getInstance();
                        if (is == null)
                            continue loop;
                    }
                    else
                    {
                        is = UMLFactory.eINSTANCE.createInstanceSpecification();
                        uroot.getPackagedElements().add(is);
                        InstanceValue iv = UMLFactory.eINSTANCE.createInstanceValue();
                        iv.setInstance(is);
                        prop.setDefaultValue(iv);
                        log.info("Creating instance specification for " + sm.getName());
                    }
                }
                // not first member
                else
                {
                    if (sm instanceof SNode || sm instanceof SComponentReference)
                    {
                        InstanceValue iv = UMLFactory.eINSTANCE.createInstanceValue();
                        Slot s = UMLFactory.eINSTANCE.createSlot();
                        is.getSlots().add(s);
                        s.setDefiningFeature(prop);
                        s.getValues().add(iv);
                        is = UMLFactory.eINSTANCE.createInstanceSpecification();
                        uroot.getPackagedElements().add(is);
                        iv.setInstance(is);
                        log.info("Creating slot for " + sm.getName());
                    }
                    else
                    {
                        Slot s = UMLFactory.eINSTANCE.createSlot();
                        is.getSlots().add(s);
                        s.setDefiningFeature(prop);
                        if (sma.getAssignedValue() != null)
                        {
                            ValueSpecification vs = getValueSpecification(sma.getAssignedValue());
                            if (vs != null)
                            {
                                s.getValues().add(vs);
                                log.info("Creating slot for " + sm.getName() + " with value "
                                        + sma.getAssignedValue().getRawValue());
                            }
                            else
                                log.error("Unable to get value for " + sma.getAssignedValue().getRawValue());
                        }
                        else if (sma.getAssignedReference() != null)
                            log.error("Unsupported reference assignment to " + sma.getAssignedReference());
                        else
                            log.error("Member assignment without value nor reference");
                    }
                }
            }
        }
        
        // branches with both from and to will have their equations put
        
        // branches (including inherited) with one from/to have
        // their variable replaced by the from/to path
        substitutions.putAll(getComponentSubstitutions(scomponent));
        for (SBranch sbranch : scomponent.getBranches())
        {
            SVariable svar = sbranch.getVariable();
            SMemberPath sfrom = sbranch.getFrom();
            SMemberPath sto = sbranch.getTo();
            
            if (svar == null || (sfrom == null && sto == null))
                continue;
            // sto and/or sfrom are not null
            
            if (sto != null && sfrom != null)
            {
                String from = join(sfrom.getPath(), ".");
                String to = join(sto.getPath(), ".");
                
                OpaqueExpression oe1 = UMLFactory.eINSTANCE.createOpaqueExpression();
                oe1.getBodies().add(svar.getName() + "=" + from);
                oe1.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                Constraint uc1 = UMLFactory.eINSTANCE.createConstraint();
                uc1.setSpecification(oe1);
                uconstraint.getOwnedRules().add(uc1);
                
                OpaqueExpression oe2 = UMLFactory.eINSTANCE.createOpaqueExpression();
                oe2.getBodies().add(from + "+" + to + "=0");
                oe2.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                Constraint uc2 = UMLFactory.eINSTANCE.createConstraint();
                uc2.setSpecification(oe2);
                uconstraint.getOwnedRules().add(uc2);
            }
        }
        
        // equations
        List<SEquation> equations = scomponent.getOwnedEquations();
        if (equations.size() > 0)
        {
            // members
            // TODO: create from equations
            for (SMember smember : scomponent.getAllMembers())
            {
                Property uproperty = createProperty(smember);
                Property uconstraintparameter = UMLFactory.eINSTANCE.createProperty();
                uconstraint.getOwnedAttributes().add(uconstraintparameter);
                SysMLUtil.setName(uconstraintparameter, uproperty.getName());
                uconstraintparameter.setType(sysmlutil.getSysMLDouble());
                if (smember instanceof SNode)
                {
                    for (SMember smember2 : ((SNode) smember).getDomain().getVariables())
                    {
                        Property uproperty2 = createProperty(smember2);
                        
                        Property uparam = UMLFactory.eINSTANCE.createProperty();
                        uconstraint.getOwnedAttributes().add(uparam);
                        SysMLUtil.setName(uparam, uproperty.getName() + "_" + uproperty2.getName());
                        uparam.setType(sysmlutil.getSysMLDouble());
                        
                        Connector uconnector = createConnector(ublock, uproperty, uproperty2, uconstraintproperty,
                                uparam);
                        uconnector.applyStereotype(sysmlutil.getBindingConnector());
                    }
                }
                else if (smember instanceof SParameter || smember instanceof SVariable || smember instanceof SInput
                        || smember instanceof SOutput)
                {
                    Connector uconnector = createConnector(ublock, uproperty, null, uconstraintproperty,
                            uconstraintparameter);
                    uconnector.applyStereotype(sysmlutil.getBindingConnector());
                    // same name, no substitutions
                }
            }
            for (SEquation sequation : equations)
            {
                ParseTree pt = expressions.get(sequation);
                if (pt != null)
                {
                    see.prepareNextEquationParsing(substitutions);
                    log.info("Parsing " + pt.getText());
                    ParseTreeWalker ptw = new ParseTreeWalker();
                    ptw.walk(see, pt);
                    log.info("Translating " + pt.getText() + " into " + see.getValue(pt));
                    
                    OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
                    oe.getBodies().add(see.getValue(pt));
                    oe.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
                    Constraint uc = UMLFactory.eINSTANCE.createConstraint();
                    uc.setSpecification(oe);
                    uconstraint.getOwnedRules().add(uc);
                }
                else
                    log.error("No tree for equation " + sequation.getExpression());
            }
        }
        
        return ublock;
    }
    
    @SuppressWarnings("null")
    private Hashtable<String, String> getComponentSubstitutions(SComponent scomponent)
    {
        Class uclass = (Class) refs.get(getKey(scomponent));
        if (uclass == null)
            return null;
        Hashtable<String, String> ret = subs.get(uclass);
        if (ret != null)
            return ret;
        ret = new Hashtable<String, String>();
        subs.put(uclass, ret);
        log.info("Analyzing branches for " + scomponent.getName());
        for (int i = 0; i < scomponent.getBranches().size(); i++)
        {
            SBranch sbranch = scomponent.getBranches().get(i);
            SVariable svar = sbranch.getVariable();
            SMemberPath sfrom = sbranch.getFrom();
            SMemberPath sto = sbranch.getTo();
            if (svar == null || (sfrom == null && sto == null))
                continue;
            // sto and/or sfrom are not null
            
            if (sfrom == null)
            {
                log.info("Removing " + svar.getName() + " from " + scomponent.getName());
                ret.put(svar.getName(), "-" + join(sto.getPath(), "_"));
                scomponent.getBranches().remove(i--);
                scomponent.getOwnedMembers().remove(svar);
            }
            if (sto == null)
            {
                log.info("Removing " + svar.getName() + " from " + scomponent.getName());
                ret.put(svar.getName(), join(sfrom.getPath(), "_"));
                scomponent.getBranches().remove(i--);
                scomponent.getOwnedMembers().remove(svar);
            }
        }
        for (SMember smember : scomponent.getOwnedMembers())
        {
            if (smember instanceof SNode)
            {
                for (SMember smember2 : ((SNode) smember).getDomain().getVariables())
                {
                    String to = smember.getName() + "_" + smember2.getName();
                    
                    String from = smember.getName() + "." + smember2.getName();
                    
                    log.info("Replacing " + from + " by " + to);
                    ret.put(from, to);
                }
            }
        }
        
        SComponent base = scomponent.getBaseComponent();
        if (base != null)
        {
            
            Hashtable<String, String> basesub = getComponentSubstitutions(base);
            if (basesub == null)
                log.error("Can't get substitutions for base component " + base.getName());
            else
                ret.putAll(basesub);
        }
        return ret;
    }
    
    private Hashtable<String, String> getClassSubstitutions(Class uclass)
    {
        if (uclass == null)
            return null;
        
        Hashtable<String, String> ret = subs.get(uclass);
        if (ret != null)
            return ret;
        
        ret = new Hashtable<String, String>();
        subs.put(uclass, ret);
        
        return ret;
    }
    
    private static String join(List<SMember> sms, String link)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < sms.size(); i++)
        {
            if (i != 0)
                sb.append(link);
            sb.append(sms.get(i).getName());
        }
        return sb.toString();
    }
    
    @SuppressWarnings("boxing")
    private void processSFunction2(SSystem ssystem, SBlock sblock)
    {
        SFunction sfunction = null;
        if (sblock instanceof SFunction2Block)
            sfunction = ((SFunction2Block) sblock).getSfunction();
        if (sfunction == null)
        {
            log.error("The object must be a SFunctionBlock with a SFunction: " + sblock);
            return;
        }
        Class ublock = createBlock(ssystem);
        
        // create corresponding bindings & variables
        for (SFVariable sv : sfunction.getVariables())
        {
            Property uparameter = (Property) refs.get(getKey(sv));
            Property uconstraintproperty = (Property) refs.get(getKey(sblock));
            
            if (sv instanceof SFInputVariable)
            {
                SInport sinport = ((SFInputVariable) sv).getInport();
                if (sinport == null)
                {
                    log.error("The output variable " + sv + " has no associated inport");
                    continue;
                }
                Property uport = createProperty(sinport);
                
                if (uport != null)
                {
                    Connector uconnector = createConnector(ublock, uconstraintproperty, uparameter, uport,
                            getSignalPortProperty());
                    uconnector.applyStereotype(sysmlutil.getBindingConnector());
                }
                else
                {
                    log.error("Couldn't bind input variable");
                }
                
            }
            else if (sv instanceof SFOutputVariable)
            {
                SOutport soutport = ((SFOutputVariable) sv).getOutport();
                if (soutport == null)
                {
                    log.error("The output variable " + sv + " has no associated outport");
                    continue;
                }
                Property uport = createProperty(soutport);
                if (uport != null)
                {
                    Connector uconnector = createConnector(ublock, uconstraintproperty, uparameter, uport,
                            getSignalPortProperty());
                    uconnector.applyStereotype(sysmlutil.getBindingConnector());
                }
                else
                {
                    log.error("Couldn't bind input variable");
                }
            }
            else if (sv instanceof SFParameter)
            {
                Property uproperty = UMLFactory.eINSTANCE.createProperty();
                ublock.getOwnedAttributes().add(uproperty);
                SysMLUtil.setName(uproperty, sblock.getName() + sv.getName());
                uproperty.applyStereotype(sysmlutil.getPhSConstant());
                uproperty.setType(sysmlutil.getSysMLDouble());
                
                Connector uconnector = createConnector(ublock, uconstraintproperty, uparameter, uproperty, null);
                uconnector.applyStereotype(sysmlutil.getBindingConnector());
            }
            else if (sv instanceof SFDiscreteStateVariable)
            {
                Property uproperty = UMLFactory.eINSTANCE.createProperty();
                ublock.getOwnedAttributes().add(uproperty);
                SysMLUtil.setName(uproperty, sblock.getName() + sv.getName());
                uproperty.applyStereotype(sysmlutil.getPhSVariable());
                uproperty.setValue(sysmlutil.getPhSVariable(), "isContinuous", false);
                uproperty.setType(sysmlutil.getSysMLDouble());
                
                Connector uconnector = createConnector(ublock, uconstraintproperty, uparameter, uproperty, null);
                uconnector.applyStereotype(sysmlutil.getBindingConnector());
            }
            else if (sv instanceof SFContinuousStateVariable)
            {
                Property uproperty = UMLFactory.eINSTANCE.createProperty();
                ublock.getOwnedAttributes().add(uproperty);
                SysMLUtil.setName(uproperty, sblock.getName() + sv.getName());
                uproperty.applyStereotype(sysmlutil.getPhSVariable());
                uproperty.setValue(sysmlutil.getPhSVariable(), "isContinuous", true);
                uproperty.setType(sysmlutil.getSysMLDouble());
                
                Connector uconnector = createConnector(ublock, uconstraintproperty, uparameter, uproperty, null);
                uconnector.applyStereotype(sysmlutil.getBindingConnector());
            }
        }
        
    }
    
    private Property processProperty(SBlock sblock)
    {
        Property uproperty = createProperty(sblock);
        if (uproperty == null)
        {
            log.error("Can't retrieve property");
            return null;
        }
        
        log.info("Processing property " + sblock.getName());
        SysMLUtil.setName(uproperty, sblock.getName());
        
        if (sblock instanceof SConnectionPortBlock)
        {
            SMember smember = ((SConnectionPortBlock) sblock).getClosestMember();
            if (smember instanceof SInput)
            {
                SInput sinput = (SInput) smember;
                uproperty.setType(getTypeForUnit(sinput.getUnit()));
            }
            else if (smember instanceof SOutput)
            {
                SOutput soutput = (SOutput) smember;
                ((Port) uproperty).setIsConjugated(true);
                uproperty.setType(getTypeForUnit(soutput.getUnit()));
            }
            else if (smember instanceof SNode)
            {
                uproperty.setType(createBlock(((SNode) smember).getDomain()));
            }
            else
                log.error("Can't set type for block " + sblock.getName());
        }
        else if (sblock instanceof SChartBlock)
            uproperty.setType(createBlock(((SChartBlock) sblock).getSystem()));
        else if (sblock instanceof SSubsystem)
            uproperty.setType(createBlock(((SSubsystem) sblock).getSystem()));
        else if (sblock instanceof SReference)
            uproperty.setType(createBlock(((SReference) sblock).getSystem()));
        else if (sblock instanceof SPhysicalBlock)
            uproperty.setType(createBlock(((SPhysicalBlock) sblock).getComponent()));
        else if (sblock instanceof SFunction2Block)
            uproperty.setType(createBlock(((SFunction2Block) sblock).getSfunction()));
        else if (sblock instanceof SInport)
            uproperty.setType(getSignalPortType());
        else if (sblock instanceof SOutport)
        {
            uproperty.setType(getSignalPortType());
            ((Port) uproperty).setIsConjugated(true);
        }
        else if (sblock instanceof SInterface)
        {
            // look for library blocks
            String name = ((SInterface) sblock).getSystem().getName();
            loop: for (LibraryComponent lc : sysmlutil.getSimulinkComponents())
            {
                if (!lc.getName().equals(name))
                    continue;
                List<SSystemParameterAssignment> todelete = new ArrayList<SSystemParameterAssignment>(
                        ((SInterface) sblock).getAssignments());
                log.debug("Potential type " + name);
                // all parameters with values match
                loop2: for (LibraryParameter lp : lc.getParameters())
                {
                    for (SSystemParameterAssignment sspa : ((SInterface) sblock).getAssignments())
                        if (sspa.getParameter().getName().equals(lp.getName()))
                        {
                            if (lp.getValue() == null)
                            {
                                todelete.remove(sspa);
                                continue loop2;
                            }
                            
                            SDataValue val = sspa.getValue();
                            if (val.getRawValue().equals(lp.getValue()))
                            {
                                continue loop2;
                            }
                        }
                    log.debug("Skipping potential type due to parameter " + lp.getName() + "/" + lp.getValue());
                    continue loop;
                }
                
                log.info("Assigning type " + lc.getClas().getQualifiedName() + " for block " + sblock.getName());
                
                uproperty.setType(lc.getClas());
                refs.put(getKey(((SInterface) sblock).getSystem()), lc.getClas());
                ((SInterface) sblock).getAssignments().removeAll(todelete);
                
                InstanceSpecification is = UMLFactory.eINSTANCE.createInstanceSpecification();
                
                loop2: for (SSystemParameterAssignment sspa : ((SInterface) sblock).getAssignments())
                {
                    for (LibraryParameter lp : lc.getParameters())
                        if (lp.getName().equals(sspa.getParameter().getName()))
                        {
                            log.info("Linking parameter " + sspa.getParameter().getName() + ":" + sspa.getValue());
                            refs.put(getKey(sspa.getParameter()), lp.getProperty());
                            Slot slot = UMLFactory.eINSTANCE.createSlot();
                            slot.setDefiningFeature(lp.getProperty());
                            slot.getValues().add(getValueSpecification(sspa.getValue()));
                            is.getSlots().add(slot);
                            continue loop2;
                        }
                }
                
                if (is.getSlots().size() > 0)
                {
                    uproperty.getNearestPackage().getPackagedElements().add(is);
                    InstanceValue iv = UMLFactory.eINSTANCE.createInstanceValue();
                    iv.setInstance(is);
                    uproperty.setDefaultValue(iv);
                }
                
                List<SInport> linports = ((SInterface) sblock).getSystem().getInports();
                List<LibraryPort> linputports = lc.getInputPorts();
                if (linports.size() != linputports.size())
                    log.error("Mismatch in the number of inports " + linports.size() + " vs " + linputports.size());
                for (int i = 0; i < linports.size(); i++)
                {
                    SInport sinport = linports.get(i);
                    if (i < linputports.size())
                    {
                        refs.put(getKey(sinport), linputports.get(i).getPort());
                        log.info("Linking inport " + sinport.getName());
                    }
                }
                List<SOutport> loutports = ((SInterface) sblock).getSystem().getOutports();
                List<LibraryPort> loutputports = lc.getOutputPorts();
                if (loutports.size() != loutputports.size())
                    log.error("Mismatch in the number of outports " + loutports.size() + " vs " + loutputports.size());
                for (int i = 0; i < loutports.size(); i++)
                {
                    SOutport soutport = loutports.get(i);
                    if (i < loutputports.size())
                    {
                        refs.put(getKey(soutport), loutputports.get(i).getPort());
                        log.info("Linking outport " + soutport.getName());
                    }
                }
                
                break;
            }
            if (uproperty.getType() == null)
                log.error("Unable to find a type for block " + sblock.getName());
        }
        else
            log.error("Unsupported block type: " + sblock.getClass());
        
        return uproperty;
    }
    
    private Property processProperty(SMember smember)
    {
        Property uproperty = createProperty(smember);
        if (uproperty == null)
        {
            log.error("Can't retrieve property");
            return null;
        }
        
        log.info("Processing property " + smember.getName());
        SysMLUtil.setName(uproperty, smember.getName());
        
        if (smember instanceof SInput)
        {
            SInput sinput = (SInput) smember;
            uproperty.setType(getTypeForUnit(sinput.getUnit()));
        }
        else if (smember instanceof SOutput)
        {
            SOutput soutput = (SOutput) smember;
            ((Port) uproperty).setIsConjugated(true);
            uproperty.setType(getTypeForUnit(soutput.getUnit()));
        }
        else if (smember instanceof SParameter)
        {
            SParameter sparameter = ((SParameter) smember);
            uproperty.setType(sysmlutil.getSysMLDouble());
            uproperty.applyStereotype(sysmlutil.getPhSConstant());
            String unit = sparameter.getUnit();
            if (unit == null || unit.isEmpty())
                uproperty.setType(sysmlutil.getSysMLDouble());
            else
                uproperty.setType(getValueType(unit));
            uproperty.setDefaultValue(getValueSpecification(sparameter.getValue()));
        }
        else if (smember instanceof SVariable)
        {
            SVariable svariable = ((SVariable) smember);
            uproperty.applyStereotype(sysmlutil.getPhSVariable());
            if (svariable.isBalancing())
                uproperty.setValue(sysmlutil.getPhSVariable(), "isConserved", true);
            String unit = svariable.getUnit();
            if (unit == null || unit.isEmpty())
                uproperty.setType(sysmlutil.getSysMLDouble());
            else
                uproperty.setType(getValueType(unit));
            uproperty.setDefaultValue(getValueSpecification(svariable.getValue()));
        }
        else if (smember instanceof SNode)
            uproperty.setType(createBlock(((SNode) smember).getDomain()));
        else if (smember instanceof SComponentReference)
        {
            uproperty.setType(createBlock(((SComponentReference) smember).getComponent()));
        }
        else
            log.error("Unsupported member type: " + smember.getClass());
        
        return uproperty;
    }
    
    private Class usignalporttype;
    private Property usignalportproperty;
    
    private Class getSignalPortType()
    {
        if (usignalporttype != null)
            return usignalporttype;
        
        Stereotype fp = sysmlutil.getFlowProperty();
        Enumeration en = null;
        for (Property p : fp.getOwnedAttributes())
            if (p.getName().equals("direction"))
                en = (Enumeration) p.getType();
            
        if (en == null)
        {
            log.error("Can't get flow property direction");
            return null;
        }
        
        EnumerationLiteral el = en.getOwnedLiteral("in");
        
        usignalporttype = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(usignalporttype);
        usignalporttype.setName("SignalPort");
        usignalporttype.applyStereotype(sysmlutil.getBlock());
        
        usignalportproperty = UMLFactory.eINSTANCE.createProperty();
        usignalporttype.getOwnedAttributes().add(usignalportproperty);
        usignalportproperty.setName("sfp");
        usignalportproperty.setType(sysmlutil.getSysMLDouble());
        usignalportproperty.applyStereotype(sysmlutil.getFlowProperty());
        usignalportproperty.setValue(sysmlutil.getFlowProperty(), "direction", el);
        usignalportproperty.applyStereotype(sysmlutil.getPhSVariable());
        
        return usignalporttype;
    }
    
    private Property processProperty2(SPhysicalBlock sphysicalblock)
    {
        log.info("Processing physical block " + sphysicalblock.getName() + " a second time");
        Property prop = createProperty(sphysicalblock);
        if (sphysicalblock.getAssignments().size() > 0)
        {
            log.info("Creating instance specification for " + prop.getName());
            InstanceSpecification is = UMLFactory.eINSTANCE.createInstanceSpecification();
            uroot.getPackagedElements().add(is);
            InstanceValue iv = UMLFactory.eINSTANCE.createInstanceValue();
            iv.setInstance(is);
            prop.setDefaultValue(iv);
            
            for (SMemberAssignment sma : sphysicalblock.getAssignments())
            {
                if (sma == null)
                    continue;
                if (sma.getMemberPath().size() != 1)
                {
                    log.error("Unexpected number of members");
                    continue;
                }
                SMember sm = sma.getMemberPath().get(0);
                prop = createProperty(sm);
                Slot s = UMLFactory.eINSTANCE.createSlot();
                is.getSlots().add(s);
                s.setDefiningFeature(prop);
                s.getValues().add(getValueSpecification(sma.getAssignedValue()));
                log.info("Added initial value for " + prop.getName());
            }
            
        }
        
        return prop;
    }
    
    private Property processProperty2(SInterface sinterface)
    {
        log.info("Processing interface block " + sinterface.getName() + " a second time");
        Property prop = createProperty(sinterface);
        if (sinterface.getAssignments().size() > 0)
        {
            InstanceSpecification is = UMLFactory.eINSTANCE.createInstanceSpecification();
            uroot.getPackagedElements().add(is);
            InstanceValue iv = UMLFactory.eINSTANCE.createInstanceValue();
            iv.setInstance(is);
            prop.setDefaultValue(iv);
            for (SSystemParameterAssignment sspa : sinterface.getAssignments())
            {
                Property uparam = (Property) refs.get(getKey(sspa.getParameter()));
                Slot uslot = UMLFactory.eINSTANCE.createSlot();
                uslot.setDefiningFeature(uparam);
                uslot.getValues().add(getValueSpecification(sspa.getValue()));
                is.getSlots().add(uslot);
            }
        }
        return prop;
    }
    
    private Property getSignalPortProperty()
    {
        if (usignalportproperty == null)
            getSignalPortType();
        return usignalportproperty;
    }
    
    private Property getOrCreateConstraintProperty(Class ublock)
    {
        for (Property uproperty : ublock.getOwnedAttributes())
        {
            if (uproperty.getType() instanceof Class
                    && uproperty.getType().isStereotypeApplied(sysmlutil.getConstraintBlock()))
                return uproperty;
        }
        
        Class uconstraintblock = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(uconstraintblock);
        uconstraintblock.applyStereotype(sysmlutil.getConstraintBlock());
        
        Property uconstraintproperty = UMLFactory.eINSTANCE.createProperty();
        ublock.getOwnedAttributes().add(uconstraintproperty);
        uconstraintproperty.setType(uconstraintblock);
        
        return uconstraintproperty;
    }
    
    private static ValueSpecification getValueSpecification(SDataValue value)
    {
        if (value == null)
            return null;
        if (value instanceof SDoubleValue)
        {
            LiteralReal real = UMLFactory.eINSTANCE.createLiteralReal();
            real.setValue(((SDoubleValue) value).getValue());
            return real;
        }
        else if (value instanceof SExpressionValue)
        {
            OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
            oe.getBodies().add(value.getRawValue());
            oe.getLanguages().add(SysMLUtil.SYSPHS_LANGUAGE);
            return oe;
        }
        return null;
    }
    
    private Class getTypeForUnit(String unit)
    {
        if (unit == null || unit.isEmpty())
            return getSignalPortType();
        return getPortType(getValueType(unit));
    }
    
    private Class getPortType(DataType valueType)
    {
        Class ret = porttypes.get(valueType);
        if (ret != null)
            return ret;
        Class porttype = UMLFactory.eINSTANCE.createClass();
        uroot.getPackagedElements().add(porttype);
        porttype.applyStereotype(sysmlutil.getBlock());
        SysMLUtil.setName(porttype, "PhS" + valueType.getName());
        Property sp = UMLFactory.eINSTANCE.createProperty();
        porttype.getOwnedAttributes().add(sp);
        SysMLUtil.setName(sp, "phs" + valueType.getName());
        sp.applyStereotype(sysmlutil.getFlowProperty());
        sp.setValue(sysmlutil.getFlowProperty(), "direction", sysmlutil.getFlowPropertyIn());
        sp.applyStereotype(sysmlutil.getPhSVariable());
        sp.setType(valueType);
        
        return porttype;
    }
    
    private DataType getValueType(String unit)
    {
        DataType dt = units.get(unit);
        if (dt != null)
            return dt;
        
        dt = UMLFactory.eINSTANCE.createDataType();
        units.put(unit, dt);
        uroot.getPackagedElements().add(dt);
        
        Generalization ugen = UMLFactory.eINSTANCE.createGeneralization();
        dt.getGeneralizations().add(ugen);
        ugen.setGeneral(sysmlutil.getSysMLDouble());
        ugen.setSpecific(dt);
        
        dt.applyStereotype(sysmlutil.getValueType());
        
        StringBuilder sbunit = new StringBuilder();
        for (char c : unit.toCharArray())
            if (Character.isLetterOrDigit(c))
                sbunit.append(c);
        SysMLUtil.setName(dt, "VT" + sbunit.toString());
        
        InstanceSpecification isunit = UMLFactory.eINSTANCE.createInstanceSpecification();
        uroot.getPackagedElements().add(isunit);
        SysMLUtil.setName(isunit, "SU" + sbunit.toString());
        isunit.getClassifiers().add(sysmlutil.getUnit());
        Slot slot = UMLFactory.eINSTANCE.createSlot();
        isunit.getSlots().add(slot);
        slot.setDefiningFeature(sysmlutil.getUnit().getAttribute("symbol", null));
        LiteralString ls = UMLFactory.eINSTANCE.createLiteralString();
        slot.getValues().add(ls);
        ls.setValue(unit);
        
        dt.setValue(sysmlutil.getValueType(), "unit", isunit);
        
        return dt;
    }
    
    private List<Property> convertList(List<SMember> path)
    {
        List<Property> ret = new ArrayList<Property>(path.size());
        for (SMember smember : path)
            ret.add(createProperty(smember));
        return ret;
    }
    
    private Connector createConnector(Class ublock, List<Property> lp0, List<Property> lp1)
    {
        Connector uconnector = UMLFactory.eINSTANCE.createConnector();
        ublock.getOwnedConnectors().add(uconnector);
        ConnectorEnd uce0 = UMLFactory.eINSTANCE.createConnectorEnd();
        uconnector.getEnds().add(uce0);
        populateConnectorEnd(uce0, lp0);
        
        ConnectorEnd uce1 = UMLFactory.eINSTANCE.createConnectorEnd();
        uconnector.getEnds().add(uce1);
        populateConnectorEnd(uce1, lp1);
        
        return uconnector;
    }
    
    private Connector createConnector(Class ublock, Property uproperty00, Property uproperty01, Property uproperty10,
            Property uproperty11)
    {
        String sp00 = uproperty00 == null ? null : uproperty00.getName();
        String sp01 = uproperty01 == null ? null : uproperty01.getName();
        String sp10 = uproperty10 == null ? null : uproperty10.getName();
        String sp11 = uproperty11 == null ? null : uproperty11.getName();
        log.info("Creating connector from " + sp00 + "." + sp01 + " to " + sp10 + "." + sp11);
        Connector uconnector = UMLFactory.eINSTANCE.createConnector();
        ublock.getOwnedConnectors().add(uconnector);
        
        ConnectorEnd ce0 = UMLFactory.eINSTANCE.createConnectorEnd();
        uconnector.getEnds().add(ce0);
        populateConnectorEnd(ce0, uproperty00, uproperty01);
        
        ConnectorEnd ce1 = UMLFactory.eINSTANCE.createConnectorEnd();
        uconnector.getEnds().add(ce1);
        populateConnectorEnd(ce1, uproperty10, uproperty11);
        
        return uconnector;
    }
    
    private void populateConnectorEnd(ConnectorEnd uconnectorend, Property uproperty0, Property property1)
    {
        if (property1 != null)
        {
            uconnectorend.setPartWithPort(uproperty0);
            uconnectorend.setRole(property1);
            List<Property> np0 = new LinkedList<Property>();
            np0.add(uproperty0);
            uconnectorend.applyStereotype(sysmlutil.getNestedConnectorEnd());
            uconnectorend.setValue(sysmlutil.getNestedConnectorEnd(), "propertyPath", np0);
        }
        else
            uconnectorend.setRole(uproperty0);
    }
    
    private void populateConnectorEnd(ConnectorEnd uconnectorend, List<Property> lp)
    {
        List<Property> pp = new LinkedList<Property>(lp);
        int pps = pp.size();
        if (pps > 0)
        {
            uconnectorend.setRole(pp.get(pps - 1));
            pp.remove(pps - 1);
        }
        if (pps > 1)
            uconnectorend.setPartWithPort(pp.get(pps - 2));
        
        if (pps > 1)
        {
            uconnectorend.applyStereotype(sysmlutil.getNestedConnectorEnd());
            uconnectorend.setValue(sysmlutil.getNestedConnectorEnd(), "propertyPath", pp);
        }
    }
    
    @Override
    public Set<java.lang.Class<?>> getOptions()
    {
        return null;
    }
    
}
