package com.engisis.sysphs.translation.modelica;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.uml2.uml.Activity;
import org.eclipse.uml2.uml.ActivityNode;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.BehavioredClassifier;
import org.eclipse.uml2.uml.CallEvent;
import org.eclipse.uml2.uml.CallOperationAction;
import org.eclipse.uml2.uml.ChangeEvent;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.ConnectableElement;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Event;
import org.eclipse.uml2.uml.InstanceSpecification;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.LiteralBoolean;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralReal;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.PseudostateKind;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.StructuralFeature;
import org.eclipse.uml2.uml.TimeEvent;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.Vertex;
import org.eclipse.uml2.uml.VisibilityKind;

import com.engisis.sysphs.language.modelica.MAccessControl;
import com.engisis.sysphs.language.modelica.MAlgorithm;
import com.engisis.sysphs.language.modelica.MBlock;
import com.engisis.sysphs.language.modelica.MBooleanValue;
import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MComponent;
import com.engisis.sysphs.language.modelica.MConnect;
import com.engisis.sysphs.language.modelica.MConnector;
import com.engisis.sysphs.language.modelica.MDataFlow;
import com.engisis.sysphs.language.modelica.MDataValue;
import com.engisis.sysphs.language.modelica.MDirection;
import com.engisis.sysphs.language.modelica.MElement;
import com.engisis.sysphs.language.modelica.MEquation;
import com.engisis.sysphs.language.modelica.MExpressionValue;
import com.engisis.sysphs.language.modelica.MExtension;
import com.engisis.sysphs.language.modelica.MHierarchy;
import com.engisis.sysphs.language.modelica.MIntegerValue;
import com.engisis.sysphs.language.modelica.MModel;
import com.engisis.sysphs.language.modelica.MModification;
import com.engisis.sysphs.language.modelica.MRealValue;
import com.engisis.sysphs.language.modelica.MStringValue;
import com.engisis.sysphs.language.modelica.MType;
import com.engisis.sysphs.language.modelica.MVariability;
import com.engisis.sysphs.language.modelica.ModelicaFactory;
import com.engisis.sysphs.serialization.modelica.ModelicaSerializer;
import com.engisis.sysphs.util.ExpressionLanguageToSimulation;
import com.engisis.sysphs.util.SysMLToSimulationTranslator;
import com.engisis.sysphs.util.SysMLUtil;
import com.engisis.sysphs.util.UMLModelErrorException;

/**
 * Translator from SysML to Modelica
 * 
 * @author barbau
 *
 */
public class SysMLToModelicaTranslator extends SysMLToSimulationTranslator
{
    // NOTES
    // - Naming convention: modelica elements start with 'm', UML elements start
    // with 'u'
    // - Translated blocks: I am NOT translating every class/block. What I do is
    // focusing on a "root" block,
    // and calculating and translating all its dependencies. This is to avoid
    // unused elements.
    // - Reference management: Blocks will be created and processed in 2
    // different passes,
    // to avoid cross-references issues
    // - Logging: Fatal errors are assumed to be logged by the calling methods
    // through exceptions, not here.
    // Warnings are logged.
    // - surjective mode:
    // - moves the content of SimBlocks into the owning block, which becomes a
    // connector
    // - moves the equations of constraint blocks to the owning block
    // - assign the parameters according to the BindingConnectors
    // - bijective mode:
    // - transforms SimBlocks into connectors
    // - transforms constraint blocks into blocks
    // - transforms binding connectors into equations
    
    private static final Logger log = Logger.getLogger(SysMLToModelicaTranslator.class);
    
    /**
     * Reference table
     */
    private Hashtable<ReferenceKey, MElement> refs = new Hashtable<ReferenceKey, MElement>();
    
    /**
     * first-pass processing stack
     */
    private Stack<ReferenceKey> toProcess = new Stack<ReferenceKey>();
    /**
     * second-pass processing stack
     */
    private Stack<ReferenceKey> toProcess2 = new Stack<ReferenceKey>();
    
    /**
     * root Modelica model
     */
    private MModel mmodel;
    /**
     * SysML utility class
     */
    private SysMLUtil sysmlutil;
    /**
     * name of the root model
     */
    private String modelblock;
    /**
     * use library for state machines
     */
    private static final boolean USE_SM_LIBRARY = true;
    /**
     * expression language translator
     */
    private ExpressionLanguageToModelicaTranslator el2m = new ExpressionLanguageToModelicaTranslator();
    
    public SysMLToModelicaTranslator()
    {
    }
    
    public SysMLToModelicaTranslator(Set<Object> options)
    {
        loadOptions(options);
    }
    
    @Override
    public void reset()
    {
        super.reset();
        refs.clear();
        mmodel = null;
    }
    
    @Override
    public void loadOptions(Set<Object> options)
    {
        super.loadOptions(options);
    }
    
    @Override
    public Set<java.lang.Class<?>> getOptions()
    {
        HashSet<java.lang.Class<?>> res = new HashSet<java.lang.Class<?>>();
        return res;
    }
    
    @Override
    protected ExpressionLanguageToSimulation getExpressionTranslator()
    {
        return el2m;
    }
    
    /**
     * Returns the name of the root model
     * 
     * @return name of the root model
     */
    public String getModelBlock()
    {
        return modelblock;
    }
    
    // Extensions can't override if static
    public String getFileExtension()
    {
        return "mo";
    }
    
    @Override
    protected String getOpaqueLanguage()
    {
        return "modelica";
    }
    
    private void init() throws UMLModelErrorException
    {
        sysmlutil = new SysMLUtil(r.getResourceSet());
        setModelNameSuffix("Model");
        
        // Create the primitive types
        // Components with these primitive types as type are meant to have a
        // data value
        
        refs.put(getKey(sysmlutil.getUMLDouble()), ModelicaUtil.getReal());
        refs.put(getKey(sysmlutil.getSysMLDouble()), ModelicaUtil.getReal());
        refs.put(getKey(sysmlutil.getUMLInteger()), ModelicaUtil.getInteger());
        refs.put(getKey(sysmlutil.getSysMLInteger()), ModelicaUtil.getInteger());
        refs.put(getKey(sysmlutil.getUMLBoolean()), ModelicaUtil.getBoolean());
        refs.put(getKey(sysmlutil.getSysMLBoolean()), ModelicaUtil.getBoolean());
        refs.put(getKey(sysmlutil.getUMLString()), ModelicaUtil.getString());
        refs.put(getKey(sysmlutil.getSysMLString()), ModelicaUtil.getString());
    }
    
    /**
     * Type of SysML element
     * 
     * @author barbau
     *
     */
    private enum TranslatedType
    {
        NONE, MODEL, CONNECTOR, TYPE, STATEMACHINE, STATE, COMPONENT, INTERFACE;
    }
    
    /**
     * Returns the type of the given SysML element
     * 
     * @param unamedelement
     *            SysML element
     * @return type
     */
    private TranslatedType getTranslatedType(NamedElement unamedelement)
    {
        if (unamedelement == null)
            throw new IllegalArgumentException("You must provide a UML named element");
        if (unamedelement instanceof StateMachine)
            return TranslatedType.STATEMACHINE;
        if (unamedelement instanceof Vertex)
            return TranslatedType.STATE;
        if (unamedelement instanceof Class && unamedelement.isStereotypeApplied(sysmlutil.getModelicaBlock()))
            return TranslatedType.INTERFACE;
        if (unamedelement instanceof Class && (((unamedelement.isStereotypeApplied(sysmlutil.getBlock())
                || unamedelement.isStereotypeApplied(sysmlutil.getInterfaceBlock()))
                && !sysmlutil.isSimBlock((Classifier) unamedelement))
                || unamedelement.isStereotypeApplied(sysmlutil.getConstraintBlock())))
        {
            // a block with SimProperties is a connector
            List<Property> lsp = sysmlutil.getAllPhSProperties((Class) unamedelement);
            for (Property uproperty : lsp)
            {
                Type utype = uproperty.getType();
                if (utype != null)
                {
                    if (utype instanceof Classifier && sysmlutil.isSimBlock((Classifier) utype))
                        return TranslatedType.CONNECTOR;
                    else if (utype instanceof DataType)
                        return TranslatedType.TYPE;
                }
            }
            return TranslatedType.MODEL;
        }
        // an interfaceblock is also a connector
        if (unamedelement instanceof Class && unamedelement.isStereotypeApplied(sysmlutil.getInterfaceBlock()))
            return TranslatedType.CONNECTOR;
        if (unamedelement instanceof DataType
                && !Collections.disjoint(SysMLUtil.getAllCompatibleTypes(((DataType) unamedelement)),
                        Arrays.asList(new Classifier[] { sysmlutil.getUMLBoolean(), sysmlutil.getUMLInteger(),
                                sysmlutil.getUMLDouble(), sysmlutil.getSysMLBoolean(), sysmlutil.getSysMLInteger(),
                                sysmlutil.getSysMLDouble() })))
            return TranslatedType.TYPE;
        if (unamedelement instanceof Property)
            return TranslatedType.COMPONENT;
        return TranslatedType.NONE;
    }
    
    @Override
    public void execute(Class urootblock, Resource r, File outputdirectory) throws UMLModelErrorException, IOException
    {
        if (urootblock == null)
            throw new IllegalArgumentException("You must provide a root block");
        if (r == null)
            throw new IllegalArgumentException("You must provide a resouce");
        this.r = r;
        log.debug("Starting translation");
        init();
        
        log.debug("Creating the root Modelica model");
        mmodel = ModelicaFactory.eINSTANCE.createMModel();
        String urootblockname = SysMLUtil.getName(urootblock);
        modelblock = urootblockname + getModelNameSuffix();
        ModelicaUtil.setName(mmodel, modelblock);
        
        // Add predefined types
        
        // Create the modelica block
        MClass mrootblock = createClass(urootblock);
        MComponent mrootcomponent = ModelicaFactory.eINSTANCE.createMComponent();
        mmodel.getComponents().add(mrootcomponent);
        ModelicaUtil.setName(mrootcomponent, "_" + urootblockname);
        mrootcomponent.setType(mrootblock);
        // Blocks are processed one by one
        // there can't be two blocks being processed at the same time
        while (toProcess.size() != 0)
        {
            ReferenceKey rk = toProcess.pop();
            Element uelem = rk.getKey()[0];
            if (!(uelem instanceof NamedElement))
                throw new IllegalStateException(
                        "The element to process " + rk + " is expected to be a named element: " + uelem.getClass());
            MClass mclass = null;
            switch (getTranslatedType((NamedElement) uelem))
            {
            case MODEL:
                mclass = processModel((Class) uelem);
                break;
            case CONNECTOR:
                mclass = processConnector((Class) uelem);
                break;
            case TYPE:
                if (uelem instanceof DataType)
                    mclass = processType((DataType) uelem);
                break;
            case COMPONENT:
                processProperty((Property) uelem);
                break;
            case STATEMACHINE:
                if (USE_SM_LIBRARY)
                    processStateMachineLibrary((StateMachine) uelem);
                else
                    processStateMachineLanguage((StateMachine) uelem);
                break;
            case INTERFACE:
                processInterface((Class) uelem);
                break;
            case STATE:
                break;
            case NONE:
            default:
                log.warn("The element " + print(uelem) + " must be a named element to be processed");
            }
            if (mclass != null && mclass.getOwningClass() == null
                    && !(uelem.isStereotypeApplied(sysmlutil.getConstraintBlock())))
                mmodel.getOwnedClasses().add(mclass);
        }
        while (toProcess2.size() != 0)
        {
            ReferenceKey rk = toProcess2.pop();
            Element uelem = rk.getKey()[0];
            if (!(uelem instanceof NamedElement))
                throw new IllegalStateException(
                        "The element to process " + rk + " is expected to be a named element: " + uelem.getClass());
            switch (getTranslatedType((NamedElement) uelem))
            {
            case CONNECTOR:
                processConnector2((Class) uelem);
                break;
            case MODEL:
                processModel2((Class) uelem);
                break;
            case STATE:
                if (USE_SM_LIBRARY)
                    processStateLibrary2((Vertex) uelem);
                else
                    processStateLanguage2((State) uelem);
                break;
            default:
                log.warn("The element " + print(uelem) + " can't be processed a second time");
            }
        }
        
        ModelicaSerializer ms = new ModelicaSerializer();
        
        if (outputdirectory != null)
            outputfilename = ms.serialize(outputdirectory.getAbsolutePath(), mmodel, getFileNameSuffix());
        else
        {
            URI target = r.getURI().trimSegments(1);
            try
            {
                if (target.isFile())
                {
                    outputfilename = ms.serialize(target.toFileString(), mmodel, getFileNameSuffix());
                }
                else
                    log.error("The target is not a file");
            }
            catch (IOException e)
            {
                log.error("Can't save the model at the location " + target, e);
                throw e;
            }
        }
    }
    
    private MClass createClass(Type utype) throws UMLModelErrorException
    {
        ReferenceKey rk = getKey(utype);
        if (utype == null)
            throw new IllegalArgumentException("You must provide a UML type");
        if (utype.getVisibility() == VisibilityKind.PRIVATE_LITERAL)
            throw new UMLModelErrorException(r, "The type " + print(utype) + " is private and can't be translated");
        MElement melement = null;
        melement = refs.get(rk);
        if (melement != null)
        {
            if (!(melement instanceof MClass))
                throw new ClassCastException(
                        "The element " + print(utype) + " is expected to be a class, but it is not");
            return (MClass) melement;
        }
        
        MClass mclass = null;
        switch (getTranslatedType(utype))
        {
        case MODEL:
            log.debug("Creating block " + print(utype));
            mclass = ModelicaFactory.eINSTANCE.createMModel();
            break;
        case CONNECTOR:
            log.debug("Creating simulation block " + print(utype));
            mclass = ModelicaFactory.eINSTANCE.createMConnector();
            break;
        case TYPE:
            if (utype instanceof DataType)
            {
                log.debug("Creating datatype " + print(utype));
                mclass = ModelicaFactory.eINSTANCE.createMType();
            }
            else
            {
                Property sp = sysmlutil.getAllPhSProperties((Class) utype).get(0);
                mclass = createClass(sp.getType());
            }
            break;
        case STATEMACHINE:
            log.debug("Creating StateMachine " + print(utype));
            mclass = ModelicaFactory.eINSTANCE.createMBlock();
            break;
        case INTERFACE:
            log.debug("Creating Interface " + print(utype));
            mclass = ModelicaFactory.eINSTANCE.createMModel();
            break;
        default:
            log.error("Unexpected type:" + print(utype));
            log.error("Stereotypes: " + utype.getStereotypeApplications().size());
            for (EObject eobj : utype.getStereotypeApplications())
                log.error(eobj);
        }
        if (mclass == null)
            throw new IllegalArgumentException("The element " + print(utype) + " can't be translated as block or type");
        
        refs.put(rk, mclass);
        toProcess.push(rk);
        
        return mclass;
    }
    
    private MComponent createProperty(Property uproperty)
    {
        ReferenceKey rk = getKey(uproperty);
        if (uproperty == null)
            throw new IllegalArgumentException("You must provide a UML property");
        
        // Returns null (do not translate) if the property is private
        if (uproperty.getVisibility() == VisibilityKind.PRIVATE_LITERAL)
        {
            log.warn("Property " + print(uproperty) + " skipped because it is private");
            return null;
        }
        else if (uproperty.getType() == null)
        {
            log.warn("Property " + print(uproperty) + " skipped because it has no type");
            return null;
        }
        else if (getTranslatedType(uproperty.getType()) == TranslatedType.NONE)
        {
            log.warn("Property " + print(uproperty) + " skipped because its type has no translation");
            return null;
        }
        
        MElement melement = refs.get(rk);
        MComponent mcomponent = null;
        if (melement != null)
        {
            if (melement instanceof MComponent)
                mcomponent = (MComponent) melement;
            else
                throw new ClassCastException(
                        ("The element " + print(uproperty) + " is expected to be a property, but it is not"));
        }
        if (mcomponent == null)
        {
            mcomponent = ModelicaFactory.eINSTANCE.createMComponent();
            refs.put(rk, mcomponent);
            toProcess.push(rk);
        }
        return mcomponent;
    }
    
    private MModel processModel(Class ublock) throws UMLModelErrorException
    {
        return (MModel) processClass(ublock);
    }
    
    private MConnector processConnector(Class ublock) throws UMLModelErrorException
    {
        return (MConnector) processClass(ublock);
    }
    
    private MClass processClass(Class ublock) throws UMLModelErrorException
    {
        if (ublock == null)
            throw new IllegalArgumentException("You must provide a UML class");
        log.debug("Processing block " + print(ublock));
        
        // Create and register the block
        MClass mclass = createClass(ublock);
        
        // Element::ownedComment not translated yet
        // NamedElement::name
        ModelicaUtil.setName(mclass, SysMLUtil.getName(ublock));
        
        // NamedElement::visibility
        // TODO: implement that
        
        // Namespace::elementImport not translated
        // Namespace::packageImport not translated
        // Namespace::ownedRule
        for (Constraint uconstraint : ublock.getOwnedRules())
        {
            // check for type?
            MEquation mequation = ModelicaFactory.eINSTANCE.createMEquation();
            mequation.setExpression(getEquation(uconstraint.getSpecification()));
            mclass.getEquations().add(mequation);
        }
        // Classifier::generalization//Classifier::general
        for (Classifier uclassifier : ublock.getGenerals())
        {
            TranslatedType tself = getTranslatedType(ublock);
            TranslatedType tgen = getTranslatedType(uclassifier);
            if ((tself == TranslatedType.MODEL && tgen == TranslatedType.MODEL)
                    || (tself == TranslatedType.CONNECTOR && tgen == TranslatedType.CONNECTOR))
            {
                MExtension mextansion = ModelicaFactory.eINSTANCE.createMExtension();
                // If it has the Block stereotype applied, it must be a class
                MClass mgeneral = createClass(uclassifier);
                mextansion.setExtendedClass(mgeneral);
                mextansion.setOwningClass(mclass);
                // Add modifications (Connectors linking properties of general
                // blocks)
            }
            else
                log.debug("Skipping non-block general classifier " + print(uclassifier));
            if (tself == TranslatedType.MODEL && tgen == TranslatedType.CONNECTOR)
            {
                for (Operation uoperation : uclassifier.getAllOperations())
                {
                    if (uoperation.isStereotypeApplied(sysmlutil.getDirectedFeature()))
                    {
                        List<Property> lp = new LinkedList<Property>();
                        for (Property uproperty : SysMLUtil.getAllAttributes(ublock))
                            if (getTranslatedType(uproperty.getType()) == TranslatedType.CONNECTOR)
                                if (((Class) uproperty.getType()).getAllOperations().contains(uoperation))
                                    lp.add(uproperty);
                        if (lp.size() == 0)
                            continue;
                        MComponent moperation = ModelicaFactory.eINSTANCE.createMComponent();
                        mclass.getComponents().add(moperation);
                        ModelicaUtil.setName(moperation, SysMLUtil.getName(uoperation));
                        moperation.setType((MClass) refs.get(getKey(sysmlutil.getSysMLBoolean())));
                        String dir = ((EnumerationLiteral) uoperation.getValue(sysmlutil.getDirectedFeature(),
                                "featureDirection")).getName();
                        if (dir.equals("provided"))
                        {
                            for (Property uproperty : lp)
                            {
                                MEquation meq = ModelicaFactory.eINSTANCE.createMEquation();
                                meq.setExpression(moperation.getName() + "=" + SysMLUtil.getName(uproperty) + "."
                                        + moperation.getName());
                                mclass.getEquations().add(meq);
                            }
                        }
                        else if (dir.equals("required"))
                        {
                            StringBuilder sb = new StringBuilder();
                            for (int i = 0; i < lp.size(); i++)
                            {
                                if (i != 0)
                                    sb.append(" or ");
                                sb.append(SysMLUtil.getName(lp.get(i)) + "." + moperation.getName());
                            }
                            MEquation meq = ModelicaFactory.eINSTANCE.createMEquation();
                            meq.setExpression(moperation.getName() + "=" + sb.toString());
                            mclass.getEquations().add(meq);
                        }
                    }
                }
            }
        }
        
        // Classifier::ownedUseCase not translated
        // Classifier::collaborationUse not translated
        // Classifier::redefinedClassifier not translated
        // Classifier::powertypeExtent not translated
        // StructuredClassifier::ownedAttribute (SimProperty, SimConstant,
        // SimVariable)
        List<Property> uproperties = new ArrayList<Property>(ublock.getOwnedAttributes());
        
        for (Property uproperty : uproperties)
        {
            if (uproperty.getType() == null)
                throw new UMLModelErrorException(r, "The property " + print(uproperty) + " must have a type");
            // We are in a port type. Copy all attributes of SimVariable instead
            // of adding the
            // property. Still process
            if (sysmlutil.isPhSProperty(uproperty))
            {
                for (Property uproperty2 : SysMLUtil.getAllAttributes(((Classifier) uproperty.getType())))
                {
                    MComponent mcomponent = createProperty(uproperty2);
                    if (mcomponent != null)
                    {
                        log.info("Adding " + print(uproperty2) + " to " + print(ublock));
                        mclass.getComponents().add(mcomponent);
                        // Modifications
                    }
                }
                continue;
            }
            if (uproperty.getType().isStereotypeApplied(sysmlutil.getConstraintBlock()))
            {
                // Create/Get the constraint block
                // createClass(uproperty.getType());
                // skip property
                continue;
            }
            
            // otherwise, add the property
            MComponent mcomponent = createProperty(uproperty);
            if (mcomponent != null)
            {
                log.info("Adding " + print(uproperty) + " to " + print(ublock));
                mclass.getComponents().add(mcomponent);
                // Modifications
            }
            else
                log.debug("Skipping property " + print(uproperty));
        }
        
        // StructuredClassifier::ownedConnector
        for (Connector uconnector : ublock.getOwnedConnectors())
        {
            // Connected properties
            ConnectableElement uproperty1 = uconnector.getEnds().get(0).getRole();
            ConnectableElement uproperty2 = uconnector.getEnds().get(1).getRole();
            if (uproperty1 == null || uproperty2 == null)
            {
                log.warn("Skipped connector " + print(uconnector) + " because of missing role");
                continue;
            }
            
            // Types of connected properties
            Type uproperty1type = uproperty1.getType();
            Type uproperty2type = uproperty2.getType();
            
            // Property path
            List<Property> lp0 = sysmlutil.getPropertyPath(uconnector.getEnds().get(0));
            List<Property> lp1 = sysmlutil.getPropertyPath(uconnector.getEnds().get(1));
            if (uproperty1type instanceof Class && uproperty2type instanceof Class)
            {
                // SimProperties contained in the connected types
                List<Property> simproperties1 = sysmlutil.getAllPhSProperties((Class) uproperty1type);
                List<Property> simproperties2 = sysmlutil.getAllPhSProperties((Class) uproperty2type);
                
                // connect the SimProperties
                if (simproperties1.size() > 0 && simproperties2.size() > 0)
                {
                    MConnect mconnect = ModelicaFactory.eINSTANCE.createMConnect();
                    for (Property uprop : lp0)
                        mconnect.getRef1().add(createProperty(uprop));
                    for (Property uprop : lp1)
                        mconnect.getRef2().add(createProperty(uprop));
                    mclass.getEquations().add(mconnect);
                    continue;
                }
                // binding behavioral port
                if (lp0.get(lp0.size() - 1).getType().isStereotypeApplied(sysmlutil.getInterfaceBlock())
                        && lp1.get(lp1.size() - 1).getType().isStereotypeApplied(sysmlutil.getInterfaceBlock()))
                {
                    MEquation mequation = ModelicaFactory.eINSTANCE.createMEquation();
                    mequation.setExpression(
                            sysmlutil.joinProperties(lp0, ".") + "=" + sysmlutil.joinProperties(lp1, "."));
                    mclass.getEquations().add(mequation);
                }
            }
            if (uconnector.isStereotypeApplied(sysmlutil.getBindingConnector()))
            {
                // refactor
                if (lp0.size() == 2 && lp0.get(0).getType().isStereotypeApplied(sysmlutil.getModelicaBlock())
                        && lp0.get(1).isStereotypeApplied(sysmlutil.getModelicaParameter()))
                {
                    // add modification
                    MComponent mc1 = createProperty(lp0.get(0));
                    MComponent mc2 = createProperty(lp0.get(1));
                    MModification mmodification = ModelicaFactory.eINSTANCE.createMModification();
                    if (mc2 != null)
                        mmodification.getComponentPath().add(mc2);
                    else
                        log.error("Can't add component " + lp0.get(1).getQualifiedName() + " to the modification in "
                                + ublock.getQualifiedName());
                    
                    for (Property property : lp1)
                    {
                        MComponent mc3 = createProperty(property);
                        if (mc3 != null)
                            mmodification.getAssignedReference().add(mc3);
                        else
                            log.error("Can't add reference " + property.getQualifiedName() + " to the modification in "
                                    + ublock.getQualifiedName());
                    }
                    mc1.getModifications().add(mmodification);
                }
                if (lp1.size() == 2 && lp1.get(0).getType().isStereotypeApplied(sysmlutil.getModelicaBlock())
                        && lp1.get(1).isStereotypeApplied(sysmlutil.getModelicaParameter()))
                {
                    // add modification
                    MComponent mc1 = createProperty(lp1.get(0));
                    MComponent mc2 = createProperty(lp1.get(1));
                    MModification mmodification = ModelicaFactory.eINSTANCE.createMModification();
                    if (mc2 != null)
                        mmodification.getComponentPath().add(mc2);
                    else
                        log.error("Can't add component " + lp1.get(1).getQualifiedName() + " to the modification in "
                                + ublock.getQualifiedName());
                    for (Property property : lp0)
                    {
                        MComponent mc3 = createProperty(property);
                        if (mc3 != null)
                            mmodification.getAssignedReference().add(mc3);
                        else
                            log.error("Can't add reference " + property.getQualifiedName() + " to the modification in "
                                    + ublock.getQualifiedName());
                    }
                    mc1.getModifications().add(mmodification);
                }
                if (lp0.size() == 2 && lp1.size() == 1 && lp0.get(0).getType().isStereotypeApplied(sysmlutil.getBlock())
                        && lp0.get(1).isStereotypeApplied(sysmlutil.getPhSConstant())
                        && lp1.get(0).isStereotypeApplied(sysmlutil.getPhSConstant()))
                {
                    MComponent mc1 = createProperty(lp0.get(0));
                    MComponent mc2 = createProperty(lp0.get(1));
                    MComponent mc3 = createProperty(lp1.get(0));
                    MModification mmod = ModelicaFactory.eINSTANCE.createMModification();
                    if (mc2 != null)
                        mmod.getComponentPath().add(mc2);
                    else
                        log.error("Can't add component " + lp0.get(1).getQualifiedName() + " to the modification in "
                                + ublock.getQualifiedName());
                    if (mc3 != null)
                        mmod.getAssignedReference().add(mc3);
                    else
                        log.error("Can't add reference " + lp1.get(0).getQualifiedName() + " to the modification in "
                                + ublock.getQualifiedName());
                    mc1.getModifications().add(mmod);
                }
                if (lp1.size() == 2 && lp0.size() == 1 && lp1.get(0).getType().isStereotypeApplied(sysmlutil.getBlock())
                        && lp1.get(1).isStereotypeApplied(sysmlutil.getPhSConstant())
                        && lp0.get(0).isStereotypeApplied(sysmlutil.getPhSConstant()))
                {
                    MComponent mc1 = createProperty(lp1.get(0));
                    MComponent mc2 = createProperty(lp1.get(1));
                    MComponent mc3 = createProperty(lp0.get(0));
                    MModification mmod = ModelicaFactory.eINSTANCE.createMModification();
                    if (mc2 != null)
                        mmod.getComponentPath().add(mc2);
                    else
                        log.error("Can't add component " + lp1.get(1).getQualifiedName() + " to the modification in "
                                + ublock.getQualifiedName());
                    if (mc3 != null)
                        mmod.getAssignedReference().add(mc3);
                    else
                        log.error("Can't add reference " + lp0.get(0).getQualifiedName() + " to the modification in "
                                + ublock.getQualifiedName());
                    mc1.getModifications().add(mmod);
                }
                // Surjective mode for later, but I want to avoid the warning
                continue;
            }
            log.warn(
                    "Connector " + print(uconnector) + " doesn't connect two simblocks and is not a binding connector");
            
        }
        // EncapsulatedClassifier::ownedPort
        
        // BehavioredClassifier::classifierBehavior
        if (ublock.getClassifierBehavior() != null && ublock.getClassifierBehavior() instanceof StateMachine)
        {
            // MBlock mstatemachine =
            // createStateMachine((StateMachine)ublock.getClassifierBehavior());
            StateMachine sm = (StateMachine) ublock.getClassifierBehavior();
            MBlock mstatemachine = createStateMachine(sm);
            mstatemachine.setOwningClass(mclass);
        }
        
        // Class::nestedClassifier not processed (except if used)
        
        // Class::ownedOperation not translated
        for (Operation uoperation : ublock.getOperations())
        {
            if (uoperation.isStereotypeApplied(sysmlutil.getDirectedFeature()))
            {
                String dir = ((EnumerationLiteral) uoperation.getValue(sysmlutil.getDirectedFeature(),
                        "featureDirection")).getName();
                MComponent moperation = ModelicaFactory.eINSTANCE.createMComponent();
                mclass.getComponents().add(moperation);
                ModelicaUtil.setName(moperation, SysMLUtil.getName(uoperation));
                if (dir.equals("provided"))
                    moperation.setDirection(MDirection.OUTPUT);
                else if (dir.equals("required"))
                    moperation.setDirection(MDirection.INPUT);
                moperation.setType((MClass) refs.get(getKey(sysmlutil.getSysMLBoolean())));
                refs.put(getKey(uoperation), moperation);
            }
        }
        // Class::isAbstract
        
        // Class::isActive
        
        // Block::isEncapsulated
        
        // process a second time for initial values
        toProcess2.push(getKey(ublock));
        
        return mclass;
    }
    
    private MClass processInterface(Class ublock) throws UMLModelErrorException
    {
        if (ublock == null)
            throw new IllegalArgumentException("You must provide a UML class");
        log.debug("Processing " + print(ublock));
        
        // Create and register the block
        MClass mclass = createClass(ublock);
        mclass.setName((String) ublock.getValue(sysmlutil.getModelicaBlock(), "name"));
        
        for (Property uproperty : SysMLUtil.getAllAttributes(ublock))
        {
            if (uproperty.isStereotypeApplied(sysmlutil.getModelicaParameter())
                    || uproperty.isStereotypeApplied(sysmlutil.getModelicaPort()))
            {
                MComponent mcomponent = createProperty(uproperty);
                if (mcomponent != null)
                {
                    log.info("Adding " + print(uproperty) + " to " + print(ublock));
                    mclass.getComponents().add(mcomponent);
                }
                else
                    log.debug("Skipping property " + print(uproperty));
            }
        }
        
        return mclass;
    }
    
    private MBlock createStateMachine(StateMachine ustatemachine)
    {
        if (ustatemachine == null)
            throw new IllegalArgumentException("You must provide a UML state machine");
        ReferenceKey rk = getKey(ustatemachine);
        
        MBlock mblock = null;
        
        MElement melement = refs.get(rk);
        if (melement != null)
        {
            if (melement instanceof MBlock)
                mblock = (MBlock) melement;
            else
                throw new ClassCastException(
                        ("The element " + print(ustatemachine) + " is expected to be a block, but it is not"));
        }
        
        if (mblock == null)
        {
            mblock = ModelicaFactory.eINSTANCE.createMBlock();
            refs.put(rk, mblock);
            toProcess.push(rk);
        }
        return mblock;
    }
    
    private MBlock createStateLanguage(State uvertex)
    {
        if (uvertex == null)
            throw new IllegalArgumentException("You must provide a UML state");
        ReferenceKey rk = getKey(uvertex);
        
        MBlock mblock = null;
        
        MElement melement = refs.get(rk);
        if (melement != null)
        {
            if (melement instanceof MBlock)
                mblock = (MBlock) melement;
            else
                throw new ClassCastException(
                        ("The element " + print(uvertex) + " is expected to be a block, but it is not"));
        }
        
        if (mblock == null)
        {
            mblock = ModelicaFactory.eINSTANCE.createMBlock();
            refs.put(rk, mblock);
            toProcess2.push(rk);
        }
        return mblock;
    }
    
    private MComponent createStateLibrary(Vertex uvertex)
    {
        if (uvertex == null)
            throw new IllegalArgumentException("You must provide a UML state");
        ReferenceKey rk = getKey(uvertex);
        
        MComponent mcomponent = null;
        
        MElement melement = refs.get(rk);
        if (melement != null)
        {
            if (melement instanceof MComponent)
                mcomponent = (MComponent) melement;
            else
                throw new ClassCastException(
                        ("The element " + print(uvertex) + " is expected to be a component, but it is not"));
        }
        
        if (mcomponent == null)
        {
            mcomponent = ModelicaFactory.eINSTANCE.createMComponent();
            ModelicaUtil.setName(mcomponent, SysMLUtil.getName(uvertex));
            refs.put(rk, mcomponent);
            toProcess2.push(rk);
        }
        return mcomponent;
    }
    
    private MBlock processStateMachineLanguage(StateMachine ustatemachine) throws UMLModelErrorException
    {
        if (ustatemachine == null)
            throw new IllegalArgumentException("You must provide a UML state machine");
        log.debug("Processing " + print(ustatemachine));
        
        // Create and register the block
        MBlock mblock = (MBlock) processClass(ustatemachine);
        
        // all components set to inner so that they can be referred to
        MElement mcontext = refs.get(getKey(ustatemachine.getContext()));
        if (mcontext instanceof MClass)
        {
            for (MComponent mcomponent : ((MClass) mcontext).getComponents())
                mcomponent.setHierarchy(MHierarchy.INNER);
        }
        
        // Change components to inner
        for (Region uregion : ustatemachine.getRegions())
        {
            // processRegionLanguage(uregion, mblock);
            processRegionLanguage(uregion, mblock);
        }
        return mblock;
        
    }
    
    private MBlock processStateMachineLibrary(StateMachine ustatemachine) throws UMLModelErrorException
    {
        if (ustatemachine == null)
            throw new IllegalArgumentException("You must provide a UML state machine");
        log.debug("Processing " + print(ustatemachine));
        
        // Create and register the block, will add to second process
        MBlock mblock = (MBlock) processClass(ustatemachine);
        
        // Change components to inner
        for (Region uregion : ustatemachine.getRegions())
        {
            // processRegionLanguage(uregion, mblock);
            processRegionLibrary(uregion, mblock);
        }
        return mblock;
        
    }
    
    private void processRegionLanguage(Region uregion, MBlock mblock) throws UMLModelErrorException
    {
        HashMap<String, String> substitutions = new HashMap<String, String>();
        BehavioredClassifier context = uregion.getStateMachine().getContext();
        for (Property uproperty : SysMLUtil.getAllAttributes(context))
        {
            if (uproperty.getType() instanceof Classifier)
            {
                List<Property> sps = sysmlutil.getAllPhSProperties((Classifier) uproperty.getType());
                if (sps.size() == 1)
                    substitutions.put(uproperty.getName() + "." + sps.get(0).getName(), uproperty.getName());
            }
        }
        for (Vertex uvertex : uregion.getSubvertices())
        {
            if (uvertex instanceof State)
            {
                State ustate = (State) uvertex;
                
                MBlock mstatetype = createStateLanguage(ustate);
                mstatetype.setOwningClass(mblock);
                
                MComponent mstate = ModelicaFactory.eINSTANCE.createMComponent();
                mblock.getComponents().add(mstate);
                ModelicaUtil.setName(mstate, "_" + SysMLUtil.getName(ustate));
                mstate.setType(mstatetype);
                
                Behavior uentry = ustate.getEntry();
                if (uentry != null)
                {
                    if (uentry instanceof OpaqueBehavior)
                    {
                        MAlgorithm mal = ModelicaFactory.eINSTANCE.createMAlgorithm();
                        String exp = replace(getAssignments((OpaqueBehavior) uentry), "\\w+(\\.\\w+)*", substitutions);
                        mal.setExpression(exp);
                        mstatetype.getAlgorithms().add(mal);
                    }
                    else
                        log.warn("Entry behavior unsupported " + uentry);
                }
                Behavior udo = ustate.getDoActivity();
                if (udo != null)
                {
                    log.warn("DoActivity not translated");
                    // if (udo instanceof OpaqueBehavior)
                    // {
                    // MEquation meq =
                    // ModelicaFactory.eINSTANCE.createMEquation();
                    // meq.setExpression(getOpaqueBehavior((OpaqueBehavior)
                    // udo));
                    // mstatetype.getEquations().add(meq);
                    // }
                    // else
                    // log.warn("Do behavior unsupported " + uentry);
                }
                Behavior uexit = ustate.getExit();
                if (uexit != null)
                {
                    log.warn("Exit not translated");
                    // if (uexit instanceof OpaqueBehavior)
                    // {
                    // log.warn("Exit behavior not guaranteed to occur last " +
                    // print(uentry));
                    // MEquation meq =
                    // ModelicaFactory.eINSTANCE.createMEquation();
                    // meq.setExpression(getOpaqueBehavior((OpaqueBehavior)
                    // uexit));
                    // mstatetype.getEquations().add(meq);
                    // }
                    // else
                    // log.warn("Exit behavior unsupported " + uentry);
                }
                log.debug("Adding state " + print(ustate));
                for (Region uregion2 : ustate.getRegions())
                    processRegionLanguage(uregion2, mstatetype);
            }
        }
        for (Transition utransition : uregion.getTransitions())
        {
            Vertex source = utransition.getSource();
            if (source == null)
                throw new UMLModelErrorException(r, "The transition has no source: " + print(utransition));
            
            Vertex target = utransition.getTarget();
            if (target == null)
                throw new UMLModelErrorException(r, "The transition has no target: " + print(utransition));
            
            if (source instanceof Pseudostate && ((Pseudostate) source).getKind() == PseudostateKind.INITIAL_LITERAL)
            {
                if (target instanceof State)
                {
                    MEquation meq = ModelicaFactory.eINSTANCE.createMEquation();
                    meq.setExpression("initialState(_" + SysMLUtil.getName(target) + ")");
                    mblock.getEquations().add(meq);
                }
                else
                    log.warn("Can't set the initial state: target not a state" + print(target));
            }
            else if (source instanceof State && target instanceof State)
            {
                String condition = "true";
                String timer = "";
                for (Trigger trigger : utransition.getTriggers())
                {
                    Event event = trigger.getEvent();
                    if (event != null)
                    {
                        if (event instanceof ChangeEvent)
                            condition = replace(getExpression(((ChangeEvent) event).getChangeExpression()),
                                    "\\w+(\\.\\w+)*", substitutions);
                        else if (event instanceof TimeEvent)
                        {
                            TimeEvent te = (TimeEvent) event;
                            if (te.isRelative())
                                timer = ",enableTimer=true,waitTime=" + getExpression(((TimeEvent) event).getWhen());
                            else
                                condition = "time>" + getExpression(((TimeEvent) event).getWhen());
                        }
                        else if (event instanceof CallEvent)
                            condition = SysMLUtil.getName(((CallEvent) event).getOperation());
                        else
                            log.warn("Trigger " + trigger + " with event " + event + " couldn't be translated");
                    }
                }
                String priority = "";
                if (source.isStereotypeApplied(sysmlutil.getSimulationVertex())
                        && source.getValue(sysmlutil.getSimulationVertex(), "transitionPriority") instanceof List<?>)
                {
                    @SuppressWarnings("unchecked")
                    List<Transition> ltr = (List<Transition>) source.getValue(sysmlutil.getSimulationVertex(),
                            "transitionPriority");
                    priority = ",priority=" + (ltr.indexOf(utransition) + 1);
                }
                MEquation meq = ModelicaFactory.eINSTANCE.createMEquation();
                meq.setExpression("transition(_" + SysMLUtil.getName(source) + ", _" + SysMLUtil.getName(target) + ", "
                        + condition + timer + priority + ")");
                mblock.getEquations().add(meq);
            }
            else
                log.warn("Unsupported transition " + print(utransition));
        }
    }
    
    private void processRegionLibrary(Region uregion, MBlock mblock) throws UMLModelErrorException
    {
        HashMap<String, String> substitutions = new HashMap<String, String>();
        BehavioredClassifier context = uregion.getStateMachine().getContext();
        for (Property uproperty : SysMLUtil.getAllAttributes(context))
        {
            if (uproperty.getType() instanceof Classifier)
            {
                List<Property> sps = sysmlutil.getAllPhSProperties((Classifier) uproperty.getType());
                if (sps.size() == 1)
                    substitutions.put(uproperty.getName() + "." + sps.get(0).getName(), uproperty.getName());
            }
        }
        for (Vertex uvertex : uregion.getSubvertices())
        {
            if (uvertex instanceof State)
            {
                State ustate = (State) uvertex;
                String ustatename = SysMLUtil.getName(ustate);
                
                MComponent mc = createStateLibrary(ustate);
                mblock.getComponents().add(mc);
                
                Behavior uentry = ustate.getEntry();
                if (uentry != null)
                {
                    if (uentry instanceof OpaqueBehavior)
                    {
                        String exp = "if " + ustatename + ".active then " + getAssignments((OpaqueBehavior) uentry)
                                + " end if;";
                        MAlgorithm mal = ModelicaFactory.eINSTANCE.createMAlgorithm();
                        mal.setExpression(replace(exp, "\\w+(\\.\\w+)*", substitutions));
                        mblock.getAlgorithms().add(mal);
                    }
                    else
                        log.warn("Entry unsupported " + uentry);
                }
                Behavior udo = ustate.getDoActivity();
                if (udo != null)
                {
                    log.warn("DoActivity not translated");
                    if (udo instanceof Activity)
                    {
                        Activity activity = (Activity) udo;
                        if (activity.getOwnedNodes().size() == 1)
                        {
                            ActivityNode node = activity.getOwnedNodes().get(0);
                            if (node instanceof CallOperationAction)
                            {
                                CallOperationAction cao = (CallOperationAction) node;
                                
                                MAlgorithm mal = ModelicaFactory.eINSTANCE.createMAlgorithm();
                                String uoperationname = SysMLUtil.getName(cao.getOperation());
                                mal.setExpression("if " + ustatename + ".active and not(pre(" + ustatename
                                        + ".active)) then " + uoperationname + ":=true; endif;");
                                mblock.getAlgorithms().add(mal);
                                
                                mal = ModelicaFactory.eINSTANCE.createMAlgorithm();
                                mal.setExpression("if not(" + ustatename + ".active) and pre(" + ustatename
                                        + ".active) then " + uoperationname + ":=false; end if;");
                                mblock.getAlgorithms().add(mal);
                            }
                            else
                                log.warn("Can translate only activity with one call operation action");
                        }
                        else
                            log.warn("Can translate only activity with one node: found "
                                    + activity.getOwnedNodes().size());
                    }
                    else
                        log.warn("Do behavior unsupported " + udo);
                }
                Behavior uexit = ustate.getEntry();
                if (uexit != null)
                {
                    log.warn("Exit not translated");
                }
                log.debug("Adding state " + print(ustate));
                for (Region uregion2 : ustate.getRegions())
                    processRegionLibrary(uregion2, mblock);
            }
            else if (uvertex instanceof Pseudostate
                    && ((Pseudostate) uvertex).getKind() == PseudostateKind.INITIAL_LITERAL)
            {
                Pseudostate upseudostate = (Pseudostate) uvertex;
                
                MComponent mc = createStateLibrary(upseudostate);
                mblock.getComponents().add(mc);
            }
            else
                log.warn("Untreated vertex: " + print(uvertex));
        }
        for (Transition utransition : uregion.getTransitions())
        {
            if (utransition.getSource() == null || utransition.getTarget() == null)
            {
                log.error("The transition has a no source or target");
                continue;
            }
            MComponent source = createStateLibrary(utransition.getSource());
            MComponent target = createStateLibrary(utransition.getTarget());
            
            MComponent transition = ModelicaFactory.eINSTANCE.createMComponent();
            mblock.getComponents().add(transition);
            ModelicaUtil.setName(transition, SysMLUtil.getName(utransition));
            log.info("Creating transition " + transition.getName());
            transition.setType(ModelicaUtil.getTransition());
            
            MEquation meq1 = ModelicaFactory.eINSTANCE.createMEquation();
            meq1.setExpression("connect(" + source.getName() + ".outPort["
                    + (utransition.getSource().getOutgoings().indexOf(utransition) + 1) + "], " + transition.getName()
                    + ".inPort)");
            mblock.getEquations().add(meq1);
            
            MEquation meq2 = ModelicaFactory.eINSTANCE.createMEquation();
            meq2.setExpression("connect(" + transition.getName() + ".outPort, " + target.getName() + ".inPort["
                    + (utransition.getTarget().getIncomings().indexOf(utransition) + 1) + "])");
            mblock.getEquations().add(meq2);
            
            String condition = "true";
            for (Trigger trigger : utransition.getTriggers())
            {
                Event event = trigger.getEvent();
                if (event != null)
                {
                    if (event instanceof ChangeEvent)
                        condition = getExpression(((ChangeEvent) event).getChangeExpression());
                    else if (event instanceof TimeEvent)
                    {
                        TimeEvent te = (TimeEvent) event;
                        if (te.isRelative())
                        {
                            MModification mtimer = ModelicaFactory.eINSTANCE.createMModification();
                            MExpressionValue mtimerex = ModelicaFactory.eINSTANCE.createMExpressionValue();
                            mtimerex.setValue("true");
                            mtimer.getComponentPath()
                                    .add(ModelicaUtil.getTransition().getComponentByName("enableTimer"));
                            mtimer.setAssignedValue(mtimerex);
                            transition.getModifications().add(mtimer);
                            
                            MModification mwait = ModelicaFactory.eINSTANCE.createMModification();
                            MExpressionValue mwaitex = ModelicaFactory.eINSTANCE.createMExpressionValue();
                            mwaitex.setValue(getExpression(((TimeEvent) event).getWhen()));
                            mwait.getComponentPath().add(ModelicaUtil.getTransition().getComponentByName("waitTime"));
                            mwait.setAssignedValue(mwaitex);
                            transition.getModifications().add(mwait);
                            
                        }
                        else
                            condition = "time>" + getExpression(((TimeEvent) event).getWhen());
                    }
                    else if (event instanceof CallEvent)
                        condition = SysMLUtil.getName(((CallEvent) event).getOperation());
                    else
                        log.warn("Trigger " + print(trigger) + " couldn't be translated");
                }
            }
            MModification mcondition = ModelicaFactory.eINSTANCE.createMModification();
            MExpressionValue mex = ModelicaFactory.eINSTANCE.createMExpressionValue();
            mex.setValue(replace(condition, "\\w+(\\.\\w+)*", substitutions));
            mcondition.getComponentPath().add(ModelicaUtil.getTransition().getComponentByName("condition"));
            mcondition.setAssignedValue(mex);
            
            transition.getModifications().add(mcondition);
        }
    }
    
    private MBlock processStateLanguage2(State ustate)
    {
        if (ustate == null)
            throw new IllegalArgumentException("You must provide a UML state");
        log.debug("Processing " + print(ustate));
        
        MBlock mblock = createStateLanguage(ustate);
        ModelicaUtil.setName(mblock, SysMLUtil.getName(ustate));
        
        // copy statemachine components to state, set to outer
        List<String> lhss = new LinkedList<String>();
        
        // dirty...
        for (MAlgorithm mal : mblock.getAlgorithms())
            for (String assign : mal.getExpression().split(";"))
                for (String lhs : assign.split(":="))
                    
                    lhss.add(lhs.trim());
                
        MElement mcontext = refs.get(getKey(ustate.containingStateMachine().getContext()));
        if (mcontext instanceof MClass)
        {
            for (MComponent mcomponent : ((MClass) mcontext).getComponents())
            {
                if (mcomponent.getHierarchy() != MHierarchy.INNER)
                    continue;
                MComponent mcomponent2 = ModelicaFactory.eINSTANCE.createMComponent();
                mblock.getComponents().add(mcomponent2);
                ModelicaUtil.setName(mcomponent2, mcomponent.getName());
                mcomponent2.setType(mcomponent.getType());
                mcomponent2.setHierarchy(MHierarchy.OUTER);
                if (lhss.contains(mcomponent2.getName()))
                    mcomponent2.setDirection(MDirection.OUTPUT);
            }
        }
        
        // if (ustate.getEntry() != null && ustate.getEntry() instanceof
        // OpaqueBehavior)
        // {
        // MEquation meq = ModelicaFactory.eINSTANCE.createMEquation();
        // meq.setExpression(getAssignments((OpaqueBehavior)
        // ustate.getEntry()));
        // mblock.getEquations().add(meq);
        // }
        
        return mblock;
    }
    
    private MComponent processStateLibrary2(Vertex uvertex)
    {
        if (uvertex == null)
            throw new IllegalArgumentException("You must provide a UML state");
        log.debug("Processing state " + SysMLUtil.getName(uvertex));
        
        MComponent mcomponent = createStateLibrary(uvertex);
        // copy statemachine components, set to outer
        if (uvertex instanceof Pseudostate && ((Pseudostate) uvertex).getKind() == PseudostateKind.INITIAL_LITERAL)
            mcomponent.setType(ModelicaUtil.getInitialStep());
        else
            mcomponent.setType(ModelicaUtil.getStep());
        
        MModification mmodification1 = ModelicaFactory.eINSTANCE.createMModification();
        mmodification1.getComponentPath().add(ModelicaUtil.getStep().getComponentByName("nIn"));
        MIntegerValue miv1 = ModelicaFactory.eINSTANCE.createMIntegerValue();
        miv1.setValue(uvertex.getIncomings().size());
        mmodification1.setAssignedValue(miv1);
        mcomponent.getModifications().add(mmodification1);
        
        MModification mmodification2 = ModelicaFactory.eINSTANCE.createMModification();
        mmodification2.getComponentPath().add(ModelicaUtil.getStep().getComponentByName("nOut"));
        MIntegerValue miv2 = ModelicaFactory.eINSTANCE.createMIntegerValue();
        miv2.setValue(uvertex.getOutgoings().size());
        mmodification2.setAssignedValue(miv2);
        mcomponent.getModifications().add(mmodification2);
        return mcomponent;
    }
    
    private MModel processModel2(Class ublock) throws UMLModelErrorException
    {
        return (MModel) processClass2(ublock);
    }
    
    private MConnector processConnector2(Class ublock) throws UMLModelErrorException
    {
        return (MConnector) processClass2(ublock);
    }
    
    @SuppressWarnings("unchecked")
    private MClass processClass2(Class ublock) throws UMLModelErrorException
    {
        if (ublock == null)
            throw new IllegalArgumentException("You must provide a UML class");
        log.debug("Processing " + print(ublock) + " a second time");
        
        // Create and register the block
        MClass mclass = createClass(ublock);
        
        // connect state machine, copy components
        
        // BehavioredClassifier::classifierBehavior
        if (ublock.getClassifierBehavior() != null && ublock.getClassifierBehavior() instanceof StateMachine)
        {
            // MBlock mstatemachine =
            // createStateMachine((StateMachine)ublock.getClassifierBehavior());
            StateMachine sm = (StateMachine) ublock.getClassifierBehavior();
            MBlock mstatemachine = createStateMachine(sm);
            
            MComponent mstatemachinecomponent = ModelicaFactory.eINSTANCE.createMComponent();
            mclass.getComponents().add(mstatemachinecomponent);
            ModelicaUtil.setName(mstatemachinecomponent, "_" + SysMLUtil.getName(sm));
            mstatemachinecomponent.setType(mstatemachine);
            
            // copy elements from context to state machine, different cases:
            // ports with simvariables are copied
            // operations are copied
            // simconstant are copied
            for (Property uproperty : SysMLUtil.getAllAttributes(ublock))
            {
                if (refs.get(getKey(uproperty)) instanceof MComponent)
                {
                    MComponent copied = (MComponent) refs.get(getKey(uproperty));
                    String upropertyname = SysMLUtil.getName(uproperty);
                    MComponent mc1 = ModelicaFactory.eINSTANCE.createMComponent();
                    mstatemachine.getComponents().add(mc1);
                    ModelicaUtil.setName(mc1, upropertyname);
                    mc1.setType(copied.getType());
                    MEquation mequation = ModelicaFactory.eINSTANCE.createMEquation();
                    mequation.setExpression(
                            upropertyname + "=" + mstatemachinecomponent.getName() + "." + upropertyname);
                    mclass.getEquations().add(mequation);
                    continue;
                }
            }
            for (Classifier uclassifier : ublock.getGenerals())
            {
                TranslatedType tself = getTranslatedType(ublock);
                TranslatedType tgen = getTranslatedType(uclassifier);
                if (tself == TranslatedType.MODEL && tgen == TranslatedType.CONNECTOR)
                {
                    for (Operation uoperation : uclassifier.getAllOperations())
                    {
                        if (uoperation.isStereotypeApplied(sysmlutil.getDirectedFeature()))
                        {
                            MComponent mc1 = ModelicaFactory.eINSTANCE.createMComponent();
                            mstatemachine.getComponents().add(mc1);
                            ModelicaUtil.setName(mc1, SysMLUtil.getName(uoperation));
                            mc1.setType(ModelicaUtil.getBoolean());
                            MEquation mequation = ModelicaFactory.eINSTANCE.createMEquation();
                            mequation.setExpression(
                                    mc1.getName() + "=" + mstatemachinecomponent.getName() + "." + mc1.getName());
                            mclass.getEquations().add(mequation);
                        }
                    }
                }
            }
        }
        
        // Make the substitutions associated with this property. Should move up
        List<Connector> connectors = ublock.getOwnedConnectors();
        
        // Parse connectors for bindings
        for (Connector conn : connectors)
        {
            if (!conn.isStereotypeApplied(sysmlutil.getBindingConnector()))
                continue;
            List<Property> lp0 = sysmlutil.getPropertyPath(conn.getEnds().get(0));
            List<Property> lp1 = sysmlutil.getPropertyPath(conn.getEnds().get(1));
            
            // this part is for the _variable_ bindings
            // inefficient but easy to read
            boolean join = true;
            loop: for (List<Property> lp : new List[] { lp0, lp1 })
                for (int i = 0; i < lp.size(); i++)
                {
                    
                    // if ((i == lp.size() - 1 &&
                    // !lp.get(i).isStereotypeApplied(sysmlutil.getSimVariable()))
                    // || (i < lp.size() - 1 &&
                    // !lp.get(i).getType().isStereotypeApplied(sysmlutil.getBlock())))
                    if (lp.get(i).getType() != null
                            && lp.get(i).getType().isStereotypeApplied(sysmlutil.getConstraintBlock()))
                    {
                        join = false;
                        break loop;
                    }
                }
            if (!lp0.get(lp0.size() - 1).isStereotypeApplied(sysmlutil.getPhSVariable())
                    && !lp1.get(lp1.size() - 1).isStereotypeApplied(sysmlutil.getPhSVariable()))
                join = false;
            String expr = sysmlutil.joinProperties(lp0, ".") + "=" + sysmlutil.joinProperties(lp1, ".");
            if (join)
            {
                MEquation meq = ModelicaFactory.eINSTANCE.createMEquation();
                meq.setExpression(expr);
                mclass.getEquations().add(meq);
            }
            // else
            // log.warn("Ignoring binding " + expr);
        }
        
        for (Property uproperty : ublock.getOwnedAttributes())
        {
            if (uproperty.getType().isStereotypeApplied(sysmlutil.getConstraintBlock()))
            {
                Class uconstraintblock = (Class) uproperty.getType();
                
                // Parse connectors for bindings
                Hashtable<String, String> substitutions = new Hashtable<String, String>();
                for (Connector conn : connectors)
                {
                    if (!conn.isStereotypeApplied(sysmlutil.getBindingConnector()))
                        continue;
                    List<Property> lp0 = sysmlutil.getPropertyPath(conn.getEnds().get(0));
                    List<Property> lp1 = sysmlutil.getPropertyPath(conn.getEnds().get(1));
                    
                    // this part is for the substitutions
                    String from = null, to = null;
                    // No support for nested CB properties
                    if (lp0.size() > 1 && lp0.get(0) == uproperty)
                    {
                        from = SysMLUtil.getName(lp0.get(1));
                        sysmlutil.removeSimProperty(lp1);
                        to = sysmlutil.joinProperties(lp1, ".");
                        if (lp0.get(1).isStereotypeApplied(sysmlutil.getMultidimensionalElement()))
                            to = "{" + to + "}";
                    }
                    if (lp1.size() > 1 && lp1.get(0) == uproperty)
                    {
                        from = SysMLUtil.getName(lp1.get(1));
                        sysmlutil.removeSimProperty(lp0);
                        to = sysmlutil.joinProperties(lp0, ".");
                        if (lp1.get(1).isStereotypeApplied(sysmlutil.getMultidimensionalElement())
                                && !lp0.get(lp0.size() - 1).isStereotypeApplied(sysmlutil.getMultidimensionalElement()))
                            to = "{" + to + "}";
                    }
                    if (from != null && to != null)
                    {
                        substitutions.put(from, to);
                    }
                }
                
                List<Constraint> uconstraints = SysMLUtil.getAllConstraints(uconstraintblock);
                
                for (Constraint uconstraint : uconstraints)
                {
                    MEquation meq = ModelicaFactory.eINSTANCE.createMEquation();
                    getExpressionTranslator().setSubstitutions(substitutions);
                    meq.setExpression(getEquation(uconstraint.getSpecification()));
                    getExpressionTranslator().setSubstitutions(null);
                    mclass.getEquations().add(meq);
                }
            }
        }
        
        // Initial values from default values
        for (Property uproperty : ublock.getOwnedAttributes())
        {
            MElement melement = refs.get(getKey(uproperty));
            if (melement != null)
            {
                // Initial value
                ValueSpecification vs = uproperty.getDefaultValue();
                if (vs != null)
                {
                    if (vs instanceof InstanceValue)
                    {
                        InstanceSpecification instancespec = ((InstanceValue) vs).getInstance();
                        if (instancespec != null)
                            for (Slot slot : instancespec.getSlots())
                            {
                                for (MModification mmodification : getModificationsFromSlot(slot,
                                        (Property) slot.getDefiningFeature()))
                                {
                                    MComponent mc = createProperty((Property) slot.getDefiningFeature());
                                    if (mc != null)
                                        mmodification.getComponentPath().add(0, mc);
                                    else
                                        log.error("Can't add component " + slot.getDefiningFeature().getQualifiedName()
                                                + " to the modification in " + ublock.getQualifiedName());
                                    ((MComponent) melement).getModifications().add(mmodification);
                                }
                            }
                    }
                    else
                    {
                        // default value from property
                        MDataValue mdv = getValue(vs);
                        if (mdv != null)
                        {
                            Classifier uclassifier = (Classifier) uproperty.getType();
                            
                            MClass mc = createClass(uclassifier);
                            MComponent start = mc.getComponentByName("start");
                            MComponent fixed = mc.getComponentByName("fixed");
                            if (start != null && fixed != null)
                            {
                                MModification mmod = ModelicaFactory.eINSTANCE.createMModification();
                                mmod.getComponentPath().add(start);
                                mmod.setAssignedValue(mdv);
                                ((MComponent) melement).getModifications().add(mmod);
                                
                                MBooleanValue mbv = ModelicaFactory.eINSTANCE.createMBooleanValue();
                                mbv.setValue(true);
                                
                                MModification mmod2 = ModelicaFactory.eINSTANCE.createMModification();
                                mmod2.getComponentPath().add(fixed);
                                mmod2.setAssignedValue(mbv);
                                ((MComponent) melement).getModifications().add(mmod2);
                            }
                            else
                                log.warn("The start component of " + print(uproperty) + " couldn't be found");
                        }
                        else
                            log.warn("Could not retrieve the default value of " + print(uproperty));
                    }
                }
            }
            // initial value from the SimBlocks
            if (sysmlutil.isPhSProperty(uproperty))
            {
                for (Property uproperty2 : SysMLUtil.getAllAttributes((Classifier) uproperty.getType()))
                {
                    MComponent mcomponent = createProperty(uproperty2);
                    ValueSpecification vs = uproperty2.getDefaultValue();
                    if (mcomponent != null && vs != null)
                    {
                        MDataValue mdv = getValue(vs);
                        if (mdv != null)
                        {
                            Classifier uclassifier = (Classifier) uproperty2.getType();
                            
                            MClass mc = createClass(uclassifier);
                            MComponent start = mc.getComponentByName("start");
                            MComponent fixed = mc.getComponentByName("fixed");
                            if (start != null && fixed != null)
                            {
                                MModification mmod = ModelicaFactory.eINSTANCE.createMModification();
                                mmod.getComponentPath().add(start);
                                mmod.setAssignedValue(mdv);
                                mcomponent.getModifications().add(mmod);
                                
                                MBooleanValue mbv = ModelicaFactory.eINSTANCE.createMBooleanValue();
                                mbv.setValue(true);
                                
                                MModification mmod2 = ModelicaFactory.eINSTANCE.createMModification();
                                mmod2.getComponentPath().add(fixed);
                                mmod2.setAssignedValue(mbv);
                                mcomponent.getModifications().add(mmod2);
                            }
                            else
                                log.warn("The start component of " + print(ublock) + " couldn't be found");
                        }
                        else
                            log.warn("Could not retrieve the default value of " + print(uproperty));
                    }
                }
                continue;
            }
            
        }
        
        // Initial values from InstanceSpecification.
        // This is here because the "start" component may not be accessible
        // before
        for (Dependency dep : ublock.getClientDependencies())
        {
            if (dep.isStereotypeApplied(sysmlutil.getInitialValuesSpecification()))
            {
                log.info("Found instance specification for " + print(ublock));
                for (NamedElement supplier : dep.getSuppliers())
                    if (supplier instanceof InstanceSpecification)
                    {
                        InstanceSpecification target = (InstanceSpecification) supplier;
                        for (Slot slot : target.getSlots())
                        {
                            List<MModification> mmodifications = getModificationsFromSlot(slot,
                                    (Property) slot.getDefiningFeature());
                            
                            StructuralFeature sf = slot.getDefiningFeature();
                            MComponent mcomponent = createProperty((Property) sf);
                            
                            mcomponent.getModifications().addAll(mmodifications);
                        }
                    }
            }
        }
        
        return mclass;
    }
    
    private MType processType(DataType udatatype) throws UMLModelErrorException
    {
        if (udatatype == null)
            throw new IllegalArgumentException("You must provide a UML datatype");
        log.debug("Processing " + print(udatatype));
        
        // Create and register the datatype
        MType mtype = (MType) createClass(udatatype);
        
        // Element::ownedComment not translated yet
        // NamedElement::name
        ModelicaUtil.setName(mtype, SysMLUtil.getName(udatatype));
        
        // NamedElement::visibility
        // TODO: implement that
        
        // Namespace::elementImport not translated
        // Namespace::packageImport not translated
        // Namespace::ownedRule not translated
        // Classifier::generalization//Classifier::general
        for (Classifier uclassifier : udatatype.getGenerals())
        {
            MExtension mextansion = ModelicaFactory.eINSTANCE.createMExtension();
            MType mgeneral = (MType) createClass(uclassifier);
            mextansion.setExtendedClass(mgeneral);
            
            log.info("Adding generalization to " + mgeneral.getName());
            
            // ValueType
            mextansion.getModifications().addAll(getModificationsFromValueType(udatatype));
            mextansion.setOwningClass(mtype);
            // Add modifications (ValueType stereotype)
        }
        
        // Classifier::ownedUseCase not translated
        // Classifier::collaborationUse not translated
        // Classifier::redefinedClassifier not translated
        // Classifier::powertypeExtent not translated
        // DataType::ownedAttribute (SimProperty, SimConstant, SimVariable)
        for (Property uproperty : udatatype.getOwnedAttributes())
        {
            log.debug("Skipping datatype property " + print(uproperty));
        }
        return mtype;
    }
    
    private MComponent processProperty(Property uproperty) throws UMLModelErrorException
    {
        if (uproperty == null)
            throw new NullPointerException("You must provide a UML property");
        log.debug("Processing property " + print(uproperty));
        
        // Create and register the datatype
        MComponent mcomponent = createProperty(uproperty);
        
        // Element::ownedComment not supported yet
        // NamedElement::name
        ModelicaUtil.setName(mcomponent, SysMLUtil.getName(uproperty));
        if (uproperty.isStereotypeApplied(sysmlutil.getModelicaParameter()))
        {
            String name = (String) uproperty.getValue(sysmlutil.getModelicaParameter(), "name");
            if (name != null)
                ModelicaUtil.setName(mcomponent, name);
        }
        else if (uproperty.isStereotypeApplied(sysmlutil.getModelicaPort()))
        {
            String name = (String) uproperty.getValue(sysmlutil.getModelicaPort(), "name");
            if (name != null)
                ModelicaUtil.setName(mcomponent, name);
        }
        
        // NamedElement::visibility
        if (uproperty.getVisibility() == VisibilityKind.PUBLIC_LITERAL)
            mcomponent.setAccessControl(MAccessControl.PUBLIC);
        else
            mcomponent.setAccessControl(MAccessControl.PROTECTED);
            
        // Feature::isStatic not supported
        // MultiplicityElement::lowerValue not supported yet
        // MultiplicityElement::upperValue not supported yet
        // MultiplicityElement::isOrdered not supported
        // MultiplicityElement::isUnique not supported
        // TypedElement::type
        Type utype = uproperty.getType();
        
        if (utype == null)
            throw new UMLModelErrorException(r, "The property " + print(uproperty) + " must have a type");
        
        // setting type
        List<Property> sps = null;
        if (utype instanceof Class)
            sps = sysmlutil.getAllPhSProperties((Class) utype);
        if (sps != null && sps.size() > 0)
        {
            Property sp = sps.get(0);
            EnumerationLiteral direction = (EnumerationLiteral) sp.getValue(sysmlutil.getFlowProperty(), "direction");
            if (direction != null)
            {
                boolean conj = uproperty instanceof Port && ((Port) uproperty).isConjugated();
                String sdirection = direction.getName();
                if (("in".equals(sdirection) && !conj) || ("out".equals(sdirection) && conj))
                {
                    mcomponent.setDirection(MDirection.INPUT);
                    mcomponent.setType(createClass(sp.getType()));
                }
                else if (("out".equals(sdirection) && !conj) || ("in".equals(sdirection) && conj))
                {
                    mcomponent.setDirection(MDirection.OUTPUT);
                    mcomponent.setType(createClass(sp.getType()));
                }
                else
                    mcomponent.setType(createClass(utype));
            }
        }
        else if (utype.isStereotypeApplied(sysmlutil.getPropertySpecificType()))
        {
            // If Property-specific type, type by the classifier and do not
            // create
            // the type
            Classifier uclassifier = (Classifier) utype;
            if (!uclassifier.isStereotypeApplied(sysmlutil.getValueType()))
                log.warn("Can only process PropertySpecificType that are ValueTypes");
            else if (uclassifier.getGenerals().size() != 1)
                log.warn("The PropertySpecificType must have 1 general classifier to be processed");
            else if (uclassifier.getGenerals().get(0).isStereotypeApplied(sysmlutil.getPropertySpecificType()))
                log.warn("The general classifier of PropertySpecificType must not me a PropertySpecificType");
            else
            {
                Classifier general = uclassifier.getGenerals().get(0);
                mcomponent.setType(createClass(general));
                mcomponent.getModifications().addAll(getModificationsFromValueType((DataType) general));
            }
        }
        else
        {
            // Typical case
            MClass mtype = createClass(utype);
            mcomponent.setType(mtype);
        }
        // Modifications: ValueType
        // Modifications: Connectors
        // StructuralFeature::isReadOnly not supported
        // Property::subsettedProperty not supported
        // Property::redefinedProperty
        for (Property redefined : uproperty.getRedefinedProperties())
        {
            if (redefined.getName().equals(uproperty.getName()))
            {
                MComponent mcomponent2 = createProperty(redefined);
                mcomponent.setRedefinedComponent(mcomponent2);
                log.info("Assigning redefined property " + print(redefined));
                break;
            }
            log.warn("Redefined property has different name: " + print(redefined));
        }
        // Property::defaultValue
        // Property::qualifier not supported
        // Property::owningAssociation not supported
        // Property::association not supported
        // SimConstant
        if (uproperty.isReadOnly())
            mcomponent.setVariability(MVariability.CONSTANT);
        else if (uproperty.isStereotypeApplied(sysmlutil.getPhSConstant()))
            mcomponent.setVariability(MVariability.PARAMETER);
        else
        {
            // SimVariable
            if (uproperty.isStereotypeApplied(sysmlutil.getPhSVariable()))
            {
                Object o = uproperty.getValue(sysmlutil.getPhSVariable(), "isContinuous");
                if (o != null && o instanceof Boolean && ((Boolean) o).booleanValue() == false)
                    mcomponent.setVariability(MVariability.DISCRETE);
                o = uproperty.getValue(sysmlutil.getPhSVariable(), "isConserved");
                if (o != null && o instanceof Boolean && ((Boolean) o).booleanValue() == true)
                {
                    log.info("Setting component as flow");
                    mcomponent.setDataFlow(MDataFlow.FLOW);
                }
            }
        }
        // SimProperty
        if (sysmlutil.isPhSProperty(uproperty))
        {
            EnumerationLiteral el = (EnumerationLiteral) uproperty.getValue(sysmlutil.getFlowProperty(), "direction");
            if (el != null)
            {
                if (el.getName().equals("in"))
                    mcomponent.setDirection(MDirection.INPUT);
                else if (el.getName().equals("out"))
                    mcomponent.setDirection(MDirection.OUTPUT);
            }
        }
        // MultidimensionalElement::dimensions
        if (uproperty.isStereotypeApplied(sysmlutil.getMultidimensionalElement())
                && uproperty.getValue(sysmlutil.getMultidimensionalElement(), "dimension") instanceof List<?>)
        {
            @SuppressWarnings("unchecked")
            List<Object> dimensions = (List<Object>) uproperty.getValue(sysmlutil.getMultidimensionalElement(),
                    "dimension");
            for (Object o : dimensions)
                if (o instanceof Integer)
                {
                    Integer dim = (Integer) o;
                    mcomponent.getDimensions().add(dim);
                }
        }
        // modifications due to interface
        if (getTranslatedType(utype) == TranslatedType.INTERFACE)
        {
            for (Property uproperty2 : SysMLUtil.getAllAttributes(((Class) utype)))
            {
                if (uproperty2.isStereotypeApplied(sysmlutil.getModelicaParameter()))
                {
                    Object val = uproperty2.getValue(sysmlutil.getModelicaParameter(), "value");
                    if (val != null)
                    {
                        if (sysmlutil.getModelicaParameter().getAttribute("format", null) != null)
                        {
                            Object oformat = uproperty2.getValue(sysmlutil.getModelicaParameter(), "format");
                            if (oformat instanceof String)
                                val = ((String) oformat).replaceAll("%s", val.toString());
                        }
                        MModification mmod = ModelicaFactory.eINSTANCE.createMModification();
                        MComponent mc = createProperty(uproperty2);
                        if (mc != null)
                            mmod.getComponentPath().add(mc);
                        else
                            log.error("Can't add component " + uproperty2.getQualifiedName()
                                    + " to the modification in " + utype.getQualifiedName());
                        MExpressionValue mexp = ModelicaFactory.eINSTANCE.createMExpressionValue();
                        mexp.setValue(val.toString());
                        mmod.setAssignedValue(mexp);
                        mcomponent.getModifications().add(mmod);
                    }
                }
            }
        }
        return mcomponent;
    }
    
    private void addInitialValue(MComponent mcomponent, LinkedList<MComponent> linkedList, Slot slot)
            throws UMLModelErrorException
    {
        
        if (slot.getValues().size() == 0)
            log.warn("Slot " + print(slot) + " has no values");
        else
        {
            StructuralFeature sf = slot.getDefiningFeature();
            if (!(sf instanceof Property))
            {
                log.warn("The feature of slot " + print(slot) + " should be a property");
                return;
            }
            MComponent lastcomp = createProperty((Property) sf);
            if (lastcomp == null)
            {
                log.error("Can't get component for property " + sf.getQualifiedName() + " in modification");
                return;
            }
            if (slot.isStereotypeApplied(sysmlutil.getInitialValueReference())
                    && slot.getValue(sysmlutil.getInitialValueReference(), "propertyPath") instanceof List<?>)
            {
                @SuppressWarnings("unchecked")
                List<Property> references = (List<Property>) slot.getValue(sysmlutil.getInitialValueReference(),
                        "propertyPath");
                List<MComponent> components = new LinkedList<MComponent>();
                for (Property prop : references)
                {
                    MComponent comp = createProperty(prop);
                    if (comp != null)
                        components.add(comp);
                    else
                        log.error("Can't get component for property " + prop.getQualifiedName() + " in modification");
                }
                MModification mmodification = ModelicaFactory.eINSTANCE.createMModification();
                mmodification.getComponentPath().addAll(linkedList);
                mmodification.getComponentPath().add(lastcomp);
                mmodification.getComponentPath().add(lastcomp.getComponentByName("start"));
                mmodification.getAssignedReference().addAll(components);
                mcomponent.getModifications().add(mmodification);
                
                MBooleanValue mbv = ModelicaFactory.eINSTANCE.createMBooleanValue();
                mbv.setValue(true);
                
                MModification mmod2 = ModelicaFactory.eINSTANCE.createMModification();
                mmod2.getComponentPath().addAll(linkedList);
                mmod2.getComponentPath().add(lastcomp);
                mmod2.getComponentPath().add(lastcomp.getComponentByName("fixed"));
                mmod2.setAssignedValue(mbv);
                mcomponent.getModifications().add(mmod2);
                
            }
            else
            {
                ValueSpecification vs = slot.getValues().get(0);
                MDataValue mdv = getValue(vs);
                if (mdv != null)
                {
                    MModification mmodification = ModelicaFactory.eINSTANCE.createMModification();
                    mmodification.getComponentPath().addAll(linkedList);
                    mmodification.getComponentPath().add(lastcomp);
                    
                    // TODO: find a fix
                    if (!sf.isStereotypeApplied(sysmlutil.getModelicaParameter()))
                        mmodification.getComponentPath().add(lastcomp.getComponentByName("start"));
                    mmodification.setAssignedValue(mdv);
                    mcomponent.getModifications().add(mmodification);
                    
                    MBooleanValue mbv = ModelicaFactory.eINSTANCE.createMBooleanValue();
                    mbv.setValue(true);
                    
                    MModification mmod2 = ModelicaFactory.eINSTANCE.createMModification();
                    mmod2.getComponentPath().addAll(linkedList);
                    mmod2.getComponentPath().add(lastcomp);
                    mmod2.getComponentPath().add(lastcomp.getComponentByName("fixed"));
                    mmod2.setAssignedValue(mbv);
                    mcomponent.getModifications().add(mmod2);
                }
                else if (vs instanceof InstanceValue)
                {
                    LinkedList<MComponent> newlist = new LinkedList<MComponent>(linkedList);
                    newlist.add(lastcomp);
                    for (Slot subslot : ((InstanceValue) vs).getInstance().getSlots())
                    {
                        addInitialValue(mcomponent, newlist, subslot);
                    }
                }
                else
                    log.warn("Unable to process the value of " + print(vs)
                            + ". Use String, Boolean, Integer, Real or SysML initial values");
            }
        }
    }
    
    private List<MModification> getModificationsFromSlot(Slot uslot, Property definingfeature)
            throws UMLModelErrorException
    {
        List<MModification> ret = new ArrayList<MModification>();
        MComponent mcomponent = createProperty(definingfeature);
        if (mcomponent == null)
        {
            log.error("Can't get component for property " + definingfeature.getQualifiedName() + " for modification");
            return ret;
        }
        if (uslot.isStereotypeApplied(sysmlutil.getInitialValueReference())
                && uslot.getValue(sysmlutil.getInitialValueReference(), "propertyPath") instanceof List<?>)
        {
            @SuppressWarnings("unchecked")
            List<Property> propertyPath = (List<Property>) uslot.getValue(sysmlutil.getInitialValueReference(),
                    "propertyPath");
            List<MComponent> referencedcomponents = new LinkedList<MComponent>();
            for (Property prop : propertyPath)
            {
                if (sysmlutil.isPhSProperty(prop))
                    continue;
                MComponent comp = createProperty(prop);
                if (comp != null)
                    referencedcomponents.add(comp);
                else
                    log.error("Can't get component for property " + prop.getQualifiedName() + " in modification");
            }
            
            MModification mmod1 = ModelicaFactory.eINSTANCE.createMModification();
            mmod1.getComponentPath().add(mcomponent.getComponentByName("start"));
            mmod1.getAssignedReference().addAll(referencedcomponents);
            ret.add(mmod1);
            
            MBooleanValue mbv = ModelicaFactory.eINSTANCE.createMBooleanValue();
            mbv.setValue(true);
            
            MModification mmod2 = ModelicaFactory.eINSTANCE.createMModification();
            mmod2.getComponentPath().add(mcomponent.getComponentByName("fixed"));
            mmod2.setAssignedValue(mbv);
            ret.add(mmod2);
            
        }
        else
        {
            if (uslot.getValues().size() == 1)
            {
                ValueSpecification vs = uslot.getValues().get(0);
                MDataValue mdv = getValue(vs);
                if (mdv != null)
                {
                    MModification mmod1 = ModelicaFactory.eINSTANCE.createMModification();
                    // TODO: find a fix
                    if (!uslot.getDefiningFeature().isStereotypeApplied(sysmlutil.getModelicaParameter()))
                        mmod1.getComponentPath().add(mcomponent.getComponentByName("start"));
                    mmod1.setAssignedValue(mdv);
                    ret.add(mmod1);
                    
                    MBooleanValue mbv = ModelicaFactory.eINSTANCE.createMBooleanValue();
                    mbv.setValue(true);
                    
                    MModification mmod2 = ModelicaFactory.eINSTANCE.createMModification();
                    mmod2.getComponentPath().add(mcomponent.getComponentByName("fixed"));
                    mmod2.setAssignedValue(mbv);
                    ret.add(mmod2);
                }
                else if (vs instanceof InstanceValue)
                {
                    for (Slot subslot : ((InstanceValue) vs).getInstance().getSlots())
                    {
                        // should happen only is sim property
                        if (sysmlutil.isPhSProperty((Property) subslot.getDefiningFeature()))
                        {
                            List<MModification> mmodifications = getModificationsFromSlot(subslot, definingfeature);
                            
                            for (MModification mmodification : mmodifications)
                            {
                                ret.add(mmodification);
                            }
                        }
                        else
                        {
                            MComponent mcomponent2 = createProperty((Property) subslot.getDefiningFeature());
                            if (mcomponent2 == null)
                            {
                                log.error("Can't get component for property "
                                        + subslot.getDefiningFeature().getQualifiedName() + " in modification");
                                continue;
                            }
                            List<MModification> mmodifications = getModificationsFromSlot(subslot,
                                    (Property) subslot.getDefiningFeature());
                            for (MModification mmodification : mmodifications)
                            {
                                mmodification.getComponentPath().add(0, mcomponent2);
                                ret.add(mmodification);
                            }
                        }
                    }
                }
                else
                    log.warn("Unable to process the value of " + print(vs)
                            + ". Use String, Boolean, Integer, Real or SysML initial values");
                
            }
            else if (uslot.getValues().size() > 1)
            {
                StringBuilder sb = new StringBuilder();
                
                // array of when to open/close braces
                List<Integer> dimensions = new LinkedList<Integer>();
                
                if (uslot.isStereotypeApplied(sysmlutil.getMultidimensionalElement())
                        && uslot.getValue(sysmlutil.getMultidimensionalElement(), "dimension") instanceof List<?>)
                {
                    @SuppressWarnings("unchecked")
                    List<Object> dims = (List<Object>) uslot.getValue(sysmlutil.getMultidimensionalElement(),
                            "dimension");
                    for (int i = 0; i < dims.size(); i++)
                    {
                        int cumul = ((Integer) dims.get(i)).intValue();
                        for (int j = i + 1; j < dims.size(); j++)
                            cumul = cumul * ((Integer) dims.get(j)).intValue();
                        dimensions.add(Integer.valueOf(cumul));
                    }
                }
                else if (uslot.getDefiningFeature().isStereotypeApplied(sysmlutil.getMultidimensionalElement())
                        && uslot.getDefiningFeature().getValue(sysmlutil.getMultidimensionalElement(),
                                "dimension") instanceof List<?>)
                {
                    @SuppressWarnings("unchecked")
                    List<Object> dims = (List<Object>) uslot.getDefiningFeature()
                            .getValue(sysmlutil.getMultidimensionalElement(), "dimension");
                    for (int i = 0; i < dims.size(); i++)
                    {
                        int cumul = ((Integer) dims.get(i)).intValue();
                        for (int j = i + 1; j < dims.size(); j++)
                            cumul = cumul * ((Integer) dims.get(j)).intValue();
                        dimensions.add(Integer.valueOf(cumul));
                    }
                }
                else
                    dimensions.add(Integer.valueOf(uslot.getValues().size()));
                
                boolean bcom;
                for (int i = 0; i < uslot.getValues().size(); i++)
                {
                    bcom = true;
                    for (int dim : dimensions)
                        if (i % dim == 0)
                        {
                            if (i != 0)
                                sb.append(",");
                            sb.append("{");
                            bcom = false;
                        }
                    if (bcom)
                        sb.append(",");
                    sb.append(getValue(uslot.getValues().get(i)).serialize());
                    
                    for (int dim : dimensions)
                        if ((i + 1) % dim == 0)
                            sb.append("}");
                }
                MExpressionValue mexv = ModelicaFactory.eINSTANCE.createMExpressionValue();
                mexv.setValue(sb.toString());
                
                MModification mmod1 = ModelicaFactory.eINSTANCE.createMModification();
                
                // TODO: find a fix
                if (!uslot.getDefiningFeature().isStereotypeApplied(sysmlutil.getModelicaParameter()))
                    mmod1.getComponentPath().add(mcomponent.getComponentByName("start"));
                mmod1.setAssignedValue(mexv);
                ret.add(mmod1);
                
                MBooleanValue mbv = ModelicaFactory.eINSTANCE.createMBooleanValue();
                mbv.setValue(true);
                
                MModification mmod2 = ModelicaFactory.eINSTANCE.createMModification();
                mmod2.getComponentPath().add(mcomponent.getComponentByName("fixed"));
                mmod2.setAssignedValue(mbv);
                ret.add(mmod2);
                
            }
        }
        return ret;
    }
    
    private List<MModification> getModificationsFromValueType(DataType udatatype)
    {
        List<MModification> modifs = new LinkedList<MModification>();
        if (udatatype.isStereotypeApplied(sysmlutil.getValueType()))
        {
            // get the first primitive type found
            MType primitive = null;
            for (Classifier parent : udatatype.allParents())
            {
                if (parent == sysmlutil.getUMLString() || parent == sysmlutil.getUMLBoolean()
                        || parent == sysmlutil.getUMLDouble() || parent == sysmlutil.getUMLInteger()
                        || parent == sysmlutil.getSysMLString() || parent == sysmlutil.getSysMLBoolean()
                        || parent == sysmlutil.getSysMLInteger() || parent == sysmlutil.getSysMLDouble())
                    primitive = (MType) refs.get(getKey(parent));
            }
            if (primitive != null)
            {
                InstanceSpecification unit = (InstanceSpecification) udatatype.getValue(sysmlutil.getValueType(),
                        "unit");
                String usymbol = null;
                String uname = null;
                if (unit != null)
                    for (Slot unitslot : unit.getSlots())
                    {
                        if (unitslot.getDefiningFeature() == null)
                            continue;
                        if (unitslot.getDefiningFeature().getName().equals("symbol"))
                            for (ValueSpecification symbol : unitslot.getValues())
                                usymbol = symbol.stringValue();
                        if (unitslot.getDefiningFeature().getName().equals("quantityKind"))
                            for (ValueSpecification quantitykind : unitslot.getValues())
                                if (quantitykind instanceof InstanceValue)
                                    for (Slot quantitykindslot : ((InstanceValue) quantitykind).getInstance()
                                            .getSlots())
                                    {
                                        if (quantitykindslot.getDefiningFeature().getName().equals("name"))
                                            for (ValueSpecification name : quantitykindslot.getValues())
                                                uname = name.stringValue();
                                    }
                    }
                else
                    log.error("The value type " + print(udatatype) + " must have a unit: ");
                if (usymbol != null)
                {
                    MModification modif = ModelicaFactory.eINSTANCE.createMModification();
                    modif.getComponentPath().add(primitive.getComponentByName("unit"));
                    MStringValue svunit = ModelicaFactory.eINSTANCE.createMStringValue();
                    svunit.setValue(usymbol);
                    modif.setAssignedValue(svunit);
                    modifs.add(modif);
                }
                if (uname != null)
                {
                    MModification modif = ModelicaFactory.eINSTANCE.createMModification();
                    modif.getComponentPath().add(primitive.getComponentByName("quantity"));
                    MStringValue svquantity = ModelicaFactory.eINSTANCE.createMStringValue();
                    svquantity.setValue(uname);
                    modif.setAssignedValue(svquantity);
                    modifs.add(modif);
                }
            }
            else
                log.warn("The type " + print(udatatype)
                        + " has a ValueType stereotype, but can't retrieve a primitive (type) ancestor");
        }
        return modifs;
    }
    
    private MDataValue getValue(ValueSpecification vs) throws UMLModelErrorException
    {
        if (vs instanceof LiteralString)
        {
            MStringValue mstringvalue = ModelicaFactory.eINSTANCE.createMStringValue();
            mstringvalue.setValue(vs.stringValue());
            return mstringvalue;
        }
        if (vs instanceof LiteralBoolean)
        {
            MBooleanValue mbooleanvalue = ModelicaFactory.eINSTANCE.createMBooleanValue();
            mbooleanvalue.setValue(vs.booleanValue());
            return mbooleanvalue;
        }
        if (vs instanceof LiteralInteger)
        {
            MIntegerValue mintegervalue = ModelicaFactory.eINSTANCE.createMIntegerValue();
            mintegervalue.setValue(vs.integerValue());
            return mintegervalue;
        }
        if (vs instanceof LiteralReal)
        {
            MRealValue mrealvalue = ModelicaFactory.eINSTANCE.createMRealValue();
            mrealvalue.setValue(vs.realValue());
            return mrealvalue;
        }
        if (vs instanceof OpaqueExpression)
        {
            String val = getEquation(vs);
            if (val != null)
            {
                MExpressionValue mexpressionvalue = ModelicaFactory.eINSTANCE.createMExpressionValue();
                mexpressionvalue.setValue(val);
                return mexpressionvalue;
            }
        }
        return null;
    }
    
    public static String replace(String original, String pattern, HashMap<String, String> replacements)
    {
        if (original == null || replacements == null)
            return original;
        
        Pattern pat = Pattern.compile(pattern);
        Matcher mat = pat.matcher(original);
        StringBuffer sb = new StringBuffer(original.length());
        while (mat.find())
        {
            mat.appendReplacement(sb, "");
            String rep = replacements.get(mat.group());
            sb.append(rep == null ? mat.group() : rep);
        }
        mat.appendTail(sb);
        return sb.toString();
    }
}
