package com.engisis.sysphs.serialization.simulink;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.HashSet;
import java.util.Set;
import java.util.Stack;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.apache.log4j.Logger;

import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SPackage;
import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SFunction1;
import com.engisis.sysphs.language.simulink.SFunction1Block;
import com.engisis.sysphs.language.simulink.SFunction2;
import com.engisis.sysphs.language.simulink.SFunction2Block;
import com.engisis.sysphs.language.simulink.SLibrary;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SReference;
import com.engisis.sysphs.language.simulink.SSubsystem;
import com.engisis.sysphs.language.simulink.SSystem;
import com.engisis.sysphs.language.stateflow.SStateflow;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator.Format;
import com.engisis.sysphs.util.Dispatcher;

/**
 * Class in charge of serializing Simulink models.
 * 
 * @author barbau
 *
 */
public class SimulinkSerializer
{
    private static final Logger log = Logger.getLogger(SimulinkSerializer.class);
    /**
     * Output directory
     */
    private String dir;
    
    /**
     * Serializes the given Simulink model
     * 
     * @param directory
     *            output directory
     * @param smodel
     *            Model to serialize
     * @param slibrary
     *            Library to serialize
     * @param sstateflow
     *            Stateflow to serialize
     * @param spackage
     *            Simscape package to serialize
     * @param format
     *            Simulink format
     * @return the full path of the generated model file
     * @throws IOException
     */
    public String serialize(String directory, SModel smodel, SLibrary slibrary, SStateflow sstateflow,
            SPackage spackage, SysMLToSimulinkTranslator.Format format) throws IOException
    {
        if (directory == null)
            throw new IllegalArgumentException("The directory can't be null");
        if (smodel == null)
            throw new IllegalArgumentException("The model can't be null");
        if (slibrary == null)
            throw new IllegalArgumentException("The library can't be null");
        if (sstateflow == null)
            throw new IllegalArgumentException("The stateflow can't be null");
        
        dir = directory;
        if (slibrary.getSystem().getOwnedBlocks().size() != 0 || sstateflow.getMachine().getChart().size() != 0)
        {
            if (format == Format.XML)
                saveXML(slibrary, sstateflow);
            else
                saveMDL(slibrary, sstateflow);
        }
        String filename = null;
        if (format == Format.XML)
            filename = saveXML(smodel, null);
        else
            filename = saveMDL(smodel, null);
        
        saveSFunctions(smodel);
        
        if (spackage != null)
            saveSimscape(spackage);
        return filename;
    }
    
    /**
     * Saves the Model and Stateflow in MDL
     * 
     * @param smodel
     *            model to serialize
     * @param sstateflow
     *            stateflow to serialize
     * @return full path of the model file
     */
    private String saveMDL(SModel smodel, SStateflow sstateflow)
    {
        String filename = dir + File.separator + smodel.getName() + ".mdl";
        try
        {
            log.debug("Saving model to " + filename);
            FileWriter fw = new FileWriter(filename);
            // fw.write(slibrary.serialize());
            Dispatcher dispatcher = new Dispatcher();
            dispatcher.addVisitor(new SimulinkMDLWriter(fw));
            dispatcher.addVisitor(new SimulinkStateflowMDLWriter(fw));
            dispatcher.addVisitor(new SimulinkSimscapeMDLWriter(fw));
            smodel.accept(dispatcher);
            
            // write stateflow if applicable
            if (sstateflow != null && sstateflow.getMachine().getChart().size() != 0)
            {
                Dispatcher dispatcher2 = new Dispatcher();
                dispatcher2.addVisitor(new StateflowMDLWriter(fw));
                sstateflow.accept(dispatcher2);
            }
            fw.close();
        }
        catch (Exception e)
        {
            log.error("Can't save the file at the location " + filename, e);
        }
        return filename;
    }
    
    /**
     * Saves the Model and Stateflow in XML
     * 
     * @param smodel
     *            model to serialize
     * @param sstateflow
     *            stateflow to serialize
     * @return full path of the model file
     */
    private String saveXML(SModel smodel, SStateflow sstateflow)
    {
        String filename = dir + File.separator + smodel.getName() + ".slx";
        try
        {
            log.debug("Saving model to " + filename);
            XMLOutputFactory xmlof = XMLOutputFactory.newFactory();
            StringWriter sw = new StringWriter();
            XMLStreamWriter xmlsw = xmlof.createXMLStreamWriter(sw);
            xmlsw.writeStartDocument("utf-8", "1.0");
            xmlsw.writeStartElement("ModelInformation");
            xmlsw.writeAttribute("Version", "0.9");
            
            Dispatcher dispatcher = new Dispatcher();
            dispatcher.addVisitor(new SimulinkXMLWriter(xmlsw));
            dispatcher.addVisitor(new SimulinkStateflowXMLWriter(xmlsw));
            dispatcher.addVisitor(new SimulinkSimscapeXMLWriter(xmlsw));
            smodel.accept(dispatcher);
            
            // write stateflow if applicable
            if (sstateflow != null && sstateflow.getMachine().getChart().size() != 0)
            {
                Dispatcher dispatcher2 = new Dispatcher();
                dispatcher2.addVisitor(new StateflowXMLWriter(xmlsw));
                sstateflow.accept(dispatcher2);
            }
            
            xmlsw.writeEndElement();
            xmlsw.writeEndDocument();
            xmlsw.flush();
            xmlsw.close();
            
            FileOutputStream fos = new FileOutputStream(filename);
            BufferedOutputStream bod = new BufferedOutputStream(fos);
            ZipOutputStream zos = new ZipOutputStream(bod);
            
            // main file
            zos.putNextEntry(new ZipEntry("simulink/blockdiagram.xml"));
            zos.write(sw.toString().getBytes());
            zos.closeEntry();
            
            // content type
            zos.putNextEntry(new ZipEntry("[Content_Types].xml"));
            
            zos.write("<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?>".getBytes());
            zos.write("<Types xmlns=\"http://schemas.openxmlformats.org/package/2006/content-types\">".getBytes());
            zos.write("<Default ContentType=\"image/png\" Extension=\"png\"/>".getBytes());
            zos.write("<Default ContentType=\"application/vnd.openxmlformats-package.relationships+xml\" Extension=\"rels\"/>"
                    .getBytes());
            zos.write("<Default ContentType=\"application/vnd.mathworks.simulink.mdl+xml\" Extension=\"xml\"/>"
                    .getBytes());
            zos.write("<Override ContentType=\"application/vnd.openxmlformats-package.core-properties+xml\" PartName=\"/metadata/coreProperties.xml\"/>"
                    .getBytes());
            zos.write("<Override ContentType=\"application/vnd.mathworks.package.coreProperties+xml\" PartName=\"/metadata/mwcoreProperties.xml\"/>"
                    .getBytes());
            zos.write("</Types>".getBytes());
            zos.closeEntry();
            
            // rels
            zos.putNextEntry(new ZipEntry("_rels/.rels"));
            zos.write("<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?>".getBytes());
            zos.write("<Relationships xmlns=\"http://schemas.openxmlformats.org/package/2006/relationships\">"
                    .getBytes());
            // zos.write("<Relationship Id=\"Thumbnail\" Target=\"metadata/thumbnail.png\" Type=\"http://schemas.openxmlformats.org/package/2006/relationships/metadata/thumbnail\"/>".getBytes());
            zos.write("<Relationship Id=\"blockDiagram\" Target=\"simulink/blockdiagram.xml\" Type=\"http://schemas.mathworks.com/simulink/2010/relationships/blockDiagram\"/>"
                    .getBytes());
            // zos.write("<Relationship Id=\"coreprops\" Target=\"metadata/coreProperties.xml\" Type=\"http://schemas.openxmlformats.org/package/2006/relationships/metadata/core-properties\"/>".getBytes());
            // zos.write("<Relationship Id=\"rId1\" Target=\"metadata/mwcoreProperties.xml\" Type=\"http://schemas.mathworks.com/package/2012/relationships/coreProperties\"/>".getBytes());
            zos.write("</Relationships>".getBytes());
            
            zos.close();
        }
        catch (IOException e)
        {
            log.error("Can't save the file at the location " + filename, e);
        }
        catch (XMLStreamException e)
        {
            log.error("Can't save the file at the location " + filename, e);
        }
        return filename;
    }
    
    /**
     * Saves the S-Functions in the given model
     * 
     * @param smodel
     *            model in which the S-Functions are saved
     */
    private void saveSFunctions(SModel smodel)
    {
        Stack<SSystem> ssystems = new Stack<SSystem>();
        Set<SSystem> processed = new HashSet<SSystem>();
        ssystems.add(smodel.getSystem());
        while (!ssystems.isEmpty())
        {
            SSystem ss = ssystems.pop();
            processed.add(ss);
            for (SBlock sb : ss.getOwnedBlocks())
            {
                if (sb instanceof SFunction1Block)
                {
                    SFunction1 sf1 = ((SFunction1Block) sb).getSfunction();
                    if (sf1.getName().equals("sf_sfun"))
                        continue;
                    String target = dir + File.separator + sf1.getName() + ".m";
                    log.debug("Saving sfunction to " + target);
                    try
                    {
                        FileWriter fw = new FileWriter(target);
                        Dispatcher dispatcher = new Dispatcher();
                        dispatcher.addVisitor(new SimulinkSF1Writer(fw));
                        sf1.accept(dispatcher);
                        fw.close();
                    }
                    catch (IOException e)
                    {
                        log.error("Unable to save SFunction " + target, e);
                    }
                }
                else if (sb instanceof SFunction2Block)
                {
                    SFunction2 sf2 = ((SFunction2Block) sb).getSfunction();
                    String target = dir + File.separator + sf2.getName() + ".m";
                    log.debug("Saving sfunction to " + target);
                    try
                    {
                        FileWriter fw = new FileWriter(target);
                        Dispatcher dispatcher = new Dispatcher();
                        dispatcher.addVisitor(new SimulinkSF2Writer(fw));
                        sf2.accept(dispatcher);
                        fw.close();
                    }
                    catch (IOException e)
                    {
                        log.error("Unable to save SFunction " + target, e);
                    }
                }
                else if (sb instanceof SSubsystem)
                {
                    SSystem ss2 = ((SSubsystem) sb).getSystem();
                    if (!processed.contains(ss2))
                        ssystems.push(ss2);
                }
                else if (sb instanceof SReference)
                {
                    SSystem ss2 = ((SReference) sb).getSystem();
                    if (!processed.contains(ss2))
                        ssystems.push(ss2);
                }
            }
        }
    }
    
    /**
     * Saves the content of the package
     * 
     * @param spackage
     *            package to save
     * @throws IOException
     */
    private void saveSimscape(SPackage spackage) throws IOException
    {
        String dtarget = dir + File.separator + ("+" + spackage.getName());
        File dir = new File(dtarget);
        if (dir.exists())
        {
            if (!dir.isDirectory())
                throw new IOException("The target must be a directory:" + dtarget);
            for (File file : dir.listFiles())
            {
                if (file.isFile() && file.getName().endsWith(".ssc"))
                    if (!file.delete())
                        log.warn("Unable to delete file " + file.getAbsolutePath());
            }
        }
        else if (!dir.mkdir())
            throw new IOException("Unable to create the directory " + dir.getAbsolutePath());
        
        for (SDomain sdomain : spackage.getDomains())
        {
            try
            {
                String target = dtarget + File.separator + sdomain.getName() + ".ssc";
                log.info("Saving domain into " + target);
                FileWriter fw = new FileWriter(target);
                StringWriter sw = new StringWriter();
                Dispatcher dispatcher = new Dispatcher();
                dispatcher.addVisitor(new SimscapeWriter(sw));
                sdomain.accept(dispatcher);
                fw.write(sw.toString());
                fw.close();
                log.info("Saved domain into " + target);
            }
            catch (IOException e)
            {
                log.error("Can't save the element " + sdomain.getName(), e);
                throw e;
            }
        }
        for (SComponent scomponent : spackage.getComponents())
        {
            try
            {
                String target = dtarget + File.separator + scomponent.getName() + ".ssc";
                log.info("Saving component into " + target);
                FileWriter fw = new FileWriter(target);
                StringWriter sw = new StringWriter();
                Dispatcher dispatcher = new Dispatcher();
                dispatcher.addVisitor(new SimscapeWriter(sw));
                scomponent.accept(dispatcher);
                fw.write(sw.toString());
                fw.close();
                log.info("Saved component into " + target);
            }
            catch (IOException e)
            {
                log.error("Can't save the element " + scomponent.getName(), e);
                throw e;
            }
        }
    }
}
