package com.engisis.sysphs.deserialization.simulink;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Hashtable;
import java.util.LinkedList;

import org.apache.log4j.Logger;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SimulinkFactory;
import com.engisis.sysphs.language.stateflow.SChart;
import com.engisis.sysphs.language.stateflow.SChartBlock;
import com.engisis.sysphs.language.stateflow.SData;
import com.engisis.sysphs.language.stateflow.SDataScope;
import com.engisis.sysphs.language.stateflow.SInstance;
import com.engisis.sysphs.language.stateflow.SMachine;
import com.engisis.sysphs.language.stateflow.SState;
import com.engisis.sysphs.language.stateflow.SStateflow;
import com.engisis.sysphs.language.stateflow.STarget;
import com.engisis.sysphs.language.stateflow.STransition;
import com.engisis.sysphs.language.stateflow.StateflowFactory;
import com.engisis.sysphs.translation.simulink.SimulinkUtil;
import com.engisis.sysphs.util.Util;
import com.engisis.sysphs.util.XMLReader;

/**
 * First-pass parser of a Stateflow XML file.
 * 
 * @author barbau
 *
 */
public class StateflowXMLReaderPass1 extends XMLReader implements SimulinkReader
{
    private static final Logger log = Logger.getLogger(StateflowXMLReaderPass1.class);
    
    /**
     * First-pass parser
     */
    private SimulinkXMLReaderPass1 definition;
    
    /**
     * Root Stateflow object
     */
    private SStateflow sstateflow;
    
    /**
     * Collection of Simulink references
     */
    private Collection<SimulinkReference> references;
    /**
     * Map with IDs as keys, and Simulink elements as values
     */
    private Hashtable<Integer, SElement> ids;
    
    /**
     * Input stream of the XML file
     */
    private InputStream is;
    
    /**
     * Constructs a first-pass Stateflow reader
     * 
     * @param definition
     *            First-pass Simulink reader
     * @param inputStream
     *            Input stream
     */
    public StateflowXMLReaderPass1(SimulinkXMLReaderPass1 definition, InputStream inputStream)
    {
        this.definition = definition;
        references = new LinkedList<SimulinkReference>();
        ids = new Hashtable<Integer, SElement>();
        is = inputStream;
    }
    
    /**
     * Returns the root Stateflow object
     * 
     * @return the root Stateflow object
     */
    public SStateflow getStateflow()
    {
        return sstateflow;
    }
    
    @Override
    public void process()
    {
        // open xml
        try
        {
            XmlPullParserFactory xmlppf = XmlPullParserFactory.newInstance();
            xmlpp = xmlppf.newPullParser();
            xmlpp.setInput(is, null);
            lookForElement("ModelInformation");
            processModelInformation();
        }
        catch (XmlPullParserException e)
        {
            log.error(e);
        }
        catch (IOException e)
        {
            log.error(e);
        }
        finally
        {
            try
            {
                is.close();
            }
            catch (IOException e)
            {
            }
        }
        
    }
    
    /**
     * Processes a Model or Library element
     * 
     * @throws XmlPullParserException
     * @throws IOException
     */
    private void processModelInformation() throws XmlPullParserException, IOException
    {
        log.info("Defining Model");
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        String ln2;
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                ln2 = xmlpp.getName();
                if (ln2.equals("Stateflow"))
                {
                    sstateflow = processStateflow();
                }
                else
                    skipToEnd();
            }
            
        }
    }
    
    /**
     * Processes a Stateflow element
     * 
     * @return a Stateflow Object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SStateflow processStateflow() throws XmlPullParserException, IOException
    {
        log.info("Defining Stateflow");
        SStateflow sstateflow = StateflowFactory.eINSTANCE.createSStateflow();
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                String name = xmlpp.getName();
                if (name.equals("machine"))
                {
                    SMachine sma = processMachine();
                    sstateflow.setMachine(sma);
                }
                else if (name.equals("instance"))
                {
                    
                    SInstance sinstance = processInstance();
                    sstateflow.getInstance().add(sinstance);
                }
                else
                    skipToEnd();
            }
        }
        return sstateflow;
    }
    
    /**
     * Processes an Instance element
     * 
     * @return an Instance object
     */
    private SInstance processInstance()
    {
        log.info("Defining Instance");
        SInstance sinstance = StateflowFactory.eINSTANCE.createSInstance();
        associateID(sinstance);
        
        return sinstance;
    }
    
    /**
     * Processes a Machine element
     * 
     * @return a Machine object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SMachine processMachine() throws XmlPullParserException, IOException
    {
        log.info("Defining Machine");
        SMachine smachine = StateflowFactory.eINSTANCE.createSMachine();
        String id = xmlpp.getAttributeValue(null, "id");
        smachine.setId(Util.toInt(id, 0));
        
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                String name = xmlpp.getName();
                if (name.equals("P"))
                {
                    // nothing
                }
                else if (name.equals("Children"))
                {
                    int depth2 = xmlpp.getDepth();
                    while (et != XmlPullParser.END_TAG || depth2 != xmlpp.getDepth())
                    {
                        et = xmlpp.next();
                        if (et == XmlPullParser.START_TAG)
                        {
                            name = xmlpp.getName();
                            if (name.equals("target"))
                            {
                                STarget starget = processTarget();
                                smachine.setTarget(starget);
                            }
                            else if (name.equals("chart"))
                            {
                                SChart schart = processChart();
                                smachine.getChart().add(schart);
                            }
                            else
                                skipToEnd();
                        }
                    }
                }
                else
                    skipToEnd();
            }
        }
        
        return smachine;
    }
    
    /**
     * Processes a Target element
     * 
     * @return a Target object
     */
    private STarget processTarget()
    {
        log.info("Defining Target");
        STarget starget = StateflowFactory.eINSTANCE.createSTarget();
        associateID(starget);
        return starget;
    }
    
    /**
     * Processes a Chart element
     * 
     * @return a Chart object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SChart processChart() throws XmlPullParserException, IOException
    {
        log.info("Defining Chart");
        SChart schart = StateflowFactory.eINSTANCE.createSChart();
        associateID(schart);
        int depth = xmlpp.getDepth();
        int et = xmlpp.getEventType();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                String name = xmlpp.getName();
                if (name.equals("P"))
                {
                    String attname = xmlpp.getAttributeValue(null, "Name");
                    if ("name".equals(attname))
                    {
                        SimulinkUtil.setName(schart, xmlpp.nextText());
                        if (definition != null && definition.getModel() != null)
                        {
                            SBlock sb = definition.getModel().getSystem().getBlockByName(schart.getName().split("/"));
                            if (sb instanceof SChartBlock)
                            {
                                log.info("Associating chart block with chart " + schart.getName());
                                ((SChartBlock) sb).setChart(schart);
                            }
                            else
                            {
                                log.error("Unable to locate the chart block " + schart.getName() + ", " + sb);
                            }
                        }
                        else
                            log.error("Unable to get simulink definition or its root model");
                    }
                }
                else if (name.equals("Children"))
                {
                    int depth2 = xmlpp.getDepth();
                    while (et != XmlPullParser.END_TAG || depth2 != xmlpp.getDepth())
                    {
                        et = xmlpp.next();
                        if (et == XmlPullParser.START_TAG)
                        {
                            String name2 = xmlpp.getName();
                            if (name2.equals("state"))
                            {
                                SState sstate = processState();
                                schart.getTreeNode().add(sstate);
                            }
                            else if (name2.equals("data"))
                            {
                                SData sdata = processData();
                                schart.getLinkNode().add(sdata);
                            }
                            else if (name2.equals("transition"))
                            {
                                STransition stransition = processTransition();
                                schart.getLinkNode().add(stransition);
                            }
                            else
                                skipToEnd();
                        }
                    }
                }
                else
                    skipToEnd();
            }
        }
        return schart;
    }
    
    /**
     * Processes a State element
     * 
     * @return a State object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SState processState() throws XmlPullParserException, IOException
    {
        log.info("Defining State");
        SState sstate = StateflowFactory.eINSTANCE.createSState();
        associateSSID(sstate);
        
        int depth = xmlpp.getDepth();
        int et = xmlpp.getEventType();
        while (et != XmlPullParser.END_TAG || xmlpp.getDepth() != depth)
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                if (xmlpp.getName().equals("P"))
                {
                    String name = xmlpp.getAttributeValue(null, "Name");
                    if ("labelString".equals(name))
                    {
                        String val = xmlpp.nextText().trim();
                        String[] tab = val.split("\n");
                        SimulinkUtil.setName(sstate, tab[0]);
                        StringBuilder sb = new StringBuilder();
                        for (int i = 1; i < tab.length; i++)
                        {
                            if (i != 1)
                                sb.append("\n");
                            sb.append(tab[i].trim());
                        }
                        sstate.setLabel(sb.toString());
                    }
                }
                else
                    skipToEnd();
            }
        }
        return sstate;
    }
    
    /**
     * Process a Data element
     * 
     * @return a Data object
     * @throws XmlPullParserException
     * @throws IOException
     */
    @SuppressWarnings({"boxing" })
    private SData processData() throws XmlPullParserException, IOException
    {
        log.info("Defining Data");
        SData sdata = StateflowFactory.eINSTANCE.createSData();
        associateSSID(sdata);
        
        SimulinkUtil.setName(sdata, xmlpp.getAttributeValue(null, "name"));
        
        int depth = xmlpp.getDepth();
        int et = xmlpp.getEventType();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                String name = xmlpp.getName();
                if (name.equals("P"))
                {
                    name = xmlpp.getAttributeValue(null, "Name");
                    if ("scope".equals(name))
                    {
                        String val = xmlpp.nextText().trim();
                        if (val.equals("INPUT_DATA"))
                            sdata.setScope(SDataScope.INPUT);
                        else if (val.equals("OUTPUT_DATA"))
                            sdata.setScope(SDataScope.OUTPUT);
                        else if (val.equals("CONSTANT_DATA"))
                            sdata.setScope(SDataScope.CONSTANT);
                    }
                }
                else if (name.equals("props"))
                {
                    int depth2 = xmlpp.getDepth();
                    while (et != XmlPullParser.END_TAG || depth2 != xmlpp.getDepth())
                    {
                        et = xmlpp.next();
                        
                        if (xmlpp.getName().equals("P"))
                        {
                            name = xmlpp.getAttributeValue(null, "Name");
                            
                            if ("initialValue".equals(name))
                            {
                                String val = xmlpp.nextText().trim();
                                // fix
                                Double d = null;
                                try
                                {
                                    log.info("Setting initial value");
                                    d = Double.valueOf(val);
                                    SDoubleValue sdv = SimulinkFactory.eINSTANCE.createSDoubleValue();
                                    sdv.setValue(d);
                                    sdata.setValue(sdv);
                                }
                                catch (NumberFormatException e)
                                {
                                }
                            }
                            else if ("dataType".equals("name"))
                            {
                                // String val = xmlpp.nextText().trim();
                                // if ("double".equals(val))
                                // sdata.setDataType(newDataType);
                                // else if ("int32".equals(val))
                                // sdata.setDataType(newDataType);
                                // else if ("boolean".equals(val))
                                // sdata.setDataType(newDataType);
                            }
                        }
                    }
                    // if (sdata.getDataType() == null)
                    // sdata.setDataType(newDataType);
                }
                else
                    skipToEnd();
            }
        }
        return sdata;
    }
    
    /**
     * Processes a Transition element
     * 
     * @return a Transition object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private STransition processTransition() throws XmlPullParserException, IOException
    {
        log.info("Defining Transition");
        STransition stransition = StateflowFactory.eINSTANCE.createSTransition();
        associateSSID(stransition);
        
        int depth = xmlpp.getDepth();
        int et = xmlpp.getEventType();
        
        while (et != XmlPullParser.END_TAG || xmlpp.getDepth() != depth)
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                String name = xmlpp.getName();
                if (name.equals("P"))
                {
                    name = xmlpp.getAttributeValue(null, "Name");
                    String val = xmlpp.nextText();
                    if ("labelString".equals(name))
                    {
                        stransition.setLabel(val);
                    }
                }
                else
                    skipToEnd();
            }
        }
        
        return stransition;
    }
    
    /**
     * Gives an ID to the specified element
     * 
     * @param selement
     *            element that needs an ID
     */
    private void associateID(SElement selement)
    {
        String s = xmlpp.getAttributeValue(null, "id");
        if (s == null || s.length() == 0)
            return;
        try
        {
            ids.put(Integer.valueOf(s), selement);
        }
        catch (NumberFormatException e)
        {
        }
        return;
        
    }
    
    /**
     * Gives an SSID to the specified element
     * 
     * @param selement
     *            element that needs an SSID
     */
    private void associateSSID(SElement selement)
    {
        String s = xmlpp.getAttributeValue(null, "SSID");
        if (s == null || s.length() == 0)
            return;
        ids.put(Util.toInteger(s, new Integer(0)), selement);
        return;
        
    }
    
    @Override
    public Collection<SimulinkReference> getReferences()
    {
        return references;
    }
    
    /**
     * Returns the map between IDs and Simulink objects
     * 
     * @return map between integer IDs and Simulink objects
     */
    public Hashtable<Integer, SElement> getIDs()
    {
        return ids;
    }
    
    @Override
    public int getPriority()
    {
        return 5;
    }
    
    @Override
    public SElement getRootElement()
    {
        return null;
    }
    
}
