package com.engisis.sysphs.deserialization.simulink;

import java.util.Map;

import org.apache.log4j.Logger;

import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SComponentReference;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SNode;
import com.engisis.sysphs.language.simscape.SPhysicalBlock;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SFunction1;
import com.engisis.sysphs.language.simulink.SFunction1Block;
import com.engisis.sysphs.language.simulink.SFunction2;
import com.engisis.sysphs.language.simulink.SFunction2Block;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SReference;
import com.engisis.sysphs.language.simulink.SSubsystem;

/**
 * Represents a reference to a Simulink element
 * 
 * @author barbau
 *
 */
public class SimulinkReference
{
    private static final Logger log = Logger.getLogger(SimulinkReference.class);
    
    /**
     * Type of the reference
     * 
     * @author barbau
     *
     */
    public enum ReferenceType
    {
        /**
         * Reference to a Simulink block
         */
        BLOCK, 
        /**
         * Reference to a Level-1 S-Function
         */
        SF1, 
        /**
         * Reference to a Level-2 S-Function
         */
        SF2, 
        /**
         * Reference to a Simscape component
         */
        COMPONENT, 
        /**
         * Reference to a Simscape domain
         */
        DOMAIN;
    }
    
    /**
     * String representing the reference
     */
    private String ref;
    /**
     * Type of reference
     */
    private ReferenceType type;
    /**
     * Element that has to be resolved
     */
    private SElement selement;
    
    /**
     * Constructor
     * 
     * @param selement element to resolve
     * @param ref reference
     * @param type type of reference
     */
    public SimulinkReference(SElement selement, String ref, ReferenceType type)
    {
        this.selement = selement;
        this.ref = ref;
        this.type = type;
    }
    
    /**
     * Returns a string giving the location (path, file name without extension)
     * of the reference
     * 
     * @return file path, without extension
     */
    public String getLocation()
    {
        switch (type)
        {
        case BLOCK:
            return ref.split("/")[0];
        case SF1:
            return ref;
        case SF2:
            return ref;
        case COMPONENT:
            return "+" + ref;
        case DOMAIN:
            return "+" + ref;
        default:
            return null;
        }
    }
    
    /**
     * Returns the type of reference
     * 
     * @return type of the reference
     */
    public ReferenceType getReferenceType()
    {
        return type;
    }
    
    /**
     * Resolves the reference
     * 
     * @param mapFiles
     */
    public void resolve(Map<String, SimulinkFile> mapFiles)
    {
        SimulinkFile sf = mapFiles.get(getLocation());
        if (sf == null)
        {
            log.error("No file for location " + getLocation());
            return;
        }
        SElement root = sf.getRootElement();
        
        if (root instanceof SModel)
        {
            SElement ref = ((SModel) root).getBlockByName(this.ref.split("/"));
            if (ref instanceof SSubsystem)
            {
                if (selement instanceof SReference)
                {
                    ((SReference) selement).setSystem(((SSubsystem) ref).getSystem());
                    log.info("Resolved reference " + selement + " to " + ((SSubsystem) ref).getSystem());
                }
                else
                    log.error("Only a Reference block can refer to another block: " + selement);
            }
            else
                log.error("Block " + ref + " is expected to be a subsystem");
        }
        else if (root instanceof SFunction1)
        {
            if (selement instanceof SFunction1Block)
            {
                ((SFunction1Block) selement).setSfunction((SFunction1) root);
                log.info("Resolved sfunction1 " + selement + " to " + root);
            }
            else
                log.error("Only a SFunction block can refer to an SFunction: " + selement);
            
        }
        else if (root instanceof SFunction2)
        {
            if (selement instanceof SFunction2Block)
            {
                ((SFunction2Block) selement).setSfunction((SFunction2) root);
                log.info("Resolved sfunction2 " + selement + " to " + root);
            }
            else
                log.error("Only a SFunction block can refer to an SFunction: " + selement);
        }
        else if (root instanceof SDomain)
        {
            if (selement instanceof SNode)
            {
                ((SNode) selement).setDomain((SDomain) root);
                log.info("Resolved component " + selement + " to " + root);
            }
            else
                log.error("Only a Node can refer to a domain: " + selement);
        }
        else if (root instanceof SComponent)
        {
            if (selement instanceof SComponent)
            {
                ((SComponent) selement).setBaseComponent((SComponent) root);
                log.info("Resolved component " + selement + " to " + root);
            }
            else if (selement instanceof SComponentReference)
            {
                ((SComponentReference) selement).setComponent((SComponent) root);
                log.info("Resolved component " + selement + " to " + root);
            }
            else if (selement instanceof SPhysicalBlock)
            {
                ((SPhysicalBlock) selement).setComponent((SComponent) root);
                log.info("Resolved component " + selement + " to " + root);
            }
            else
                log.error("Only a Component, a component reference, or a physical block can refer to a Component: "
                        + selement);
        }
        else
        {
            log.error("Root not recognized: " + root);
        }
    }
    
    /**
     * Returns the resolved element
     * 
     * @return resolved element
     */
    public SElement getElement()
    {
        return selement;
    }
}
