package com.engisis.sysphs.deserialization.simulink;

import com.engisis.sysphs.deserialization.simulink.SimulinkReference.ReferenceType;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.stateflow.SStateflow;
import com.engisis.sysphs.translation.simulink.SimulinkUtil;
import com.engisis.sysphs.util.ANTLRErrorListener;
import com.engisis.sysphs.generation.simulink.MATLABLexer;
import com.engisis.sysphs.generation.simulink.MATLABParser;
import com.engisis.sysphs.generation.simulink.MDLLexer;
import com.engisis.sysphs.generation.simulink.MDLParser;
import com.engisis.sysphs.generation.simulink.SimscapeLexer;
import com.engisis.sysphs.generation.simulink.SimscapeParser;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.Set;
import java.util.Stack;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.antlr.v4.runtime.ANTLRFileStream;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTree;
import org.apache.log4j.Logger;

/**
 * Class in charge of deserializing Simulink files and their dependencies. The
 * deserialization process works by calling various readers. A Simulink file may
 * have several readers, usually one for each pass.
 * 
 * @author barbau
 *
 */
public class SimulinkDeserializer
{
    private static final Logger log = Logger.getLogger(SimulinkDeserializer.class);
    
    /**
     * Map with file paths as keys, and Simulink files as values
     */
    private Hashtable<String, SimulinkFile> mapFiles;
    /**
     * Queue of first-pass readers
     */
    private Queue<SimulinkReader> p1queue;
    /**
     * Queue of second-pass readers
     */
    private Queue<SimulinkReader> p2queue;
    /**
     * Collection of readers already processed
     */
    private Set<SimulinkReader> processed;
    /**
     * Root Simulink model
     */
    private SModel rootmodel;
    
    /**
     * List of Stateflow objects
     */
    private List<SStateflow> lstateflow;
    /**
     * Objects to close
     */
    private Stack<Closeable> toclose;
    
    /**
     * Map with Simulink objects as keys, and syntax tree nodes as values
     */
    private Hashtable<SElement, ParseTree> expressions;
    
    /**
     * Working directory
     */
    private File wd;
    
    /**
     * Returns a maps with Simulink objects as keys, and their syntax tree nodes
     * as values
     * 
     * @return map from Simulink objects to syntax tree nodes
     */
    public Hashtable<SElement, ParseTree> getExpressions()
    {
        return expressions;
    }
    
    /**
     * Initiates the deserialization
     * 
     * @param filename
     *            path to the Simulink file to deserialize
     * @throws IOException
     */
    public void deserialize(String filename) throws IOException
    {
        toclose = new Stack<Closeable>();
        rootmodel = null;
        lstateflow = new LinkedList<SStateflow>();
        mapFiles = new Hashtable<String, SimulinkFile>();
        
        // queue ordered by priority
        p1queue = new PriorityQueue<SimulinkReader>(new SimulinkReaderComparator());
        p2queue = new PriorityQueue<SimulinkReader>(new SimulinkReaderComparator());
        
        processed = new HashSet<SimulinkReader>();
        
        expressions = new Hashtable<SElement, ParseTree>();
        
        File f = new File(filename);
        wd = f.getParentFile();
        
        SimulinkFile sf = new SimulinkFile(filename, Format.SIM);
        // simulink file name rules disallow multiple "."
        String shortname = f.getName().split("\\.")[0];
        mapFiles.put(shortname, sf);
        
        populateReaders(sf);
        
        SimulinkReader sr = null;
        while (!p1queue.isEmpty())
        {
            sr = p1queue.poll();
            processed.add(sr);
            log.debug("Popped " + sr);
            // if references are present, resolve them
            if (sr.getReferences() != null)
                for (SimulinkReference ref : sr.getReferences())
                    ref.resolve(mapFiles);
            sr.process();
            
            if (sr instanceof SimulinkXMLReaderPass1)
            {
                SModel smodel = ((SimulinkXMLReaderPass1) sr).getModel();
                SimulinkUtil.setName(smodel, shortname);
                if (rootmodel == null)
                    rootmodel = smodel;
            }
            else if (sr instanceof StateflowXMLReaderPass1)
                lstateflow.add(((StateflowXMLReaderPass1) sr).getStateflow());
            
            if (sr.getReferences() != null)
                loop: for (SimulinkReference ref : sr.getReferences())
                {
                    // won't handle odd extensions on Linux
                    ReferenceType rt = ref.getReferenceType();
                    if (rt == ReferenceType.BLOCK)
                    {
                        String fn1 = wd.getAbsolutePath() + File.separator + ref.getLocation() + ".mdl";
                        String fn2 = wd.getAbsolutePath() + File.separator + ref.getLocation() + ".slx";
                        
                        // processed files will be in mapFile
                        for (SimulinkFile sf2 : mapFiles.values())
                            if (sf2.getFilename().equals(fn1) || sf2.getFilename().equals(fn2))
                                continue loop;
                        
                        File f1 = new File(fn1);
                        File f2 = new File(fn2);
                        
                        if (f1.isFile() && f1.canRead())
                        {
                            log.info("Adding reference " + fn1);
                            SimulinkFile sf2 = new SimulinkFile(fn1, Format.SIM);
                            mapFiles.put(ref.getLocation(), sf2);
                            populateReaders(sf2);
                        }
                        else if (f2.isFile() && f2.canRead())
                        {
                            log.info("Adding reference " + fn2);
                            SimulinkFile sf2 = new SimulinkFile(fn2, Format.SIM);
                            mapFiles.put(ref.getLocation(), sf2);
                            populateReaders(sf2);
                        }
                    }
                    else if (rt == ReferenceType.SF1 || rt == ReferenceType.SF2)
                    {
                        String fn = wd.getAbsolutePath() + File.separator + ref.getLocation() + ".m";
                        
                        for (SimulinkFile sf2 : mapFiles.values())
                            if (sf2.getFilename().equals(fn))
                                continue loop;
                        
                        f = new File(fn);
                        if (f.isFile() && f.canRead())
                        {
                            log.info("Adding reference " + fn);
                            if (ref.getReferenceType() == ReferenceType.SF1)
                            {
                                SimulinkFile sf2 = new SimulinkFile(fn, Format.SF1);
                                mapFiles.put(ref.getLocation(), sf2);
                                populateReaders(sf2);
                            }
                            else
                            {
                                SimulinkFile sf2 = new SimulinkFile(fn, Format.SF2);
                                mapFiles.put(ref.getLocation(), sf2);
                                populateReaders(sf2);
                            }
                        }
                    }
                    else if (rt == ReferenceType.COMPONENT || rt == ReferenceType.DOMAIN)
                    {
                        // how to deal with foundation components
                        String fn = wd.getAbsolutePath() + File.separator
                                + ref.getLocation().replace('.', File.separatorChar) + ".ssc";
                        
                        for (SimulinkFile sf2 : mapFiles.values())
                            if (sf2.getFilename().equals(fn))
                                continue loop;
                        
                        f = new File(fn);
                        if (f.isFile() && f.canRead())
                        {
                            log.info("Adding reference " + fn);
                            SimulinkFile sf2 = new SimulinkFile(fn, Format.SSC);
                            mapFiles.put(ref.getLocation(), sf2);
                            populateReaders(sf2);
                        }
                        else
                            log.error("Can't find " + f);
                    }
                    else
                        log.error("Untreated reference: " + f);
                }
            
        }
        
        // external reference resolution
        for (SimulinkReader reader : processed)
            if (reader.getReferences() != null)
                for (SimulinkReference ref : reader.getReferences())
                    ref.resolve(mapFiles);
        
        // internal reference resolution
        while (!p2queue.isEmpty())
        {
            sr = p2queue.poll();
            sr.process();
        }
        while (toclose.size() != 0)
        {
            Closeable c = toclose.pop();
            try
            {
                c.close();
            }
            catch (IOException e)
            {
            }
        }
    }
    
    /**
     * Adds the appropriate readers for a given Simulink file
     * 
     * @param sf
     *            Simulink file for which readers need to be added
     * @throws IOException
     */
    private void populateReaders(SimulinkFile sf) throws IOException
    {
        String filename = sf.getFilename();
        Format format = sf.getFormat();
        
        if (format == Format.SIM)
        {
            log.info("Determining format of " + filename);
            // Chose between MDL and SLX
            File f = new File(filename);
            if (!f.canRead())
                throw new IOException("Can't read the file " + filename);
            boolean slx = false;
            FileInputStream fis = new FileInputStream(f);
            byte[] zipbytes = new byte[4];
            fis.read(zipbytes);
            if (zipbytes[0] == 'P' && zipbytes[1] == 'K' && zipbytes[2] == 0x3 && zipbytes[3] == 0x4)
                slx = true;
            fis.close();
            if (slx)
            {
                log.info("XML format detected");
                // open zip
                ZipFile zf = new ZipFile(filename);
                ZipEntry ze = zf.getEntry("simulink/blockdiagram.xml");
                
                if (ze == null)
                {
                    zf.close();
                    throw new IOException("The given model does not contain simulink/blockdiagram.xml");
                }
                
                log.debug("Adding XML Simulink Definition reader to " + filename);
                SimulinkXMLReaderPass1 sxmlr = new SimulinkXMLReaderPass1(zf.getInputStream(ze));
                sf.getReaders().add(sxmlr);
                p1queue.add(sxmlr);
                
                log.debug("Adding XML Stateflow Definition reader to " + filename);
                StateflowXMLReaderPass1 sxmlr2 = new StateflowXMLReaderPass1(sxmlr, zf.getInputStream(ze));
                sf.getReaders().add(sxmlr2);
                p1queue.add(sxmlr2);
                
                log.debug("Adding XML Simulink Resolution reader to " + filename);
                SimulinkXMLReaderPass2 sxmlr3 = new SimulinkXMLReaderPass2(sxmlr, zf.getInputStream(ze));
                sf.getReaders().add(sxmlr3);
                p2queue.add(sxmlr3);
                
                log.debug("Adding XML Stateflow Resolution reader to " + filename);
                StateflowXMLReaderPass2 sxmlr4 = new StateflowXMLReaderPass2(sxmlr2, zf.getInputStream(ze));
                sf.getReaders().add(sxmlr4);
                p2queue.add(sxmlr4);
                
                toclose.push(zf);
            }
            else
            {
                log.info("Text format detected");
                
                MDLLexer ml = new MDLLexer(new ANTLRFileStream(filename));
                CommonTokenStream cts = new CommonTokenStream(ml);
                MDLParser mp = new MDLParser(cts);
                mp.addErrorListener(new ANTLRErrorListener(log));
                ParserRuleContext prc = mp.file();
                
                log.debug("Adding MDL Simulink Definition reader to " + filename);
                SimulinkMDLReader smdlr = new SimulinkMDLReader(prc);
                sf.getReaders().add(smdlr);
                p1queue.add(smdlr);
                
                log.debug("Adding MDL Stateflow Definition reader to " + filename);
                StateflowMDLReader smdlr2 = new StateflowMDLReader(prc);
                sf.getReaders().add(smdlr2);
                p1queue.add(smdlr2);
            }
        }
        else if (format == Format.SF1)
        {
            MATLABLexer ml = new MATLABLexer(new ANTLRFileStream(filename));
            CommonTokenStream cts = new CommonTokenStream(ml);
            MATLABParser mp = new MATLABParser(cts);
            mp.addErrorListener(new ANTLRErrorListener(log));
            ParserRuleContext prc = mp.file();
            
            log.debug("Adding SF1 reader to " + filename);
            SimulinkSF1Reader sr = new SimulinkSF1Reader(prc);
            sf.getReaders().add(sr);
            p1queue.add(sr);
        }
        else if (format == Format.SF2)
        {
            MATLABLexer ml = new MATLABLexer(new ANTLRFileStream(filename));
            CommonTokenStream cts = new CommonTokenStream(ml);
            MATLABParser mp = new MATLABParser(cts);
            mp.addErrorListener(new ANTLRErrorListener(log));
            ParserRuleContext prc = mp.file();
            
            log.debug("Adding SF2 reader to " + filename);
            SimulinkSF2Reader sr = new SimulinkSF2Reader(prc);
            sf.getReaders().add(sr);
            p1queue.add(sr);
        }
        else if (format == Format.SSC)
        {
            SimscapeLexer sl = new SimscapeLexer(new ANTLRFileStream(filename));
            CommonTokenStream cts = new CommonTokenStream(sl);
            SimscapeParser sp = new SimscapeParser(cts);
            sp.addErrorListener(new ANTLRErrorListener(log));
            ParserRuleContext prc = sp.file();
            
            log.debug("Adding Simscape Definition reader to " + filename);
            SimscapeReaderPass1 srd = new SimscapeReaderPass1(cts, prc);
            srd.setExpressions(expressions);
            sf.getReaders().add(srd);
            p1queue.add(srd);
            
            log.debug("Adding Simscape Resolution reader to " + filename);
            SimscapeReaderPass2 srr = new SimscapeReaderPass2(srd, prc);
            sf.getReaders().add(srr);
            p2queue.add(srr);
        }
        else
            log.error("Can't find format of " + filename);
    }
    
    /**
     * Returns the root Simulink model
     * 
     * @return room Simulink model
     */
    public SModel getModel()
    {
        return rootmodel;
    }
    
    /**
     * Returns a list of Stateflow objects
     * 
     * @return list of Stateflow objects
     */
    public List<SStateflow> getStateflow()
    {
        return Collections.unmodifiableList(lstateflow);
    }
    
    /**
     * Comparator used to sort readers based on their priority.
     * 
     * @author barbau
     *
     */
    private static class SimulinkReaderComparator implements Comparator<SimulinkReader>
    {
        @Override
        public int compare(SimulinkReader o1, SimulinkReader o2)
        {
            return o1.getPriority() - o2.getPriority();
        }
    }
    
}
