package com.engisis.sysphs.deserialization.modelica;

import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MElement;
import com.engisis.sysphs.translation.modelica.ModelicaUtil;
import com.engisis.sysphs.generation.modelica.ModelicaBaseListener;
import com.engisis.sysphs.generation.modelica.ModelicaLexer;
import com.engisis.sysphs.generation.modelica.ModelicaParser;

import java.io.File;
import java.io.IOException;
import java.util.Hashtable;
import java.util.List;

import org.antlr.v4.runtime.ANTLRFileStream;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeWalker;

/**
 * Class in charge of deserializing Modelica files.
 * 
 * @author barbau
 *
 */
public class ModelicaDeserializer extends ModelicaBaseListener
{
    /**
     * Root classes extracted from the Modelica file
     */
    private List<MClass> classes;
    /**
     * Map with Modelica objects as keys, and syntax tree nodes as values.
     */
    private Hashtable<MElement, ParseTree> expressions;
    
    /**
     * Constructs a Modelica deserializer for a file
     * 
     * @param file
     *            Modelica file to be deserialized
     * @throws IOException if an error occurs when the file is read
     */
    public ModelicaDeserializer(File file) throws IOException
    {
        ModelicaLexer ml = new ModelicaLexer(new ANTLRFileStream(file.getAbsolutePath()));
        CommonTokenStream cts = new CommonTokenStream(ml);
        ModelicaParser mp = new ModelicaParser(cts);
        ParserRuleContext tree = mp.stored_definition();
        ParseTreeWalker ptw = new ParseTreeWalker();
        
        ModelicaScope globalscope = new ModelicaScope("");
        
        // primitive types
        
        ModelicaScope msstring = new ModelicaScope("String");
        msstring.setMNamedElement(ModelicaUtil.getString());
        msstring.setParent(globalscope);
        
        ModelicaScope msreal = new ModelicaScope("Real");
        msreal.setMNamedElement(ModelicaUtil.getReal());
        msreal.setParent(globalscope);
        
        ModelicaScope msboolean = new ModelicaScope("Boolean");
        msboolean.setMNamedElement(ModelicaUtil.getBoolean());
        msboolean.setParent(globalscope);
        
        ModelicaScope msinteger = new ModelicaScope("Integer");
        msinteger.setMNamedElement(ModelicaUtil.getInteger());
        msinteger.setParent(globalscope);
        
        ModelicaScope msinitialstep = new ModelicaScope(ModelicaUtil.getInitialStep().getName());
        msinitialstep.setMNamedElement(ModelicaUtil.getInitialStep());
        msinitialstep.setParent(globalscope);
        
        ModelicaScope msstep = new ModelicaScope(ModelicaUtil.getStep().getName());
        msstep.setMNamedElement(ModelicaUtil.getStep());
        msstep.setParent(globalscope);
        
        ModelicaScope mstransition = new ModelicaScope(ModelicaUtil.getTransition().getName());
        mstransition.setMNamedElement(ModelicaUtil.getTransition());
        mstransition.setParent(globalscope);
        
        ModelicaReaderPass1 mpd = new ModelicaReaderPass1(globalscope);
        ptw.walk(mpd, tree);
        mpd.resolve();
        
        ModelicaReaderPass2 mpr = new ModelicaReaderPass2(mpd);
        ptw.walk(mpr, tree);
        
        classes = globalscope.getAllMClasses();
        expressions = mpd.getExpressions();
    }
    
    /**
     * Returns the list of all root classes in the file
     * 
     * @return list of all root classes in the file
     */
    public List<MClass> getClasses()
    {
        return classes;
    }
    
    /**
     * Returns a map with Modelica objects as keys, and their corresponding
     * syntax tree nodes as values
     * 
     * @return a map with Modelica objects as keys, and syntax tree nodes as
     *         values
     */
    public Hashtable<MElement, ParseTree> getExpressions()
    {
        return expressions;
    }
}
