package com.engisis.sysphs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Stack;

import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLStreamException;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;

import com.engisis.sysphs.util.Configuration;
import com.engisis.sysphs.util.EMFUtil;
import com.engisis.sysphs.util.SysMLPreProcessor;
import com.engisis.sysphs.util.SysMLToSimulationTranslator;
import com.engisis.sysphs.util.SysMLUtil;
import com.engisis.sysphs.util.UMLModelErrorException;
import com.engisis.sysphs.util.XMIReferenceFinder;

/**
 * This class is used to manage the translation of SysML models into simulation
 * models
 * 
 * @author barbau
 *
 */
public class SysMLToSimulationTranslationManager
{
    private static final Logger log = Logger.getLogger(SysMLToSimulationTranslationManager.class);

    /**
     * UML class that will be translated
     */
    private Class rootclass;

    /**
     * name of the UML class that will be translated
     */
    private String rootclassname;
    
    /**
     * Resource that holds the SysML model
     */
    private Resource resource;
    
    /**
     * Translator used to create the simulation model
     */
    private SysMLToSimulationTranslator translator;
    
    /**
     * Output directory
     */
    private File outputdirectory;
    
    /**
     * Whether the user wants the preprocessing
     */
    private boolean preproc = false;
    
    /**
     * Whether the preprocessing has been done
     */
    private boolean preprocdone = false;
    
    /**
     * Constructs a SysML to simulation transformation manager. There are 3
     * phases in the translation: 1) the SysML model is loaded when the
     * constructor is called, 2) the root class is selected among the loaded
     * classes - the simulation generator is set, 3) the translation is executed
     * The SysML model is not loaded in the generator to allow different
     * generators to be applied.
     * 
     * @param model
     *            file location of a SysML model
     * @param paths
     *            paths where to look for XMI files
     * @param preproc
     *            whether SysML preprocessing is enabled
     * @throws IOException
     * @throws FileNotFoundException
     * @throws UMLModelErrorException
     * @throws XMLStreamException
     */
    public SysMLToSimulationTranslationManager(String model, String[] paths) throws IOException, FileNotFoundException,
            UMLModelErrorException, XMLStreamException
    {
        if (model == null)
            throw new IllegalArgumentException("The model can't be null");
        
        if (paths != null)
            for (int i = 0; i < paths.length; i++)
                if (!paths[i].endsWith(File.separator))
                    paths[i] += File.separator;
        
        // TODO: move readers to SysMLGeneration
        ResourceSet rs = EMFUtil.createResourceSet();
        
        // The files need to be processed in order because of the stereotypes
        // This part determines the order (simply put dependencies at the end)
        
        // stack of URIs to parse
        Stack<URI> toprepare = new Stack<URI>();
        
        // add main model
        URI modeluri = URI.createFileURI(new File(model).getAbsolutePath());
        toprepare.push(modeluri);
        
        // build map model -> dependencies
        Hashtable<URI, HashSet<URI>> deps = new Hashtable<URI, HashSet<URI>>();
        // get all dependencies
        while (toprepare.size() != 0)
        {
            URI input = toprepare.pop();
            try
            {
                InputStream is = rs.getURIConverter().createInputStream(input);
                XMIReferenceFinder xmir = new XMIReferenceFinder(rs, is, Configuration.getReaderIDMappings(),
                        Configuration.getLocationMappings(), modeluri, paths);
                xmir.run();
                is.close();
                deps.put(input, xmir.getReferences());
                // get the external references, and make sure they are loaded
                // before
                for (URI ref : xmir.getReferences())
                {
                    log.debug("Dependency between " + input.lastSegment() + " and " + ref.lastSegment());
                    if (!deps.containsKey(ref))
                        toprepare.push(ref);
                }
            }
            catch (IOException e)
            {
                log.error("Couldn't parse the XMI for dependencies", e);
                throw e;
            }
            catch (XMLStreamException e)
            {
                log.error("Couldn't parse the XMI for dependencies", e);
                throw e;
            }
            catch (FactoryConfigurationError e)
            {
                log.error("Couldn't parse the XMI for dependencies", e);
                throw e;
            }
        }
        
        // List of URIs to load
        List<URI> toload = new ArrayList<URI>();
        // create loading list in the right order
        // put the sources first
        while (toload.size() != deps.keySet().size())
        {
            boolean action = false;
            // inefficient
            loop: for (URI uri : deps.keySet())
            {
                if (!toload.contains(uri))
                {
                    // URI not loaded yet
                    // skip if one of the dependencies is not loaded
                    for (URI ref : deps.get(uri))
                        if (!toload.contains(ref) && !ref.equals(uri))
                            continue loop;
                    log.debug("Added " + uri.lastSegment() + " to load list");
                    toload.add(uri);
                    action = true;
                }
            }
            // break if no action was taken and loop not finished
            if (!action)
            {
                log.error("Dependency resolution problem");
                break;
            }
        }
        // This part filters and loads the files in the order
        for (int i = 0; i < toload.size(); i++)
        {
            resource = EMFUtil.loadResource(rs, toload.get(i), paths);
        }
        
        // To load the indirect profiles, may be unnecessary
        EcoreUtil.resolveAll(rs);
        // return error if no SysML profile
        if (rs.getPackageRegistry().get(SysMLUtil.uriSysML.toString()) == null)
            throw new UMLModelErrorException(resource, "The model doesn't import the SysML profile");
        if (rs.getPackageRegistry().get(SysMLUtil.uriSysPhS.toString()) == null)
            throw new UMLModelErrorException(resource, "The model doesn't import the SysPhS profile");
        // DebugUtil.checkResource(r);
    }
    
    /**
     * Returns a list of classes in the current resource
     * 
     * @return a list of qualified names
     */
    public List<String> getClasses()
    {
        LinkedList<String> ret = new LinkedList<String>();
        TreeIterator<EObject> iter = resource.getAllContents();
        while (iter.hasNext())
        {
            EObject eo = iter.next();
            if (!(eo instanceof NamedElement))
            {
                iter.prune();
            }
            else if (eo instanceof Class)
            {
                ret.add(((Class) eo).getQualifiedName());
            }
        }
        return ret;
    }
    
    /**
     * Sets the UML Class from which the translation works
     * 
     * @param rootclass
     *            a qualified name of the root class
     * @throws UMLModelErrorException
     */
    public void setInputRootName(String rootclass) throws UMLModelErrorException
    {
        if (rootclass == null)
            throw new IllegalArgumentException("The root element can't be null");
        this.rootclassname = rootclass;
        
        String[] fqname = rootclass.split("::");
        
        // look for the model (I assume there's only one)
        Model umodel = null;
        for (EObject object : resource.getContents())
        {
            if (object instanceof Model && fqname[0].equals(((Model) object).getName()))
                umodel = (Model) object;
        }
        
        // exit if none is found
        if (umodel == null)
        {
            throw new UMLModelErrorException(resource, "Could not find a root model in the XMI file named " + fqname[0]);
        }
        
        // Look for the root block (more efficient than parsing the entire file)
        Namespace currentelement = umodel;
        NamedElement nerootblock = null;
        
        // Parse the block's qualified name
        for (int i = 1; i < fqname.length; i++)
        {
            // get the next name
            String nextname = fqname[i];
            
            NamedElement nextelement = currentelement.getMember(nextname);
            if (nextelement == null)
            {
                log.error("Unknown element: " + currentelement.getQualifiedName() + "::" + fqname[i]);
                break;
            }
            if (rootclass.equals(nextelement.getQualifiedName()))
            {
                nerootblock = nextelement;
                // we have to be at the end of the loop, so no break
            }
            else
            {
                if (nextelement instanceof Namespace)
                    currentelement = (Namespace) nextelement;
                else
                {
                    log.error("Not a namespace: " + currentelement.getQualifiedName() + "::" + fqname[i]);
                    break;
                }
            }
        }
        
        if (nerootblock == null)
        {
            throw new UMLModelErrorException(resource, "Could not find the root block in the model");
        }
        if (!(nerootblock instanceof Class))
        {
            throw new UMLModelErrorException(resource, "The root block is not a Class");
        }
        
        this.rootclass = ((Class) nerootblock);
    }
    
    /**
     * Returns the name of the input root class
     * @return the name of the input root class
     */
    public String getInputRootName()
    {
        return rootclassname;
    }
    
    /**
     * Enables/disables the SysML preprocessing
     * 
     * @param preproc
     *            true if preprocessing is enabled
     */
    public void setPreprocessing(boolean preproc)
    {
        this.preproc = preproc;
    }
    
    /**
     * Sets the output directory
     * 
     * @param outputdirectory
     *            file representing the output directory
     */
    public void setOutputDirectory(File outputdirectory)
    {
        this.outputdirectory = outputdirectory;
    }
    
    /**
     * Sets the translator to be used
     * 
     * @param translator
     *            simulation translator
     */
    public void setTranslator(SysMLToSimulationTranslator translator)
    {
        this.translator = translator;
    }
    
    /**
     * Executes the translation on the resource
     * 
     * @throws UMLModelErrorException
     * @throws IOException
     */
    public void execute() throws UMLModelErrorException, IOException
    {
        if (translator == null)
            throw new IllegalStateException("A generator must be provided first");
        if (rootclass == null)
            throw new IllegalStateException("A root class must be provided first");
        log.info("started");
        
        // list stereotyped elements
        // DebugUtil.printStereotypedElements(r);
        
        if (preproc && !preprocdone)
        {
            if (outputdirectory == null)
                resource.setURI(resource.getURI().appendFileExtension("bak"));
            else
                resource.setURI(URI.createFileURI(outputdirectory.getAbsolutePath())
                        .appendSegment(resource.getURI().lastSegment()).appendFileExtension("bak"));
            SysMLPreProcessor sysmlt = new SysMLPreProcessor(resource);
            sysmlt.execute();
            resource.save(null);
            preprocdone = true;
        }
        
        // Execute the translation
        translator.execute(rootclass, resource, outputdirectory);
        
    }
}
