package com.engisis.sysphs.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.Stereotype;

/**
 * SysML utility class
 * 
 * @author barbau
 *
 */
public class SysMLUtil
{
    public static final String SYSPHS_LANGUAGE = "sysphs";
    
    private ResourceSet rs;
    
    private Stereotype sinitialvaluesspec;
    private Stereotype sblock;
    private Stereotype sconstraintblock;
    private Stereotype sbindingconnector;
    private Stereotype snestedconnectorend;
    private Stereotype svaluetype;
    private Class sunit;
    private Class squantitykind;
    private Stereotype spropertyspecifictype;
    private Stereotype sflowproperty;
    private Stereotype sinterfaceblock;
    private Stereotype sdirectedfeature;
    private Stereotype sconnectorproperty;
    private Stereotype sparticipantproperty;

    private Class sconservedsubstance;
    private Class srealsignalin;
    private Class srealsignalout;
    
    private Stereotype sphsvariable;
    private Stereotype sphsconstant;
    private Stereotype smultidimelement;
    private Stereotype sinitialvalueref;
    private Stereotype ssimulationvertex;
    
    private Stereotype smodelicablock;
    private Stereotype smodelicaport;
    private Stereotype smodelicaparameter;
    private Stereotype ssimulinkblock;
    private Stereotype ssimulinkport;
    private Stereotype ssimulinkparameter;
    
    private DataType ustring;
    private DataType uboolean;
    private DataType uinteger;
    private DataType udouble;
    private DataType sstring;
    private DataType sboolean;
    private DataType sinteger;
    private DataType sdouble;
    
    private List<LibraryComponent> lscomponents;
    private List<LibraryComponent> lmcomponents;
    
    private static final URI locationTypes = URI.createURI("http://www.omg.org/spec/UML/20110701/PrimitiveTypes.xmi#");
    public static final URI uriSysML = URI.createURI("http://www.omg.org/spec/SysML/20161101/SysML");
    public static final URI uriSysPhS = URI.createURI("http://www.omg.org/spec/SysPhS/20171215");
    private static final URI uriSysPhSExtended = URI
            .createURI("http://www.omg.org/spec/SysPhS/20171215/SysPhSExtended");
    private static final URI uriSysPhSPlatformProfile = URI
            .createURI("http://www.omg.org/spec/SysPhS/20171215/SysPhSPlatformProfile");
    
    /**
     * Constructs an utility object for the specified resource set
     * 
     * @param rs
     *            resource set in which the models are loaded
     * @throws UMLModelErrorException
     */
    public SysMLUtil(ResourceSet rs) throws UMLModelErrorException
    {
        if (rs == null)
            throw new IllegalArgumentException("You must provide a ResourceSet");
        
        this.rs = rs;
        loadSysML();
        loadSysPhS();
        loadSysPhSExtended();
        loadSysPhSPlatform();
    }
    
    /**
     * Loads the SysML stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysML() throws UMLModelErrorException
    {
        // get and check all the stereotypes
        sblock = loadStereotype(rs, uriSysML, "Block");
        sconstraintblock = loadStereotype(rs, uriSysML, "ConstraintBlock");
        sbindingconnector = loadStereotype(rs, uriSysML, "BindingConnector");
        snestedconnectorend = loadStereotype(rs, uriSysML, "NestedConnectorEnd");
        svaluetype = loadStereotype(rs, uriSysML, "ValueType");
        sunit = loadClass(rs, uriSysML, "Unit");
        squantitykind = loadClass(rs, uriSysML, "QuantityKind");
        spropertyspecifictype = loadStereotype(rs, uriSysML, "PropertySpecificType");
        sflowproperty = loadStereotype(rs, uriSysML, "FlowProperty");
        sinterfaceblock = loadStereotype(rs, uriSysML, "InterfaceBlock");
        sdirectedfeature = loadStereotype(rs, uriSysML, "DirectedFeature");
        sconnectorproperty = loadStereotype(rs, uriSysML, "ConnectorProperty");
        sparticipantproperty = loadStereotype(rs, uriSysML, "ParticipantProperty");
        
        ustring = loadType(rs, locationTypes, "String");
        uboolean = loadType(rs, locationTypes, "Boolean");
        uinteger = loadType(rs, locationTypes, "Integer");
        udouble = loadType(rs, locationTypes, "Real");
        sstring = loadType(rs, uriSysML, "SysML_dataType.String");
        sboolean = loadType(rs, uriSysML, "SysML_dataType.Boolean");
        sinteger = loadType(rs, uriSysML, "SysML_dataType.Integer");
        sdouble = loadType(rs, uriSysML, "SysML_dataType.Real");
        
    }
    
    /**
     * Loads the SysPhS stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysPhS() throws UMLModelErrorException
    {
        sphsvariable = loadStereotype(rs, uriSysPhS, "PhSVariable");
        sphsconstant = loadStereotype(rs, uriSysPhS, "PhSConstant");
    }
    
    /**
     * Loads the extended SysPhS stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysPhSExtended() throws UMLModelErrorException
    {
        if (rs.getResource(uriSysPhSExtended, false) != null)
        {
            sinitialvalueref = loadStereotype(rs, uriSysPhSExtended, "InitialValueReference");
            ssimulationvertex = loadStereotype(rs, uriSysPhSExtended, "SimulationVertex");
            sinitialvaluesspec = loadStereotype(rs, uriSysPhSExtended, "InitialValuesSpecification");
        }
    }
    
    /**
     * Loads the SysPhS platform stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysPhSPlatform() throws UMLModelErrorException
    {
        Resource r = rs.getResource(uriSysPhSPlatformProfile, false);
        if (r != null)
        {
            smultidimelement = loadStereotype(rs, uriSysPhSPlatformProfile, "MultidimensionalElement");
            smodelicablock = loadStereotype(rs, uriSysPhSPlatformProfile, "ModelicaBlock");
            smodelicaparameter = loadStereotype(rs, uriSysPhSPlatformProfile, "ModelicaParameter");
            smodelicaport = loadStereotype(rs, uriSysPhSPlatformProfile, "ModelicaPort");
            
            ssimulinkblock = loadStereotype(rs, uriSysPhSPlatformProfile, "SimulinkBlock");
            ssimulinkparameter = loadStereotype(rs, uriSysPhSPlatformProfile, "SimulinkParameter");
            ssimulinkport = loadStereotype(rs, uriSysPhSPlatformProfile, "SimulinkPort");
            
            lscomponents = new LinkedList<LibraryComponent>();
            lmcomponents = new LinkedList<LibraryComponent>();
            TreeIterator<EObject> ti = r.getAllContents();
            while (ti.hasNext())
            {
                EObject eo = ti.next();
                if (!(eo instanceof Namespace))
                {
                    ti.prune();
                    continue;
                }
                if (eo instanceof Class)
                {
                    Class uclass = (Class) eo;
                    processLibraryComponent(lmcomponents, uclass, smodelicablock, smodelicaparameter, smodelicaport);
                    processLibraryComponent(lscomponents, uclass, ssimulinkblock, ssimulinkparameter, ssimulinkport);
                    if ("ConservedSubstance".equals(uclass.getName())
                            || "ConservedQuantityKind".equals(uclass.getName()))
                        sconservedsubstance = uclass;
                    else if ("RealSignalInElement".equals(uclass.getName()))
                        srealsignalin = uclass;
                    else if ("RealSignalOutElement".equals(uclass.getName()))
                        srealsignalout = uclass;
                }
            }
        }
    }
    
    /**
     * Adds the given library component to the list
     * 
     * @param list
     *            list in which the library components should be added
     * @param uclass
     *            library component to parse
     * @param stclass
     *            library block stereotype
     * @param stparameter
     *            library parameter stereotype
     * @param stport
     *            library port stereotype
     */
    private void processLibraryComponent(List<LibraryComponent> list, Class uclass, Stereotype stclass,
            Stereotype stparameter, Stereotype stport)
    {
        if (uclass.isStereotypeApplied(stclass))
        {
            String name = (String) uclass.getValue(stclass, "name");
            if (name == null)
                name = uclass.getName();
            LibraryComponent lc = new LibraryComponent(uclass, name);
            for (Property uproperty : SysMLUtil.getAllAttributes(uclass))
            {
                if (uproperty.isStereotypeApplied(stparameter))
                {
                    name = (String) uproperty.getValue(stparameter, "name");
                    if (name == null)
                        name = uproperty.getName();
                    String value = (String) uproperty.getValue(stparameter, "value");
                    LibraryParameter lp = new LibraryParameter(uproperty, name, value);
                    lc.parameters.add(lp);
                }
                else if (uproperty.isStereotypeApplied(stport))
                {
                    name = (String) uproperty.getValue(stport, "name");
                    if (name == null)
                        name = uproperty.getName();
                    LibraryPort lp = new LibraryPort((Port) uproperty, name);
                    lc.ports.add(lp);
                }
                
            }
            list.add(lc);
        }
    }
    
    /**
     * Returns the resource set corresponding to the utility class
     * 
     * @return resource set in which the models are loaded
     */
    public ResourceSet getResourceSet()
    {
        return rs;
    }
    
    /**
     * Loads the stereotype with the specified namespace and name into the
     * resource set
     * 
     * @param rs
     *            resource set in which to load the stereotype
     * @param namespace
     *            namespace of the stereotype
     * @param name
     *            name of the stereotype
     * @return loaded stereotype
     * @throws UMLModelErrorException
     */
    private static Stereotype loadStereotype(ResourceSet rs, URI namespace, String name) throws UMLModelErrorException
    {
        NamedElement namedelement = loadFromNamespaceAndName(rs, namespace, name);
        if (!(namedelement instanceof Stereotype))
            throw new ClassCastException("The element named " + name + " in " + namespace + " is not a stereotype");
        return (Stereotype) namedelement;
    }
    
    /**
     * Loads the class with the specified namespace and name into the resource
     * set
     * 
     * @param rs
     *            resource set in which to load the class
     * @param namespace
     *            namespace of the class
     * @param name
     *            name of the class
     * @return loaded class
     * @throws UMLModelErrorException
     */
    private static Class loadClass(ResourceSet rs, URI namespace, String name) throws UMLModelErrorException
    {
        Element element = loadFromNamespaceAndName(rs, namespace, name);
        if (!(element instanceof Class))
            throw new ClassCastException("The element named " + name + " in " + namespace + " is not a class");
        return (Class) element;
    }
    
    /**
     * Loads the type with the specified location and ID into the resource set
     * 
     * @param rs
     *            resource set in which to load the type
     * @param location
     *            location of the type
     * @param id
     *            id of the type
     * @return loaded type
     * @throws UMLModelErrorException
     */
    private static DataType loadType(ResourceSet rs, URI location, String id)
    {
        Element element = loadFromLocationAndID(rs, location, id);
        if (!(element instanceof DataType))
            throw new ClassCastException("The element identified by " + id + " at " + location + " is not a type");
        return (DataType) element;
    }
    
    /**
     * Loads an element from a location and an ID into the resource set
     * 
     * @param rs
     *            resource set in which to load the element
     * @param location
     *            location of the element
     * @param id
     *            id of the element
     * @return loaded element
     */
    private static Element loadFromLocationAndID(ResourceSet rs, URI location, String id)
    {
        if (location == null)
            throw new NullPointerException("You must provide a location");
        if (id == null)
            throw new NullPointerException("You must provide an id");
        return (Element) rs.getEObject(location.appendFragment(id), true);
    }
    
    /**
     * Loads an element from a namespace and a name into the resource set
     * 
     * @param rs
     *            resource set in which to load the element
     * @param namespace
     *            namespace of the element
     * @param name
     *            name of the element
     * @return loaded element
     */
    private static NamedElement loadFromNamespaceAndName(ResourceSet rs, URI namespace, String name)
            throws UMLModelErrorException
    {
        if (rs == null)
            throw new IllegalArgumentException("You must provide a resource set");
        if (namespace == null)
            throw new IllegalArgumentException("You must provide a namespace");
        if (name == null)
            throw new IllegalArgumentException("You must provide a name");
        
        Resource resource = rs.getResource(namespace, true);
        TreeIterator<EObject> treeiterator = resource.getAllContents();
        while (treeiterator.hasNext())
        {
            EObject currenteobject = treeiterator.next();
            if (!(currenteobject instanceof Namespace))
                treeiterator.prune();
            else
            {
                Namespace namedelement = (Namespace) currenteobject;
                if (name.equals(namedelement.getName()))
                {
                    return namedelement;
                }
            }
        }
        throw new UMLModelErrorException(resource, "Can't retrieve an element named " + name + " in " + namespace);
    }
    
    public Stereotype getInitialValuesSpecification()
    {
        return sinitialvaluesspec;
    }
    
    public Stereotype getBlock()
    {
        return sblock;
    }
    
    public Stereotype getConstraintBlock()
    {
        return sconstraintblock;
    }
    
    public Stereotype getBindingConnector()
    {
        return sbindingconnector;
    }
    
    public Stereotype getNestedConnectorEnd()
    {
        return snestedconnectorend;
    }
    
    public Stereotype getValueType()
    {
        return svaluetype;
    }
    
    public Class getUnit()
    {
        return sunit;
    }
    
    public Class getQuantityKind()
    {
        return squantitykind;
    }
    
    public Stereotype getPropertySpecificType()
    {
        return spropertyspecifictype;
    }
    
    public Stereotype getFlowProperty()
    {
        return sflowproperty;
    }
    
    public EnumerationLiteral getFlowPropertyIn()
    {
        return ((Enumeration) getFlowProperty().getOwnedAttribute("direction", null).getType()).getOwnedLiteral("in");
    }
    
    public EnumerationLiteral getFlowPropertyOut()
    {
        return ((Enumeration) getFlowProperty().getOwnedAttribute("direction", null).getType()).getOwnedLiteral("out");
    }
    
    public EnumerationLiteral getFlowPropertyInout()
    {
        return ((Enumeration) getFlowProperty().getOwnedAttribute("direction", null).getType())
                .getOwnedLiteral("inout");
    }
    
    public Stereotype getInterfaceBlock()
    {
        return sinterfaceblock;
    }
    
    public Stereotype getDirectedFeature()
    {
        return sdirectedfeature;
    }
    
    public Stereotype getConnectorProperty()
    {
        return sconnectorproperty;
    }
    
    public Stereotype getParticipantProperty()
    {
        return sparticipantproperty;
    }
    
    public Stereotype getPhSVariable()
    {
        return sphsvariable;
    }
    
    public Stereotype getPhSConstant()
    {
        return sphsconstant;
    }
    
    public Stereotype getMultidimensionalElement()
    {
        return smultidimelement;
    }
    
    public Stereotype getInitialValueReference()
    {
        return sinitialvalueref;
    }
    
    public Stereotype getSimulationVertex()
    {
        return ssimulationvertex;
    }
    
    public DataType getUMLString()
    {
        return ustring;
    }
    
    public DataType getUMLBoolean()
    {
        return uboolean;
    }
    
    public DataType getUMLInteger()
    {
        return uinteger;
    }
    
    public DataType getUMLDouble()
    {
        return udouble;
    }
    
    public DataType getSysMLString()
    {
        return sstring;
    }
    
    public DataType getSysMLBoolean()
    {
        return sboolean;
    }
    
    public DataType getSysMLInteger()
    {
        return sinteger;
    }
    
    public DataType getSysMLDouble()
    {
        return sdouble;
    }
    
    public Class getConservedSubstance()
    {
        return sconservedsubstance;
    }
    
    public Class getRealSignalIn()
    {
        return srealsignalin;
    }
    
    public Class getRealSignalOut()
    {
        return srealsignalout;
    }
    
    public Stereotype getModelicaBlock()
    {
        return smodelicablock;
    }
    
    public Stereotype getModelicaParameter()
    {
        return smodelicaparameter;
    }
    
    public Stereotype getModelicaPort()
    {
        return smodelicaport;
    }
    
    public Stereotype getSimulinkBlock()
    {
        return ssimulinkblock;
    }
    
    public Stereotype getSimulinkParameter()
    {
        return ssimulinkparameter;
    }
    
    public Stereotype getSimulinkPort()
    {
        return ssimulinkport;
    }
    
    /**
     * Returns a list of Simulink components
     * 
     * @return list of Simulink components
     */
    public List<LibraryComponent> getSimulinkComponents()
    {
        return Collections.unmodifiableList(lscomponents);
    }
    
    /**
     * Returns a list of Modelica components
     * 
     * @return list of Modelica components
     */
    public List<LibraryComponent> getModelicaComponents()
    {
        return Collections.unmodifiableList(lmcomponents);
    }
    
    /**
     * Returns the path of a connector end
     * 
     * @param end
     *            connector end
     * @return list of property (path)
     */
    public List<Property> getPropertyPath(ConnectorEnd end)
    {
        List<Property> result = new ArrayList<Property>();
        if (end.isStereotypeApplied(snestedconnectorend))
        {
            @SuppressWarnings("unchecked")
            List<Object> properties = (List<Object>) (end.getValue(snestedconnectorend, "propertyPath"));
            for (Object property : properties)
                result.add((Property) property);
        }
        else if (end.getPartWithPort() != null)
            result.add(end.getPartWithPort());
        if (!(end.getRole() instanceof Property))
            throw new IllegalArgumentException("The connector role " + end.getRole() + " should be a property");
        result.add((Property) end.getRole());
        return result;
    }
    
    /**
     * Updates the connector end with the given path
     * 
     * @param end
     *            connector end to update
     * @param lp
     *            path for the connector end
     */
    public void updatePropertyPath(ConnectorEnd end, List<Property> lp)
    {
        if (lp.size() > 0)
            end.setRole(lp.get(lp.size() - 1));
        if (lp.size() > 1)
        {
            if (!end.isStereotypeApplied(getNestedConnectorEnd()))
                end.applyStereotype(getNestedConnectorEnd());
            List<Property> pp = (List<Property>) end.getValue(getNestedConnectorEnd(), "propertyPath");
            if (pp == null)
            {
                pp = new ArrayList<Property>(lp.size() - 1);
                end.setValue(getNestedConnectorEnd(), "propertyPath", pp);
            }
            pp.clear();
            for (int i = 0; i < lp.size() - 1; i++)
                pp.add(lp.get(i));
        }
    }
    
    /**
     * Returns a list of property joined by the specified separator
     * 
     * @param properties
     *            path to join
     * @param separator
     *            string to add between the elements
     * @return joined properties
     */
    public String joinProperties(List<Property> properties, String separator)
    {
        if (properties == null || separator == null)
            return "";
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < properties.size(); i++)
        {
            Property p = properties.get(i);
            if (i != 0)
                sb.append(separator);
            sb.append(getName(p));
            if (p.isStereotypeApplied(smultidimelement)
                    && p.getValue(smultidimelement, "dimensions") instanceof List<?>)
            {
                @SuppressWarnings("unchecked")
                List<Integer> dims = (List<Integer>) p.getValue(smultidimelement, "dimensions");
                if (dims.size() == 1 && dims.get(0).equals(Integer.valueOf(1)))
                    sb.append("[1]");
            }
        }
        return sb.toString();
    }
    
    /**
     * Returns all the PhS flow properties
     * 
     * @param uclass
     *            classifier in which the sim properties are looked for
     * @return list of sim properties
     */
    public List<Property> getOwnedPhSProperties(Classifier uclassifier)
    {
        List<Property> lp = getOwnedAttributes(uclassifier);
        ArrayList<Property> properties = new ArrayList<Property>();
        for (Property uproperty : lp)
        {
            if (!uproperty.isStereotypeApplied(sflowproperty))
                continue;
            if (uproperty.isStereotypeApplied(sphsvariable) || isSimBlock((Classifier) uproperty.getType()))
                properties.add(uproperty);
        }
        return properties;
    }
    
    public boolean isPhSProperty(Property uproperty)
    {
        if (uproperty == null)
            return false;
        if (!uproperty.isStereotypeApplied(sflowproperty))
            return false;
        if (uproperty.getType() instanceof DataType && uproperty.isStereotypeApplied(sphsvariable))
            return true;
        else if (uproperty.getType() instanceof Classifier && isSimBlock((Classifier) uproperty.getType()))
            return true;
        return false;
    }
    
    public boolean isSimBlock(Classifier uclassifier)
    {
        if (uclassifier == null)
            return false;
        for (Classifier ugeneral : uclassifier.getGenerals())
            if (ugeneral == sconservedsubstance)
                return true;
            else if (isSimBlock(ugeneral))
                return true;
        return false;
        
    }
    
    /**
     * Returns all the sim properties, even inherited and accounting for
     * redefinitions.
     * 
     * @param uclass
     *            classifier in which the sim properties are looked for
     * @return list of sim properties
     */
    public List<Property> getAllPhSProperties(Classifier uclassifier)
    {
        List<Property> lp = getAllCorrectedAttributes(uclassifier);
        
        ArrayList<Property> ret = new ArrayList<Property>();
        for (Property uproperty : lp)
        {
            if (isPhSProperty(uproperty))
                ret.add(uproperty);
        }
        return ret;
    }
    
    /**
     * Returns all the owned flow properties.
     * 
     * @param uclass
     *            classifier in which the flow properties are looked for
     * @return list of flow properties
     */
    public List<Property> getOwnedFlowProperties(Classifier uclassifier)
    {
        List<Property> lp = getOwnedAttributes(uclassifier);
        ArrayList<Property> properties = new ArrayList<Property>();
        for (Property uproperty : lp)
        {
            if (uproperty.isStereotypeApplied(sflowproperty) && uproperty.getType() != null)
                properties.add(uproperty);
        }
        return properties;
    }
    
    /**
     * Returns all the flow properties, even inherited and accounting for
     * redefinitions.
     * 
     * @param uclass
     *            classifier in which the flow properties are looked for
     * @return list of flow properties
     */
    public List<Property> getAllFlowProperties(Classifier uclassifier)
    {
        List<Property> lp = getAllCorrectedAttributes(uclassifier);
        
        ArrayList<Property> ret = new ArrayList<Property>();
        for (Property uproperty : lp)
        {
            if (uproperty.isStereotypeApplied(sflowproperty) && uproperty.getType() != null)
                ret.add(uproperty);
        }
        return ret;
    }
    
    /**
     * Returns all the sim variables in the given property's type
     * 
     * @param uproperty
     *            property whose type is used to look for sim variables
     * @return list of sim variable in the property's type
     */
    public List<Property> getSimVariables(Property uproperty)
    {
        ArrayList<Property> properties = new ArrayList<Property>();
        if (uproperty.getType() instanceof Class)
        {
            for (Property property : SysMLUtil.getAllAttributes((Class) uproperty.getType()))
                if (property.isStereotypeApplied(getPhSVariable()))
                    properties.add(property);
        }
        return properties;
    }
    
    /**
     * Looked for the final redefining property of the specified property
     * 
     * @param uproperty
     *            property for which a redefining property is looked for
     * @param context
     *            context of the redefinitions
     * @return final redefining property
     */
    public static Property correctProperty(Property uproperty, Classifier context)
    {
        if (uproperty == null)
            return null;
        Property ret = uproperty;
        List<Property> uproperties = getAllAttributes(context);
        boolean cont = true;
        while (cont)
        {
            for (Property uprop : uproperties)
            {
                if (uprop.getRedefinedProperties().contains(ret))
                {
                    ret = uprop;
                    continue;
                }
            }
            cont = false;
        }
        return ret;
    }
    
    /**
     * Returns all owned and inherited constraints of the specified classifier
     * 
     * @param uclassifier
     *            classifier in which the constraints are looked for
     * @return list of all constraints
     */
    public static List<Constraint> getAllConstraints(Classifier uclassifier)
    {
        List<Constraint> ret = new LinkedList<Constraint>();
        ret.addAll(uclassifier.getOwnedRules());
        for (Classifier general : uclassifier.getGenerals())
            ret.addAll(getAllConstraints(general));
        return ret;
    }
    
    /**
     * Returns all owned and inherited connectors of the specified class
     * 
     * @param uclass
     *            class in which connectors are looked for
     * @return list of connectors
     */
    public static List<Connector> getAllConnectors(Class uclass)
    {
        List<Connector> ret = new LinkedList<Connector>();
        ret.addAll(uclass.getOwnedConnectors());
        for (Generalization ugeneralization : uclass.getGeneralizations())
        {
            if (ugeneralization.getSpecific() == uclass && ugeneralization.getGeneral() instanceof Class)
                ret.addAll(getAllConnectors((Class) ugeneralization.getGeneral()));
        }
        return ret;
    }
    
    public static List<Property> getOwnedAttributes(Classifier uclassifier)
    {
        if (uclassifier instanceof Class)
            return ((Class) uclassifier).getOwnedAttributes();
        else if (uclassifier instanceof Interface)
            return ((Interface) uclassifier).getOwnedAttributes();
        else if (uclassifier instanceof DataType)
            return ((DataType) uclassifier).getOwnedAttributes();
        else
            return new ArrayList<Property>();
    }
    
    public static List<Property> getAllAttributes(Classifier uclassifier)
    {
        List<Property> ret = new ArrayList<Property>();
        for (Classifier ugeneral : uclassifier.getGenerals())
            ret.addAll(getAllAttributes(ugeneral));
        if (uclassifier instanceof Class)
            ret.addAll(((Class) uclassifier).getOwnedAttributes());
        else if (uclassifier instanceof Interface)
            ret.addAll(((Interface) uclassifier).getOwnedAttributes());
        else if (uclassifier instanceof DataType)
            ret.addAll(((DataType) uclassifier).getOwnedAttributes());
        return ret;
    }
    
    public static List<Classifier> getAllCompatibleTypes(Classifier uclassifier)
    {
        LinkedList<Classifier> ret = new LinkedList<Classifier>();
        if (uclassifier == null)
            return ret;
        ret.add(uclassifier);
        for (Classifier general : uclassifier.getGenerals())
        {
            ret.addAll(getAllCompatibleTypes(general));
        }
        return ret;
    }
    
    /*
     * Removes any sim property in the specified path
     *
     * @param properties list in which sim properties should be removed
     */
    public void removeSimProperty(List<Property> properties)
    {
        int i = 0;
        while (i != properties.size())
        {
            if (properties.get(i).isStereotypeApplied(sflowproperty))
            {
                properties.remove(i);
                continue;
            }
            i++;
        }
    }
    
    public static List<Property> getAllCorrectedAttributes(Classifier uclassifier)
    {
        List<Property> ret = new ArrayList<Property>(getAllAttributes(uclassifier));
        List<Property> rem = new LinkedList<Property>();
        for (Property p : ret)
            rem.addAll(p.getRedefinedProperties());
        ret.removeAll(rem);
        return ret;
    }
    
    public List<Property> getCorrectedPropertyPath(ConnectorEnd ce, Classifier context) throws UMLModelErrorException
    {
        if (ce == null)
            throw new IllegalArgumentException("The connector end can't be null");
        if (context == null)
            throw new IllegalArgumentException("The context end can't be null");
        List<Property> lp = getPropertyPath(ce);
        List<Property> ret = new ArrayList<Property>(lp.size());
        Classifier c = context;
        for (Property p : lp)
        {
            Property p2 = correctProperty(p, c);
            ret.add(p2);
            if (ret.size() != lp.size())
            {
                if (p2.getType() instanceof Classifier)
                    c = (Classifier) p2.getType();
                else
                    throw new UMLModelErrorException(p2.eResource(),
                            "The property has no classifier type: " + p2.getQualifiedName());
            }
        }
        return ret;
    }
    
    /**
     * Returns a name for the specified named elements. Generates one if null or
     * empty.
     * 
     * @param nes
     *            named elements
     * @return generated name
     */
    public static String getName(NamedElement... nes)
    {
        StringBuilder ret = new StringBuilder();
        for (int i = 0; i < nes.length; i++)
        {
            if (i != 0)
                ret.append("_");
            NamedElement ne = nes[i];
            String n = ne.getName();
            if (n != null)
            {
                // remove disallowed characters
                for (char c : n.toCharArray())
                {
                    if (c == '_' || Character.isLetter(c) || (Character.isDigit(c) && ret.length() != 0))
                        ret.append(c);
                }
            }
            if (ret.length() == 0)
            {
                ret.append(getName((NamedElement) ne.getOwner()));
                ret.append("_");
                ret.append(ne.eClass().getName().toLowerCase());
                if (ne.getNamespace() != null)
                    ret.append(ne.getNamespace().getOwnedMembers().indexOf(ne));
            }
        }
        return ret.toString();
    }
    
    public static void setName(NamedElement ne, String name)
    {
        if (name == null)
        {
            ne.setName(null);
            return;
        }
        // raw name, not filtered since everything is allowed
        String rname = name.length() == 0 ? "_" + ne.getClass().getName() : name;
        
        List<NamedElement> nes = new LinkedList<NamedElement>();
        Namespace ns = ne.getNamespace();
        if (ns != null)
            nes.addAll(ns.getOwnedMembers());
        
        if (!isPresent(rname, nes))
        {
            ne.setName(rname);
            return;
        }
        
        int i = 0;
        while (true)
        {
            String nname = rname + (i++);
            if (!isPresent(nname, nes))
            {
                ne.setName(nname);
                return;
            }
        }
    }
    
    private static boolean isPresent(String name, List<NamedElement> nes)
    {
        if (name != null)
            for (NamedElement ne : nes)
                if (name.equals(ne.getName()))
                    return true;
        return false;
    }
    
    /**
     * Library block class
     * 
     * @author barbau
     *
     */
    public class LibraryComponent
    {
        private String name;
        private Class clas;
        
        private List<LibraryParameter> parameters = new LinkedList<LibraryParameter>();
        private List<LibraryPort> ports = new LinkedList<LibraryPort>();
        
        private LibraryComponent(Class clas, String name)
        {
            this.clas = clas;
            this.name = name;
        }
        
        public String getName()
        {
            return name;
        }
        
        public Class getClas()
        {
            return clas;
        }
        
        public List<LibraryParameter> getParameters()
        {
            return Collections.unmodifiableList(parameters);
        }
        
        public List<LibraryPort> getPorts()
        {
            return Collections.unmodifiableList(ports);
        }
        
        public List<LibraryPort> getInputPorts()
        {
            List<LibraryPort> ret = new LinkedList<LibraryPort>();
            for (LibraryPort lp : ports)
            {
                if (!(lp.getPort().getType() instanceof Classifier))
                    continue;
                List<Property> lfp = getAllFlowProperties((Classifier) lp.getPort().getType());
                if (lfp.size() == 1)
                {
                    Object o = lfp.get(0).getValue(sflowproperty, "direction");
                    if (o instanceof EnumerationLiteral)
                    {
                        EnumerationLiteral el = ((EnumerationLiteral) o);
                        if ((el.getName().equals("in") && !lp.getPort().isConjugated())
                                || (el.getName().equals("out") && lp.getPort().isConjugated()))
                            
                            ret.add(lp);
                    }
                }
            }
            return Collections.unmodifiableList(ret);
        }
        
        public List<LibraryPort> getOutputPorts()
        {
            List<LibraryPort> ret = new LinkedList<LibraryPort>();
            for (LibraryPort lp : ports)
            {
                if (!(lp.getPort().getType() instanceof Classifier))
                    continue;
                List<Property> lfp = getAllFlowProperties((Classifier) lp.getPort().getType());
                if (lfp.size() == 1)
                {
                    Object o = lfp.get(0).getValue(sflowproperty, "direction");
                    if (o instanceof EnumerationLiteral)
                    {
                        EnumerationLiteral el = ((EnumerationLiteral) o);
                        if ((el.getName().equals("out") && !lp.getPort().isConjugated())
                                || (el.getName().equals("in") && lp.getPort().isConjugated()))
                            ret.add(lp);
                    }
                }
            }
            return Collections.unmodifiableList(ret);
        }
    }
    
    /**
     * Library parameter class
     * 
     * @author barbau
     *
     */
    public class LibraryParameter
    {
        private Property property;
        private String name;
        private String value;
        
        private LibraryParameter(Property property, String name, String value)
        {
            this.property = property;
            this.name = name;
            this.value = value;
        }
        
        public Property getProperty()
        {
            return property;
        }
        
        public String getName()
        {
            return name;
        }
        
        public String getValue()
        {
            return value;
        }
    }
    
    /**
     * Library port class
     * 
     * @author barbau
     *
     */
    public class LibraryPort
    {
        private Port port;
        private String name;
        
        private LibraryPort(Port port, String name)
        {
            this.port = port;
            this.name = name;
        }
        
        public Port getPort()
        {
            return port;
        }
        
        public String getName()
        {
            return name;
        }
    }
    
    public List<Region> getAllCorrectedRegions(StateMachine ustatemachine)
    {
        List<Region> ret = new LinkedList<Region>();
        for (Classifier c : ustatemachine.getGenerals())
            if (c instanceof StateMachine)
                ret.addAll(getAllCorrectedRegions((StateMachine) c));
        for (Region r : ustatemachine.getRegions())
            ret.removeAll(r.getRedefinedElements());
        ret.addAll(ustatemachine.getRegions());
        return ret;
    }

    public Behavior getClassifierBehavior(Class uclass)
    {
        Behavior b = uclass.getClassifierBehavior();
        if (b != null)
            return b;
        for(Classifier c : uclass.getGenerals())
        {
            if (c instanceof Class)
            {
                b = getClassifierBehavior((Class) c);
                if (b != null)
                    return b;
            }
        }
        return null;
    }
    
}
