package com.engisis.sysphs.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLStreamException;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EPackage.Registry;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.impl.DynamicEObjectImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xml.type.impl.AnyTypeImpl;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.UMLPlugin;
import org.eclipse.uml2.uml.profile.standard.StandardPackage;
import org.eclipse.uml2.uml.resource.UMLResource;
import org.eclipse.uml2.uml.resources.util.UMLResourcesUtil;

/**
 * Utility class for EMF
 * 
 * @author barbau
 *
 */
public class EMFUtil
{
    private static final Logger log = Logger.getLogger(EMFUtil.class);
    
    /**
     * Create a basic ResourceSet, without URL mappings
     * 
     * @return resource set
     * @throws FileNotFoundException
     */
    public static ResourceSet createBasicResourceSet() throws FileNotFoundException
    {
        // setup the ResourceSet environment
        ResourceSet rs = new ResourceSetImpl();
        rs.getLoadOptions().put(XMLResource.OPTION_RECORD_UNKNOWN_FEATURE, Boolean.TRUE);
        rs.getLoadOptions().put(XMLResource.OPTION_RECORD_ANY_TYPE_NAMESPACE_DECLARATIONS, Boolean.TRUE);
        rs.getLoadOptions().put(XMLResource.OPTION_DEFER_IDREF_RESOLUTION, Boolean.valueOf(false));
        Registry registry = rs.getPackageRegistry();
        
        // Namespace used by Papyrus
        registry.put(UMLPackage.eNS_URI, UMLPackage.eINSTANCE);
        registry.put(StandardPackage.eNS_URI, StandardPackage.eINSTANCE);
        
        Map<URI, URI> urimap = rs.getURIConverter().getURIMap();
        
        // Where to get the resources from (from UMLResourceUtil, JAR assumed)
        java.lang.Class<UMLResourcesUtil> umlre = UMLResourcesUtil.class;
        String url = umlre.getResource("/" + umlre.getName().replace('.', '/') + ".class").toExternalForm();
        if (url.indexOf("!/") == -1)
            throw new FileNotFoundException("The UML2 resource library must be in a JAR: " + url);
        URI umljar = URI.createURI(url.substring(0, url.lastIndexOf('!') + 2));
        
        UMLPlugin.getEPackageNsURIToProfileLocationMap().put(StandardPackage.eNS_URI,
                umljar.appendSegment("profiles").appendSegment("Standard.profile.uml").appendFragment("_0"));
        
        // map locations used in UML profiles and libraries
        urimap.put(URI.createURI(UMLResource.UML_METAMODEL_URI),
                umljar.appendSegment("metamodels").appendSegment("UML.metamodel.uml"));
        urimap.put(URI.createURI(UMLResource.UML_PRIMITIVE_TYPES_LIBRARY_URI),
                umljar.appendSegment("libraries").appendSegment("UMLPrimitiveTypes.library.uml"));
        
        // uris of MD17 and MD18
        String[] umluris = new String[] { "http://www.omg.org/spec/UML/20110701",
                "http://www.omg.org/spec/UML/20131001" };
        for (String umluri : umluris)
        {
            // register namespaces
            registry.put(umluri, UMLPackage.eINSTANCE);
            EPackage.Registry.INSTANCE.put(umluri, UMLPackage.eINSTANCE);
            
            // register standard profile namespace
            registry.put(umluri + "/StandardProfile", StandardPackage.eINSTANCE);
            
            // register standard profile location
            UMLPlugin.getEPackageNsURIToProfileLocationMap().put(umluri + "/StandardProfile",
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml").appendFragment("_0"));
            
            // register metamodel and library locations
            urimap.put(URI.createURI(umluri + "/UML.xmi"),
                    umljar.appendSegment("metamodels").appendSegment("UML.metamodel.uml"));
            urimap.put(URI.createURI(umluri + "/PrimitiveTypes.xmi"),
                    umljar.appendSegment("libraries").appendSegment("UMLPrimitiveTypes.library.uml"));
            urimap.put(URI.createURI(umluri + "/StandardProfile.xmi"),
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml"));
            urimap.put(URI.createURI(umluri + "/StandardProfileL2.xmi"),
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml"));
            urimap.put(URI.createURI(umluri + "/StandardProfileL3.xmi"),
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml"));
            
        }
        
        Map<String, Object> extension2factory = rs.getResourceFactoryRegistry().getExtensionToFactoryMap();
        
        // Extension used by Papyrus
        extension2factory.put(UMLResource.FILE_EXTENSION, UMLResource.Factory.INSTANCE);
        // Extension used by MD
        extension2factory.put("xmi", UMLResource.Factory.INSTANCE);
        extension2factory.put("xml", UMLResource.Factory.INSTANCE);
        
        // Old way
        
        // Mapping between the pathmaps used in Papyrus and the actual models
        urimap.put(URI.createURI(UMLResource.LIBRARIES_PATHMAP), umljar.appendSegment("libraries").appendSegment(""));
        urimap.put(URI.createURI(UMLResource.METAMODELS_PATHMAP), umljar.appendSegment("metamodels").appendSegment(""));
        urimap.put(URI.createURI(UMLResource.PROFILES_PATHMAP), umljar.appendSegment("profiles").appendSegment(""));
        
        return rs;
    }
    
    /**
     * Creates an EMF resource set with URL mappings
     * 
     * @return a ResourceSet able to read OMG UML files
     * @throws FileNotFoundException
     */
    public static ResourceSet createResourceSet() throws FileNotFoundException
    {
        ResourceSet rs = createBasicResourceSet();
        Map<URI, URI> urimap = rs.getURIConverter().getURIMap();
        for (Entry<URI, URI> localmapentry : Configuration.getLocationMappings().entrySet())
            urimap.put(localmapentry.getKey(), localmapentry.getValue());
        return rs;
    }
    
    /**
     * Filters and loads a file. Also, Defines UML profiles.
     * 
     * @param rs
     *            the ResourceSet in which the UML file should be loaded
     * @param fileuri
     *            the location of the file
     * @param nsmappings
     * @return loaded resource
     */
    public static Resource loadResource(final ResourceSet rs, URI fileuri, String[] paths)
    {
        log.info("Loading file " + fileuri);
        Resource r = rs.createResource(fileuri);
        try
        {
            PipedInputStream pis = new PipedInputStream(2048);
            PipedOutputStream pos = new PipedOutputStream(pis);
            
            XMIPreReader xmif = new XMIPreReader(rs, rs.getURIConverter().createInputStream(fileuri), pos, fileuri,
                    Configuration.getReaderIDMappings(), Configuration.getReaderNSMappings(), paths);
            Thread th = new Thread(xmif);
            // feeds pos
            th.start();
            // consumes pos
            try
            {
                r.load(pis, null);
            }
            catch (Exception e)
            {
                log.error(e);
            }
            
            pis.close();
            pos.close();
            
            log.debug("Resource " + r + " loaded");
            
            // profiles to load
            LinkedList<Profile> profiles = new LinkedList<Profile>();
            
            // profiles to apply
            Hashtable<Package, LinkedList<Profile>> applications = new Hashtable<Package, LinkedList<Profile>>();
            
            // parse content
            TreeIterator<EObject> allcontent = r.getAllContents();
            while (allcontent.hasNext())
            {
                EObject eo = allcontent.next();
                if (!(eo instanceof Package))
                {
                    allcontent.prune();
                    continue;
                }
                Package p = (Package) eo;
                
                LinkedList<Profile> lpa = new LinkedList<Profile>();
                applications.put(p, lpa);
                
                for (Profile prof : p.getAppliedProfiles())
                {
                    log.info("Identified applied profile " + prof + " in " + p);
                    lpa.add(prof);
                }
                if (p instanceof Profile)
                {
                    Profile prof = (Profile) p;
                    // profiles.add(prof);
                    profiles.add(prof);
                    flatten(prof);
                    String profURI = prof.getURI();
                    if (!prof.isDefined())
                    {
                        EPackage definition = prof.define();
                        log.debug("Definition: " + definition.getName() + "  with " + definition.getNsPrefix() + " :"
                                + definition.getNsURI() + " at " + EcoreUtil.getURI(definition) + "--"
                                + EcoreUtil.getURI(prof));
                        
                        log.debug("Registering " + profURI + " for " + definition.getNsURI());
                        UMLPlugin.getEPackageNsURIToProfileLocationMap().put(profURI, EcoreUtil.getURI(definition));
                        rs.getPackageRegistry().put(profURI, definition);
                        String alt = Configuration.getReaderNSMappings().get(profURI);
                        if (alt != null)
                        {
                            log.debug("Registering " + alt + " for " + definition.getNsURI());
                            UMLPlugin.getEPackageNsURIToProfileLocationMap().put(alt, EcoreUtil.getURI(definition));
                            rs.getPackageRegistry().put(alt, definition);
                        }
                    }
                    else
                        log.debug("Profile " + profURI + " defined");
                    log.debug("Profile " + prof + " defined by " + prof.getDefinition());
                }
            }
            // attempt to recreate stereotype applications
            List<DynamicEObjectImpl> ldeos = new LinkedList<DynamicEObjectImpl>();
            for (EObject eo : r.getContents())
            {
                if (eo instanceof AnyTypeImpl)
                {
                    AnyTypeImpl at = (AnyTypeImpl) eo;
                    String name = at.eClass().getName();
                    for (Profile profile : profiles)
                    {
                        for (EClassifier ec : profile.getDefinition().getEClassifiers())
                        {
                            if (ec instanceof EClass && ec.getName().equals(name))
                            {
                                DynamicEObjectImpl deo = new DynamicEObjectImpl((EClass) ec);
                                ldeos.add(deo);
                                for (FeatureMap.Entry fm : at.getAnyAttribute())
                                {
                                    String attname = fm.getEStructuralFeature().getName();
                                    EStructuralFeature esf = ((EClass) ec).getEStructuralFeature(attname);
                                    if (esf != null)
                                    {
                                        if (esf instanceof EAttribute)
                                        {
                                            String etypename = ((EAttribute) esf).getEType().getInstanceClassName();
                                            try
                                            {
                                                if (etypename.equals("boolean"))
                                                {
                                                    deo.eSet(esf, Util.toBoolean(fm.getValue().toString(), false));
                                                }
                                                else if (etypename.equals("int"))
                                                {
                                                    deo.eSet(esf, Util.toInt(fm.getValue().toString(), 0));
                                                }
                                                else if (etypename.equals("double"))
                                                {
                                                    deo.eSet(esf, Util.toDouble(fm.getValue().toString(), 0.0));
                                                }
                                                else if (etypename.equals("java.lang.String"))
                                                {
                                                    deo.eSet(esf, fm.getValue().toString());
                                                }
                                                else
                                                    log.warn("Unsupported type: " + etypename);
                                            }
                                            catch (Exception e)
                                            {
                                                log.warn("Can't assign value " + fm.getValue() + " to " + esf);
                                            }
                                            
                                            continue;
                                        }
                                        else if (esf instanceof EReference)
                                        {
                                            deo.eSet(esf, r.getEObject(fm.getValue().toString()));
                                            continue;
                                        }
                                    }
                                    log.error("Coundn't set value of " + ec + "::" + esf);
                                }
                            }
                        }
                    }
                }
            }
            r.getContents().addAll(ldeos);
            for (Entry<Package, LinkedList<Profile>> app : applications.entrySet())
            {
                for (Profile prof : app.getValue())
                {
                    log.info("Applying profile " + prof + " at " + EcoreUtil.getURI(prof) + " defined by "
                            + prof.getDefinition());
                    app.getKey().applyProfile(prof);
                }
            }
            
            // r.save(new
            // FileOutputStream(r.getURI().appendFileExtension("defined").toFileString()),
            // null);
            
        }
        catch (IOException e)
        {
            log.error("Couldn't read the file", e);
        }
        catch (XMLStreamException e)
        {
            log.error("Error during the XML parsing", e);
        }
        catch (FactoryConfigurationError e)
        {
            log.error("Error during the StAX configuration", e);
        }
        catch (Exception e)
        {
            log.error("Reading error", e);
        }
        EcoreUtil.resolveAll(r);
        return r;
    }
    
    /**
     * Saves the resource using the specified XMI serialization target
     * 
     * @param r
     *            resource to save
     * @param target
     *            XMI serialization target
     * @throws IOException
     */
    public static void saveResource(final Resource r, String target) throws IOException
    {
        // annoying
        final Map<Object, Object> opts = new HashMap<Object, Object>();
        opts.put(XMLResource.OPTION_DECLARE_XML, Boolean.TRUE);
        opts.put(XMLResource.OPTION_ENCODING, "UTF-8");
        
        if (r instanceof XMLResource)
        {
            ((XMLResource) r).setEncoding("UTF-8");
        }
        
        PipedInputStream pis = new PipedInputStream();
        final PipedOutputStream pos = new PipedOutputStream(pis);
        XMIPostWriter cleaner = new XMIPostWriter(pis, r.getURI().toFileString(),
                Configuration.getWriterIDMappings(target), Configuration.getWriterNSMappings(target));
        Thread t = new Thread(new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    r.save(pos, opts);
                    pos.close();
                }
                catch (IOException e)
                {
                    log.error("Can't save " + r.getURI(), e);
                }
            }
        });
        t.start();
        cleaner.run();
        pis.close();
        pos.close();
        log.info("Saved " + r.getURI());
    }
    
    /**
     * Flattens the profile, due to UML2 not being able to access stereotypes in
     * nested profiles
     * 
     * @param profile
     *            profile to flatten
     */
    private static void flatten(Profile profile)
    {
        move(profile, profile);
    }
    
    /**
     * Moves the content of a package to specified profile
     * 
     * @param from
     *            package from which the content is moved
     * @param to
     *            destination of the moved content
     */
    private static void move(Package from, Profile to)
    {
        int i = 0;
        while (i != from.getPackagedElements().size())
        {
            PackageableElement pe = from.getPackagedElements().get(i);
            if (pe instanceof Package)
            {
                move((Package) pe, to);
                from.getPackagedElements().remove(i);
            }
            else if (from != to)
            {
                from.getPackagedElements().remove(i);
                to.getPackagedElements().add(pe);
            }
            else
                i++;
        }
    }
    
    /**
     * Creates a basic SysML model
     * 
     * @param rs
     *            resource set in which the resource is created
     * @param file
     *            location of the model
     * @param paths
     *            List of directories
     * @return resource representing the model
     */
    public static Resource createSysMLModel(ResourceSet rs, URI file, String[] paths)
    {
        log.info("Creating " + file);
        Resource r = rs.createResource(file);
        
        Model model = UMLFactory.eINSTANCE.createModel();
        model.setName("data");
        r.getContents().add(model);
        
        Resource sysml = loadResource(rs, URI.createURI("http://www.omg.org/spec/SysML/20161101/SysML.xmi"), null);
        Profile sysmlprofile = (Profile) EcoreUtil.getObjectByType(sysml.getContents(), UMLPackage.Literals.PROFILE);
        model.applyProfile(sysmlprofile);
        
//        File fprofile = locateFile(paths, "SysPhSProfile.xmi");
//        if (fprofile != null)
//        {
//            Resource simulation = loadResource(rs, URI.createFileURI(fprofile.getAbsolutePath()), null);
//            Profile simulationprofile = (Profile) EcoreUtil.getObjectByType(simulation.getContents(),
//                    UMLPackage.Literals.PROFILE);
//            model.applyProfile(simulationprofile);
//        }
//        else
//            log.error("Unable to locate SysPhS profile file in the path");
        
        Resource sysphs = loadResource(rs, URI.createURI("http://www.omg.org/spec/SysPhS/20171215/SysPhSProfile.xmi"), null);
        Profile sysphsprofile = (Profile) EcoreUtil.getObjectByType(sysphs.getContents(), UMLPackage.Literals.PROFILE);
        model.applyProfile(sysphsprofile);
        
        return r;
    }
    
    /**
     * Returns a resource for the simulation library
     * 
     * @param rs
     *            resource set in which the library is loaded
     * @param file
     *            location from which the library is loaded
     * @param paths
     *            List of directories
     * @return the simulation library
     */
    public static Resource getSimulationLibrary(ResourceSet rs, URI file, String[] paths)
    {
        return loadResource(rs, URI.createURI("http://www.omg.org/spec/SysPhS/20171215/SysPhSLibrary.xmi"), null);
//        File flibrary = locateFile(paths, "SysPhSLibrary.xmi");
//        if (flibrary != null)
//            return loadResource(rs, URI.createFileURI(flibrary.getAbsolutePath()), null);
//        log.error("Unable to locate SysPhS library file in the path");
//        return null;
    }
    
    private static File locateFile(String[] paths, String name)
    {
        StringBuilder sb = new StringBuilder();
        for (String path : paths)
        {
            sb.append(" " + path);
            File f = new File(path, name);
            if (f.exists())
                return f;
        }
        log.error("Couldn't locate " + name + " in" + sb.toString());
        return null;
    }
}
