package com.engisis.sysphs.serialization.simulink;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.simulink.SFContinuousStateVariable;
import com.engisis.sysphs.language.simulink.SFDWorkAssignment;
import com.engisis.sysphs.language.simulink.SFDWorkVariable;
import com.engisis.sysphs.language.simulink.SFDerivativeAssignment;
import com.engisis.sysphs.language.simulink.SFDiscreteStateVariable;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputAssignment;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFUpdateAssignment;
import com.engisis.sysphs.language.simulink.SFVariable;
import com.engisis.sysphs.language.simulink.SFVariableAssignment;
import com.engisis.sysphs.language.simulink.SFunction1;
import com.engisis.sysphs.util.TextWriter;

/**
 * Writes Level-1 S-Functions
 * 
 * @author barbau
 *
 */
public class SimulinkSF1Writer extends TextWriter
{
    /**
     * Constructs a Level-1 S-Functions writer
     * 
     * @param w
     *            object to write the content into
     */
    protected SimulinkSF1Writer(Writer w)
    {
        super(w);
    }
    
    public void visit(SFunction1 object) throws IOException
    {
        // outputs:
        // sys: output (depends on the flag), x0: initial state values,str:
        // ??,ts: sample time,ssc: ??
        // inputs:
        // t: time, x: state variables, u: inputs, flag: function that is called
        List<SFInputVariable> in = object.getInputs();
        List<SFContinuousStateVariable> cst = object.getContinuousStates();
        List<SFDiscreteStateVariable> dst = object.getDiscreteStates();
        List<SFOutputVariable> out = object.getOutputs();
        List<SFVariable> st = new ArrayList<SFVariable>(cst);
        st.addAll(dst);
        
        // build parameter list
        StringBuilder sbpar = new StringBuilder();
        for (SFDWorkVariable sdw : object.getDWorkStates())
            sbpar.append("," + sdw.getName());
        
        // build assignments
        List<String> lUpdate = new LinkedList<String>();
        List<String> lDeriv = new LinkedList<String>();
        List<String> lOutput = new LinkedList<String>();
        List<String> lDWork = new LinkedList<String>();
        
        for (SFVariableAssignment sassignment : object.getAssignments())
        {
            String expr = sassignment.getExpression() + ";";
            if (sassignment instanceof SFUpdateAssignment)
                lUpdate.add(expr);
            else if (sassignment instanceof SFDerivativeAssignment)
                lDeriv.add(expr);
            else if (sassignment instanceof SFOutputAssignment)
                lOutput.add(expr);
            else if (sassignment instanceof SFDWorkAssignment)
                lDWork.add(expr);
            
        }
        
        append("function [sys,x0,str,ts] = " + object.getName() + "(t,x,u,flag)", TL);
        indent();
        for (String s : lDWork)
            append(s, TL);
        append("switch flag,", TL);
        append("case 0,", TL);
        append("[sys,x0,str,ts]=mdlInitializeSizes;", TL);
        
        // Derivatives
        append("case 1,", TL);
        indent();
        if (cst.size() != 0)
            append("sys=mdlDerivatives(t,x,u" + sbpar.toString() + ");", TL);
        else
            append("sys=[];", TL);
        unindent();
        
        // Update
        append("case 2,", TL);
        indent();
        if (dst.size() != 0)
            append("sys=mdlUpdate(t,x,u" + sbpar.toString() + ");", TL);
        else
            append("sys=[];", TL);
        unindent();
        
        // Outputs
        append("case 3,", TL);
        indent();
        if (out.size() != 0)
            append("sys=mdlOutputs(t,x,u" + sbpar.toString() + ");", TL);
        else
            append("sys=[];", TL);
        unindent();
        
        // To decide variable discrete sample time
        append("case 4,", TL);
        indent();
        append("sys=[];", TL);
        unindent();
        
        // Terminate
        append("case 9,", TL);
        indent();
        append("sys=[];", TL);
        unindent();
        append("end", TL);
        
        append("function [sys,x0,str,ts] = mdlInitializeSizes()", TL);
        indent();
        append("sizes = simsizes;", TL);
        
        // either the first dimension of the first state, or the number of
        // states
        if (cst.size() == 1 && cst.get(0).getDimensions().size() == 1)
            append("sizes.NumContStates = " + cst.get(0).getDimensions().get(0) + ";", TL);
        else
            append("sizes.NumContStates = " + cst.size() + ";", TL);
        
        append("sizes.NumDiscStates = " + dst.size() + ";", TL);
        append("sizes.NumOutputs = " + out.size() + ";", TL);
        append("sizes.NumInputs = " + in.size() + ";", TL);
        append("sizes.DirFeedthrough = 1;", TL);
        append("sizes.NumSampleTimes = 1;", TL);
        append("sys = simsizes(sizes);", TL);
        append("str = [];", TL);
        
        append("x0  = [", T);
        // build state variable definition
        for (SFVariable svar : st)
        {
            if (svar != st.get(0))
                append(";");
            if (svar.getValue() != null)
            {
                String v = getValue(svar);
                if (v.isEmpty())
                    append("0");
                else
                    append(v);
            }
            else
            {
                if (svar.getDimensions().size() == 0)
                    append("0");
                else
                {
                    for (int i = 0; i < (svar.getDimensions().get(0)).intValue(); i++)
                    {
                        if (i != 0)
                            append(";");
                        append("0");
                    }
                }
            }
        }
        append("];", L);
        append("ts  = [0 0];", TL);
        unindent();
        append("end", TL);
        
        if (cst.size() != 0)
        {
            append("function sys=mdlDerivatives(t,x,u" + sbpar.toString() + ")", TL);
            indent();
            for (String s : lDeriv)
                append(s, TL);
            unindent();
            append("end", TL);
        }
        if (dst.size() != 0)
        {
            append("function sys=mdlUpdate(t,x,u" + sbpar.toString() + ")", TL);
            indent();
            for (String s : lUpdate)
                append(s, TL);
            unindent();
            append("end", TL);
        }
        if (out.size() != 0)
        {
            append("function sys=mdlOutputs(t,x,u" + sbpar.toString() + ")", TL);
            indent();
            for (String s : lOutput)
                append(s, TL);
            unindent();
            append("end", TL);
        }
        
        append("end", TL);
    }
    
    public static String getValue(SFVariable svar)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < svar.getValue().size(); i++)
        {
            SDataValue sdv = svar.getValue().get(i);
            if (i != 0)
                sb.append(";");
            if (sdv instanceof SExpressionValue)
                sb.append(((SExpressionValue) sdv).getValue());
            else if (sdv instanceof SDoubleValue)
                sb.append(((SDoubleValue) sdv).getValue());
        }
        return sb.toString();
    }
}
