package com.engisis.sysphs.deserialization.simulink;

import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SFContinuousStateVariable;
import com.engisis.sysphs.language.simulink.SFDerivativeAssignment;
import com.engisis.sysphs.language.simulink.SFDiscreteStateVariable;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputAssignment;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFUpdateAssignment;
import com.engisis.sysphs.language.simulink.SFunction2;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SimulinkFactory;
import com.engisis.sysphs.translation.simulink.SimulinkUtil;
import com.engisis.sysphs.generation.simulink.MATLABBaseListener;
import com.engisis.sysphs.generation.simulink.MATLABParser.AssignmentContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr10Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr11Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr2Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr3Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr4Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr5Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr6Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr7Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr8Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr9Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.ExprContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.F_inputContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.FileContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.FunctionContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Id_argContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.ReferenceContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.StatementContext;

import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Map.Entry;

import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeProperty;
import org.antlr.v4.runtime.tree.ParseTreeWalker;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.apache.log4j.Logger;

/**
 * Parser of level 2 S-Function
 * 
 * @author barbau
 *
 */
public class SimulinkSF2Reader extends MATLABBaseListener implements SimulinkReader
{
    private static final Logger log = Logger.getLogger(SimulinkSF2Reader.class);
    /**
     * Object values associated with syntax tree nodes
     */
    private ParseTreeProperty<Object> values;
    /**
     * Substitutions of string to perform
     */
    private Hashtable<String, String> substitutions;
    /**
     * Root of the syntax tree
     */
    private ParserRuleContext prc;
    
    /**
     * Constructs a parser for a Level-2 S-Function
     * 
     * @param prc
     */
    public SimulinkSF2Reader(ParserRuleContext prc)
    {
        values = new ParseTreeProperty<Object>();
        substitutions = new Hashtable<String, String>();
        this.prc = prc;
    }
    
    @Override
    public void process()
    {
        new ParseTreeWalker().walk(this, prc);
    }
    
    /**
     * Names of the update, derivative, and output functions
     */
    private String fupd, fder, fout;
    /**
     * Root S-Function
     */
    private SFunction2 sf2;
    
    @Override
    public void enterFile(FileContext ctx)
    {
        sf2 = SimulinkFactory.eINSTANCE.createSFunction2();
        SimulinkUtil.setName(sf2, ctx.function(0).f_def_line().ID(0).getText());
        log.info("Creating SFunction " + sf2.getName());
    }
    
    @SuppressWarnings("boxing")
    @Override
    public void exitFunction(FunctionContext ctx)
    {
        String block;
        String name = ctx.f_def_line().ID(0).getText();
        
        F_inputContext fic = ctx.f_def_line().f_input();
        if (fic != null && fic.ID().size() == 1)
        {
            block = fic.ID().get(0).getText();
            
            if (name.equals("setup"))
            {
                log.debug("Setup function detected");
                for (StatementContext sc : ctx.statement())
                {
                    AssignmentContext ac = sc.assignment();
                    if (ac != null)
                    {
                        if (ac.reference().id_arg().size() == 2
                                && ac.reference().id_arg().get(0).ID().getText().equals(block)
                                && ac.reference().id_arg().get(0).argument_list() == null)
                        {
                            String prop = ac.reference().id_arg().get(1).ID().getText();
                            Object val = values.get(ac.expr());
                            if (prop.equals("NumInputPorts"))
                            {
                                if (val instanceof Integer)
                                {
                                    for (int i = 0; i < (Integer) val; i++)
                                    {
                                        log.debug("Adding input variable");
                                        SFInputVariable sfiv = SimulinkFactory.eINSTANCE.createSFInputVariable();
                                        sf2.getVariables().add(sfiv);
                                        SimulinkUtil.setName(sfiv, "_iv" + i);
                                        substitutions.put(".InputPort(" + (i + 1) + ").Data", sfiv.getName());
                                        
                                        SInport sinport = SimulinkFactory.eINSTANCE.createSInport();
                                        sinport.setOwningSystem(sf2);
                                        SimulinkUtil.setName(sinport, sfiv.getName());
                                    }
                                }
                            }
                            else if (prop.equals("NumOutputPorts"))
                            {
                                if (val instanceof Integer)
                                {
                                    for (int i = 0; i < (Integer) val; i++)
                                    {
                                        log.debug("Adding output variable");
                                        SFOutputVariable sfov = SimulinkFactory.eINSTANCE.createSFOutputVariable();
                                        sf2.getVariables().add(sfov);
                                        SimulinkUtil.setName(sfov, "_ov" + i);
                                        substitutions.put(".OutputPort(" + (i + 1) + ").Data", sfov.getName());
                                        
                                        SOutport soutport = SimulinkFactory.eINSTANCE.createSOutport();
                                        soutport.setOwningSystem(sf2);
                                        SimulinkUtil.setName(soutport, sfov.getName());
                                    }
                                }
                            }
                            else if (prop.equals("NumContStates"))
                            {
                                if (val instanceof Integer)
                                {
                                    for (int i = 0; i < (Integer) val; i++)
                                    {
                                        log.debug("Adding continuous state");
                                        SFContinuousStateVariable sfcs = SimulinkFactory.eINSTANCE
                                                .createSFContinuousStateVariable();
                                        sf2.getVariables().add(sfcs);
                                        SimulinkUtil.setName(sfcs, "_cs" + i);
                                        substitutions.put(".ContStates.Data(" + (i + 1) + ")", sfcs.getName());
                                        substitutions.put(".Derivatives.Data(" + (i + 1) + ")", "der(" + sfcs.getName()
                                                + ")");
                                    }
                                }
                            }
                            else if (prop.equals("NumDiscStates"))
                            {
                                if (val instanceof Integer)
                                {
                                    for (int i = 0; i < (Integer) val; i++)
                                    {
                                        log.debug("Adding discrete state");
                                        SFDiscreteStateVariable sfds = SimulinkFactory.eINSTANCE
                                                .createSFDiscreteStateVariable();
                                        sf2.getVariables().add(sfds);
                                        SimulinkUtil.setName(sfds, "_ds" + i);
                                        substitutions.put(".DiscStates.Data(" + (i + 1) + ")", "pre(" + sfds.getName()
                                                + ")");
                                        substitutions.put(".Dwork(" + (i + 1) + ").Data", sfds.getName());
                                    }
                                }
                            }
                            
                        }
                    }
                    ExprContext ec = sc.expr();
                    if (ec != null)
                    {
                        if (values.get(ec) instanceof ReferenceContext)
                        {
                            ReferenceContext rc = (ReferenceContext) values.get(ec);
                            List<Id_argContext> ref = rc.id_arg();
                            if (ref.size() == 2)
                            {
                                Id_argContext ref0 = ref.get(0);
                                Id_argContext ref1 = ref.get(1);
                                if (ref0.getText().equals(block) && ref1.ID().getText().equals("RegBlockMethod")
                                        && ref1.argument_list() != null)
                                {
                                    List<ExprContext> exprs = ref1.argument_list().expr();
                                    if (exprs.size() == 2)
                                    {
                                        Object o0 = values.get(exprs.get(0));
                                        Object o1 = values.get(exprs.get(1));
                                        if (o0 instanceof String && o1 instanceof TerminalNode)
                                        {
                                            if (o0.equals("Outputs"))
                                                fout = ((TerminalNode) o1).getText();
                                            else if (o0.equals("Updates"))
                                                fupd = ((TerminalNode) o1).getText();
                                            else if (o0.equals("Derivatives"))
                                                fder = ((TerminalNode) o1).getText();
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if (name.equals(fder))
            {
                log.debug("Derivative function detected");
                for (StatementContext sc : ctx.statement())
                {
                    AssignmentContext ac = sc.assignment();
                    if (ac != null)
                    {
                        String lhs = ac.reference().getText();
                        String rhs = ac.expr().getText();
                        for (int i = 0; i < sf2.getContinuousStates().size(); i++)
                        {
                            if (lhs.equals(block + ".Derivatives.Data(" + (i + 1) + ")"))
                            {
                                SFDerivativeAssignment sfda = SimulinkFactory.eINSTANCE.createSFDerivativeAssignment();
                                sfda.setVariable(sf2.getContinuousStates().get(i));
                                for (Entry<String, String> ent : substitutions.entrySet())
                                {
                                    lhs = lhs.replace(block + ent.getKey(), ent.getValue());
                                    rhs = rhs.replace(block + ent.getKey(), ent.getValue());
                                }
                                sfda.setExpression(lhs + "=" + rhs);
                                sf2.getAssignments().add(sfda);
                                log.debug("Found derivative assignment");
                            }
                        }
                    }
                }
            }
            else if (name.equals(fupd))
            {
                log.debug("Update function detected");
                for (StatementContext sc : ctx.statement())
                {
                    AssignmentContext ac = sc.assignment();
                    if (ac != null)
                    {
                        String lhs = ac.reference().getText();
                        String rhs = ac.expr().getText();
                        for (int i = 0; i < sf2.getDiscreteStates().size(); i++)
                        {
                            if (lhs.equals(block + ".DiscStates.Data(" + (i + 1) + ")"))
                            {
                                SFUpdateAssignment sfua = SimulinkFactory.eINSTANCE.createSFUpdateAssignment();
                                sfua.setVariable(sf2.getDiscreteStates().get(i));
                                for (Entry<String, String> ent : substitutions.entrySet())
                                {
                                    lhs = lhs.replace(block + ent.getKey(), ent.getValue());
                                    rhs = rhs.replace(block + ent.getKey(), ent.getValue());
                                }
                                sfua.setExpression(lhs + "=" + rhs);
                                sf2.getAssignments().add(sfua);
                                log.debug("Found update assignment");
                            }
                        }
                    }
                }
            }
            else if (name.equals(fout))
            {
                log.debug("Output function detected");
                for (StatementContext sc : ctx.statement())
                {
                    AssignmentContext ac = sc.assignment();
                    if (ac != null)
                    {
                        String lhs = ac.reference().getText();
                        String rhs = ac.expr().getText();
                        for (int i = 0; i < sf2.getOutputs().size(); i++)
                        {
                            if (lhs.equals(block + ".OutputPort(" + (i + 1) + ").Data"))
                            {
                                SFOutputAssignment sfoa = SimulinkFactory.eINSTANCE.createSFOutputAssignment();
                                sfoa.setVariable(sf2.getOutputs().get(i));
                                for (Entry<String, String> ent : substitutions.entrySet())
                                {
                                    lhs = lhs.replace(block + ent.getKey(), ent.getValue());
                                    rhs = rhs.replace(block + ent.getKey(), ent.getValue());
                                }
                                sfoa.setExpression(lhs + "=" + rhs);
                                sf2.getAssignments().add(sfoa);
                                log.debug("Found output assignment");
                            }
                        }
                    }
                }
            }
        }
    }
    
    @Override
    public void exitExpr(ExprContext ctx)
    {
        if (ctx.expr2().size() == 1)
            transfer(ctx, ctx.expr2().get(0));
    }
    
    @Override
    public void exitExpr2(Expr2Context ctx)
    {
        if (ctx.expr3().size() == 1)
            transfer(ctx, ctx.expr3().get(0));
    }
    
    @Override
    public void exitExpr3(Expr3Context ctx)
    {
        if (ctx.expr4().size() == 1)
            transfer(ctx, ctx.expr4().get(0));
    }
    
    @Override
    public void exitExpr4(Expr4Context ctx)
    {
        if (ctx.expr5().size() == 1)
            transfer(ctx, ctx.expr5().get(0));
    }
    
    @Override
    public void exitExpr5(Expr5Context ctx)
    {
        if (ctx.expr6().size() == 1)
            transfer(ctx, ctx.expr6().get(0));
    }
    
    @Override
    public void exitExpr6(Expr6Context ctx)
    {
        if (ctx.expr7().size() == 1)
            transfer(ctx, ctx.expr7().get(0));
    }
    
    @Override
    public void exitExpr7(Expr7Context ctx)
    {
        if (ctx.expr8().size() == 1)
            transfer(ctx, ctx.expr8().get(0));
    }
    
    @Override
    public void exitExpr8(Expr8Context ctx)
    {
        if (ctx.expr9().size() == 1)
            transfer(ctx, ctx.expr9().get(0));
    }
    
    @Override
    public void exitExpr9(Expr9Context ctx)
    {
        transfer(ctx, ctx.expr10());
    }
    
    @Override
    public void exitExpr10(Expr10Context ctx)
    {
        if (ctx.expr11().size() == 1)
            transfer(ctx, ctx.expr11().get(0));
    }
    
    @Override
    public void exitExpr11(Expr11Context ctx)
    {
        
        if (ctx.INT() != null)
            values.put(ctx, Integer.valueOf(ctx.INT().getText()));
        else if (ctx.FLOAT() != null)
            values.put(ctx, Double.valueOf(ctx.FLOAT().getText()));
        else if (ctx.STRING() != null)
            values.put(ctx, ctx.STRING().getText().substring(1, ctx.STRING().getText().length() - 1));
        else if (ctx.reference() != null)
            values.put(ctx, ctx.reference());
        else if (ctx.ID() != null)
            values.put(ctx, ctx.ID());
        else if (ctx.expr() != null)
            transfer(ctx, ctx.expr());
    }
    
    private void transfer(ParseTree to, ParseTree from)
    {
        values.put(to, values.get(from));
    }
    
    @Override
    public Collection<SimulinkReference> getReferences()
    {
        return null;
    }
    
    @Override
    public int getPriority()
    {
        return 2;
    }
    
    @Override
    public SElement getRootElement()
    {
        return sf2;
    }
}
