package com.engisis.sysphs.deserialization.simulink;

import com.engisis.sysphs.deserialization.simulink.SimulinkReference.ReferenceType;
import com.engisis.sysphs.language.simscape.SBranch;
import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SComponentReference;
import com.engisis.sysphs.language.simscape.SConnection;
import com.engisis.sysphs.language.simscape.SConnectionPortBlock;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SEquation;
import com.engisis.sysphs.language.simscape.SInput;
import com.engisis.sysphs.language.simscape.SLocation;
import com.engisis.sysphs.language.simscape.SMember;
import com.engisis.sysphs.language.simscape.SMemberAccess;
import com.engisis.sysphs.language.simscape.SNode;
import com.engisis.sysphs.language.simscape.SOutput;
import com.engisis.sysphs.language.simscape.SParameter;
import com.engisis.sysphs.language.simscape.SVariable;
import com.engisis.sysphs.language.simscape.SimscapeFactory;
import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SimulinkFactory;
import com.engisis.sysphs.translation.simulink.SimulinkUtil;
import com.engisis.sysphs.generation.simulink.SimscapeBaseListener;
import com.engisis.sysphs.generation.simulink.SimscapeLexer;
import com.engisis.sysphs.generation.simulink.SimscapeParser.BranchContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.BranchesContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ComponentContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ComponentsContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ConnectionContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ConnectionsContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.DeclarationContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.DomainContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.EquationContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.EquationsContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr10Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr11Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr2Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr3Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr4Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr5Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr6Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr7Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr8Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.Expr9Context;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ExprContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.InputsContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.NodesContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ObjectContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.OutputsContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ParametersContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.VariablesContext;

import java.util.Collection;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;

import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.Lexer;
import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.Token;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeProperty;
import org.antlr.v4.runtime.tree.ParseTreeWalker;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.apache.log4j.Logger;

/**
 * First-pass parser of a Simscape file
 * 
 * @author barbau
 *
 */
public class SimscapeReaderPass1 extends SimscapeBaseListener implements SimulinkReader
{
    private static final Logger log = Logger.getLogger(SimscapeReaderPass1.class);
    
    /**
     * Collection of references
     */
    private Collection<SimulinkReference> references;
    /**
     * Objects associated with syntax tree nodes
     */
    private ParseTreeProperty<SElement> objects;
    /**
     * Simple expressions associated with syntax tree nodes
     */
    private ParseTreeProperty<ParseTree> exprs;
    
    /**
     * Map with Simscape objects as keys, and syntax tree nodes as values
     */
    private Hashtable<SElement, ParseTree> expressions;
    
    /**
     * Current Simscape component
     */
    private SComponent scomponent = null;
    /**
     * Current Simscape domain
     */
    private SDomain sdomain = null;
    
    /**
     * Root of the syntax tree
     */
    private ParserRuleContext prc;
    /**
     * Original token stream
     */
    private CommonTokenStream cts;
    
    /**
     * Constructs a first-pass reader
     * 
     * @param cts
     *            original token stream
     * @param prc
     *            root syntax tree node
     */
    public SimscapeReaderPass1(CommonTokenStream cts, ParserRuleContext prc)
    {
        this.cts = cts;
        this.prc = prc;
        references = new LinkedList<SimulinkReference>();
        objects = new ParseTreeProperty<SElement>();
        exprs = new ParseTreeProperty<ParseTree>();
    }
    
    /**
     * Sets the map between Simulink objects and their syntax tree nodes
     * 
     * @param expressions
     */
    public void setExpressions(Hashtable<SElement, ParseTree> expressions)
    {
        this.expressions = expressions;
    }
    
    @Override
    public void process()
    {
        new ParseTreeWalker().walk(this, prc);
    }
    
    @Override
    public void enterComponent(ComponentContext ctx)
    {
        log.info("Defining component " + ctx.ID().getText());
        scomponent = SimscapeFactory.eINSTANCE.createSComponent();
        objects.put(ctx, scomponent);
        SimulinkUtil.setName(scomponent, ctx.ID().getText());
        if (ctx.reference() != null)
        {
            references.add(new SimulinkReference(scomponent, ctx.reference().getText(), ReferenceType.COMPONENT));
        }
    }
    
    @Override
    public void enterDomain(DomainContext ctx)
    {
        log.info("Defining domain " + ctx.ID());
        sdomain = SimscapeFactory.eINSTANCE.createSDomain();
        objects.put(ctx, sdomain);
        SimulinkUtil.setName(sdomain, ctx.ID().getText());
    }
    
    @Override
    public void enterNodes(NodesContext ctx)
    {
        for (int i = 0; i < ctx.declaration().size(); i++)
        {
            DeclarationContext dc = ctx.declaration(i);
            String name = dc.ID().getText();
            log.info("Defining node " + name);
            SNode snode = SimscapeFactory.eINSTANCE.createSNode();
            if (scomponent != null)
                scomponent.getOwnedMembers().add(snode);
            
            SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            scpb.setMember(snode);
            scpb.setOwningSystem(scomponent);
            
            SimulinkUtil.setName(snode, name);
            objects.put(dc, snode);
            
            // get position
            Token semi = ctx.SEMI(i).getSymbol();
            int j = semi.getTokenIndex();
            List<Token> lts = cts.getHiddenTokensToRight(j, Lexer.HIDDEN);
            if (lts != null)
            {
                String str = lts.get(0).getText();
                if (str.contains(":left"))
                {
                    snode.setLocation(SLocation.LEFT);
                    scpb.setLocation(SLocation.LEFT);
                }
                else if (str.contains(":right"))
                {
                    snode.setLocation(SLocation.RIGHT);
                    scpb.setLocation(SLocation.RIGHT);
                }
            }
            
            references.add(new SimulinkReference(snode, dc.expr().getText(), ReferenceType.DOMAIN));
        }
    }
    
    @Override
    public void enterVariables(VariablesContext ctx)
    {
        SMemberAccess access = SMemberAccess.PUBLIC;
        boolean balancing = false;
        
        if (ctx.modifiers() != null)
            for (DeclarationContext ac : ctx.modifiers().declaration())
            {
                if (ac.ID().getText().equals("Access"))
                {
                    String val = ac.expr().getText();
                    if (val.equals("private"))
                        access = SMemberAccess.PRIVATE;
                    else if (val.equals("protected"))
                        access = SMemberAccess.PROTECTED;
                }
                else if (ac.ID().getText().equals("Balancing"))
                {
                    if (ac.expr().getText().equals("true"))
                    {
                        balancing = true;
                    }
                }
            }
        
        for (DeclarationContext dc : ctx.declaration())
        {
            String name = dc.ID().getText();
            log.info("Defining variable " + name);
            SVariable svariable = SimscapeFactory.eINSTANCE.createSVariable();
            if (scomponent != null)
                scomponent.getOwnedMembers().add(svariable);
            if (sdomain != null)
                sdomain.getVariables().add(svariable);
            objects.put(dc, svariable);
            SimulinkUtil.setName(svariable, name);
            svariable.setAccess(access);
            svariable.setBalancing(balancing);
            
        }
    }
    
    @Override
    public void exitVariables(VariablesContext ctx)
    {
        for (DeclarationContext dc : ctx.declaration())
        {
            SVariable svariable = (SVariable) objects.get(dc);
            setMemberValue(svariable, dc.expr());
        }
    }
    
    @Override
    public void enterParameters(ParametersContext ctx)
    {
        SMemberAccess access = SMemberAccess.PUBLIC;
        
        if (ctx.modifiers() != null)
            for (DeclarationContext ac : ctx.modifiers().declaration())
            {
                if (ac.ID().equals("Access"))
                {
                    String val = ac.expr().getText();
                    if (val.equals("private"))
                        access = SMemberAccess.PRIVATE;
                    else if (val.equals("protected"))
                        access = SMemberAccess.PROTECTED;
                }
            }
        
        for (DeclarationContext dc : ctx.declaration())
        {
            String name = dc.ID().getText();
            log.info("Defining parameter " + name);
            SParameter sparameter = SimscapeFactory.eINSTANCE.createSParameter();
            if (scomponent != null)
                scomponent.getOwnedMembers().add(sparameter);
            if (sdomain != null)
                sdomain.getParameters().add(sparameter);
            SimulinkUtil.setName(sparameter, name);
            objects.put(dc, sparameter);
            sparameter.setAccess(access);
        }
    }
    
    @Override
    public void exitParameters(ParametersContext ctx)
    {
        for (DeclarationContext dc : ctx.declaration())
        {
            SParameter sparameter = (SParameter) objects.get(dc);
            setMemberValue(sparameter, dc.expr());
        }
    }
    
    @Override
    public void enterInputs(InputsContext ctx)
    {
        for (int i = 0; i < ctx.declaration().size(); i++)
        {
            DeclarationContext dc = ctx.declaration(i);
            String name = dc.ID().getText();
            log.info("Defining input " + name);
            SInput sinput = SimscapeFactory.eINSTANCE.createSInput();
            if (scomponent != null)
                scomponent.getOwnedMembers().add(sinput);
            
            SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            scpb.setMember(sinput);
            scpb.setOwningSystem(scomponent);
            
            SimulinkUtil.setName(sinput, name);
            objects.put(dc, sinput);
            
            sinput.setLocation(SLocation.LEFT);
            scpb.setLocation(SLocation.LEFT);
            
            // get position
            // Token semi = ctx.SEMI(i).getSymbol();
            // int j = semi.getTokenIndex();
            // List<Token> lts = cts.getHiddenTokensToRight(j, Lexer.HIDDEN);
            // if (lts != null)
            // {
            // String str = lts.get(0).getText();
            // if (str.contains(":left"))
            // {
            // sinput.setLocation(SLocation.LEFT);
            // scpb.setLocation(SLocation.LEFT);
            // }
            // else if (str.contains(":right"))
            // {
            // sinput.setLocation(SLocation.RIGHT);
            // scpb.setLocation(SLocation.RIGHT);
            // }
            // }
            //
        }
    }
    
    @Override
    public void exitInputs(InputsContext ctx)
    {
        for (int i = 0; i < ctx.declaration().size(); i++)
        {
            DeclarationContext dc = ctx.declaration(i);
            SInput sinput = (SInput) objects.get(dc);
            setMemberValue(sinput, dc.expr());
        }
    }
    
    @Override
    public void enterOutputs(OutputsContext ctx)
    {
        for (int i = 0; i < ctx.declaration().size(); i++)
        {
            DeclarationContext dc = ctx.declaration(i);
            String name = dc.ID().getText();
            log.info("Defining output " + name);
            SOutput soutput = SimscapeFactory.eINSTANCE.createSOutput();
            if (scomponent != null)
                scomponent.getOwnedMembers().add(soutput);
            
            SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            scpb.setMember(soutput);
            scpb.setOwningSystem(scomponent);
            
            SimulinkUtil.setName(soutput, name);
            objects.put(dc, soutput);
            
            // get position
            soutput.setLocation(SLocation.RIGHT);
            scpb.setLocation(SLocation.RIGHT);
            
            // Token semi = ctx.SEMI(i).getSymbol();
            // int j = semi.getTokenIndex();
            // List<Token> lts = cts.getHiddenTokensToRight(j, Lexer.HIDDEN);
            // if (lts != null)
            // {
            // String str = lts.get(0).getText();
            // if (str.contains(":left"))
            // {
            // soutput.setLocation(SLocation.LEFT);
            // soutput.setLocation(SLocation.LEFT);
            // }
            // else if (str.contains(":right"))
            // {
            // soutput.setLocation(SLocation.RIGHT);
            // soutput.setLocation(SLocation.RIGHT);
            // }
            // }
            
        }
    }
    
    @Override
    public void exitOutputs(OutputsContext ctx)
    {
        for (int i = 0; i < ctx.declaration().size(); i++)
        {
            DeclarationContext dc = ctx.declaration(i);
            SOutput soutput = (SOutput) objects.get(dc);
            setMemberValue(soutput, dc.expr());
        }
    }
    
    @Override
    public void enterComponents(ComponentsContext ctx)
    {
        for (DeclarationContext dc : ctx.declaration())
        {
            String name = dc.ID().getText();
            SComponentReference scomponentreference = SimscapeFactory.eINSTANCE.createSComponentReference();
            if (scomponent != null)
                scomponent.getOwnedMembers().add(scomponentreference);
            SimulinkUtil.setName(scomponentreference, name);
            objects.put(dc, scomponentreference);
            references.add(new SimulinkReference(scomponentreference, dc.expr().getText(), ReferenceType.COMPONENT));
        }
    }
    
    @Override
    public void enterEquations(EquationsContext ctx)
    {
        if (scomponent == null)
            return;
        for (EquationContext eq : ctx.equation())
        {
            SEquation seq = SimscapeFactory.eINSTANCE.createSEquation();
            objects.put(eq, seq);
            seq.setExpression(eq.getText());
            scomponent.getOwnedEquations().add(seq);
            if (expressions != null)
                expressions.put(seq, eq);
        }
    }
    
    @Override
    public void enterConnections(ConnectionsContext ctx)
    {
        if (scomponent == null)
            return;
        for (ConnectionContext cc : ctx.connection())
        {
            SConnection sc = SimscapeFactory.eINSTANCE.createSConnection();
            objects.put(cc, sc);
            scomponent.getOwnedConnections().add(sc);
        }
    }
    
    @Override
    public void enterBranches(BranchesContext ctx)
    {
        if (scomponent == null)
            return;
        for (BranchContext bc : ctx.branch())
        {
            SBranch sbranch = SimscapeFactory.eINSTANCE.createSBranch();
            objects.put(bc, sbranch);
            scomponent.getBranches().add(sbranch);
        }
    }
    
    @Override
    public void exitExpr(ExprContext ctx)
    {
        if (ctx.expr2().size() == 1)
            assignProperty(ctx, ctx.expr2(0));
    }
    
    @Override
    public void exitExpr2(Expr2Context ctx)
    {
        if (ctx.expr3().size() == 1)
            assignProperty(ctx, ctx.expr3(0));
    }
    
    @Override
    public void exitExpr3(Expr3Context ctx)
    {
        if (ctx.expr4().size() == 1)
            assignProperty(ctx, ctx.expr4(0));
    }
    
    @Override
    public void exitExpr4(Expr4Context ctx)
    {
        if (ctx.expr5().size() == 1)
            assignProperty(ctx, ctx.expr5(0));
    }
    
    @Override
    public void exitExpr5(Expr5Context ctx)
    {
        if (ctx.expr6().size() == 1)
            assignProperty(ctx, ctx.expr6(0));
    }
    
    @Override
    public void exitExpr6(Expr6Context ctx)
    {
        if (ctx.expr7().size() == 1)
            assignProperty(ctx, ctx.expr7(0));
    }
    
    @Override
    public void exitExpr7(Expr7Context ctx)
    {
        if (ctx.expr8().size() == 1)
            assignProperty(ctx, ctx.expr8(0));
    }
    
    @Override
    public void exitExpr8(Expr8Context ctx)
    {
        if (ctx.expr9().size() == 1)
            assignProperty(ctx, ctx.expr9(0));
    }
    
    @Override
    public void exitExpr9(Expr9Context ctx)
    {
        if (ctx.unary() == null)
            assignProperty(ctx, ctx.expr10());
    }
    
    @Override
    public void exitExpr10(Expr10Context ctx)
    {
        if (ctx.expr11().size() == 1)
            assignProperty(ctx, ctx.expr11(0));
    }
    
    @Override
    public void exitExpr11(Expr11Context ctx)
    {
        if (ctx.INT() != null)
            exprs.put(ctx, ctx.INT());
        else if (ctx.FLOAT() != null)
            exprs.put(ctx, ctx.FLOAT());
        else if (ctx.BOOL() != null)
            exprs.put(ctx, ctx.BOOL());
        else if (ctx.STRING() != null)
            exprs.put(ctx, ctx.STRING());
        else if (ctx.array() != null)
            exprs.put(ctx, ctx.array());
        else if (ctx.reference() != null)
            exprs.put(ctx, ctx.reference());
        else if (ctx.expr() != null)
            exprs.put(ctx, ctx.expr());
        else if (ctx.object() != null)
            exprs.put(ctx, ctx.object());
    }
    
    /**
     * Sets value, unit, priority of Simscape member
     * 
     * @param smember
     *            Simscape member
     * @param expr
     *            expression containing value, unit, and priority
     */
    private void setMemberValue(SMember smember, ExprContext expr)
    {
        SDataValue sdv = null;
        String unit = null;
        ParseTree pt = exprs.get(expr);
        if (pt instanceof TerminalNode)
        {
            sdv = getValue(expr);
        }
        else if (pt instanceof ObjectContext)
        {
            // cases:
            // - {value={val, unit}}
            // - {val, unit}
            // - {value={val, unit}, priority=}
            ObjectContext oc = (ObjectContext) pt;
            if (oc.val().size() == 1)
            {
                
                if (oc.val(0).ID().getText().equals("value"))
                {
                    ParseTree exp = exprs.get(oc.val(0).expr());
                    if (exp instanceof ObjectContext)
                    {
                        ObjectContext oc2 = (ObjectContext) exp;
                        if (oc2.val().size() == 2)
                        {
                            sdv = getValue(oc2.val(0));
                            unit = getUnit(exprs.get(oc2.val(1)));
                        }
                        else
                            log.error("Expecting two values in object " + oc2.getText());
                    }
                    else
                        log.error("Expecting object for value " + oc.getText());
                }
                else
                    log.error("Expecting value key for object " + oc.getText());
            }
            else if (oc.val().size() == 2)
            {
                for (int i = 0; i < oc.val().size(); i++)
                {
                    if (oc.val(i).ID() == null)
                    {
                        if (i == 0)
                        {
                            sdv = getValue(oc.val(i).expr());
                        }
                        else
                        {
                            unit = getUnit(exprs.get(oc.val(i).expr()));
                        }
                    }
                    else if (oc.val(i).ID().getText().equals("value"))
                    {
                        ParseTree exp = exprs.get(oc.val(i).expr());
                        if (exp instanceof ObjectContext)
                        {
                            ObjectContext oc2 = (ObjectContext) exp;
                            if (oc2.val().size() == 2)
                            {
                                sdv = getValue(oc2.val(0));
                                unit = getUnit(exprs.get(oc2.val(1)));
                            }
                            else
                                log.error("Expecting two values in object " + oc2.getText());
                        }
                        else
                            log.error("Expecting object for value " + exp.getText());
                    }
                    else
                        log.warn("Unexpected key " + oc.val(i).getText());
                }
            }
            else
                log.error("Expecting object of size 1 or 2 " + oc.getText());
        }
        if (smember instanceof SVariable)
        {
            log.info("Assigning value " + sdv + " and unit " + unit + " to variable " + smember.getName());
            SVariable sv = (SVariable) smember;
            sv.setValue(sdv);
            sv.setUnit(unit);
        }
        else if (smember instanceof SParameter)
        {
            log.info("Assigning value " + sdv + " and unit " + unit + " to parameter " + smember.getName());
            SParameter sp = (SParameter) smember;
            sp.setValue(sdv);
            sp.setUnit(unit);
        }
        else if (smember instanceof SInput)
        {
            log.info("Assigning value " + sdv + " and unit " + unit + " to input " + smember.getName());
            SInput si = (SInput) smember;
            si.setValue(sdv);
            si.setUnit(unit);
        }
        else if (smember instanceof SOutput)
        {
            log.info("Assigning value " + sdv + " and unit " + unit + " to output " + smember.getName());
            SOutput so = (SOutput) smember;
            so.setValue(sdv);
            so.setUnit(unit);
        }
    }
    
    /**
     * Returns the Simscape value from a terminal node
     * 
     * @param pt
     * @return Simscape value
     */
    public SDataValue getValue(ParseTree pt)
    {
        ParseTree expr = exprs.get(pt);
        if (expr == null)
        {
            log.warn("No value for " + pt.getText());
        }
        else if (expr instanceof TerminalNode)
        {
            TerminalNode tn = (TerminalNode) expr;
            if (tn.getSymbol().getType() == SimscapeLexer.INT || tn.getSymbol().getType() == SimscapeLexer.FLOAT)
            {
                SDoubleValue sdv = SimulinkFactory.eINSTANCE.createSDoubleValue();
                sdv.setValue(Double.parseDouble(tn.getText()));
                return sdv;
            }
            log.warn("Can't parse variable value " + tn.getText());
        }
        else
            log.warn("Unrecognized value " + expr.getText());
        return null;
    }
    
    /**
     * Returns the Simscape unit from a terminal node
     * 
     * @param pt
     * @return
     */
    static String getUnit(ParseTree pt)
    {
        if (pt instanceof TerminalNode && ((TerminalNode) pt).getSymbol().getType() == SimscapeLexer.STRING)
        {
            String sunit = pt.getText();
            return sunit.substring(1, sunit.length() - 1);
        }
        return null;
    }
    
    @Override
    public Collection<SimulinkReference> getReferences()
    {
        return references;
    }
    
    private void assignProperty(ParseTree from, ParseTree to)
    {
        exprs.put(from, exprs.get(to));
    }
    
    @Override
    public int getPriority()
    {
        return 5;
    }
    
    @Override
    public SElement getRootElement()
    {
        return scomponent != null ? scomponent : sdomain;
    }
    
    /**
     * Returns the association between syntax tree nodes and Simscape objects
     * 
     * @return map from syntax tree nodes to Simscape objects
     */
    public ParseTreeProperty<SElement> getObjects()
    {
        return objects;
    }
    
}
