package com.engisis.sysphs.deserialization.modelica;

import java.util.LinkedList;
import java.util.List;

import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MComponent;
import com.engisis.sysphs.language.modelica.MConnect;
import com.engisis.sysphs.language.modelica.MDataValue;
import com.engisis.sysphs.language.modelica.MExpressionValue;
import com.engisis.sysphs.language.modelica.MExtension;
import com.engisis.sysphs.language.modelica.MIntegerValue;
import com.engisis.sysphs.language.modelica.MModification;
import com.engisis.sysphs.language.modelica.MRealValue;
import com.engisis.sysphs.language.modelica.MStringValue;
import com.engisis.sysphs.language.modelica.ModelicaFactory;
import com.engisis.sysphs.translation.modelica.ModelicaUtil;
import com.engisis.sysphs.generation.modelica.ModelicaBaseListener;
import com.engisis.sysphs.generation.modelica.ModelicaParser.ArgumentContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Arithmetic_expressionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Class_definitionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Class_specifierContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Component_clause1Context;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Component_declarationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Connect_clauseContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.DeclarationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Element_modificationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.FactorContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Logical_expressionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Logical_factorContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Logical_termContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.ModificationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.PrimaryContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.RelationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Simple_expressionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Stored_definitionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.TermContext;

import org.antlr.v4.runtime.tree.ParseTreeProperty;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.apache.log4j.Logger;

/**
 * Class managing the second pass of the Modelica deserialization
 * 
 * @author barbau
 *
 */
public class ModelicaReaderPass2 extends ModelicaBaseListener
{
    private static final Logger log = Logger.getLogger(ModelicaReaderPass2.class);
    
    /**
     * First pass parser
     */
    private ModelicaReaderPass1 mpd;
    
    /**
     * Modelica scopes associated with syntax tree nodes
     */
    private ParseTreeProperty<ModelicaScope> scopelist;
    /**
     * Current scope
     */
    private ModelicaScope scope;
    /**
     * Object values associated with syntax tree nodes
     */
    private ParseTreeProperty<Object> values;
    
    /**
     * Constructs a second pass Modelica reader
     * 
     * @param mpd
     *            First pass parser
     */
    public ModelicaReaderPass2(ModelicaReaderPass1 mpd)
    {
        this.mpd = mpd;
        this.scopelist = mpd.getScopeList();
        values = new ParseTreeProperty<Object>();
    }
    
    @Override
    public void enterStored_definition(Stored_definitionContext ctx)
    {
        super.enterStored_definition(ctx);
        scope = scopelist.get(ctx);
    }
    
    @Override
    public void exitStored_definition(Stored_definitionContext ctx)
    {
        super.exitStored_definition(ctx);
        scope = scope.getParent();
    }
    
    @Override
    public void enterClass_definition(Class_definitionContext ctx)
    {
        super.enterClass_definition(ctx);
        scope = scopelist.get(ctx);
    }
    
    @Override
    public void exitClass_definition(Class_definitionContext ctx)
    {
        super.exitClass_definition(ctx);
        scope = scope.getParent();
    }
    
    @Override
    public void exitClass_specifier(Class_specifierContext ctx)
    {
        super.exitClass_specifier(ctx);
        ModelicaScope ms = scopelist.get(ctx);
        if (ms == null || !(ms.getMNamedElement() instanceof MClass))
            return;
        
        MClass mclass = (MClass) ms.getMNamedElement();
        
        if (ctx.name_path() == null)
            return;
        
        MClass mclass2 = (MClass) ms.lookForClass(ctx.name_path().IDENT()).getMNamedElement();
        
        if (ctx.class_modification() == null || ctx.class_modification().argument_list() == null)
            return;
        for (ArgumentContext ac : ctx.class_modification().argument_list().argument())
        {
            log.info("Extension modification for class " + scope.getMNamedElement().getName());
            MModification modif = ModelicaFactory.eINSTANCE.createMModification();
            Element_modificationContext emc = ac.element_modification_or_replaceable().element_modification();
            
            for (TerminalNode tn : emc.component_reference().IDENT())
            {
                MComponent mc = mclass.getComponentByName(tn.getText());
                if (mc == null)
                {
                    modif.getComponentPath().clear();
                    break;
                }
                mclass = mc.getType();
                modif.getComponentPath().add(mc);
            }
            if (emc.modification() != null && emc.modification().class_modification() == null)
            {
                Simple_expressionContext sec = emc.modification().expression().simple_expression();
                if (sec != null)
                {
                    Object o = values.get(sec);
                    if (o instanceof List)
                    {
                        modif.getAssignedReference().addAll((List) o);
                        log.info("Assigning reference " + modif.getComponentPath() + " = " + o);
                    }
                    else
                    {
                        modif.setAssignedValue(getValueFromExpression(sec));
                        log.info("Assigning value " + modif.getComponentPath() + " = " + modif.getAssignedValue());
                    }
                }
            }
            if (modif.getComponentPath().size() > 0
                    && (modif.getAssignedValue() != null || modif.getAssignedReference().size() > 0))
            {
                for (MExtension me : ((MClass) scope.getMNamedElement()).getExtensions())
                    if (me.getExtendedClass() == mclass2)
                        me.getModifications().add(modif);
            }
            else
                log.error("No modification was added");
        }
    }
    
    @Override
    public void enterComponent_declaration(Component_declarationContext ctx)
    {
        super.enterComponent_declaration(ctx);
        scope = scopelist.get(ctx);
    }

    @Override
    public void exitComponent_declaration(Component_declarationContext ctx)
    {
        super.exitComponent_declaration(ctx);
        scope = scope.getParent();
        
    }
    
    @Override
    public void exitDeclaration(DeclarationContext ctx)
    {
        super.exitDeclaration(ctx);
        ModificationContext modification = ctx.modification();
        if (modification != null && scope.getMNamedElement() instanceof MComponent)
        {
            MComponent mcomponent = (MComponent) scope.getMNamedElement();
            // default value
            if (modification.expression() != null && modification.expression().simple_expression() != null)
                mcomponent.setValue(getValueFromExpression(modification.expression().simple_expression()));
            else if (modification.class_modification() != null
                    && modification.class_modification().argument_list() != null)
            {
                loop: for (ArgumentContext ac : modification.class_modification().argument_list().argument())
                {
                    log.info("Component modification for component " + mcomponent.getName());
                    MComponent mc = mcomponent;
                    if (ac.element_modification_or_replaceable() != null
                            && ac.element_modification_or_replaceable().element_modification() != null)
                    {
                        MModification modif = ModelicaFactory.eINSTANCE.createMModification();
                        Element_modificationContext emc = ac.element_modification_or_replaceable()
                                .element_modification();
                        for (TerminalNode tn : emc.component_reference().IDENT())
                        {
                            MClass type = mc.getType();
                            if (type == null)
                            {
                                log.error("The type of component " + mc.getName() + " in null");
                                continue loop;
                            }
                            mc = type.getComponentByName(tn.getText());
                            if (mc == null)
                            {
                                log.error("Couldn't find component " + tn.getText() + " in class " + type.getName()
                                        + ", creating it");
                                mc = ModelicaFactory.eINSTANCE.createMComponent();
                                mc.setOwningClass(type);
                                ModelicaUtil.setName(mc, tn.getText());
                                continue loop;
                            }
                            modif.getComponentPath().add(mc);
                        }
                        if (emc.modification() != null && emc.modification().class_modification() == null)
                        {
                            Simple_expressionContext sec = emc.modification().expression().simple_expression();
                            if (sec != null)
                            {
                                Object o = values.get(sec);
                                if (o instanceof List)
                                {
                                    modif.getAssignedReference().addAll((List) o);
                                    log.info("Assigning reference " + modif.getComponentPath() + " = " + o);
                                }
                                else
                                {
                                    mpd.getExpressions().put(modif, sec);
                                    modif.setAssignedValue(getValueFromExpression(sec));
                                    log.info("Assigning value " + modif.getComponentPath() + " = "
                                            + modif.getAssignedValue());
                                }
                            }
                        }
                        if (modif.getAssignedValue() != null || modif.getAssignedReference().size() > 0)
                            ((MComponent) scope.getMNamedElement()).getModifications().add(modif);
                        else
                            log.error("Coundn't get the modification of " + scope);
                    }
                    
                }
            }
        }

    }

    @Override
    public void enterComponent_clause1(Component_clause1Context ctx)
    {
        super.enterComponent_clause1(ctx);
        scope = scopelist.get(ctx.component_declaration1());
        if (scope.getMNamedElement() instanceof MComponent)
        {
            MComponent mc = (MComponent) scope.getMNamedElement();
            MClass owner = mc.getOwningClass();
            for(MExtension mextension : owner.getExtensions())
            {
                MClass mextended = mextension.getExtendedClass();
                MComponent mc2 = mextended.getComponentByName(mc.getName());
                if (mc2 != null)
                {
                    mc.setRedefinedComponent(mc2);
                    log.info("Setting redefined property for " + mc.getName());
                    return;
                }
            }
        }
        log.error("Unable to set redefined component");
    }
    
    @Override
    public void exitComponent_clause1(Component_clause1Context ctx)
    {
        super.exitComponent_clause1(ctx);
        scope = scope.getParent();
    }
    
    
    
    @Override
    public void enterConnect_clause(Connect_clauseContext ctx)
    {
        super.enterConnect_clause(ctx);
        if (!(scope.getMNamedElement() instanceof MClass))
        {
            log.debug("Connect clause skipped, " + scope.getMNamedElement() + " not a class");
            return;
        }
        MConnect mconnect = ModelicaFactory.eINSTANCE.createMConnect();
        List<TerminalNode> idents1 = ctx.connector_ref(0).IDENT();
        addComponentToList(mconnect.getRef1(), idents1);
        if (mconnect.getRef1().size() == idents1.size())
        {
            List<TerminalNode> idents2 = ctx.connector_ref(1).IDENT();
            addComponentToList(mconnect.getRef2(), idents2);
            if (mconnect.getRef2().size() == idents2.size())
            {
                ((MClass) scope.getMNamedElement()).getEquations().add(mconnect);
                log.info("Connect equation added to " + scope);
            }
            else
                log.warn("Connect equation failed: found " + mconnect.getRef2().size() + "/" + idents2.size() + " in "
                        + idents2);
        }
        else
            log.warn("Connect equation failed: found " + mconnect.getRef1().size() + "/" + idents1.size() + " in "
                    + idents1);
    }
    
    @Override
    public void exitArgument(ArgumentContext ctx)
    {
        super.enterArgument(ctx);
    }
    
    /**
     * Adds a component to a list of component
     * 
     * @param list
     *            list to which the component should be added
     * @param idents
     *            identifier of the component, given as list of terminals
     */
    private void addComponentToList(List<MComponent> list, List<TerminalNode> idents)
    {
        ModelicaScope ms = scope.lookForElementInScope(idents.get(0));
        if (ms.getMNamedElement() instanceof MComponent)
        {
            MComponent mc = (MComponent) ms.getMNamedElement();
            list.add(mc);
            for (int i = 1; i < idents.size(); i++)
            {
                MClass type = mc.getType();
                if (type == null)
                {
                    log.warn("Type of component " + mc.getName() + " is null");
                    break;
                }
                String sclass = type.getName();
                String scomponent = idents.get(i).getText();
                mc = type.getComponentByName(scomponent);
                
                if (mc == null)
                {
                    log.warn("Component " + scomponent + " not found in class " + sclass + ", creating it");
                    mc = ModelicaFactory.eINSTANCE.createMComponent();
                    mc.setOwningClass(type);
                    ModelicaUtil.setName(mc, scomponent);
                }
                
                list.add(mc);
            }
        }
    }
    
    @Override
    public void exitPrimary(PrimaryContext ctx)
    {
        super.exitPrimary(ctx);
        if (ctx.STRING() != null)
        {
            String v = ctx.STRING().getText();
            values.put(ctx, v.substring(1, v.length() - 1));
        }
        else if (ctx.FALSE() != null)
            values.put(ctx, Boolean.valueOf(false));
        else if (ctx.TRUE() != null)
            values.put(ctx, Boolean.valueOf(true));
        else if (ctx.UNSIGNED_NUMBER() != null)
        {
            values.put(ctx, Double.valueOf(ctx.UNSIGNED_NUMBER().getText()));
        }
        else if (ctx.component_reference() != null)
        {
            if (scope.getMNamedElement() instanceof MComponent)
            {
                List<MComponent> lmc = new LinkedList<MComponent>();
                // component references with a component scope are resolved from
                // the component owner
                MClass mc = (MClass) scope.getParent().getMNamedElement();
                for (TerminalNode tn : ctx.component_reference().IDENT())
                {
                    MComponent mcomp = mc.getComponentByName(tn.getText());
                    if (mcomp != null)
                    {
                        lmc.add(mcomp);
                        mc = mcomp.getType();
                    }
                    else
                    {
                        log.error("Can't find component " + tn.getText() + " in " + mc.getName());
                        break;
                    }
                }
                values.put(ctx, lmc);
            }
        }
    }
    
    @Override
    public void exitFactor(FactorContext ctx)
    {
        super.exitFactor(ctx);
        List<PrimaryContext> primary = ctx.primary();
        if (primary.size() == 1 && values.get(primary.get(0)) != null)
            values.put(ctx, values.get(primary.get(0)));
    }
    
    @Override
    public void exitTerm(TermContext ctx)
    {
        super.exitTerm(ctx);
        List<FactorContext> factor = ctx.factor();
        if (factor.size() == 1 && values.get(factor.get(0)) != null)
            values.put(ctx, values.get(factor.get(0)));
    }
    
    @Override
    public void exitArithmetic_expression(Arithmetic_expressionContext ctx)
    {
        super.exitArithmetic_expression(ctx);
        List<TermContext> term = ctx.term();
        if (term.size() == 1 && values.get(term.get(0)) != null)
        {
            if (values.get(term.get(0)) instanceof Double)
            {
                if (ctx.add_op().size() == 0 || ctx.add_op(0).PLUS() != null)
                    values.put(ctx, values.get(term.get(0)));
                else if (ctx.add_op().size() == 1 && ctx.add_op(0).MINUS() != null)
                    values.put(ctx, Double.valueOf(-((Double) values.get(term.get(0))).doubleValue()));
            }
            else if (ctx.add_op().size() == 0 || ctx.add_op(0).PLUS() != null)
                values.put(ctx, values.get(term.get(0)));
        }
    }
    
    @Override
    public void exitRelation(RelationContext ctx)
    {
        super.exitRelation(ctx);
        List<Arithmetic_expressionContext> exp = ctx.arithmetic_expression();
        if (exp.size() == 1 && values.get(exp.get(0)) != null)
            values.put(ctx, values.get(exp.get(0)));
    }
    
    @Override
    public void exitLogical_factor(Logical_factorContext ctx)
    {
        super.exitLogical_factor(ctx);
        if (values.get(ctx.relation()) != null)
            values.put(ctx, values.get(ctx.relation()));
    }
    
    @Override
    public void exitLogical_term(Logical_termContext ctx)
    {
        super.exitLogical_term(ctx);
        List<Logical_factorContext> term = ctx.logical_factor();
        if (term.size() == 1 && values.get(term.get(0)) != null)
            values.put(ctx, values.get(term.get(0)));
    }
    
    @Override
    public void exitLogical_expression(Logical_expressionContext ctx)
    {
        super.exitLogical_expression(ctx);
        List<Logical_termContext> term = ctx.logical_term();
        if (term.size() == 1 && values.get(term.get(0)) != null)
            values.put(ctx, values.get(term.get(0)));
    }
    
    @Override
    public void exitSimple_expression(Simple_expressionContext ctx)
    {
        super.exitSimple_expression(ctx);
        List<Logical_expressionContext> exp = ctx.logical_expression();
        if (exp.size() == 1 && values.get(exp.get(0)) != null)
            values.put(ctx, values.get(exp.get(0)));
    }
    
    /**
     * Retrieves the Modelica value from a simple expression
     * 
     * @param exp
     *            simple expression with a value
     * @return a Modelica data value
     */
    public MDataValue getValueFromExpression(Simple_expressionContext exp)
    {
        if (values.get(exp) != null)
        {
            Object o = values.get(exp);
            if (o instanceof Integer)
            {
                MIntegerValue mdv = ModelicaFactory.eINSTANCE.createMIntegerValue();
                mdv.setValue(((Integer) o).intValue());
                return mdv;
            }
            else if (o instanceof String)
            {
                MStringValue mdv = ModelicaFactory.eINSTANCE.createMStringValue();
                mdv.setValue(o.toString());
                return mdv;
            }
            else if (o instanceof Double)
            {
                MRealValue mdv = ModelicaFactory.eINSTANCE.createMRealValue();
                mdv.setValue(((Double) o).doubleValue());
                return mdv;
            }
        }
        MExpressionValue mdv = ModelicaFactory.eINSTANCE.createMExpressionValue();
        mdv.setValue(exp.getText());
        return mdv;
    }
}
