package com.engisis.sysphs.client.gui;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.GroupLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.GroupLayout.Alignment;

import org.apache.log4j.Logger;

import com.engisis.sysphs.SimulationToSysMLTranslationManager;
import com.engisis.sysphs.client.gui.WorkerDialog.Worker;
import com.engisis.sysphs.translation.modelica.ModelicaToSysMLTranslator;
import com.engisis.sysphs.translation.simulink.SimulinkToSysMLTranslator;
import com.engisis.sysphs.util.Configuration;
import com.engisis.sysphs.util.SimulationToSysMLTranslator;
import com.engisis.sysphs.util.UMLModelErrorException;

/**
 * Frame used to launch the translation of simulation models to SysML models.
 * 
 * @author barbau
 *
 */
public class SimulationToSysMLFrame extends JFrame
{
    private static final long serialVersionUID = -1105097960685575427L;
    
    private static final Logger log = Logger.getLogger(SimulationToSysMLFrame.class);
    
    /**
     * File containing the simulation model
     */
    private File fmodel;
    
    /**
     * Output directory
     */
    private File foutput;
    
    /**
     * Translator
     */
    private SimulationToSysMLTranslationManager simulation2sysml;
    
    /**
     * Translator to use for the simulation model
     */
    private SimulationToSysMLTranslator translator;
    
    /**
     * Dialog used to display progress in the translation
     */
    private WorkerDialog swd;
    
    /**
     * Panel containing various information on the simulation model
     */
    private JPanel jpInformation;
    
    /**
     * Label showing the simulation file name
     */
    private JLabel jlFiles;
    
    /**
     * Panel containing options for the translator
     */
    private JPanel jpTranslator;
    
    /**
     * Label showing the available translators
     */
    private JLabel jlTranslator;
    
    /**
     * Selector of Modelica translator
     */
    private JRadioButton jrbModelica;
    
    /**
     * Selector of Simulink translator
     */
    private JRadioButton jrbSimulink;
    
    /**
     * Button group
     */
    private ButtonGroup bgTranslator;
    
    /**
     * Panel containing the target option
     */
    private JPanel jpTarget;
    
    /**
     * Combo box for selecting the XMI setialization target
     */
    private JComboBox<String> jcbTarget;
    
    /**
     * Panel containing the output directory information
     */
    private JPanel jpOutput;
    
    /**
     * Button for selecting the output directory
     */
    private JButton jbOutput;
    
    /**
     * Output directory label
     */
    private JLabel jlOutput;
    
    /**
     * Translation button
     */
    private JButton jbTranslate;
    
    /**
     * Constructs a translation frame.
     * 
     * @param model
     *            path to the simulation file
     * @param rootblock
     *            block to start the translation with
     * @param paths
     *            paths in which XMI dependencies are looked for
     */
    public SimulationToSysMLFrame(final String model, final String rootblock, final String[] paths)
    {
        super("Simulation frame");
        fmodel = new File(model);
        foutput = fmodel.getParentFile();
        if (!isWritable(foutput))
            foutput = new File(System.getProperty("user.home"));
        
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        swd = new WorkerDialog(this);
        
        try
        {
            simulation2sysml = new SimulationToSysMLTranslationManager(model, paths);
        }
        catch (FileNotFoundException e1)
        {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }
        
        addWindowListener(new WindowListener()
        {
            @Override
            public void windowOpened(WindowEvent e)
            {
            }
            
            @Override
            public void windowIconified(WindowEvent e)
            {
            }
            
            @Override
            public void windowDeiconified(WindowEvent e)
            {
            }
            
            @Override
            public void windowDeactivated(WindowEvent e)
            {
            }
            
            @Override
            public void windowClosing(WindowEvent e)
            {
                try
                {
                    if (simulation2sysml != null)
                        simulation2sysml.setTranslator(null);
                }
                catch (Exception ex)
                {
                }
                
            }
            
            @Override
            public void windowClosed(WindowEvent e)
            {
            }
            
            @Override
            public void windowActivated(WindowEvent e)
            {
            }
        });
        
        // content
        
        getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.Y_AXIS));
        
        // Information panel
        jpInformation = new JPanel();
        getContentPane().add(jpInformation);
        jpInformation.setBorder(BorderFactory.createTitledBorder("Information"));
        
        jlFiles = new JLabel();
        jpInformation.add(jlFiles);
        
        StringBuilder sbinfo = new StringBuilder();
        sbinfo.append("Files: " + model);
        jlFiles.setText(sbinfo.toString());
        
        // Translator panel
        jpTranslator = new JPanel();
        getContentPane().add(jpTranslator);
        GroupLayout glTranslator = new GroupLayout(jpTranslator);
        jpTranslator.setLayout(glTranslator);
        jpTranslator.setBorder(BorderFactory.createTitledBorder("Translator"));
        
        jlTranslator = new JLabel("Source language:");
        
        jrbModelica = new JRadioButton("Modelica");
        jrbModelica.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(ItemEvent e)
            {
                if (e.getStateChange() == 1)
                {
                    translator = new ModelicaToSysMLTranslator(null);
                }
            }
        });
        
        jrbSimulink = new JRadioButton("Simulink");
        jrbSimulink.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(ItemEvent e)
            {
                if (e.getStateChange() == 1)
                {
                    translator = new SimulinkToSysMLTranslator();
                }
            }
        });
        
        bgTranslator = new ButtonGroup();
        bgTranslator.add(jrbModelica);
        bgTranslator.add(jrbSimulink);
        
        String ext = model.toLowerCase();
        if (ext.endsWith(".slx") || ext.endsWith(".mdl"))
        {
            jrbSimulink.setSelected(true);
        }
        else if (ext.endsWith(".mo"))
        {
            jrbModelica.setSelected(true);
        }
        
        glTranslator.setHorizontalGroup(glTranslator.createSequentialGroup()
                .addGroup(glTranslator.createParallelGroup(Alignment.LEADING).addComponent(jlTranslator))
                .addGroup(glTranslator.createParallelGroup(Alignment.LEADING).addComponent(jrbModelica))
                .addGroup(glTranslator.createParallelGroup(Alignment.LEADING).addComponent(jrbSimulink)));
        
        glTranslator.setVerticalGroup(glTranslator.createSequentialGroup().addGroup(
                glTranslator.createParallelGroup(Alignment.BASELINE).addComponent(jlTranslator)
                        .addComponent(jrbModelica).addComponent(jrbSimulink)));
        
        jpTarget = new JPanel();
        getContentPane().add(jpTarget);
        GroupLayout glTarget = new GroupLayout(jpTarget);
        
        jpTarget.setLayout(glTarget);
        jpTarget.setBorder(BorderFactory.createTitledBorder("XMI target"));
        
        JLabel jlTarget = new JLabel("Target:");
        
        jcbTarget = new JComboBox<String>();
        jcbTarget.setEditable(false);
        for (String target : Configuration.getTargets())
        {
            jcbTarget.addItem(target);
        }
        
        glTarget.setHorizontalGroup(glTarget.createSequentialGroup()
                .addGroup(glTarget.createParallelGroup(Alignment.LEADING).addComponent(jlTarget))
                .addGroup(glTarget.createParallelGroup(Alignment.LEADING).addComponent(jcbTarget)));
        
        glTarget.setVerticalGroup(glTarget.createSequentialGroup().addGroup(
                glTarget.createParallelGroup(Alignment.BASELINE).addComponent(jlTarget).addComponent(jcbTarget)));
        
        jpOutput = new JPanel();
        getContentPane().add(jpOutput);
        GroupLayout glOutput = new GroupLayout(jpOutput);
        jpOutput.setLayout(glOutput);
        jpOutput.setBorder(BorderFactory.createTitledBorder("Output directory"));
        
        jbOutput = new JButton();
        jbOutput.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                JFileChooser jfc = new JFileChooser();
                jfc.setCurrentDirectory(foutput);
                jfc.setDialogTitle("Select output directory");
                jfc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
                jfc.setAcceptAllFileFilterUsed(false);
                
                if (jfc.showOpenDialog(SimulationToSysMLFrame.this) == JFileChooser.APPROVE_OPTION)
                {
                    foutput = jfc.getSelectedFile();
                    updateOutput();
                }
            }
        });
        
        jlOutput = new JLabel();
        
        glOutput.setHorizontalGroup(glOutput.createSequentialGroup().addComponent(jbOutput).addComponent(jlOutput));
        glOutput.setVerticalGroup(glOutput.createParallelGroup(Alignment.BASELINE).addComponent(jbOutput)
                .addComponent(jlOutput));
        
        jbTranslate = new JButton();
        jbTranslate.setText("Translate");
        jbTranslate.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                try
                {
                    if (translator == null)
                    {
                        JOptionPane.showMessageDialog(SimulationToSysMLFrame.this, "You must select a translator");
                        return;
                    }
                    // generation.reset();
                    // simulation2sysml.setInputRootName(jlBlocks.getSelectedValue());
                    simulation2sysml.setTranslator(translator);
                    simulation2sysml.setOutputDirectory(foutput);
                    int i = jcbTarget.getSelectedIndex();
                    if (i != -1)
                        simulation2sysml.setTarget(jcbTarget.getItemAt(i));
                }
                catch (Exception e)
                {
                    log.error("Could not create the translator", e);
                }
                Worker<Void, Void> sw = swd.new Worker<Void, Void>()
                {
                    @Override
                    public Void doInBackground()
                    {
                        try
                        {
                            simulation2sysml.execute();
                        }
                        catch (UMLModelErrorException e)
                        {
                            JOptionPane.showMessageDialog(swd, "The translation failed");
                            log.error("Bad model error during the translation", e);
                        }
                        catch (IOException e)
                        {
                            JOptionPane.showMessageDialog(swd, "The translation failed");
                            log.error("IO error during the translation", e);
                        }
                        catch (Exception e)
                        {
                            JOptionPane.showMessageDialog(swd, "Unexpected exception");
                            log.error("Other error during the translation", e);
                        }
                        return null;
                    }
                };
                swd.show(sw, "Translating file");
            }
        });
        
        // Translator button
        getContentPane().add(jbTranslate);
        
    }
    
    /**
     * Updates the output directory labels
     */
    private void updateOutput()
    {
        log.debug("Changed output directory to " + foutput);
        jbOutput.setText(foutput.getAbsolutePath());
        if (isWritable(foutput))
        {
            jlOutput.setText("");
            jlOutput.setForeground(Color.BLACK);
        }
        else
        {
            jlOutput.setText("Readonly directory");
            jlOutput.setForeground(Color.RED);
        }
    }
    
    /**
     * Determines whether the given directory can be written into
     * 
     * @param dir
     *            directory to test
     * @return
     */
    private static boolean isWritable(File dir)
    {
        if (dir == null || !dir.exists() || !dir.isDirectory() || !dir.canWrite())
            return false;
        try
        {
            File tmp = File.createTempFile("sim", "tmp", dir);
            tmp.delete();
        }
        catch (IOException e)
        {
            return false;
        }
        return true;
    }
}
