package com.engisis.sysphs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.JFileChooser;
import javax.xml.stream.XMLStreamException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import com.engisis.sysphs.client.gui.SimulationToSysMLFrame;
import com.engisis.sysphs.client.gui.SysMLToSimulationFrame;
import com.engisis.sysphs.translation.modelica.ModelicaToSysMLTranslator;
import com.engisis.sysphs.translation.modelica.SysMLToModelicaTranslator;
import com.engisis.sysphs.translation.simulink.SimulinkToSysMLTranslator;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator.Domains;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator.Format;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator.SFunction;
import com.engisis.sysphs.util.SimulationToSysMLTranslator;
import com.engisis.sysphs.util.SysMLToSimulationTranslator;
import com.engisis.sysphs.util.UMLModelErrorException;

/**
 * Entry to the translator.
 * 
 * @author barbau
 *
 */
public class TranslationManager
{
    private static final Logger log = Logger.getLogger(TranslationManager.class);
    
    private enum Target
    {
        MODELICA, SIMULINK
    }
    
    private static final String PATH = "path";
    private static final String REVERSE = "reverse";
    private static final String TARGET = "target";
    private static final String TARGET_SIMULINK = "simulink";
    private static final String TARGET_MODELICA = "modelica";
    private static final String SFUNCTION = "sfunction";
    private static final String SFUNCTION_LEVEL1 = "level1";
    private static final String SFUNCTION_LEVEL2 = "level2";
    private static final String SFUNCTION_SIMSCAPE = "simscape";
    private static final String FORMAT = "format";
    private static final String FORMAT_TEXT = "text";
    private static final String FORMAT_XML = "xml";
    private static final String DOMAINS = "domains";
    private static final String DOMAINS_CREATE = "create";
    private static final String DOMAINS_REUSE = "reuse";
    private static final String PREPROC = "preproc";
    
    public static void main(String[] args)
    {
        log.debug("Program started");
        
        boolean reverse = false;
        
        Target target = null;
        SFunction sfunction = null;
        Format format = null;
        Domains domains = null;
        String rootblock = null;
        String model = null;
        String[] paths = null;
        
        boolean preproc = false;
        // CL options
        HelpFormatter formatter = new HelpFormatter();
        Options options = new Options();
        options.addOption(PATH, true, "Paths separated by " + File.pathSeparator);
        options.addOption(REVERSE, false, "Reverse translation (disabled by default)");
        options.addOption(TARGET, true, "Target platform ('" + TARGET_MODELICA + "' or '" + TARGET_SIMULINK + "')");
        options.addOption(SFUNCTION, true, "Type of SFunction ('" + SFUNCTION_LEVEL1 + "', '" + SFUNCTION_LEVEL2
                + "', '" + SFUNCTION_SIMSCAPE + "')");
        options.addOption(FORMAT, true, "Simulink format ('" + FORMAT_TEXT + "' or '" + FORMAT_XML + "')");
        options.addOption(DOMAINS, true, "Simscape domains ('" + DOMAINS_CREATE + "', '" + DOMAINS_REUSE + "')");
        options.addOption(PREPROC, false, "Enables preprocessing transformations (disabled by default)");
        
        CommandLineParser clp = new org.apache.commons.cli.BasicParser();
        try
        {
            CommandLine cl = clp.parse(options, args);
            if (cl.hasOption(REVERSE))
                reverse = true;
            if (cl.hasOption(TARGET))
            {
                String s = cl.getOptionValue(TARGET);
                if (TARGET_MODELICA.equals(s))
                    target = Target.MODELICA;
                else if (TARGET_SIMULINK.equals(s))
                    target = Target.SIMULINK;
                else
                {
                    log.warn("Unrecognized target " + s);
                    formatter.printHelp("", options);
                }
            }
            if (cl.hasOption(SFUNCTION))
            {
                String s = cl.getOptionValue(SFUNCTION);
                if (SFUNCTION_LEVEL1.equals(s))
                    sfunction = SFunction.Level1;
                else if (SFUNCTION_LEVEL2.equals(s))
                    sfunction = SFunction.Level2;
                else if (SFUNCTION_SIMSCAPE.equals(s))
                    sfunction = SFunction.Simscape;
                else
                {
                    log.warn("Unrecognized sfunction " + s);
                    formatter.printHelp("", options);
                }
            }
            if (cl.hasOption(FORMAT))
            {
                String s = cl.getOptionValue(FORMAT);
                if (FORMAT_TEXT.equals(s))
                    format = Format.TEXT;
                else if (FORMAT_XML.equals(s))
                    format = Format.XML;
                else
                {
                    log.warn("Unrecognized format " + s);
                    formatter.printHelp("", options);
                }
            }
            if (cl.hasOption(DOMAINS))
            {
                String s = cl.getOptionValue(DOMAINS);
                if (DOMAINS_CREATE.equals(s))
                    domains = Domains.CREATE;
                else if (DOMAINS_REUSE.equals(s))
                    domains = Domains.REUSE;
                else
                {
                    log.warn("Unrecognized domains " + s);
                    formatter.printHelp("", options);
                }
            }
            if (cl.hasOption(PREPROC))
                preproc = true;
            if (cl.hasOption(PATH))
                paths = cl.getOptionValue(PATH).split(File.pathSeparator);
            
            @SuppressWarnings("unchecked")
            List<String> arglist = cl.getArgList();
            if (arglist.size() > 0)
                model = arglist.get(0);
            if (arglist.size() > 1)
                rootblock = arglist.get(1);
            
        }
        catch (ParseException e)
        {
            log.error("Could not parse the command line", e);
            formatter.printHelp("", options);
            System.exit(1);
        }
        
        if (!reverse)
        {
            log.info("Straight translation");
            if (model != null && rootblock != null && target != null)
            {
                log.info("No GUI necessary");
                SysMLToSimulationTranslationManager sysml2simulation;
                try
                {
                    sysml2simulation = new SysMLToSimulationTranslationManager(model, paths);
                    if (preproc)
                        sysml2simulation.setPreprocessing(true);
                    sysml2simulation.setInputRootName(rootblock);
                    
                    SysMLToSimulationTranslator translator = null;
                    if (target == Target.MODELICA)
                    {
                        translator = new SysMLToModelicaTranslator();
                    }
                    else
                    {
                        translator = new SysMLToSimulinkTranslator();
                        Set<Object> os = new HashSet<Object>();
                        if (domains != null)
                            os.add(domains);
                        if (format != null)
                            os.add(format);
                        if (sfunction != null)
                            os.add(sfunction);
                        translator.loadOptions(os);
                    }
                    sysml2simulation.setTranslator(translator);
                    
                    sysml2simulation.execute();
                }
                catch (FileNotFoundException e)
                {
                    log.error("File not found during the initialization", e);
                    System.exit(2);
                }
                catch (UMLModelErrorException e)
                {
                    log.error("Bad model error during the initialization", e);
                    System.exit(2);
                }
                catch (XMLStreamException e)
                {
                    log.error("XML error during the initialization", e);
                    System.exit(2);
                }
                catch (IOException e)
                {
                    log.error("IO error during the translation", e);
                    System.exit(2);
                }
            }
            else
            {
                log.info("GUI necessary");
                if (model == null)
                {
                    JFileChooser jfc = new JFileChooser(".");
                    if (jfc.showDialog(null, "Select") != JFileChooser.APPROVE_OPTION)
                        System.exit(1);
                    model = jfc.getSelectedFile().getAbsolutePath();
                }
                
                SysMLToSimulationFrame simulationframe = new SysMLToSimulationFrame(model, rootblock, paths);
                simulationframe.pack();
                simulationframe.setVisible(true);
            }
        }
        else
        {
            log.info("Reverse translation");
            if (model != null && target != null)
            {
                log.info("No GUI necessary");
                SimulationToSysMLTranslationManager simulation2sysml;
                try
                {
                    simulation2sysml = new SimulationToSysMLTranslationManager(model, paths);
                    SimulationToSysMLTranslator translator = null;
                    if (target == Target.MODELICA)
                        translator = new ModelicaToSysMLTranslator(null);
                    else
                        translator = new SimulinkToSysMLTranslator();
                    
                    simulation2sysml.setTranslator(translator);
                    simulation2sysml.execute();
                }
                catch (FileNotFoundException e)
                {
                    log.error("File not found during the initialization", e);
                    System.exit(2);
                }
                catch (IOException e)
                {
                    log.error("IO error during the translation", e);
                    System.exit(2);
                }
                catch (UMLModelErrorException e)
                {
                    log.error("Bad model error during the initialization", e);
                    System.exit(2);
                }
            }
            else
            {
                log.info("GUI necessary");
                if (model == null)
                {
                    JFileChooser jfc = new JFileChooser(".");
                    if (jfc.showDialog(null, "Select") != JFileChooser.APPROVE_OPTION)
                        System.exit(1);
                    model = jfc.getSelectedFile().getAbsolutePath();
                }
                
                SimulationToSysMLFrame simulationframe = new SimulationToSysMLFrame(model, rootblock, paths);
                simulationframe.pack();
                simulationframe.setVisible(true);
            }
        }
        
        // error if no input file
        
    }
}
