grammar ExpressionLanguage;

// taken from openmodelica grammar
// changed after Modelica 3.3 grammar
// simplified to cover only expressions
// expressions used for: constraints, default values, change/time expressions, opaquebehaviors
// need to support: equations, modification, statement

FUNCTION	: 'function'	;

IF		: 'if'		;
THEN		: 'then'	;
ELSEIF		: 'elseif'	;
ELSE		: 'else'	;
END			: 'end'		;

FOR		: 'for'		;
LOOP		: 'loop'	;
IN		: 'in'		;

WHILE		: 'while'	;

WHEN		: 'when'	;
ELSEWHEN	: 'elsewhen'	;

OR		: 'or'		;
AND			: 'and'	;
NOT		: 'not'		;

LPAR		: '('	;
RPAR		: ')'	;

LBRACK		: '['	;
RBRACK		: ']'	;
LBRACE		: '{'	;
RBRACE		: '}'	;

EQUALS		: '='	;
ASSIGN		: ':='	;

COMMA		: ','	;
LESS		: '<'	;
LESSEQ		: '<='	;
GREATER		: '>'	;
GREATEREQ	: '>='	;
EQEQ		: '=='	;
LESSGT		: '<>'	;

PLUS		: '+'	;
MINUS		: '-'	;
PLUS_EW     : '.+'  ;
MINUS_EW	: '.-'	;

STAR		: '*'	;
SLASH		: '/'	;
STAR_EW		: '.*'	;
SLASH_EW	: './'	;

POWER		: '^'	;
POWER_EW	: '.^'	;

DOT         : '.';
COLON		: ':'	;
SEMICOLON	: ';'	;

TRUE		: 'true'	;
FALSE		: 'false'	;

DER         : 'der';
INITIAL		: 'initial'	;

IDENT:
		NONDIGIT (NONDIGIT | DIGIT)* | QIDENT
		;

QIDENT:
         '\'' (QCHAR | SESCAPE) (QCHAR | SESCAPE)* '\'' ;

fragment
QCHAR :	DIGIT | NONDIGIT|'!' | '#' | '$' | '%' | '&' | '(' | ')' | '*' | '+' | 
        ',"' | '-' | '.' | '/' | ':' | ';' | '<' | '>' | '=' | '?' | '@' | '[' |
        ']' | '^' | '{' | '}' | '|' | '~' | ' ';

fragment
SESCAPE : '\\' ('\\' | '"' | '\'' | '?' | 'a' | 'b' | 'f' | 'n' | 'r' | 't' | 'v');

fragment
SCHAR :	~["\\];

STRING : '"' (SCHAR | SESCAPE)* '"';

fragment
NONDIGIT : 	[a-zA-Z_];

fragment
DIGIT : [0-9];

fragment
UNSIGNED_INTEGER :
        (DIGIT)+;

UNSIGNED_NUMBER : UNSIGNED_INTEGER ('.' UNSIGNED_INTEGER? )?
                  (('e'|'E') ('+' | '-')? UNSIGNED_INTEGER)?;

equation :
		(   simple_expression EQUALS expression
		|	if_equation
		|   component_reference function_call_args
		)
		;

statements :
	(statement (SEMICOLON? (statement SEMICOLON)*));

statement :
		(   simple_expression ((ASSIGN expression) |function_call_args)
		|	LPAR expression_list RPAR ASSIGN component_reference function_call_args
		|	if_statement
		|	for_statement
		|	while_statement
		)
		;

if_equation :
		IF expression THEN (equation SEMICOLON)*
                (ELSEIF expression THEN (equation SEMICOLON)*)*
                (ELSE (equation SEMICOLON)* )?
		END IF
		;

if_statement :
		IF expression THEN (statement SEMICOLON)*
                (ELSEIF expression THEN (statement SEMICOLON)*)*
                (ELSE (statement SEMICOLON)* )?
		END IF
		;

for_statement :
		FOR for_indices LOOP
		(statement SEMICOLON)*
		END FOR
		;

for_indices :
        for_index (COMMA for_index)*
    ;

for_index:
        IDENT (IN expression)?
;

while_statement :
		WHILE expression LOOP
		(statement SEMICOLON)*
		END WHILE
		;

expression :
		( simple_expression
		)
		;

simple_expression :
		logical_expression
		( COLON logical_expression
			( COLON logical_expression
			)?
		)?
		;


logical_expression :
		logical_term ( OR logical_term )*
		;

logical_term :
		logical_factor ( AND logical_factor )*
		;

logical_factor :
		( NOT )? relation
		;

relation :
		arithmetic_expression ( rel_op arithmetic_expression )?
		;

rel_op :
		( LESS | LESSEQ | GREATER | GREATEREQ | EQEQ | LESSGT )
		;

arithmetic_expression :
		add_op? term (add_op term)*
		;

add_op :
           PLUS|MINUS|PLUS_EW|MINUS_EW;

term :
		factor ( mul_op factor )*
		;

mul_op:
          STAR | SLASH | STAR_EW | SLASH_EW;

factor :
		primary ( ( POWER | POWER_EW ) primary )?
		;

primary :
		( UNSIGNED_NUMBER
		| STRING
		| FALSE
		| TRUE
		| (name_path |DER) function_call_args
        | component_reference
		| LPAR output_expression_list RPAR
		| LBRACK expression_list (SEMICOLON expression_list)* RBRACK
		)
    ;

name_path :
		IDENT (DOT IDENT)*
		;

component_reference :
		DOT? IDENT array_subscripts? (DOT IDENT array_subscripts?)*
		;

function_call_args :
		LPAR function_arguments? RPAR
		;

function_arguments :
                       (function_argument ((COMMA function_arguments))?)|
                       named_arguments
		;

named_arguments :
		named_argument ( COMMA named_argument)*
		;

named_argument :
		IDENT EQUALS function_argument
		;

function_argument:
                     (FUNCTION IDENT LBRACE named_arguments? RBRACE)|expression;

output_expression_list:
                          expression? (COMMA expression?)*;

expression_list :
		expression (COMMA expression_list)*
		;

array_subscripts :
		LBRACK subscript ( COMMA subscript )* RBRACK
	;

subscript :
		expression | COLON
		;
