package com.engisis.sysphs.translation.simulink;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Stack;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.uml2.uml.Activity;
import org.eclipse.uml2.uml.ActivityNode;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.BehavioredClassifier;
import org.eclipse.uml2.uml.CallEvent;
import org.eclipse.uml2.uml.CallOperationAction;
import org.eclipse.uml2.uml.ChangeEvent;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Event;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.InstanceSpecification;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.LiteralReal;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.PseudostateKind;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.StructuralFeature;
import org.eclipse.uml2.uml.TimeEvent;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.Vertex;
import org.eclipse.uml2.uml.VisibilityKind;

import com.engisis.sysphs.language.simscape.SBranch;
import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SComponentReference;
import com.engisis.sysphs.language.simscape.SConnection;
import com.engisis.sysphs.language.simscape.SConnectionPortBlock;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SEquation;
import com.engisis.sysphs.language.simscape.SInput;
import com.engisis.sysphs.language.simscape.SLocation;
import com.engisis.sysphs.language.simscape.SMember;
import com.engisis.sysphs.language.simscape.SMemberAccess;
import com.engisis.sysphs.language.simscape.SMemberAssignment;
import com.engisis.sysphs.language.simscape.SMemberPath;
import com.engisis.sysphs.language.simscape.SNode;
import com.engisis.sysphs.language.simscape.SOutput;
import com.engisis.sysphs.language.simscape.SPackage;
import com.engisis.sysphs.language.simscape.SParameter;
import com.engisis.sysphs.language.simscape.SPhysicalBlock;
import com.engisis.sysphs.language.simscape.SPhysicalConnectionPoint;
import com.engisis.sysphs.language.simscape.SPhysicalLine;
import com.engisis.sysphs.language.simscape.SVariable;
import com.engisis.sysphs.language.simscape.SimscapeFactory;
import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SConnectionPoint;
import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.simulink.SFContinuousStateVariable;
import com.engisis.sysphs.language.simulink.SFDWorkVariable;
import com.engisis.sysphs.language.simulink.SFDiscreteStateVariable;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFParameter;
import com.engisis.sysphs.language.simulink.SFVariable;
import com.engisis.sysphs.language.simulink.SFVariableAssignment;
import com.engisis.sysphs.language.simulink.SFunction1;
import com.engisis.sysphs.language.simulink.SFunction1Block;
import com.engisis.sysphs.language.simulink.SFunction2;
import com.engisis.sysphs.language.simulink.SFunction2Block;
import com.engisis.sysphs.language.simulink.SFunctionBlock;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SInterface;
import com.engisis.sysphs.language.simulink.SLibrary;
import com.engisis.sysphs.language.simulink.SLine;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SReference;
import com.engisis.sysphs.language.simulink.SSubsystem;
import com.engisis.sysphs.language.simulink.SSystem;
import com.engisis.sysphs.language.simulink.SSystemParameter;
import com.engisis.sysphs.language.simulink.SSystemParameterAssignment;
import com.engisis.sysphs.language.simulink.SimulinkFactory;
import com.engisis.sysphs.language.stateflow.SChart;
import com.engisis.sysphs.language.stateflow.SChartBlock;
import com.engisis.sysphs.language.stateflow.SChartSystem;
import com.engisis.sysphs.language.stateflow.SConnectNode;
import com.engisis.sysphs.language.stateflow.SData;
import com.engisis.sysphs.language.stateflow.SDataScope;
import com.engisis.sysphs.language.stateflow.SInstance;
import com.engisis.sysphs.language.stateflow.SJunction;
import com.engisis.sysphs.language.stateflow.SMachine;
import com.engisis.sysphs.language.stateflow.SState;
import com.engisis.sysphs.language.stateflow.SStateType;
import com.engisis.sysphs.language.stateflow.SStateflow;
import com.engisis.sysphs.language.stateflow.STarget;
import com.engisis.sysphs.language.stateflow.STransition;
import com.engisis.sysphs.language.stateflow.STreeNode;
import com.engisis.sysphs.language.stateflow.StateflowFactory;
import com.engisis.sysphs.serialization.simulink.SimulinkSerializer;
import com.engisis.sysphs.util.Configuration;
import com.engisis.sysphs.util.ExpressionLanguageToSimulation;
import com.engisis.sysphs.util.SysMLToSimulationTranslator;
import com.engisis.sysphs.util.SysMLUtil;
import com.engisis.sysphs.util.UMLModelErrorException;

/**
 * Translator from SysML to Simulink.
 * 
 * @author barbau
 *
 */
public class SysMLToSimulinkTranslator extends SysMLToSimulationTranslator
{
    // TODO: context of state charts should be property
    // TODO: refactor initial values
    // NOTES
    // - Naming convention: simulink elements start with 's', UML elements start
    // with 'u'
    private static final Logger log = Logger.getLogger(SysMLToSimulinkTranslator.class);
    
    /**
     * name of the block parameter in level-2 SFunctions
     */
    public static final String SF2_BLOCK_NAME = "block";
    
    /**
     * reference table
     */
    private Hashtable<ReferenceKey, SElement> refs = new Hashtable<ReferenceKey, SElement>();
    
    // reference of block to property.
    
    //
    /**
     * first-pass processing stack
     */
    private Stack<ReferenceKey> toProcess = new Stack<ReferenceKey>();
    /**
     * second-pass processing stack
     */
    private Stack<ReferenceKey> toProcessForConnectors = new Stack<ReferenceKey>();
    
    /**
     * root Simulink model
     */
    private SModel smodel;
    /**
     * root Simulink library
     */
    private SLibrary slibrary;
    /**
     * root Stateflow object
     */
    private SStateflow sstateflow;
    /**
     * SysML utility class
     */
    private SysMLUtil sysmlutil;
    
    /**
     * root Simscape package
     */
    private SPackage spackage;
    
    /**
     * how to translate blocks with constraint blocks
     */
    private SFunction sfmode = SFunction.Level2;
    /**
     * whether new domains are created, or existing domains are reused
     */
    private Domains domains = Domains.REUSE;
    /**
     * serialization format for Simulink models
     */
    private Format format = Format.XML;
    
    /**
     * expression translator
     */
    private ExpressionLanguageToMatlabTranslator el2m = new ExpressionLanguageToMatlabTranslator();
    
    public SysMLToSimulinkTranslator()
    {
    }
    
    public SysMLToSimulinkTranslator(Set<Object> options)
    {
        loadOptions(options);
    }
    
    @Override
    public void loadOptions(Set<Object> options)
    {
        super.loadOptions(options);
        if (options == null)
            return;
        loop: for (Object option : options)
        {
            for (SFunction sf : SFunction.values())
                if (sf.equals(option))
                {
                    log.info("Option " + option + " recognized");
                    sfmode = sf;
                    continue loop;
                }
            for (Domains d : Domains.values())
                if (d.equals(option))
                {
                    log.info("Option " + option + " recognized");
                    domains = d;
                    continue loop;
                }
            for (Format f : Format.values())
                if (f.equals(option))
                {
                    log.info("Option " + option + " recognized");
                    format = f;
                    continue loop;
                }
            log.error("Option " + option + " not recognized");
        }
    }
    
    @Override
    public Set<java.lang.Class<?>> getOptions()
    {
        HashSet<java.lang.Class<?>> res = new HashSet<java.lang.Class<?>>();
        res.add(SFunction.class);
        res.add(Domains.class);
        res.add(Format.class);
        return res;
    }
    
    @Override
    protected ExpressionLanguageToSimulation getExpressionTranslator()
    {
        return el2m;
    }
    
    private enum TranslatedType
    {
        NONE, SYSTEM, CHART, INPORT, OUTPORT, SFUNCTION, COMPONENT, DOMAIN, DATATYPE;
    }
    
    private enum TranslatedVertex
    {
        STATE, INITIAL, JUNCTION, TRANSITION, NONE
    }
    
    public enum SFunction
    {
        Simscape, Level2, Level1;
    }
    
    public enum Domains
    {
        
        CREATE, REUSE;
    }
    
    public enum Format
    {
        
        XML, TEXT;
    }
    
    @Override
    public void reset()
    {
        refs.clear();
        smodel = null;
        sstateflow = null;
        spackage = null;
        slibrary = null;
    }
    
    public TranslatedType getTranslatedType(NamedElement unamedelement)
    {
        if (unamedelement == null)
            throw new IllegalArgumentException("A named element must be provided");
        if (unamedelement.isStereotypeApplied(sysmlutil.getBlock())
                || unamedelement.isStereotypeApplied(sysmlutil.getInterfaceBlock()))
        // && unamedelement.isStereotypeApplied(sysmlutil.getSysMLSimBlock())
        {
            Class uclass = (Class) unamedelement;
            
            // return domain if conserved variables are present and Simscape is
            // available
            if (sfmode == SFunction.Simscape && hasConservedVariable(uclass))
                return TranslatedType.DOMAIN;
            
            List<Property> simprops = sysmlutil.getAllPhSProperties(uclass);
            if (simprops.size() == 0)
            {
                // if associated state machine, translate as chart
                Behavior ubehavior = sysmlutil.getClassifierBehavior(uclass);
                if (ubehavior != null && ubehavior instanceof StateMachine)
                    return TranslatedType.CHART;
                
                int nbBindingConnectors = 0;
                boolean hasComponentGeneral = false;
                boolean hasInoutFlowProperty = false;
                // to determine the type, count constraint properties including
                // in specialization
                int nbConstraintBlock = 0;
                for (Property uproperty : SysMLUtil.getAllCorrectedAttributes(uclass))
                {
                    // skip ports
                    if (uproperty instanceof Port && uproperty.getType() instanceof Classifier && !hasInoutFlowProperty)
                    {
                        List<Property> phsprops = sysmlutil.getAllPhSProperties((Classifier) uproperty.getType());
                        for (Property phsprop : phsprops)
                            if (((EnumerationLiteral) phsprop.getValue(sysmlutil.getFlowProperty(),
                                    "direction")) == sysmlutil.getFlowPropertyInout())
                                hasInoutFlowProperty = true;
                    }
                    Type utype = uproperty.getType();
                    if (utype instanceof Class && utype.isStereotypeApplied(sysmlutil.getConstraintBlock()))
                        nbConstraintBlock++;
                }
                // getGenerals is inefficient, can think on building the
                // dependencies once at the beginning
                TreeIterator<EObject> tieo = r.getAllContents();
                while (tieo.hasNext())
                {
                    EObject eo = tieo.next();
                    if (!(eo instanceof Namespace))
                        tieo.prune();
                    else if (eo instanceof Class)
                    {
                        Class cl = (Class) eo;
                        if (cl.getGenerals().contains(uclass))
                            for (Property uproperty : cl.getAttributes())
                                if (uproperty.getType().isStereotypeApplied(sysmlutil.getConstraintBlock()))
                                    nbConstraintBlock++;
                    }
                }
                
                for (Classifier ugeneral : uclass.getGenerals())
                {
                    if (getTranslatedType(ugeneral) == TranslatedType.COMPONENT)
                    {
                        hasComponentGeneral = true;
                        break;
                    }
                }
                
                for (Connector uconnector : SysMLUtil.getAllConnectors(uclass))
                    if (uconnector.isStereotypeApplied(sysmlutil.getBindingConnector()))
                    {
                        /*
                         * int s1 =
                         * sysmlutil.getPropertyPath(uconnector.getEnds(
                         * ).get(0)).size(); int s2 =
                         * sysmlutil.getPropertyPath(uconnector
                         * .getEnds().get(1)).size(); if ((s1 > 1 && s2 > 1
                         * )||s1 > 2 || s2 > 2)
                         */
                        nbBindingConnectors++;
                        
                    }
                    
                // if the class has parts and no constraint property, translate
                // as system
                if (sfmode == SFunction.Simscape)
                {
                    if (nbConstraintBlock != 0 || nbBindingConnectors != 0 || hasComponentGeneral)
                        return TranslatedType.COMPONENT;
                    return TranslatedType.SYSTEM;
                }
                return TranslatedType.SYSTEM;
            }
            
            // if has simproperty, must be one AND having only one attribute
            if (simprops.size() == 1 && simprops.get(0).getType() instanceof DataType)
            {
                // otherwise, return inport or outport
                EnumerationLiteral lit = (EnumerationLiteral) simprops.get(0).getValue(sysmlutil.getFlowProperty(),
                        "direction");
                
                if ("in".equals(lit.getName()))
                    return TranslatedType.INPORT;
                if ("out".equals(lit.getName()))
                    return TranslatedType.OUTPORT;
                log.warn("The simulink translator does not support inout flow properties in a non-Simscape mode");
            }
            log.warn("Can't translate a type that has several simproperties or not one simvariable:"
                    + print(unamedelement));
            return TranslatedType.NONE;
        }
        if (unamedelement.isStereotypeApplied(sysmlutil.getConstraintBlock()))
        {
            if (sfmode == SFunction.Simscape)
                return TranslatedType.COMPONENT;
            return TranslatedType.SFUNCTION;
        }
        if (unamedelement instanceof DataType
                && !Collections.disjoint(SysMLUtil.getAllCompatibleTypes(((DataType) unamedelement)),
                        Arrays.asList(new Classifier[] { sysmlutil.getUMLBoolean(), sysmlutil.getUMLInteger(),
                                sysmlutil.getUMLDouble(), sysmlutil.getSysMLBoolean(), sysmlutil.getSysMLInteger(),
                                sysmlutil.getSysMLDouble() })))
            return TranslatedType.DATATYPE;
        return TranslatedType.NONE;
    }
    
    @SuppressWarnings("boxing")
    private boolean hasConservedVariable(Class uclass)
    {
        for (Property uproperty : sysmlutil.getAllPhSProperties(uclass))
        {
            if (uproperty.getType() instanceof Class)
            {
                Class utype = (Class) uproperty.getType();
                for (Property uproperty2 : SysMLUtil.getAllCorrectedAttributes(utype))
                    if (uproperty2.isStereotypeApplied(sysmlutil.getPhSVariable())
                            && (Boolean) uproperty2.getValue(sysmlutil.getPhSVariable(), "isConserved"))
                        return true;
            }
        }
        return false;
    }
    
    public static TranslatedVertex getTranslatedVertex(Vertex uvertex)
    {
        if (uvertex instanceof State)
            return TranslatedVertex.STATE;
        if (uvertex instanceof Transition)
            return TranslatedVertex.TRANSITION;
        if (uvertex instanceof Pseudostate && ((Pseudostate) uvertex).getKind() == PseudostateKind.INITIAL_LITERAL)
            return TranslatedVertex.INITIAL;
        return TranslatedVertex.NONE;
    }
    
    private void init()
    {
        if (sfmode == SFunction.Simscape && domains == Domains.REUSE)
        {
            SimscapeUtil.mapDomains(r, refs);
        }
    }
    
    @SuppressWarnings("null")
    @Override
    public void execute(Class urootblock, Resource r, File outputdirectory) throws UMLModelErrorException, IOException
    {
        if (urootblock == null)
            throw new IllegalArgumentException("You must provide a root block");
        if (r == null)
            throw new IllegalArgumentException("You must provide a resouce");
        this.r = r;
        init();
        
        log.debug("Starting translation, mode=" + format + "/" + sfmode + "/" + domains);
        sysmlutil = new SysMLUtil(r.getResourceSet());
        
        String urootblockname = SysMLUtil.getName(urootblock) + getModelNameSuffix();
        
        log.debug("Creating the root library");
        slibrary = SimulinkFactory.eINSTANCE.createSLibrary();
        String slibraryname = urootblockname + "Library";
        SimulinkUtil.setName(slibrary, slibraryname);
        
        SSystem slibrarysystem = SimulinkFactory.eINSTANCE.createSSystem();
        slibrary.setSystem(slibrarysystem);
        SimulinkUtil.setName(slibrarysystem, slibraryname);
        
        log.debug("Creating the root model");
        smodel = SimulinkFactory.eINSTANCE.createSModel();
        SimulinkUtil.setName(smodel, urootblockname);
        
        SSystem smodelsystem = SimulinkFactory.eINSTANCE.createSSystem();
        smodel.setSystem(smodelsystem);
        
        if (sfmode == SFunction.Simscape)
        {
            spackage = SimscapeFactory.eINSTANCE.createSPackage();
            SimulinkUtil.setName(spackage, slibraryname + getFileNameSuffix());
            spackage.setGenerated(true);
        }
        
        TranslatedType tt = getTranslatedType(urootblock);
        
        if (tt == TranslatedType.SYSTEM)
        {
            // associate the root block to the Simulink model
            ReferenceKey rkmodelsystem = getKey(urootblock);
            refs.put(rkmodelsystem, smodelsystem);
            toProcess.add(rkmodelsystem);
        }
        else if (tt == TranslatedType.COMPONENT)
        {
            createComponent(urootblock);
        }
        else
        {
            log.warn("Aborting generation, the root block is not a system not a chart: " + tt);
            return;
        }
        if (urootblock.getVisibility() != VisibilityKind.PUBLIC_LITERAL)
        {
            log.warn("Aborting generation, the root block is not public");
            return;
        }
        
        sstateflow = StateflowFactory.eINSTANCE.createSStateflow();
        SMachine smachine = StateflowFactory.eINSTANCE.createSMachine();
        sstateflow.setMachine(smachine);
        smachine.setID();
        SimulinkUtil.setName(smachine, slibraryname);
        
        STarget starget = StateflowFactory.eINSTANCE.createSTarget();
        starget.setOwningMachine(smachine);
        starget.setID();
        
        while (toProcess.size() != 0)
        {
            ReferenceKey rk = toProcess.pop();
            log.debug("Treating " + print(rk));
            SElement selement = refs.get(rk);
            if (selement == null)
                throw new IllegalStateException("Unable to process an element that is not referenced: " + print(rk));
            // must be before SSystem
            if (selement instanceof SChartSystem)
            {
                // do not add to the library as it is already added to chart
                // block
                processChartSystem((Class) rk.getKey()[0]);
                toProcessForConnectors.push(rk);
            }
            // must be before SSystem
            else if (selement instanceof SComponent)
            {
                SComponent scomponent = processComponent((Class) rk.getKey()[0]);
                if (!rk.getKey()[0].isStereotypeApplied(sysmlutil.getConstraintBlock()))
                    scomponent.setOwningPackage(spackage);
            }
            else if (selement instanceof SDomain)
            {
                processDomain((Class) rk.getKey()[0]).setOwningPackage(spackage);
            }
            else if (selement instanceof SFunction1)
            {
                processSFunction1((Class) rk.getKey()[0], (Class) rk.getKey()[1], (Property) rk.getKey()[2]);
                toProcessForConnectors.add(rk);
            }
            else if (selement instanceof SFunction2)
            {
                processSFunction2((Class) rk.getKey()[0], (Class) rk.getKey()[1], (Property) rk.getKey()[2]);
                toProcessForConnectors.add(rk);
            }
            else if (selement instanceof SSystem)
            {
                // cast guaranteed at this point
                processSystem((Class) rk.getKey()[0]);
                
                toProcessForConnectors.push(rk);
            }
            else if (selement instanceof STreeNode)
                processRegion2((Class) rk.getKey()[0], (Region) rk.getKey()[1]);
            else
                log.warn("Unprocessed element: " + selement);
        }
        // process systems again for connectors.
        // needed because the structure has to exist before connections are made
        while (toProcessForConnectors.size() != 0)
        {
            ReferenceKey rk = toProcessForConnectors.pop();
            log.debug("Treating " + print(rk));
            SElement selement = refs.get(rk);
            
            if (selement instanceof SComponent)
            {
                processComponent2((Class) rk.getKey()[0]);
            }
            else if (selement instanceof SFunction1)
            {
                processSFunction12((Class) rk.getKey()[0], (Class) rk.getKey()[1], (Property) rk.getKey()[2]);
            }
            else if (selement instanceof SFunction2)
            {
                processSFunction22((Class) rk.getKey()[0], (Class) rk.getKey()[1], (Property) rk.getKey()[2]);
            }
            else if (selement instanceof SSystem)
            {
                processSystemForConnectors((Class) rk.getKey()[0]);
            }
        }
        
        // if root is simscape component...
        if (tt == TranslatedType.COMPONENT)
        {
            // create component and physical block
            SComponent sc1 = createComponent(urootblock);
            sc1.setHidden(false);
            SPhysicalBlock spb1 = SimscapeFactory.eINSTANCE.createSPhysicalBlock();
            spb1.setOwningSystem(smodelsystem);
            SimulinkUtil.setName(spb1, sc1.getName().toLowerCase());
            spb1.setComponent(sc1);
            spb1.setSID();
            
            SNode sn1 = null;
            // if no node, create one and add connection
            if (sc1.getNodes().size() == 0)
            {
                if (sc1.getOwnedConnections().size() == 0)
                {
                    log.error("Couldn't find an connection in the root component");
                    return;
                }
                SNode scopiednode = null;
                SConnection scopiedconn = null;
                loop: for (SConnection sconn : sc1.getOwnedConnections())
                {
                    for (SMemberPath smp : sconn.getPoints())
                    {
                        List<SMember> path = smp.getPath();
                        if (path.size() != 0)
                        {
                            SMember sml = path.get(path.size() - 1);
                            if (sml instanceof SNode)
                            {
                                scopiednode = (SNode) sml;
                                scopiedconn = sconn;
                                break loop;
                            }
                        }
                    }
                }
                if (scopiednode == null)
                {
                    log.error("Couldn't find an internal node to copy in the root component");
                    return;
                }
                sn1 = SimscapeFactory.eINSTANCE.createSNode();
                sc1.getOwnedMembers().add(sn1);
                SimulinkUtil.setName(sn1, scopiednode.getName());
                sn1.setDomain(scopiednode.getDomain());
                sn1.setLocation(SLocation.LEFT);
                createConnectionPortBlock(sn1, SLocation.LEFT).setOwningSystem(sc1);
                
                SMemberPath smp = SimscapeFactory.eINSTANCE.createSMemberPath();
                smp.getPath().add(sn1);
                scopiedconn.getPoints().add(smp);
            }
            else
                sn1 = sc1.getNodes().get(0);
            
            SConnectionPortBlock scpb1 = sn1.getConnectionPortBlock();
            
            // create dummy component, physical block, and line
            SComponent sc2 = SimscapeFactory.eINSTANCE.createSComponent();
            sc2.setOwningPackage(spackage);
            SimulinkUtil.setName(sc2, sn1.getDomain().getName() + "Dummy");
            sc2.setHidden(false);
            
            SNode sn2 = SimscapeFactory.eINSTANCE.createSNode();
            sc2.getOwnedMembers().add(sn2);
            SimulinkUtil.setName(sn2, sn1.getName());
            sn2.setDomain(sn1.getDomain());
            sn2.setLocation(SLocation.LEFT);
            SConnectionPortBlock scpb2 = createConnectionPortBlock(sn2, SLocation.LEFT);
            scpb2.setOwningSystem(sc2);
            
            
            for (SVariable svar : sn1.getDomain().getVariables())
            {
                if (!svar.isBalancing())
                {
                    SVariable svar2 = SimscapeFactory.eINSTANCE.createSVariable();
                    sc2.getOwnedMembers().add(svar2);
                    SimulinkUtil.setName(svar2, svar.getName());
                    svar2.setUnit(svar.getUnit());
                    SEquation seq = SimscapeFactory.eINSTANCE.createSEquation();
                    seq.setExpression(svar2.getName() + "==" + sn2.getName() + "." + svar.getName());
                    sc2.getOwnedEquations().add(seq);
                }
                else
                {
                    SVariable svar2 = SimscapeFactory.eINSTANCE.createSVariable();
                    sc2.getOwnedMembers().add(svar2);
                    SimulinkUtil.setName(svar2, svar.getName());
                    svar2.setUnit(svar.getUnit());
                    
                    SBranch sbranch = SimscapeFactory.eINSTANCE.createSBranch();
                    sbranch.setVariable(svar2);
                    SMemberPath smp = SimscapeFactory.eINSTANCE.createSMemberPath();
                    smp.getPath().add(sn1);
                    smp.getPath().add(svar);
                    sbranch.setFrom(smp);
                    sc2.getBranches().add(sbranch);
                    
                    SEquation seq = SimscapeFactory.eINSTANCE.createSEquation();
                    seq.setExpression(svar2.getName() + "==0");
                    sc2.getOwnedEquations().add(seq);
                }
            }
            
            SPhysicalBlock spb2 = SimscapeFactory.eINSTANCE.createSPhysicalBlock();
            spb2.setOwningSystem(smodelsystem);
            SimulinkUtil.setName(spb2, sc2.getName().toLowerCase());
            spb2.setComponent(sc2);
            spb2.setSID();
            
            SPhysicalLine spl = SimscapeFactory.eINSTANCE.createSPhysicalLine();
            SPhysicalConnectionPoint spcp1 = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
            spcp1.setBlock(spb1);
            spcp1.setPort(scpb1);
            SPhysicalConnectionPoint spcp2 = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
            spcp2.setBlock(spb2);
            spcp2.setPort(scpb2);
            spl.getPoints().add(spcp1);
            spl.getPoints().add(spcp2);
            
            smodelsystem.getOwnedLines().add(spl);
            
            // create physical line
        }
        
        // add solver if physical lines are present
        loop: for (SLine sl : smodelsystem.getOwnedLines())
        {
            if (sl instanceof SPhysicalLine)
            {
                SPhysicalLine spl = (SPhysicalLine) sl;
                for (SPhysicalConnectionPoint spc : new ArrayList<SPhysicalConnectionPoint>(spl.getPoints()))
                {
                    if (spc.getPort() != null)
                    {
                        SPhysicalBlock spb = SimscapeFactory.eINSTANCE.createSPhysicalBlock();
                        spb.setOwningSystem(smodelsystem);
                        SimulinkUtil.setName(spb, SimscapeUtil.getSolverConfiguration().getName());
                        spb.setComponent(SimscapeUtil.getSolverConfiguration());
                        spb.setSID();
                        
                        SPhysicalConnectionPoint spc2 = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                        spc2.setBlock(spb);
                        spc2.setPort(SimscapeUtil.getSolverConfiguration().getRConnectionPorts().get(0));
                        spl.getPoints().add(spc2);
                        break loop;
                    }
                }
            }
        }
        
        // save main model in both formats
        SimulinkSerializer ss = new SimulinkSerializer();
        if (outputdirectory != null)
        {
            outputfilename = ss.serialize(outputdirectory.getAbsolutePath(), smodel, slibrary, sstateflow, spackage,
                    format);
        }
        else
        {
            String dir = r.getURI().trimFileExtension().trimSegments(1).toFileString();
            outputfilename = ss.serialize(dir, smodel, slibrary, sstateflow, spackage, format);
        }
    }
    
    private SSystem createSystem(Class ublock)
    {
        ReferenceKey rk = getKey(ublock);
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (selement instanceof SSystem)
                return (SSystem) selement;
            throw new IllegalArgumentException(
                    "The element " + print(rk) + " should be a system: " + selement.getClass());
        }
        log.debug("Creating system " + print(rk));
        SSystem ssystem = SimulinkFactory.eINSTANCE.createSSystem();
        
        // Attach the system to a subsystem
        if (!ublock.isStereotypeApplied(sysmlutil.getSimulinkBlock()))
        {
            SSubsystem ssubsystem = SimulinkFactory.eINSTANCE.createSSubsystem();
            ssubsystem.setSystem(ssystem);
            
            // add the system's subsystem to the library
            // slibrary.getSystem().getOwnedBlocks().add(ssystem2.getOwningSubsystem());
            // opposite reference doesn't seem to work
            ssubsystem.setOwningSystem(slibrary.getSystem());
            ssubsystem.setSID();
        }
        refs.put(rk, ssystem);
        toProcess.add(rk);
        return ssystem;
    }
    
    private SChartSystem createChartSystem(Class ublock)
    {
        ReferenceKey rk = getKey(ublock);
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (selement instanceof SChartSystem)
                return (SChartSystem) selement;
            throw new IllegalArgumentException(
                    "The element " + print(rk) + " should be a chart system: " + selement.getClass());
        }
        log.debug("Creating chart system " + print(rk));
        SChartSystem schartsystem = StateflowFactory.eINSTANCE.createSChartSystem();
        
        SChartBlock ssubsystem = StateflowFactory.eINSTANCE.createSChartBlock();
        ssubsystem.setSystem(schartsystem);
        ssubsystem.setOwningSystem(slibrary.getSystem());
        ssubsystem.setSID();
        
        refs.put(rk, schartsystem);
        toProcess.add(rk);
        
        createChart(ublock, (StateMachine) sysmlutil.getClassifierBehavior(ublock));
        
        return schartsystem;
    }
    
    private SBlock createBlock(Class ublock, Property uproperty)
    {
        ReferenceKey rk = getKey(ublock, uproperty);
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (!(selement instanceof SBlock))
                throw new IllegalArgumentException(
                        "The element " + print(rk) + " should be a block: " + selement.getClass());
            return (SBlock) selement;
        }
        // if none, create the block
        
        if (uproperty.getVisibility() != VisibilityKind.PUBLIC_LITERAL)
        {
            log.warn("Skipping non-public property " + print(uproperty));
            return null;
        }
        
        boolean conjugated = uproperty instanceof Port && ((Port) uproperty).isConjugated();
        Type utype = uproperty.getType();
        if (utype == null)
        {
            log.warn("Skipping property " + print(uproperty) + " because it has no type");
            return null;
        }
        SBlock sblock = null;
        switch (getTranslatedType(utype))
        {
        case SYSTEM:
            // create an interface to the library component
            if (utype.isStereotypeApplied(sysmlutil.getSimulinkBlock()))
            {
                SInterface sinterface = SimulinkFactory.eINSTANCE.createSInterface();
                sinterface.setSystem(createSystem((Class) utype));
                sblock = sinterface;
            }
            // create a reference to the system's subsystem
            else
            {
                SReference sreference = SimulinkFactory.eINSTANCE.createSReference();
                sreference.setSystem(createSystem((Class) utype));
                sblock = sreference;
            }
            break;
        case CHART:
            // create a subsystem and a chart
            SReference sreference = SimulinkFactory.eINSTANCE.createSReference();
            // sreference.setSystem(createChartSystem((Class) utype));
            sreference.setSystem(createChartSystem((Class) utype));
            sblock = sreference;
            /*
             * previously used SChartBlock schartblock =
             * StateflowFactory.eINSTANCE.createSChartBlock(); SChartSystem
             * schartsystem = createChartSystem((Class) utype);
             * schartblock.setSystem(schartsystem);
             * schartsystem.setOwningSubsystem(schartblock);
             * schartblock.setChart(createChart((StateMachine)((Class)
             * utype).getClassifierBehavior())); sblock = schartblock;
             */
            break;
        case SFUNCTION:
            if (sfmode == SFunction.Level1)
            {
                SFunction1Block sfunctionblock = SimulinkFactory.eINSTANCE.createSFunction1Block();
                sfunctionblock.setSfunction(createSFunction1((Class) utype, ublock, uproperty));
                sblock = sfunctionblock;
            }
            else
            {
                SFunction2Block sfunctionblock = SimulinkFactory.eINSTANCE.createSFunction2Block();
                sfunctionblock.setSfunction(createSFunction2((Class) utype, ublock, uproperty));
                sblock = sfunctionblock;
            }
            break;
        case INPORT:
            if (sfmode == SFunction.Simscape)
            {
                SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
                scpb.setComponent(SimscapeUtil.getConnectionPortComponent());
                if (conjugated)
                    scpb.setLocation(SLocation.RIGHT);
                else
                    scpb.setLocation(SLocation.LEFT);
                sblock = scpb;
            }
            else
            {
                if (conjugated)
                {
                    SOutport soutport = SimulinkFactory.eINSTANCE.createSOutport();
                    soutport.setUnit(
                            getUnitFromType(sysmlutil.getAllPhSProperties((Classifier) utype).get(0).getType()));
                    sblock = soutport;
                }
                else
                {
                    SInport sinport = SimulinkFactory.eINSTANCE.createSInport();
                    sinport.setUnit(
                            getUnitFromType(sysmlutil.getAllPhSProperties((Classifier) utype).get(0).getType()));
                    sblock = sinport;
                }
            }
            break;
        case OUTPORT:
            if (sfmode == SFunction.Simscape)
            {
                SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
                scpb.setComponent(SimscapeUtil.getConnectionPortComponent());
                if (conjugated)
                    scpb.setLocation(SLocation.LEFT);
                else
                    scpb.setLocation(SLocation.RIGHT);
                sblock = scpb;
            }
            else
            {
                if (conjugated)
                {
                    SInport sinport = SimulinkFactory.eINSTANCE.createSInport();
                    sinport.setUnit(
                            getUnitFromType(sysmlutil.getAllPhSProperties((Classifier) utype).get(0).getType()));
                    sblock = sinport;
                }
                else
                {
                    SOutport soutport = SimulinkFactory.eINSTANCE.createSOutport();
                    soutport.setUnit(
                            getUnitFromType(sysmlutil.getAllPhSProperties((Classifier) utype).get(0).getType()));
                    sblock = soutport;
                }
            }
            break;
        case DATATYPE:
            if (uproperty.isStereotypeApplied(sysmlutil.getPhSConstant()) && uproperty.getDefaultValue() != null)
            {
                // used to create a constant block
                // now binding connectors
                /*
                 * SInterface sconstant =
                 * SimulinkFactory.eINSTANCE.createSInterface();
                 * 
                 * SSystem ssconstant =
                 * SimulinkFactory.eINSTANCE.createSSystem();
                 * ssconstant.setName("Constant");
                 * ssconstant.getOwnedBlocks().add
                 * (SimulinkFactory.eINSTANCE.createSOutport());
                 * sconstant.setSystem(ssconstant);
                 * 
                 * SSystemParameter sSFParameter =
                 * SimulinkFactory.eINSTANCE.createSSystemParameter();
                 * sSFParameter.setName("Value");
                 * ssconstant.getSystemparameters().add(sSFParameter);
                 * 
                 * SDataValue sdv = getValue(uproperty.getDefaultValue()); if
                 * (sdv == null) throw new UMLModelErrorException(r,
                 * "The constant " + print(uproperty) +
                 * " has no valid default value"); SSystemParameterAssignment
                 * sassignment =
                 * SimulinkFactory.eINSTANCE.createSSystemParameterAssignment();
                 * sassignment.setParameter(sSFParameter);
                 * sassignment.setValue(sdv);
                 * sconstant.getAssignments().add(sassignment); sblock =
                 * sconstant;
                 */
            }
            else if (uproperty.isStereotypeApplied(sysmlutil.getPhSVariable()))
            {
                
                // SInterface sscope =
                // SimulinkFactory.eINSTANCE.createSInterface();
                //
                // SSystem ssscope = SimulinkFactory.eINSTANCE.createSSystem();
                // ssscope.setName("Scope");
                // SimulinkFactory.eINSTANCE.createSInport().setOwningSystem(ssscope);
                // sscope.setSystem(ssscope);
                //
                // sblock = sscope;
                
            }
            else
                log.warn("Skipping datatype property " + print(rk));
            break;
        case COMPONENT:
            SPhysicalBlock sphysicalblock = SimscapeFactory.eINSTANCE.createSPhysicalBlock();
            SComponent scomponent = createComponent((Class) uproperty.getType());
            sphysicalblock.setComponent(scomponent);
            SimulinkUtil.setName(sphysicalblock, SysMLUtil.getName(uproperty));
            
            sblock = sphysicalblock;
            break;
        case DOMAIN:
            // Domains can't be accessed from here
            break;
        case NONE:
            log.warn("Skipping property " + print(rk) + " because the type is not supported");
            break;
        default:
            break;
        }
        if (sblock != null)
        {
            log.debug("Creating block " + print(rk));
            String name = SysMLUtil.getName(uproperty);
            if (uproperty.isStereotypeApplied(sysmlutil.getSimulinkPort()))
            {
                Object o = uproperty.getValue(sysmlutil.getSimulinkPort(), "name");
                if (o != null)
                    name = o.toString();
            }
            SimulinkUtil.setName(sblock, name);
            refs.put(rk, sblock);
        }
        return sblock;
    }
    
    private SFunction1 createSFunction1(Class ublock, Class ccontext, Property pcontext)
    {
        ReferenceKey rk = getKey(ublock, ccontext, pcontext);
        // SFunctions can be created and processed at the same time
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (selement instanceof SFunction1)
                return (SFunction1) selement;
            throw new IllegalArgumentException(
                    "The element corresponding to " + print(rk) + " is supposed to be a S-Function");
        }
        SFunction1 sfunction = SimulinkFactory.eINSTANCE.createSFunction1();
        log.debug("Creating sfunction " + print(rk));
        refs.put(rk, sfunction);
        toProcess.push(rk);
        return sfunction;
    }
    
    private SFunction2 createSFunction2(Class ublock, Class ccontext, Property pcontext)
    {
        ReferenceKey rk = getKey(ublock, ccontext, pcontext);
        // SFunctions can be created and processed at the same time
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (selement instanceof SFunction2)
                return (SFunction2) selement;
            throw new IllegalArgumentException(
                    "The element corresponding to " + print(rk) + " is supposed to be a S-Function");
        }
        SFunction2 sfunction = SimulinkFactory.eINSTANCE.createSFunction2();
        log.debug("Creating sfunction " + print(rk));
        refs.put(rk, sfunction);
        toProcess.push(rk);
        return sfunction;
    }
    
    private SFVariable createVariable(Class ublock, Property uproperty, Class ccontext, Property pcontext)
            throws UMLModelErrorException
    {
        ReferenceKey rk = getKey(ublock, uproperty, ccontext, pcontext);
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (!(selement instanceof SFVariable))
                throw new IllegalArgumentException("Expecting a variable: " + selement.getClass());
            return (SFVariable) selement;
        }
        // if none, create the variable
        SFVariable sv = null;
        
        // actual variable type determined by the connected variable
        for (Connector uconnector : SysMLUtil.getAllConnectors(ccontext))
        {
            if (uconnector.isStereotypeApplied(sysmlutil.getBindingConnector()))
            {
                // can bind only properties with values (size 1) and nested
                // connectorends (size 3)
                // to sfunction variable (size 2)
                ConnectorEnd uend1 = uconnector.getEnds().get(0);
                ConnectorEnd uend2 = uconnector.getEnds().get(1);
                
                List<Property> lprop1 = sysmlutil.getCorrectedPropertyPath(uend1, ccontext);
                List<Property> lprop2 = sysmlutil.getCorrectedPropertyPath(uend2, ccontext);
                
                List<Property> lblock = null;
                List<Property> lfunction = null;
                
                // determine between lprop1/lprop2 which is the function and
                // which is the block
                if (lprop1.size() == 2 && lprop1.get(0) == pcontext && lprop1.get(1) == uproperty)
                {
                    lfunction = lprop1;
                    lblock = lprop2;
                }
                if (lprop2.size() == 2 && lprop2.get(0) == pcontext && lprop2.get(1) == uproperty)
                {
                    lfunction = lprop2;
                    lblock = lprop1;
                }
                
                // this connector doesn't match
                if (lblock == null || lfunction == null)
                    continue;
                
                // size 1: look for continuous/discrete variables, parameters
                
                if (lblock.size() == 1)
                {
                    Property ubound = lblock.get(0);
                    if (ubound.isStereotypeApplied(sysmlutil.getPhSVariable()))
                    {
                        if (((Boolean) ubound.getValue(sysmlutil.getPhSVariable(), "isContinuous")).booleanValue())
                        {
                            SFContinuousStateVariable scsv = SimulinkFactory.eINSTANCE
                                    .createSFContinuousStateVariable();
                            if (ubound.getDefaultValue() != null)
                                scsv.getValue().add(getValue(ubound.getDefaultValue()));
                            else
                                log.warn("The property " + print(ubound) + " should have a default value");
                            log.debug("Creating continuous variable " + print(rk));
                            sv = scsv;
                        }
                        else
                        {
                            SFDiscreteStateVariable sdsv = SimulinkFactory.eINSTANCE.createSFDiscreteStateVariable();
                            if (ubound.getDefaultValue() != null)
                                sdsv.getValue().add(getValue(ubound.getDefaultValue()));
                            else
                                log.warn("The property " + print(ubound) + " should have a default value");
                            log.debug("Creating discrete variable " + print(rk));
                            sv = sdsv;
                        }
                    }
                    else if (ubound.isStereotypeApplied(sysmlutil.getPhSConstant()))
                    {
                        SFParameter sp = SimulinkFactory.eINSTANCE.createSFParameter();
                        if (ubound.getDefaultValue() == null)
                        {
                            SDoubleValue sdv = SimulinkFactory.eINSTANCE.createSDoubleValue();
                            sdv.setValue(0);
                            sp.getValue().add(sdv);
                        }
                        else
                            sp.getValue().add(getValue(ubound.getDefaultValue()));
                        log.debug("Creating parameter " + print(rk));
                        sv = sp;
                    }
                    else
                        log.warn("Skipping property " + print(uproperty)
                                + " as it is not in, out, continuous, or discrete");
                }
                
                // size 2: look for input/output
                // problem: calling createBlock
                else if (lblock.size() == 2)
                {
                    SBlock sblock = createBlock(ccontext, lfunction.get(0));
                    if (!(sblock instanceof SFunctionBlock))
                    {
                        log.warn("Skipping " + print(uproperty) + " as an s-function can't be found");
                        continue;
                    }
                    SFunctionBlock sfb = (SFunctionBlock) sblock;
                    
                    sblock = createBlock(ccontext, lblock.get(0));
                    if (sblock == null)
                    {
                        log.warn("Skipping " + print(uproperty) + " as an inport/outport can't be found");
                        continue;
                    }
                    
                    if (sblock instanceof SInport)
                    {
                        SFInputVariable siv = SimulinkFactory.eINSTANCE.createSFInputVariable();
                        log.debug("Creating input variable " + print(rk));
                        sv = siv;
                        SInport so = (SInport) sblock;
                        siv.setInport(so);
                        siv.setDirectFeedthrough(false);
                        sfb.getInports().add(so);
                    }
                    else if (sblock instanceof SOutport)
                    {
                        SFOutputVariable sov = SimulinkFactory.eINSTANCE.createSFOutputVariable();
                        log.debug("Creating output variable " + print(rk));
                        sv = sov;
                        SOutport so = (SOutport) sblock;
                        sov.setOutport(so);
                        sfb.getOutports().add(so);
                    }
                    else
                        log.warn("Unexpected block type: " + sblock);
                }
                else
                    log.warn("Can't support this number of nested properties");
                if (sv != null)
                {
                    SimulinkUtil.setName(sv, SysMLUtil.getName(uproperty));
                    if (uproperty.isStereotypeApplied(sysmlutil.getMultidimensionalElement()) && uproperty
                            .getValue(sysmlutil.getMultidimensionalElement(), "dimension") instanceof List<?>)
                    {
                        @SuppressWarnings("unchecked")
                        List<Object> dims = (List<Object>) uproperty.getValue(sysmlutil.getMultidimensionalElement(),
                                "dimension");
                        if (dims.size() > 2)
                            throw new UMLModelErrorException(r,
                                    "The property can't have more than 2 dimensions " + print(uproperty));
                        for (int i = 0; i < dims.size(); i++)
                            if (dims.get(i) instanceof Integer)
                                sv.getDimensions().add((Integer) dims.get(i));
                    }
                    refs.put(rk, sv);
                    return sv;
                }
            }
        }
        // if nothing treated at this point, create DWork
        SFDWorkVariable sdw = SimulinkFactory.eINSTANCE.createSFDWorkVariable();
        
        sv = sdw;
        SimulinkUtil.setName(sv, SysMLUtil.getName(uproperty));
        if (uproperty.isStereotypeApplied(sysmlutil.getMultidimensionalElement())
                && uproperty.getValue(sysmlutil.getMultidimensionalElement(), "dimension") instanceof List<?>)
        {
            @SuppressWarnings("unchecked")
            List<Object> dims = (List<Object>) uproperty.getValue(sysmlutil.getMultidimensionalElement(), "dimension");
            if (dims.size() > 2)
                throw new UMLModelErrorException(r,
                        "The property can't have more than 2 dimensions " + print(uproperty));
            for (int i = 0; i < dims.size(); i++)
                if (dims.get(i) instanceof Integer)
                    sv.getDimensions().add((Integer) dims.get(i));
        }
        refs.put(rk, sv);
        return sv;
        
    }
    
    private SComponent createComponent(Class ublock)
    {
        ReferenceKey rk = getKey(ublock);
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (selement instanceof SComponent)
                return (SComponent) selement;
            throw new IllegalStateException(
                    "The element correspinding to " + print(ublock) + " is not a component: " + selement.getClass());
        }
        log.debug("Creating component " + print(rk));
        SComponent scomponent = SimscapeFactory.eINSTANCE.createSComponent();
        refs.put(rk, scomponent);
        toProcess.push(rk);
        return scomponent;
    }
    
    private SDomain createDomain(Class ublock)
    {
        ReferenceKey rk = getKey(ublock);
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (selement instanceof SDomain)
                return (SDomain) selement;
            throw new IllegalStateException(
                    "The element corresponding to " + print(ublock) + " is not a domain: " + selement.getClass());
        }
        SDomain sdomain = null;
        // if CREATE or REUSE with non-existing domains,
        // create a domain and recopy attributes
        // if REUSE with existing domains, put them directly
        if (domains == Domains.CREATE)
        {
            sdomain = SimscapeFactory.eINSTANCE.createSDomain();
            refs.put(rk, sdomain);
            toProcess.push(rk);
        }
        // domain retrieved (the class that contains the SimBlock property is
        // passed)
        else
        {
            // guaranteed 1
            selement = refs.get(getKey(sysmlutil.getAllPhSProperties(ublock).get(0).getType()));
            if (selement == null || !(selement instanceof SDomain))
            {
                log.debug("Creating domain " + print(rk));
                sdomain = SimscapeFactory.eINSTANCE.createSDomain();
                refs.put(rk, sdomain);
                toProcess.push(rk);
            }
            else
            {
                sdomain = (SDomain) selement;
                refs.put(rk, sdomain);
            }
        }
        return sdomain;
    }
    
    private SMember createMember(Property uproperty) throws UMLModelErrorException
    {
        ReferenceKey rk = getKey(uproperty);
        SElement selement = refs.get(rk);
        if (selement != null)
        {
            if (selement instanceof SMember)
                return (SMember) selement;
            throw new IllegalStateException(
                    "The element corresponding to " + print(uproperty) + " is not a member: " + selement.getClass());
        }
        SMember smember = null;
        Type utype = uproperty.getType();
        if (utype.isStereotypeApplied(sysmlutil.getConstraintBlock()))
        {
            log.info("Skipping constraint block property" + print(uproperty));
            return null;
        }
        boolean conjugated = (uproperty instanceof Port) && ((Port) uproperty).isConjugated();
        ValueSpecification vs = uproperty.getDefaultValue();
        String su = getUnitFromType(utype);
        Property simprop;
        TranslatedType tt = getTranslatedType(utype);
        switch (tt)
        {
        case COMPONENT:
        case SYSTEM:
            SComponentReference scomponentreference = SimscapeFactory.eINSTANCE.createSComponentReference();
            SComponent scomponent = createComponent((Class) utype);
            scomponentreference.setComponent(scomponent);
            scomponentreference.setHidden(true);
            smember = scomponentreference;
            break;
        case DOMAIN:
            // may be node
            SNode snode = SimscapeFactory.eINSTANCE.createSNode();
            SDomain sdomain = createDomain((Class) utype);
            snode.setDomain(sdomain);
            
            smember = snode;
            break;
        case INPORT:
            simprop = sysmlutil.getAllPhSProperties((Class) utype).get(0);
            su = getUnitFromType(simprop.getType());
            if (conjugated)
            {
                SOutput soutput = SimscapeFactory.eINSTANCE.createSOutput();
                if (vs != null)
                {
                    SDataValue sv = getValue(vs);
                    soutput.setValue(sv);
                }
                if (su != null)
                    soutput.setUnit(su);
                smember = soutput;
            }
            else
            {
                SInput sinput = SimscapeFactory.eINSTANCE.createSInput();
                if (vs != null)
                {
                    SDataValue sv = getValue(vs);
                    sinput.setValue(sv);
                }
                if (su != null)
                    sinput.setUnit(su);
                smember = sinput;
            }
            break;
        case OUTPORT:
            simprop = sysmlutil.getAllPhSProperties((Class) utype).get(0);
            su = getUnitFromType(simprop.getType());
            if (conjugated)
            {
                SInput sinput = SimscapeFactory.eINSTANCE.createSInput();
                if (vs != null)
                {
                    SDataValue sv = getValue(vs);
                    sinput.setValue(sv);
                }
                if (su != null)
                    sinput.setUnit(su);
                smember = sinput;
            }
            else
            {
                SOutput soutput = SimscapeFactory.eINSTANCE.createSOutput();
                if (vs != null)
                {
                    SDataValue sv = getValue(vs);
                    soutput.setValue(sv);
                }
                if (su != null)
                    soutput.setUnit(su);
                smember = soutput;
            }
            break;
        case DATATYPE:
            if (uproperty.isStereotypeApplied(sysmlutil.getPhSConstant()))
            {
                SParameter sparameter = SimscapeFactory.eINSTANCE.createSParameter();
                if (vs != null)
                {
                    SDataValue sv = getValue(vs);
                    sparameter.setValue(sv);
                }
                if (su != null)
                    sparameter.setUnit(su);
                smember = sparameter;
            }
            else if (uproperty.isStereotypeApplied(sysmlutil.getPhSVariable()))
            {
                if (!((Boolean) uproperty.getValue(sysmlutil.getPhSVariable(), "isContinuous")).booleanValue())
                {
                    log.warn("Don't know how to handle discrete variables, skippig");
                    return null;
                }
                SVariable svariable = SimscapeFactory.eINSTANCE.createSVariable();
                // balancing=true only in domains
                if (((Boolean) uproperty.getValue(sysmlutil.getPhSVariable(), "isConserved")).booleanValue()
                        && sysmlutil.isSimBlock(uproperty.getClass_()))
                    svariable.setBalancing(true);
                
                if (vs != null)
                {
                    SDataValue sv = getValue(vs);
                    svariable.setValue(sv);
                }
                if (su != null)
                {
                    svariable.setUnit(su);
                    log.info("SETTING UNIT " + su + " TO " + uproperty.getName());
                }
                
                smember = svariable;
            }
            else
            {
                SVariable svariable = SimscapeFactory.eINSTANCE.createSVariable();
                if (vs != null)
                {
                    SDataValue sv = getValue(vs);
                    if (!(sv instanceof SExpressionValue))
                        svariable.setValue(sv);
                }
                if (su != null)
                {
                    svariable.setUnit(su);
                    log.info("SETTING UNIT " + su + " TO " + svariable.getName());
                }
                smember = svariable;
            }
            break;
        default:
            log.warn("Member " + print(uproperty) + " skipped as " + print(utype)
                    + " can't be a component, domain, or datatype: " + tt.name());
            break;
        }
        if (smember != null)
        {
            SimulinkUtil.setName(smember, SysMLUtil.getName(uproperty));
            if (uproperty.getVisibility() == VisibilityKind.PUBLIC_LITERAL)
                smember.setAccess(SMemberAccess.PUBLIC);
            else if (uproperty.getVisibility() == VisibilityKind.PROTECTED_LITERAL)
                smember.setAccess(SMemberAccess.PROTECTED);
            smember.setHidden(false);
            refs.put(rk, smember);
        }
        else
            log.warn("Skipping property " + print(uproperty));
        return smember;
    }
    
    private SSystem processSystem(Class ublock)
    {
        log.debug("Processing system " + print(ublock));
        SSystem ssystem = createSystem(ublock);
        String name = SysMLUtil.getName(ublock);
        if (ublock.isStereotypeApplied(sysmlutil.getSimulinkBlock()))
        {
            Object o = ublock.getValue(sysmlutil.getSimulinkBlock(), "name");
            if (o != null)
                name = o.toString();
        }
        SimulinkUtil.setName(ssystem, name);
        SSubsystem ssubsystem = ssystem.getOwningSubsystem();
        if (ssubsystem != null)
            SimulinkUtil.setName(ssubsystem, name);
        
        // parse attributes, to create subsystems, inports, outports
        List<Property> uproperties = new ArrayList<Property>(SysMLUtil.getAllCorrectedAttributes(ublock));
        for (Property uproperty : uproperties)
        {
            Type utype = uproperty.getType();
            // parse type operations to add inout ports for provided/required
            // operations
            String upropertyname = SysMLUtil.getName(uproperty);
            if (utype != null && utype.isStereotypeApplied(sysmlutil.getInterfaceBlock()) && utype instanceof Class)
            {
                log.info("Processing InterfaceBlock" + print(utype));
                for (Operation uoperation : ((Class) utype).getAllOperations())
                {
                    if (uoperation.isStereotypeApplied(sysmlutil.getDirectedFeature()))
                    {
                        EnumerationLiteral direction = (EnumerationLiteral) uoperation
                                .getValue(sysmlutil.getDirectedFeature(), "featureDirection");
                        String uoperationname = SysMLUtil.getName(uoperation);
                        if (direction.getName().equals("required"))
                        {
                            ReferenceKey key = getKey(ublock, uproperty, uoperation);
                            SInport sinport = SimulinkFactory.eINSTANCE.createSInport();
                            sinport.setOwningSystem(ssystem);
                            SimulinkUtil.setName(sinport, upropertyname + "_" + uoperationname + "_required");
                            sinport.setSID();
                            /*
                             * if (ssystem instanceof SChartSystem &&
                             * ssystem.getOwningSubsystem() != null)
                             * sinport.setSID
                             * (ssystem.getOwningSubsystem().getSID() + "::" +
                             * ((SChartSystem)ssystem).getNextID()); else
                             * sinport.setSID(String.valueOf(++lsid));
                             */
                            refs.put(key, sinport);
                        }
                        else if (direction.getName().equals("provided"))
                        {
                            ReferenceKey key = getKey(ublock, uproperty, uoperation);
                            SOutport soutport = SimulinkFactory.eINSTANCE.createSOutport();
                            soutport.setOwningSystem(ssystem);
                            SimulinkUtil.setName(soutport, upropertyname + "_" + uoperationname + "_provided");
                            soutport.setSID();
                            /*
                             * if (ssystem instanceof SChartSystem &&
                             * ssystem.getOwningSubsystem() != null)
                             * soutport.setSID
                             * (ssystem.getOwningSubsystem().getSID() + "::" +
                             * ((SChartSystem)ssystem).getNextID()); else
                             * soutport.setSID(String.valueOf(++lsid));
                             */
                            refs.put(key, soutport);
                        }
                        else
                            log.warn("Provided+required features not implemented: " + print(uproperty));
                    }
                }
            }
            
            // skip block if SimulationBlock and not SimulationPort
            if (ublock.isStereotypeApplied(sysmlutil.getSimulinkBlock()))
            {
                if (uproperty.isStereotypeApplied(sysmlutil.getSimulinkParameter()))
                {
                    SSystemParameter SFParameter = SimulinkFactory.eINSTANCE.createSSystemParameter();
                    Object o = uproperty.getValue(sysmlutil.getSimulinkParameter(), "name");
                    if (o == null)
                        SimulinkUtil.setName(SFParameter, upropertyname);
                    else
                        SimulinkUtil.setName(SFParameter, o.toString());
                    o = uproperty.getValue(sysmlutil.getSimulinkParameter(), "value");
                    if (o != null)
                    {
                        Object of = null;
                        if (sysmlutil.getSimulinkParameter().getAttribute("format", null) != null)
                            of = uproperty.getValue(sysmlutil.getSimulinkParameter(), "format");
                        String f = of == null ? "%s" : of.toString();
                        f = f.replaceAll("%s", o.toString());
                        
                        SExpressionValue sev = SimulinkFactory.eINSTANCE.createSExpressionValue();
                        sev.setValue(f);
                        SFParameter.setValue(sev);
                    }
                    
                    refs.put(getKey(ublock, uproperty), SFParameter);
                    ssystem.getSystemparameters().add(SFParameter);
                    continue;
                }
                else if (!uproperty.isStereotypeApplied(sysmlutil.getSimulinkPort()))
                {
                    log.info("Skipping non-SimulationPort " + print(uproperty));
                    continue;
                }
            }
            SBlock sblock = createBlock(ublock, uproperty);
            if (sblock != null)
            {
                // ssystem.getOwnedBlocks().add(sblock);
                log.info("Adding " + sblock.getName() + " to " + ssystem.getName());
                sblock.setOwningSystem(ssystem);
                sblock.setSID();
                /*
                 * if (ssystem instanceof SChartSystem &&
                 * ssystem.getOwningSubsystem() != null)
                 * sblock.setSID(ssystem.getOwningSubsystem().getSID() + "::" +
                 * ((SChartSystem)ssystem).getNextID()); else
                 * sblock.setSID(String.valueOf(++lsid));
                 */
            }
            else if (uproperty.isStereotypeApplied(sysmlutil.getPhSConstant()))
                log.info("Constant " + print(getKey(ublock, uproperty)) + " identified");
            else
                log.warn("Block " + print(getKey(ublock, uproperty)) + " not added");
        }
        
        return ssystem;
    }
    
    private SChartSystem processChartSystem(Class ublock) throws UMLModelErrorException
    {
        log.debug("Processing chart system " + print(ublock));
        SChartSystem schartsystem = createChartSystem(ublock);
        processSystem(ublock);
        
        // need to create an SFunction, a Demux, and a Terminator plus lines
        // SFunction
        SFunction1Block sf1block = SimulinkFactory.eINSTANCE.createSFunction1Block();
        sf1block.setName(" SFunction ");
        sf1block.setOwningSystem(schartsystem);
        sf1block.setSID();
        
        SFunction1 sf1 = SimulinkFactory.eINSTANCE.createSFunction1();
        sf1.setName("sf_sfun");
        sf1block.setSfunction(sf1);
        
        SChart chart = (SChart) refs.get(getKey(ublock, sysmlutil.getClassifierBehavior(ublock)));
        
        for (SBlock sb : schartsystem.getOwnedBlocks())
        {
            if (sb instanceof SInport)
            {
                // connect inport to SF inport
                SInport inport = SimulinkFactory.eINSTANCE.createSInport();
                inport.setOwningSystem(sf1);
                SLine line = schartsystem.getLine(sb, null);
                SConnectionPoint pt = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                pt.setBlock(sf1block);
                pt.setPort(inport);
                line.getDestinations().add(pt);
                
                // create chart data
                SData indata = StateflowFactory.eINSTANCE.createSData();
                indata.setOwningTreeNode(chart);
                SimulinkUtil.setName(indata, ((SInport) sb).getName());
                indata.setMachine(chart.getOwningMachine());
                indata.setID();
                indata.setScope(SDataScope.INPUT);
                
            }
            else if (sb instanceof SOutport)
            {
                SOutport outport = SimulinkFactory.eINSTANCE.createSOutport();
                outport.setOwningSystem(sf1);
                SLine line = schartsystem.getLine(sf1block, outport);
                SConnectionPoint pt = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                pt.setBlock(sb);
                line.getDestinations().add(pt);
                
                // create chart data
                SData outdata = StateflowFactory.eINSTANCE.createSData();
                outdata.setOwningTreeNode(chart);
                SimulinkUtil.setName(outdata, ((SOutport) sb).getName());
                outdata.setMachine(chart.getOwningMachine());
                outdata.setID();
                outdata.setScope(SDataScope.OUTPUT);
            }
        }
        
        for (Property uproperty : SysMLUtil.getAllCorrectedAttributes(ublock))
        {
            if (uproperty.isStereotypeApplied(sysmlutil.getPhSConstant()))
            {
                String upropertyname = SysMLUtil.getName(uproperty);
                log.info("Constant " + upropertyname + " added to state machine");
                SData constant = StateflowFactory.eINSTANCE.createSData();
                constant.setOwningTreeNode(chart);
                SimulinkUtil.setName(constant, upropertyname);
                constant.setMachine(chart.getOwningMachine());
                constant.setID();
                constant.setScope(SDataScope.CONSTANT);
                if (uproperty.getDefaultValue() != null)
                {
                    constant.setValue(getValue(uproperty.getDefaultValue()));
                }
                else
                {
                    SDoubleValue sdv = SimulinkFactory.eINSTANCE.createSDoubleValue();
                    sdv.setValue(0.0);
                    constant.setValue(sdv);
                }
            }
        }
        
        // Ground
        if (schartsystem.getInports().size() == 0)
        {
            SInport inground = SimulinkFactory.eINSTANCE.createSInport();
            inground.setOwningSystem(sf1);
            
            SInterface ground = SimulinkFactory.eINSTANCE.createSInterface();
            ground.setName(" Ground ");
            
            SSystem grounds = SimulinkFactory.eINSTANCE.createSSystem();
            grounds.setName("Ground");
            SimulinkFactory.eINSTANCE.createSInport().setOwningSystem(grounds);
            SimulinkFactory.eINSTANCE.createSOutport().setOwningSystem(grounds);
            
            ground.setSystem(grounds);
            
            ground.setOwningSystem(schartsystem);
            ground.setSID();
            
            SLine sline = schartsystem.getLine(ground, null);
            SConnectionPoint scp = SimulinkFactory.eINSTANCE.createSConnectionPoint();
            scp.setBlock(sf1block);
            scp.setPort(inground);
            sline.getDestinations().add(scp);
        }
        
        // dummy output, added for unknown reason
        SOutport outterm = SimulinkFactory.eINSTANCE.createSOutport();
        outterm.setOwningSystem(sf1);
        
        // Demux
        SInterface demux = SimulinkFactory.eINSTANCE.createSInterface();
        demux.setName(" Demux ");
        
        // inefficient, would need to call Simulink block library
        SSystem demuxs = SimulinkFactory.eINSTANCE.createSSystem();
        demuxs.setName("Demux");
        SimulinkFactory.eINSTANCE.createSInport().setOwningSystem(demuxs);
        SimulinkFactory.eINSTANCE.createSOutport().setOwningSystem(demuxs);
        
        demux.setSystem(demuxs);
        
        SSystemParameter demuxoutput = SimulinkFactory.eINSTANCE.createSSystemParameter();
        demuxoutput.setName("Outputs");
        SExpressionValue demuxoutputv = SimulinkFactory.eINSTANCE.createSExpressionValue();
        demuxoutputv.setValue("1");
        demuxoutput.setValue(demuxoutputv);
        demuxs.getSystemparameters().add(demuxoutput);
        
        demux.setOwningSystem(schartsystem);
        demux.setSID();
        
        SLine line1 = schartsystem.getLine(sf1block, outterm);
        SConnectionPoint pt1 = SimulinkFactory.eINSTANCE.createSConnectionPoint();
        pt1.setBlock(demux);
        line1.getDestinations().add(pt1);
        
        // Terminator
        SInterface terminator = SimulinkFactory.eINSTANCE.createSInterface();
        terminator.setName(" Terminator ");
        
        SSystem terminators = SimulinkFactory.eINSTANCE.createSSystem();
        terminators.setName("Terminator");
        SimulinkFactory.eINSTANCE.createSInport().setOwningSystem(terminators);
        
        terminator.setSystem(terminators);
        
        terminator.setOwningSystem(schartsystem);
        terminator.setSID();
        
        SLine line2 = schartsystem.getLine(demux, null);
        SConnectionPoint pt2 = SimulinkFactory.eINSTANCE.createSConnectionPoint();
        pt2.setBlock(terminator);
        line2.getDestinations().add(pt2);
        
        return schartsystem;
    }
    
    @SuppressWarnings("unchecked")
    private SSystem processSystemForConnectors(Class ublock) throws UMLModelErrorException
    {
        log.debug("Processing system for connectors " + print(ublock));
        SSystem ssystem = createSystem(ublock);
        
        // add parameter values
        for (Property uproperty : SysMLUtil.getAllCorrectedAttributes(ublock))
        {
            SElement selement = refs.get(getKey(ublock, uproperty));
            ValueSpecification vs = uproperty.getDefaultValue();
            if (vs != null && vs instanceof InstanceValue && ((InstanceValue) vs).getInstance() != null)
            {
                for (Slot uslot : ((InstanceValue) vs).getInstance().getSlots())
                {
                    StructuralFeature sf = uslot.getDefiningFeature();
                    Object o = refs.get(getKey(uproperty.getType(), sf));
                    if (o == null)
                    {
                        log.warn("No reference for object defined by " + print(uproperty.getType()) + " and "
                                + print(sf));
                    }
                    if (selement instanceof SInterface && o instanceof SSystemParameter)
                    {
                        SSystemParameter SFParameter = (SSystemParameter) o;
                        SSystemParameterAssignment sassignment = SimulinkFactory.eINSTANCE
                                .createSSystemParameterAssignment();
                        sassignment.setParameter(SFParameter);
                        SDataValue sdv = getValue(uslot);
                        if (sdv != null)
                            sassignment.setValue(sdv);
                        
                        ((SInterface) selement).getAssignments().add(sassignment);
                        continue;
                    }
                    o = refs.get(getKey(sf));
                    if (o == null)
                        log.warn("The initial value of " + sf.getQualifiedName() + " in " + print(ublock)
                                + " was skipped, no reference for " + print(sf));
                    else if (selement instanceof SPhysicalBlock && o instanceof SMember)
                    {
                        if (uslot.isStereotypeApplied(sysmlutil.getInitialValueReference()) && uslot
                                .getValue(sysmlutil.getInitialValueReference(), "propertyPath") instanceof List<?>)
                        {
                            List<Property> references = (List<Property>) uslot
                                    .getValue(sysmlutil.getInitialValueReference(), "propertyPath");
                            if (references.size() == 1)
                            {
                                SMember sm = (SMember) o;
                                SMemberAssignment sassignment = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                                sassignment.getMemberPath().add(sm);
                                SExpressionValue val = SimulinkFactory.eINSTANCE.createSExpressionValue();
                                val.setValue(SysMLUtil.getName(references.get(0)));
                                sassignment.setAssignedValue(val);
                                ((SPhysicalBlock) selement).getAssignments().add(sassignment);
                            }
                        }
                        else if (uslot.getValues().size() == 1)
                        {
                            SMember sm = (SMember) o;
                            SMemberAssignment sassignment = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                            sassignment.getMemberPath().add(sm);
                            sassignment.setAssignedValue(getValue(uslot.getValues().get(0)));
                            ((SPhysicalBlock) selement).getAssignments().add(sassignment);
                        }
                        else
                            log.warn("Can't assign an array to " + ((SPhysicalBlock) selement).getName() + "."
                                    + ((SMember) o).getName());
                        continue;
                    }
                    else
                        log.warn("The initial value of " + sf.getQualifiedName() + " in " + print(ublock)
                                + " was skipped, types:" + selement.getClass() + " and " + o.getClass());
                }
            }
        }
        
        // parse connectors, to create "lines"
        for (Connector uconnector : SysMLUtil.getAllConnectors(ublock))
        {
            ConnectorEnd uend1 = uconnector.getEnds().get(0);
            ConnectorEnd uend2 = uconnector.getEnds().get(1);
            List<Property> lprop1 = sysmlutil.getCorrectedPropertyPath(uend1, ublock);
            List<Property> lprop2 = sysmlutil.getCorrectedPropertyPath(uend2, ublock);
            
            // binding connectors
            if (uconnector.isStereotypeApplied(sysmlutil.getBindingConnector()))
            {
                // to refactor
                // assign parameter value to simulink block
                if (lprop1.size() == 2 && lprop1.get(0).getType().isStereotypeApplied(sysmlutil.getSimulinkBlock())
                        && lprop1.get(1).isStereotypeApplied(sysmlutil.getSimulinkParameter()))
                {
                    SDataValue sdv = getValue(lprop2.get(0).getDefaultValue());
                    if (sdv != null)
                    {
                        SInterface si = (SInterface) refs.get(getKey(ublock, lprop1.get(0)));
                        SSystemParameter ssp = (SSystemParameter) refs
                                .get(getKey(lprop1.get(0).getType(), lprop1.get(1)));
                        SSystemParameterAssignment sspa = SimulinkFactory.eINSTANCE.createSSystemParameterAssignment();
                        sspa.setParameter(ssp);
                        sspa.setValue(sdv);
                        si.getAssignments().add(sspa);
                    }
                    else
                        log.warn("Couldn't assign parameter value");
                }
                if (lprop2.size() == 2 && lprop2.get(0).getType().isStereotypeApplied(sysmlutil.getSimulinkBlock())
                        && lprop2.get(1).isStereotypeApplied(sysmlutil.getSimulinkParameter()))
                {
                    SDataValue sdv = getValue(lprop1.get(0).getDefaultValue());
                    if (sdv != null)
                    {
                        SInterface si = (SInterface) refs.get(getKey(ublock, lprop2.get(0)));
                        SSystemParameter ssp = (SSystemParameter) refs
                                .get(getKey(lprop2.get(0).getType(), lprop2.get(1)));
                        SSystemParameterAssignment sspa = SimulinkFactory.eINSTANCE.createSSystemParameterAssignment();
                        sspa.setParameter(ssp);
                        sspa.setValue(sdv);
                        si.getAssignments().add(sspa);
                    }
                    else
                        log.warn("Couldn't assign parameter value");
                }
                // assign parameter value to simscape component parameter
                if (lprop1.size() == 1 && lprop1.get(0).isStereotypeApplied(sysmlutil.getPhSConstant())
                        && lprop2.size() == 2)
                {
                    SDataValue sdv = getValue(lprop1.get(0).getDefaultValue());
                    SElement se1 = refs.get(getKey(ublock, lprop2.get(0)));
                    SElement se2 = refs.get(getKey(lprop2.get(1)));
                    
                    if (se1 instanceof SPhysicalBlock && se2 instanceof SParameter)
                    {
                        SMemberAssignment sma = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                        sma.getMemberPath().add((SMember) se2);
                        sma.setAssignedValue(sdv);
                        ((SPhysicalBlock) se1).getAssignments().add(sma);
                    }
                }
                if (lprop2.size() == 1 && lprop2.get(0).isStereotypeApplied(sysmlutil.getPhSConstant())
                        && lprop1.size() == 2)
                {
                    SDataValue sdv = getValue(lprop2.get(0).getDefaultValue());
                    SElement se1 = refs.get(getKey(ublock, lprop1.get(0)));
                    SElement se2 = refs.get(getKey(lprop1.get(1)));
                    
                    if (se1 instanceof SPhysicalBlock && se2 instanceof SParameter)
                    {
                        SMemberAssignment sma = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                        sma.getMemberPath().add((SMember) se2);
                        sma.setAssignedValue(sdv);
                        ((SPhysicalBlock) se1).getAssignments().add(sma);
                    }
                }
                continue;
            }
            
            // connectors can only be between
            // {inport,outport,subsystem+inport/outport}
            
            // the lists can't be empty!
            if (lprop1.size() > 2 || lprop2.size() > 2)
            {
                log.warn("Connector " + print(uconnector) + " skipped as its ends must have 1 or 2 properties");
                continue;
            }
            
            // When the type is an InterfaceBlock, connect the operations
            if (lprop1.size() == 2 && lprop2.size() == 2)
            {
                Type type1 = lprop1.get(1).getType();
                Type type2 = lprop2.get(1).getType();
                
                if (type1.isStereotypeApplied(sysmlutil.getInterfaceBlock())
                        && type2.isStereotypeApplied(sysmlutil.getInterfaceBlock()))
                {
                    log.debug("Connection between two interfaces block found: " + print(uconnector));
                    // match operations
                    SBlock sb1 = (SBlock) refs.get(getKey(ublock, lprop1.get(0)));
                    SBlock sb2 = (SBlock) refs.get(getKey(ublock, lprop2.get(0)));
                    loop: for (Operation uop1 : ((Class) type1).getAllOperations())
                    {
                        for (Operation uop2 : ((Class) type2).getAllOperations())
                        {
                            // TODO: need to add much more...
                            if (uop1.getName().equals(uop2.getName()) && uop1.getType() == uop2.getType())
                            {
                                log.debug("Matching operations: " + uop1.getName());
                                SBlock srcblock = null;
                                SBlock srcport = null;
                                SBlock dstblock = null;
                                SBlock dstport = null;
                                
                                SElement e1 = refs.get(getKey(lprop1.get(0).getType(), lprop1.get(1), uop1));
                                
                                if (e1 instanceof SOutport)
                                {
                                    srcblock = sb1;
                                    srcport = (SOutport) e1;
                                }
                                if (e1 instanceof SInport)
                                {
                                    dstblock = sb1;
                                    dstport = (SInport) e1;
                                }
                                SElement e2 = refs.get(getKey(lprop2.get(0).getType(), lprop2.get(1), uop2));
                                if (e2 instanceof SOutport)
                                {
                                    srcblock = sb2;
                                    srcport = (SOutport) e2;
                                }
                                if (e2 instanceof SInport)
                                {
                                    dstblock = sb2;
                                    dstport = (SInport) e2;
                                }
                                if (srcport != null && dstport != null && dstblock != null && srcblock != null)
                                {
                                    log.debug("Adding line between " + srcblock.getName() + "/" + srcport.getName()
                                            + " and " + dstblock.getName() + "/" + dstport.getName());
                                    SLine sline = ssystem.getLine(srcblock, srcport);
                                    SConnectionPoint sdestination = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                                    sline.getDestinations().add(sdestination);
                                    sdestination.setBlock(dstblock);
                                    sdestination.setPort(dstport);
                                    continue loop;
                                }
                            }
                        }
                    }
                }
            }
            
            // deal with physical lines
            List<SPhysicalConnectionPoint> lspcp = new ArrayList<SPhysicalConnectionPoint>();
            for (List<Property> lp : new List[] { lprop1, lprop2 })
            {
                SElement sb = refs.get(getKey(ublock, lp.get(0)));
                if (sb instanceof SPhysicalBlock)
                {
                    SPhysicalBlock spb = (SPhysicalBlock) sb;
                    if (lp.size() > 1)
                    {
                        SMember sm = (SMember) refs.get(getKey(lp.get(1)));
                        if (sm != null)
                        {
                            SPhysicalConnectionPoint spcp = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                            spcp.setBlock(spb);
                            spcp.setPort(sm.getConnectionPortBlock());
                            lspcp.add(spcp);
                        }
                        else
                            log.warn("Can't find member corresponding to " + print(lp.get(1)));
                    }
                    else
                    {
                        SComponent sc = spb.getComponent();
                        if (sc != null && sc.getOwnedMembers().size() > 0)
                        {
                            SPhysicalConnectionPoint spcp = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                            spcp.setBlock(spb);
                            spcp.setPort(sc.getOwnedMembers().get(0).getConnectionPortBlock());
                            lspcp.add(spcp);
                        }
                        else
                            log.warn("No component in physical block or no member in the component");
                    }
                }
                if (lp.size() > 1 && sb instanceof SReference)
                {
                    SBlock sport = createBlock((Class) lp.get(0).getType(), lp.get(1));
                    if (sport instanceof SConnectionPortBlock)
                    {
                        SPhysicalConnectionPoint spcp = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                        spcp.setBlock((SBlock) sb);
                        spcp.setPort((SConnectionPortBlock) sport);
                        lspcp.add(spcp);
                    }
                }
            }
            if (lspcp.size() == 2)
            {
                SPhysicalConnectionPoint spc0 = lspcp.get(0);
                SPhysicalConnectionPoint spc1 = lspcp.get(1);
                
                log.debug("Adding physical line between " + spc0.getBlock().getName() + "/" + spc0.getPort().getName()
                        + " and " + spc1.getBlock().getName() + "/" + spc1.getPort().getName());
                addPhysicalLine(ssystem, spc0, spc1);
                
                continue;
            }
            
            // deal with unidirectional physical lines and Simulink lines
            SBlock srcblock = null;
            SBlock srcport = null;
            SBlock dstblock = null;
            SBlock dstport = null;
            SPhysicalLine splin = null;
            SPhysicalLine splout = null;
            for (List<Property> lp : new List[] { lprop1, lprop2 })
            {
                SBlock sblock = createBlock(ublock, lp.get(0));
                if (sblock instanceof SInport)
                    srcblock = sblock;
                else if (sblock instanceof SOutport)
                    dstblock = sblock;
                else if (lp.get(0).isStereotypeApplied(sysmlutil.getPhSConstant())
                        && lp.get(0).getDefaultValue() != null)
                {
                    srcblock = sblock;
                }
                else if (lp.get(0).isStereotypeApplied(sysmlutil.getPhSVariable()))
                {
                    dstblock = sblock;
                }
                else if (lp.size() == 2)
                {
                    if (sblock instanceof SReference || sblock instanceof SInterface || sblock instanceof SSubsystem)
                    {
                        SBlock sport = createBlock((Class) lp.get(0).getType(), lp.get(1));
                        if (sport instanceof SOutport)
                        {
                            srcblock = sblock;
                            srcport = sport;
                        }
                        else if (sport instanceof SInport)
                        {
                            dstblock = sblock;
                            dstport = sport;
                        }
                    }
                    else if (sblock instanceof SPhysicalBlock)
                    {
                        SMember smember = createMember(lp.get(1));
                        // create blocks but do not add yet
                        if (smember instanceof SInput)
                        {
                            SPhysicalBlock spb = SimscapeFactory.eINSTANCE.createSPhysicalBlock();
                            SimulinkUtil.setName(spb, spb.getSID());
                            spb.setComponent(SimscapeUtil.getSimulink2PS());
                            
                            SPhysicalConnectionPoint scp1 = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                            scp1.setBlock(spb);
                            scp1.setPort(SimscapeUtil.getSimulink2PS().getRConnectionPorts().get(0));
                            
                            SPhysicalConnectionPoint scp2 = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                            scp2.setBlock((SPhysicalBlock) sblock);
                            scp2.setPort(smember.getConnectionPortBlock());
                            
                            splin = SimscapeFactory.eINSTANCE.createSPhysicalLine();
                            splin.getPoints().add(scp1);
                            splin.getPoints().add(scp2);
                            
                            dstblock = spb;
                            dstport = spb.getComponent().getInports().get(0);
                        }
                        else if (smember instanceof SOutput)
                        {
                            SPhysicalBlock spb = SimscapeFactory.eINSTANCE.createSPhysicalBlock();
                            SimulinkUtil.setName(spb, spb.getSID());
                            spb.setComponent(SimscapeUtil.getPS2Simulink());
                            
                            SPhysicalConnectionPoint scp1 = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                            scp1.setBlock(spb);
                            scp1.setPort(SimscapeUtil.getPS2Simulink().getLConnectionPorts().get(0));
                            
                            SPhysicalConnectionPoint scp2 = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
                            scp2.setBlock((SPhysicalBlock) sblock);
                            scp2.setPort(smember.getConnectionPortBlock());
                            
                            splout = SimscapeFactory.eINSTANCE.createSPhysicalLine();
                            splout.getPoints().add(scp1);
                            splout.getPoints().add(scp2);
                            
                            srcblock = spb;
                            srcport = spb.getComponent().getOutports().get(0);
                        }
                    }
                    else
                        log.warn("Block type not supported: " + sblock.getClass());
                }
                else
                    log.warn("The connector " + print(uconnector) + " is problematic");
            }
            if (srcblock == null || dstblock == null)
                log.info("null==" + srcblock + "/" + srcport + "--" + dstblock + "/" + dstport);
            if (srcblock != null && dstblock != null)
            {
                log.debug("Adding line between " + srcblock.getName() + "/" + (srcport == null ? "" : srcport.getName())
                        + " and " + dstblock.getName() + "/" + (dstport == null ? "" : dstport.getName()));
                SLine sline = ssystem.getLine(srcblock, srcport);
                SConnectionPoint sdestination = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                sline.getDestinations().add(sdestination);
                sdestination.setBlock(dstblock);
                sdestination.setPort(dstport);
                
                // add the physical block at this point
                if (srcblock instanceof SPhysicalBlock)
                {
                    srcblock.setOwningSystem(ssystem);
                    srcblock.setSID();
                }
                if (dstblock instanceof SPhysicalBlock)
                {
                    dstblock.setOwningSystem(ssystem);
                    dstblock.setSID();
                }
                if (splin != null)
                    ssystem.getOwnedLines().add(splin);
                if (splout != null)
                    ssystem.getOwnedLines().add(splout);
                
                if (!ssystem.getOwnedLines().contains(sline))
                    ssystem.getOwnedLines().add(sline);
            }
            else
            {
                log.warn("Connector " + print(uconnector) + " skipped because it does not connect compatible elements");
            }
            
        }
        
        return ssystem;
    }
    
    @SuppressWarnings("null")
    private static void addPhysicalLine(SSystem ssystem, SPhysicalConnectionPoint spc0, SPhysicalConnectionPoint spc1)
    {
        // get an existing physical line, if any point is present
        int i = 0;
        SPhysicalLine spline = null;
        SPhysicalConnectionPoint scpconsolidate = null;
        
        // if one point is found is a line, add the other point and consolidate
        loop: while (i != ssystem.getOwnedLines().size())
        {
            SLine sline = ssystem.getOwnedLines().get(i);
            if (sline instanceof SPhysicalLine)
            {
                SPhysicalLine spline2 = (SPhysicalLine) sline;
                if (spline2.getPoints().contains(scpconsolidate))
                {
                    for (SPhysicalConnectionPoint spcp : spline2.getPoints())
                        if (!spcp.equals(scpconsolidate))
                        {
                            spline.getPoints().add(spcp);
                        }
                    // remove current line and continue with same i
                    ssystem.getOwnedLines().remove(i);
                    continue loop;
                }
                
                for (SPhysicalConnectionPoint spc : spline2.getPoints())
                {
                    if (spc.equals(spc0))
                    {
                        spline = spline2;
                        spline.getPoints().add(spc1);
                        scpconsolidate = spc1;
                        break;
                    }
                    else if (spc.equals(spc1))
                    {
                        spline = spline2;
                        spline.getPoints().add(spc0);
                        scpconsolidate = spc0;
                        break;
                    }
                }
            }
            i++;
        }
        
        // create physical line, or consolidate
        if (spline == null)
        {
            spline = SimscapeFactory.eINSTANCE.createSPhysicalLine();
            
            spline.getPoints().add(spc0);
            spline.getPoints().add(spc1);
            ssystem.getOwnedLines().add(spline);
        }
    }
    
    private SFunction1 processSFunction1(Class ublock, Class ccontext, Property pcontext) throws UMLModelErrorException
    {
        log.debug("Processing s-function " + print(ublock));
        // SFunctions can be created and processed at the same time
        SFunction1 sfunction = createSFunction1(ublock, ccontext, pcontext);
        String name = SysMLUtil.getName(ccontext) + "_" + SysMLUtil.getName(pcontext) + "_" + SysMLUtil.getName(ublock);
        SimulinkUtil.setName(sfunction, name);
        
        for (Property uproperty : SysMLUtil.getAllCorrectedAttributes(ublock))
        {
            SFVariable SFVariable = createVariable(ublock, uproperty, ccontext, pcontext);
            if (SFVariable != null)
            {
                log.debug("Adding variable " + print(uproperty));
                sfunction.getVariables().add(SFVariable);
            }
            else
                log.warn("Skipping variable " + print(uproperty));
        }
        
        // need to create mux/demux at the same time
        // can't be otherwise
        SFunction1Block sfb = (SFunction1Block) refs.get(getKey(ccontext, pcontext));
        SSystem ssys = sfb.getOwningSystem();
        List<SFInputVariable> liv = sfunction.getInputs();
        List<SFOutputVariable> lov = sfunction.getOutputs();
        
        if (liv != null)
        {
            SInterface mux = null;
            if (liv.size() > 1)
            {
                mux = SimulinkFactory.eINSTANCE.createSInterface();
                mux.setOwningSystem(ssys);
                SimulinkUtil.setName(mux, sfunction.getName() + "Mux");
                
                SSystem smux = SimulinkFactory.eINSTANCE.createSSystem();
                SimulinkUtil.setName(smux, "Mux");
                mux.setSystem(smux);
                
                SSystemParameter sinputs = SimulinkFactory.eINSTANCE.createSSystemParameter();
                smux.getSystemparameters().add(sinputs);
                sinputs.setName("Inputs");
                SExpressionValue sinputsv = SimulinkFactory.eINSTANCE.createSExpressionValue();
                sinputsv.setValue(String.valueOf(liv.size()));
                sinputs.setValue(sinputsv);
                
                SLine sline = ssys.getLine(mux, null);
                SConnectionPoint scp = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                scp.setBlock(sfb);
                sline.getDestinations().add(scp);
            }
            for (int i = 0; i < liv.size(); i++)
            {
                SFInputVariable SFInputVariable = liv.get(i);
                SInport sinport = SFInputVariable.getInport();
                if (sinport != null)
                {
                    SLine sline = ssys.getLine(sinport, null);
                    SConnectionPoint scp = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                    if (mux == null)
                        scp.setBlock(sfb);
                    else
                    {
                        scp.setBlock(mux);
                        SInport sin = SimulinkFactory.eINSTANCE.createSInport();
                        sin.setOwningSystem(mux.getSystem());
                        scp.setPort(sin);
                    }
                    sline.getDestinations().add(scp);
                }
            }
        }
        
        if (lov != null)
        {
            SInterface demux = null;
            if (lov.size() > 1)
            {
                demux = SimulinkFactory.eINSTANCE.createSInterface();
                demux.setOwningSystem(ssys);
                SimulinkUtil.setName(demux, sfunction.getName() + "Demux");
                
                SSystem sdemux = SimulinkFactory.eINSTANCE.createSSystem();
                SimulinkUtil.setName(sdemux, "Demux");
                demux.setSystem(sdemux);
                
                SSystemParameter soutputs = SimulinkFactory.eINSTANCE.createSSystemParameter();
                SimulinkUtil.setName(soutputs, "Outputs");
                SExpressionValue soutputsv = SimulinkFactory.eINSTANCE.createSExpressionValue();
                soutputsv.setValue(String.valueOf(lov.size()));
                soutputs.setValue(soutputsv);
                sdemux.getSystemparameters().add(soutputs);
                
                SLine sline = ssys.getLine(sfb, null);
                SConnectionPoint scp = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                scp.setBlock(demux);
                sline.getDestinations().add(scp);
            }
            for (int i = 0; i < lov.size(); i++)
            {
                SFOutputVariable SFOutputVariable = lov.get(i);
                SOutport soutport = SFOutputVariable.getOutport();
                
                if (soutport != null)
                {
                    if (demux == null)
                    {
                        SLine sline = ssys.getLine(sfb, null);
                        SConnectionPoint scp = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                        scp.setBlock(soutport);
                        sline.getDestinations().add(scp);
                    }
                    else
                    {
                        SOutport sout = SimulinkFactory.eINSTANCE.createSOutport();
                        sout.setOwningSystem(demux.getSystem());
                        SLine sline = ssys.getLine(demux, sout);
                        SConnectionPoint scp = SimulinkFactory.eINSTANCE.createSConnectionPoint();
                        scp.setBlock(soutport);
                        sline.getDestinations().add(scp);
                    }
                }
            }
        }
        
        return sfunction;
    }
    
    private SFunction1 processSFunction12(Class ublock, Class ccontext, Property pcontext) throws UMLModelErrorException
    {
        log.debug("Processing s-function " + print(ublock) + " a second time");
        // SFunctions can be created and processed at the same time
        SFunction1 sfunction = createSFunction1(ublock, ccontext, pcontext);
        
        Hashtable<String, String> substitutionsLHS = new Hashtable<String, String>();
        Hashtable<String, String> substitutionsRHS = new Hashtable<String, String>();
        
        List<SFInputVariable> inputs = sfunction.getInputs();
        List<SFContinuousStateVariable> cvariables = sfunction.getContinuousStates();
        List<SFDiscreteStateVariable> dvariables = sfunction.getDiscreteStates();
        List<SFOutputVariable> outputs = sfunction.getOutputs();
        List<SFVariable> variables = new ArrayList<SFVariable>(cvariables);
        variables.addAll(dvariables);
        
        substitutionsRHS.put("time", "t");
        for (SFVariable svar : sfunction.getVariables())
        {
            // two cases:
            // - multiple variables: we add an index
            // - one state with multiple dimensions: we don't add an index
            
            String add = "";
            if (svar instanceof SFInputVariable)
            {
                if (inputs.size() > 1)
                    add = "(" + String.valueOf(inputs.indexOf(svar) + 1) + ")";
                substitutionsRHS.put(svar.getName(), "u" + add);
            }
            else if (svar instanceof SFOutputVariable)
            {
                if (outputs.size() > 1)
                    add = "(" + String.valueOf(outputs.indexOf(svar) + 1) + ")";
                substitutionsLHS.put(svar.getName(), "sys" + add);
            }
            else if (svar instanceof SFDiscreteStateVariable)
            {
                if (dvariables.size() > 1)
                    add = "(" + (dvariables.indexOf(svar) + 1) + ")";
                substitutionsLHS.put("next(" + svar.getName() + ")", "sys" + add);
                
                if (variables.size() > 1)
                    add = "(" + (variables.indexOf(svar) + 1) + ")";
                substitutionsRHS.put(svar.getName(), "x" + add);
                
            }
            else if (svar instanceof SFContinuousStateVariable)
            {
                if (cvariables.size() > 1)
                    add = "(" + (cvariables.indexOf(svar) + 1) + ")";
                substitutionsLHS.put("der(" + svar.getName() + ")", "sys" + add);
                
                if (variables.size() > 1)
                    add = "(" + (variables.indexOf(svar) + 1) + ")";
                substitutionsRHS.put(svar.getName(), "x" + add);
            }
            else if (svar instanceof SFParameter)
                substitutionsRHS.put(svar.getName(), getValue(svar));
        }
        
        for (Constraint uconstraint : SysMLUtil.getAllConstraints(ublock))
        {
            el2m.prepareNextEquationParsing(sfunction, substitutionsLHS, substitutionsRHS, inputs);
            String expr = getEquation(uconstraint.getSpecification());
            SFVariableAssignment sassignment = el2m.getAssignment();
            if (sassignment == null)
                log.warn("Assignment " + expr + " unrecognized and skipped");
            else
            {
                sfunction.getAssignments().add(sassignment);
                sassignment.setExpression(expr);
            }
        }
        
        // Initial values from InstanceSpecification
        for (Dependency dep : ublock.getClientDependencies())
        {
            if (dep.isStereotypeApplied(sysmlutil.getInitialValuesSpecification()))
            {
                log.info("Found instance specification for " + print(ublock));
                for (NamedElement supplier : dep.getSuppliers())
                    if (supplier instanceof InstanceSpecification)
                    {
                        InstanceSpecification target = (InstanceSpecification) supplier;
                        for (Slot slot : target.getSlots())
                        {
                            StructuralFeature sf = slot.getDefiningFeature();
                            if (sf instanceof Property)
                            {
                                SFVariable sv = createVariable(ublock, (Property) sf, ccontext, pcontext);
                                for (ValueSpecification vs : slot.getValues())
                                {
                                    sv.getValue().add(getValue(vs));
                                }
                                // TODO: implement that
                                if (((Property) sf).getClass_() == ublock)
                                {
                                    // add as initial value
                                    
                                }
                                else
                                {
                                    // add as extension modification
                                }
                            }
                        }
                    }
            }
        }
        
        return sfunction;
    }
    
    private SFunction2 processSFunction2(Class ublock, Class ccontext, Property pcontext) throws UMLModelErrorException
    {
        log.debug("Processing s-function " + print(ublock));
        // SFunctions can be created and processed at the same time
        SFunction2 sfunction = createSFunction2(ublock, ccontext, pcontext);
        String name = SysMLUtil.getName(ccontext) + "_" + SysMLUtil.getName(pcontext) + "_" + SysMLUtil.getName(ublock);
        SimulinkUtil.setName(sfunction, name);
        
        for (Property uproperty : SysMLUtil.getAllCorrectedAttributes(ublock))
        {
            SFVariable SFVariable = createVariable(ublock, uproperty, ccontext, pcontext);
            if (SFVariable != null)
            {
                log.debug("Adding variable " + print(uproperty));
                sfunction.getVariables().add(SFVariable);
            }
            else
                log.warn("Skipping variable " + print(uproperty));
        }
        
        return sfunction;
    }
    
    private SFunction2 processSFunction22(Class ublock, Class ccontext, Property pcontext) throws UMLModelErrorException
    {
        log.debug("Processing s-function " + print(ublock) + " a second time");
        // SFunctions can be created and processed at the same time
        SFunction2 sfunction = createSFunction2(ublock, ccontext, pcontext);
        
        Hashtable<String, String> substitutionsLHS = new Hashtable<String, String>();
        Hashtable<String, String> substitutionsRHS = new Hashtable<String, String>();
        
        List<SFInputVariable> inputs = sfunction.getInputs();
        List<SFOutputVariable> outputs = sfunction.getOutputs();
        List<SFContinuousStateVariable> cvariables = sfunction.getContinuousStates();
        List<SFDiscreteStateVariable> dvariables = sfunction.getDiscreteStates();
        List<SFParameter> parameters = sfunction.getParameters();
        List<SFDWorkVariable> dworks = sfunction.getDWorkStates();
        List<SFVariable> alldworks = new ArrayList<SFVariable>(dvariables);
        alldworks.addAll(dworks);
        
        // Define substitutions
        for (SFInputVariable siv : inputs)
            substitutionsRHS.put(siv.getName(),
                    SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".InputPort(" + (inputs.indexOf(siv) + 1) + ").Data");
        for (SFOutputVariable sov : outputs)
            substitutionsLHS.put(sov.getName(),
                    SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".OutputPort(" + (outputs.indexOf(sov) + 1) + ").Data");
        for (SFContinuousStateVariable scsv : cvariables)
        {
            substitutionsRHS.put(scsv.getName(), SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".ContStates.Data("
                    + (cvariables.indexOf(scsv) + 1) + ")");
            substitutionsLHS.put(scsv.getName(), SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".ContStates.Data("
                    + (cvariables.indexOf(scsv) + 1) + ")");
            substitutionsLHS.put("der(" + scsv.getName() + ")", SysMLToSimulinkTranslator.SF2_BLOCK_NAME
                    + ".Derivatives.Data(" + (cvariables.indexOf(scsv) + 1) + ")");
        }
        for (SFParameter sp : parameters)
            substitutionsRHS.put(sp.getName(), getValue(sp));
        
        for (SFDWorkVariable sdwv : dworks)
        {
            String str = SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".Dwork(" + (alldworks.indexOf(sdwv) + 1)
                    + ").Data";
            substitutionsLHS.put(sdwv.getName(), str);
            if (sdwv.getDimensions().size() == 2)
                str = "reshape(" + str + "," + sdwv.getDimensions().get(1) + "," + sdwv.getDimensions().get(0) + ")";
            substitutionsRHS.put(sdwv.getName(), str);
        }
        for (SFDiscreteStateVariable sdsv : dvariables)
        {
            String str = SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".Dwork(" + (alldworks.indexOf(sdsv) + 1)
                    + ").Data";
            substitutionsLHS.put(sdsv.getName(), str);
            if (sdsv.getDimensions().size() == 2)
                str = "reshape(" + str + "," + sdsv.getDimensions().get(1) + "," + sdsv.getDimensions().get(0) + ")";
            substitutionsRHS.put(sdsv.getName(), str);
        }
        
        // WONT WORK BECAUSE OF STATES ARRAYS
        // for(SFVariable sv : dws)
        // if (sv.getValue() != null)
        // lInitial.add(sv.getName() + "=" + getValue(sv) + ";");
        
        for (Constraint uconstraint : SysMLUtil.getAllConstraints(ublock))
        {
            
            el2m.prepareNextEquationParsing(sfunction, substitutionsLHS, substitutionsRHS, inputs);
            String expr = getEquation(uconstraint.getSpecification());
            SFVariableAssignment sassignment = el2m.getAssignment();
            if (sassignment == null)
                log.warn("Assignment " + expr + " unrecognized and skipped");
            else
            {
                sfunction.getAssignments().add(sassignment);
                sassignment.setExpression(expr);
            }
        }
        
        // Initial values from InstanceSpecification
        for (Dependency dep : ublock.getClientDependencies())
        {
            if (dep.isStereotypeApplied(sysmlutil.getInitialValuesSpecification()))
            {
                log.info("Found instance specification for " + print(ublock));
                for (NamedElement supplier : dep.getSuppliers())
                    if (supplier instanceof InstanceSpecification)
                    {
                        InstanceSpecification target = (InstanceSpecification) supplier;
                        for (Slot slot : target.getSlots())
                        {
                            StructuralFeature sf = slot.getDefiningFeature();
                            if (sf instanceof Property)
                            {
                                SFVariable sv = createVariable(ublock, (Property) sf, ccontext, pcontext);
                                for (ValueSpecification vs : slot.getValues())
                                {
                                    sv.getValue().add(getValue(vs));
                                }
                                // TODO: implement that
                                if (((Property) sf).getClass_() == ublock)
                                {
                                    // add as initial value
                                    
                                }
                                else
                                {
                                    // add as extension modification
                                }
                            }
                        }
                    }
            }
        }
        
        return sfunction;
    }
    
    private SChart createChart(Class ublock, StateMachine ustatemachine)
    {
        log.debug("Creating state machine " + print(ustatemachine) + "/" + print(ustatemachine));
        ReferenceKey rk = getKey(ublock, ustatemachine);
        
        SElement selement = refs.get(rk);
        
        if (selement != null)
        {
            if (selement instanceof SChart)
                return (SChart) selement;
            throw new IllegalStateException("The element correspinding to " + print(ustatemachine) + "/"
                    + print(ustatemachine) + " is not a chart: " + selement.getClass());
        }
        
        SMachine smachine = sstateflow.getMachine();
        
        SChart schart = StateflowFactory.eINSTANCE.createSChart();
        refs.put(rk, schart);
        schart.setOwningMachine(smachine);
        schart.setID();
        
        String ustatemachinecontextname = SysMLUtil.getName(ublock);
        SimulinkUtil.setName(schart, ustatemachinecontextname);
        
        SInstance sinstance = StateflowFactory.eINSTANCE.createSInstance();
        sinstance.setChart(schart);
        sinstance.setOwningStateflow(sstateflow);
        sinstance.setID();
        sinstance.setMachine(smachine);
        SimulinkUtil.setName(sinstance, ustatemachinecontextname);
        
        SState scontainer = null;
        List<Region> regions = sysmlutil.getAllCorrectedRegions(ustatemachine);
        
        // if more than 1 region, create a container
        if (regions.size() > 1)
        {
            scontainer = StateflowFactory.eINSTANCE.createSState();
            scontainer.setType(SStateType.AND);
            scontainer.setOwningTreeNode(schart);
            scontainer.setID();
        }
        else if (regions.size() == 1)
            refs.put(getKey(ublock, regions.get(0)), schart);
        
        for (Region uregion : regions)
        {
            if (scontainer != null)
            {
                SState slocalcontainer = StateflowFactory.eINSTANCE.createSState();
                slocalcontainer.setType(SStateType.OR);
                slocalcontainer.setOwningTreeNode(scontainer);
                slocalcontainer.setID();
                refs.put(getKey(ublock, uregion), slocalcontainer);
            }
            processRegion1(ublock, uregion);
        }
        return schart;
        
    }
    
    private void processRegion1(Class ublock, Region uregion)
    {
        log.debug("Processing region " + print(ublock) + "/" + print(uregion) + " #1");
        ReferenceKey rk = getKey(ublock, uregion);
        STreeNode snodecontainer = (STreeNode) refs.get(rk);
        for (Vertex vertex : uregion.getSubvertices())
        {
            switch (getTranslatedVertex(vertex))
            {
            case STATE:
                SState state = StateflowFactory.eINSTANCE.createSState();
                log.debug("Creating state " + print(vertex));
                
                List<Region> regions = ((State) vertex).getRegions();
                SState scontainer = null;
                // if more than 1 region, create a container
                if (regions.size() > 1)
                {
                    scontainer = StateflowFactory.eINSTANCE.createSState();
                    scontainer.setType(SStateType.AND);
                    scontainer.setOwningTreeNode(state);
                    scontainer.setID();
                }
                else if (regions.size() == 1)
                    refs.put(getKey(ublock, regions.get(0)), state);
                
                for (Region uregion2 : regions)
                {
                    if (scontainer != null)
                    {
                        SState slocalcontainer = StateflowFactory.eINSTANCE.createSState();
                        slocalcontainer.setType(SStateType.OR);
                        slocalcontainer.setOwningTreeNode(scontainer);
                        slocalcontainer.setID();
                        refs.put(getKey(ublock, uregion2), slocalcontainer);
                    }
                    processRegion1(ublock, uregion2);
                }
                
                refs.put(getKey(ublock, vertex), state);
                state.setOwningTreeNode(snodecontainer);
                state.setID();
                break;
            case INITIAL:
                break;
            case JUNCTION:
                SJunction junction = StateflowFactory.eINSTANCE.createSJunction();
                junction.setOwningTreeNode(snodecontainer);
                junction.setID();
                log.debug("Creating junction " + print(ublock) + "/" + print(vertex));
                refs.put(getKey(ublock, vertex), junction);
                break;
            default:
                log.warn("Untranslated element: " + print(ublock) + "/" + print(vertex));
            }
        }
        toProcess.push(rk);
    }
    
    private void processRegion2(Class ublock, Region uregion) throws UMLModelErrorException
    {
        log.debug("Processing region " + print(ublock) + "/" + print(uregion) + " #2");
        STreeNode snodecontainer = (STreeNode) refs.get(getKey(ublock, uregion));
        
        Hashtable<String, String> substitutions = new Hashtable<String, String>();
        BehavioredClassifier context = ublock;
        for (Property uproperty : SysMLUtil.getAllCorrectedAttributes(context))
        {
            if (uproperty.getType() instanceof Classifier)
            {
                List<Property> sps = sysmlutil.getAllPhSProperties((Classifier) uproperty.getType());
                if (sps.size() == 1)
                    substitutions.put(uproperty.getName() + "." + sps.get(0).getName(), uproperty.getName());
            }
        }
        
        for (Vertex vertex : uregion.getSubvertices())
        {
            SElement selement = refs.get(getKey(ublock, vertex));
            if (selement instanceof SState)
            {
                SState state = (SState) selement;
                SimulinkUtil.setName(state, SysMLUtil.getName(vertex));
                StringBuilder sblabel = new StringBuilder();
                Behavior entry = ((State) vertex).getEntry();
                if (entry instanceof OpaqueBehavior)
                {
                    el2m.setSubstitutions(substitutions);
                    String expr = getAssignments((OpaqueBehavior) entry);
                    el2m.setSubstitutions(null);
                    if (sblabel.length() != 0)
                        sblabel.append('\n');
                    sblabel.append("entry:" + expr + ";");
                }
                Behavior during = ((State) vertex).getDoActivity();
                if (during instanceof Activity)
                {
                    Activity activity = (Activity) during;
                    if (activity.getOwnedNodes().size() == 1)
                    {
                        ActivityNode node = activity.getOwnedNodes().get(0);
                        if (node instanceof CallOperationAction)
                        {
                            CallOperationAction cao = (CallOperationAction) node;
                            for (Property prop : SysMLUtil.getAllCorrectedAttributes(context))
                            {
                                SElement se = refs.get(getKey(context, prop, cao.getOperation()));
                                if (se instanceof SOutport)
                                {
                                    if (sblabel.length() != 0)
                                        sblabel.append('\n');
                                    sblabel.append("entry:" + ((SOutport) se).getName() + "=1;");
                                    sblabel.append("\nexit:" + ((SOutport) se).getName() + "=0;");
                                }
                            }
                            
                        }
                        else
                            log.warn("Can translate only activity with one call operation action");
                    }
                    else
                        log.warn("Can translate only activity with one node: found " + activity.getOwnedNodes().size());
                }
                else if (during != null)
                    log.warn("DoActivity not translated");
                Behavior exit = ((State) vertex).getExit();
                if (exit != null)
                    log.warn("Exit unsupported");
                if (exit instanceof OpaqueBehavior)
                {
                    el2m.setSubstitutions(substitutions);
                    String expr = getAssignments((OpaqueBehavior) exit);
                    el2m.setSubstitutions(null);
                    if (sblabel.length() != 0)
                        sblabel.append('\n');
                    sblabel.append("exit:" + expr + ";");
                }
                state.setLabel(sblabel.toString());
            }
        }
        for (Transition transition : uregion.getTransitions())
        {
            Vertex target = transition.getTarget();
            if (target == null)
                throw new UMLModelErrorException(r,
                        "The transition has no target: " + print(ublock) + "/" + print(transition));
            Vertex source = transition.getSource();
            if (source == null)
                throw new UMLModelErrorException(r,
                        "The transition has no source: " + print(ublock) + "/" + print(transition));
            
            SElement starget = refs.get(getKey(ublock, target));
            if (starget != null)
            {
                SElement ssource = refs.get(getKey(ublock, source));
                
                if (ssource == null && getTranslatedVertex(source) != TranslatedVertex.INITIAL)
                {
                    log.warn("Transition " + print(ublock) + "/" + print(transition)
                            + " skipped as the source is not tanslated");
                    continue;
                }
                log.debug("Creating transition " + print(ublock) + "/" + print(transition));
                STransition stransition = StateflowFactory.eINSTANCE.createSTransition();
                if (ssource != null)
                    stransition.setSrc((SConnectNode) ssource);
                stransition.setDst((SConnectNode) starget);
                if (transition.getSource().isStereotypeApplied(sysmlutil.getSimulationVertex()) && transition
                        .getSource().getValue(sysmlutil.getSimulationVertex(), "transitionPriority") instanceof List<?>)
                {
                    @SuppressWarnings("unchecked")
                    List<Transition> ltr = (List<Transition>) transition.getSource()
                            .getValue(sysmlutil.getSimulationVertex(), "transitionPriority");
                    List<STransition> outgoings = stransition.getSrc().getOutgoing();
                    for (int i = ltr.indexOf(transition) + 1; i < ltr.size(); i++)
                    {
                        int j = outgoings.indexOf(refs.get(getKey(ublock, ltr.get(i))));
                        int k = ltr.indexOf(transition);
                        if (j != -1 && j < k)
                            outgoings.add(j, outgoings.remove(k));
                    }
                }
                for (Trigger trigger : transition.getTriggers())
                {
                    Event event = trigger.getEvent();
                    if (event instanceof ChangeEvent)
                    {
                        el2m.setSubstitutions(substitutions);
                        String exp = getExpression(((ChangeEvent) event).getChangeExpression());
                        el2m.setSubstitutions(null);
                        
                        stransition.setLabel("[" + exp + "]");
                    }
                    else if (event instanceof TimeEvent)
                    {
                        el2m.setSubstitutions(substitutions);
                        String exp = getExpression(((TimeEvent) event).getWhen());
                        el2m.setSubstitutions(null);
                        
                        stransition.setLabel("after(" + exp + ", sec)");
                    }
                    else if (event instanceof CallEvent)
                    {
                        Operation operation = ((CallEvent) event).getOperation();
                        for (Property prop : SysMLUtil.getAllCorrectedAttributes(context))
                        {
                            SElement se = refs.get(getKey(context, prop, operation));
                            if (se instanceof SInport)
                            {
                                stransition.setLabel("[" + ((SInport) se).getName() + "==1]");
                                break;
                            }
                        }
                    }
                }
                if (stransition.getLabel() == null && transition.getGuard() != null)
                {
                    el2m.setSubstitutions(substitutions);
                    String exp = getExpression(transition.getGuard().getSpecification());
                    el2m.setSubstitutions(null);
                    stransition.setLabel("[" + exp + "]");
                }
                stransition.setOwningTreeNode(snodecontainer);
                stransition.setID();
            }
            else
                log.warn("Transition " + print(ublock) + "/" + print(transition)
                        + " skipped as the target is not tanslated");
        }
    }
    
    private SComponent processComponent(Class ublock) throws UMLModelErrorException
    {
        log.debug("Processing component " + print(ublock));
        SComponent scomponent = createComponent(ublock);
        SimulinkUtil.setName(scomponent, SysMLUtil.getName(ublock));
        
        boolean inh = true;
        if (useInheritance(ublock))
        {
            log.info("Using inheritence");
        }
        else
        {
            inh = false;
            log.info("Not using inheritence");
        }
        
        if (inh)
        {
            for (Classifier ugeneral : ublock.getGenerals())
            {
                if (getTranslatedType(ugeneral) == TranslatedType.COMPONENT)
                {
                    SComponent sgeneral = createComponent((Class) ugeneral);
                    scomponent.setBaseComponent(sgeneral);
                    break;
                }
                log.warn("Unsupported generalization of " + print(ublock) + ": " + print(ugeneral));
            }
        }
        
        List<Property> uproperties = inh ? SysMLUtil.getOwnedAttributes(ublock)
                : SysMLUtil.getAllCorrectedAttributes(ublock);
        
        for (Property uproperty : uproperties)
        {
            // TODO: change that
            if (uproperty.getVisibility() == VisibilityKind.PRIVATE_LITERAL)
            {
                log.warn("Skipping property " + print(uproperty) + " because it is private");
                continue;
            }
            else if (uproperty.getType() == null)
            {
                log.warn("Skipping property " + print(uproperty) + " because it has no type");
                continue;
            }
            else if (getTranslatedType(uproperty.getType()) == TranslatedType.NONE)
            {
                log.warn("Skipping property " + print(uproperty) + " because its type is not translated");
                continue;
            }
            
            SMember smember = createMember(uproperty);
            if (smember != null)
            {
                if (uproperty.getDefaultValue() != null)
                {
                    if (uproperty.getDefaultValue() instanceof InstanceValue)
                    {
                        InstanceSpecification instancespec = ((InstanceValue) uproperty.getDefaultValue())
                                .getInstance();
                        if (instancespec != null)
                        {
                            LinkedList<SMember> ll = new LinkedList<SMember>();
                            ll.add(smember);
                            for (Slot slot : instancespec.getSlots())
                            {
                                addInitialValue(scomponent, ll, slot, (Property) slot.getDefiningFeature());
                            }
                        }
                    }
                    SDataValue svalue = getValue(uproperty.getDefaultValue());
                    if (svalue != null && svalue instanceof SExpressionValue)
                    {
                        SEquation sequation = SimscapeFactory.eINSTANCE.createSEquation();
                        sequation.setExpression(smember.getName() + "==" + ((SExpressionValue) svalue).getValue());
                        scomponent.getOwnedEquations().add(sequation);
                    }
                }
                log.debug("Adding " + print(uproperty) + " to " + print(ublock));
                if (smember instanceof SNode)
                {
                    if (scomponent.getNodes().size() % 2 == 0)
                    {
                        ((SNode) smember).setLocation(SLocation.LEFT);
                        createConnectionPortBlock(smember, SLocation.LEFT).setOwningSystem(scomponent);
                    }
                    else
                    {
                        ((SNode) smember).setLocation(SLocation.RIGHT);
                        createConnectionPortBlock(smember, SLocation.RIGHT).setOwningSystem(scomponent);
                    }
                }
                else if (smember instanceof SInput)
                {
                    ((SInput) smember).setLocation(SLocation.LEFT);
                    createConnectionPortBlock(smember, SLocation.LEFT).setOwningSystem(scomponent);
                }
                else if (smember instanceof SOutput)
                {
                    ((SOutput) smember).setLocation(SLocation.RIGHT);
                    createConnectionPortBlock(smember, SLocation.RIGHT).setOwningSystem(scomponent);
                }
                scomponent.getOwnedMembers().add(smember);
            }
        }
        scomponent.setHidden(true);
        for (Property p : SysMLUtil.getAllCorrectedAttributes(ublock))
            if (p instanceof Port)
            {
                scomponent.setHidden(false);
                break;
            }
        
        List<Connector> uconnectors = inh ? ublock.getOwnedConnectors() : SysMLUtil.getAllConnectors(ublock);
        // connectors
        for (Connector uconnector : uconnectors)
        {
            ConnectorEnd end0 = uconnector.getEnds().get(0);
            ConnectorEnd end1 = uconnector.getEnds().get(1);
            List<Property> lp0 = inh ? sysmlutil.getPropertyPath(end0)
                    : sysmlutil.getCorrectedPropertyPath(end0, ublock);
            List<Property> lp1 = inh ? sysmlutil.getPropertyPath(end1)
                    : sysmlutil.getCorrectedPropertyPath(end1, ublock);
            
            // Binding connectors not treated
            if (!uconnector.isStereotypeApplied(sysmlutil.getBindingConnector()))
            {
                // connect properties
                {
                    SConnection sconnection = SimscapeFactory.eINSTANCE.createSConnection();
                    SMemberPath scp0 = SimscapeFactory.eINSTANCE.createSMemberPath();
                    SMemberPath scp1 = SimscapeFactory.eINSTANCE.createSMemberPath();
                    
                    SMember sm0 = null;
                    for (Property prop2 : lp0)
                    {
                        sm0 = createMember(prop2);
                        if (sm0 != null)
                            scp0.getPath().add(sm0);
                        else
                            log.error("Can't add " + print(prop2) + " to the member path of " + print(uconnector));
                    }
                    SMember sm1 = null;
                    for (Property prop2 : lp1)
                    {
                        sm1 = createMember(prop2);
                        if (sm1 != null)
                            scp1.getPath().add(sm1);
                        else
                            log.error("Can't add " + print(prop2) + " to the member path of " + print(uconnector));
                    }
                    
                    // make sure the types match
                    if (sm0 instanceof SNode && sm1 instanceof SNode)
                    {
                        sconnection.getPoints().add(scp0);
                        sconnection.getPoints().add(scp1);
                    }
                    else if (sm0 instanceof SInput && sm1 instanceof SOutput)
                    {
                        sconnection.getPoints().add(scp1);
                        sconnection.getPoints().add(scp0);
                    }
                    else if (sm1 instanceof SInput && sm0 instanceof SOutput)
                    {
                        sconnection.getPoints().add(scp0);
                        sconnection.getPoints().add(scp1);
                    }
                    else
                    {
                        log.error("Incompatible types for connector " + print(uconnector));
                    }
                    log.info("Adding connection");
                    scomponent.getOwnedConnections().add(sconnection);
                }
            }
        }
        // process a second time
        
        ReferenceKey k = getKey(ublock);
        if (!toProcessForConnectors.contains(k))
            toProcessForConnectors.push(k);
        
        // initial values
        return scomponent;
    }
    
    private static SConnectionPortBlock createConnectionPortBlock(SMember smember, SLocation location)
    {
        SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
        scpb.setLocation(location);
        scpb.setMember(smember);
        return scpb;
    }
    
    private SDomain processDomain(Class ublock) throws UMLModelErrorException
    {
        // processed only if need to process the simproperties
        log.debug("Processing domain " + print(ublock));
        SDomain sdomain = createDomain(ublock);
        SimulinkUtil.setName(sdomain, SysMLUtil.getName(ublock));
        
        if (ublock.getGenerals().size() > 0)
            log.warn("Generalization of domains is not supported by Simscape: " + print(ublock));
        for (Property simproperty : sysmlutil.getAllPhSProperties(ublock))
        {
            if (simproperty.getType() instanceof Classifier)
                for (Property uproperty : SysMLUtil.getAllCorrectedAttributes((Classifier) simproperty.getType()))
                {
                    if (uproperty.getVisibility() == VisibilityKind.PRIVATE_LITERAL)
                    {
                        log.warn("Skipping non-public property " + print(uproperty));
                        continue;
                    }
                    SMember smember = createMember(uproperty);
                    if (smember != null)
                    {
                        if (smember instanceof SParameter)
                        {
                            log.debug("Adding " + print(uproperty) + " to " + print(ublock));
                            sdomain.getParameters().add((SParameter) smember);
                        }
                        else if (smember instanceof SVariable)
                        {
                            log.debug("Adding " + print(uproperty) + " to " + print(ublock));
                            sdomain.getVariables().add((SVariable) smember);
                        }
                    }
                    else
                        log.warn("Could not add " + print(uproperty) + " to " + print(ublock));
                }
            else
                log.error("The sim property " + print(simproperty) + " is not typed by a classifier");
        }
        return sdomain;
    }
    
    @SuppressWarnings("boxing")
    private SComponent processComponent2(Class ublock) throws UMLModelErrorException
    {
        log.debug("Processing component " + print(ublock) + " a second time");
        SComponent scomponent = createComponent(ublock);
        
        boolean inh = useInheritance(ublock);
        
        // add new variable for each conserved variable
        loop: for (SMember sm : new ArrayList<SMember>(scomponent.getOwnedMembers()))
            if (sm instanceof SNode)
            {
                // skip if bound to something
                for (SConnection sc : scomponent.getOwnedConnections())
                    for (SMemberPath smp : sc.getPoints())
                        if (smp.getPath().get(0) == sm)
                            continue loop;
                for (SVariable sv : ((SNode) sm).getDomain().getVariables())
                    if (sv.isBalancing())
                    {
                        // add variable...
                        SVariable sv2 = SimscapeFactory.eINSTANCE.createSVariable();
                        scomponent.getOwnedMembers().add(sv2);
                        SimulinkUtil.setName(sv2, sm.getName() + sv.getName());
                        sv2.setValue(sv.getValue());
                        sv2.setUnit(sv.getUnit());
                        // ...and branch
                        SBranch sb = SimscapeFactory.eINSTANCE.createSBranch();
                        sb.setVariable(sv2);
                        SMemberPath smp = SimscapeFactory.eINSTANCE.createSMemberPath();
                        smp.getPath().add(sm);
                        smp.getPath().add(sv);
                        sb.setFrom(smp);
                        scomponent.getBranches().add(sb);
                    }
            }
        
        // Make the substitutions associated with this property. Should move up
        
        // Parse connectors for bindings
        // AND for parameter assignments/equations
        
        List<Connector> uconnectors = inh ? ublock.getOwnedConnectors() : SysMLUtil.getAllConnectors(ublock);
        
        for (Connector conn : uconnectors)
        {
            if (!conn.isStereotypeApplied(sysmlutil.getBindingConnector()))
                continue;
            ConnectorEnd ce0 = conn.getEnds().get(0);
            ConnectorEnd ce1 = conn.getEnds().get(1);
            List<Property> lp0 = inh ? sysmlutil.getPropertyPath(ce0) : sysmlutil.getCorrectedPropertyPath(ce0, ublock);
            List<Property> lp1 = inh ? sysmlutil.getPropertyPath(ce1) : sysmlutil.getCorrectedPropertyPath(ce1, ublock);
            
            String from = null, to = null;
            // No support for nested CB properties
            
            Property lp0p = lp0.get(lp0.size() - 1);
            Property lp1p = lp1.get(lp1.size() - 1);
            
            // bindings/equations
            if (lp0.size() == 1 && lp1.size() == 2)
            {
                SElement se1 = refs.get(getKey(lp0.get(0)));
                SElement se2 = refs.get(getKey(lp1.get(0)));
                SElement se3 = refs.get(getKey(lp1.get(1)));
                if (se2 instanceof SComponentReference)
                {
                    if (se1 instanceof SParameter && se3 instanceof SParameter)
                    {
                        SMemberAssignment sma = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                        sma.getAssignedReference().add((SMember) se1);
                        sma.getMemberPath().add((SMember) se2);
                        sma.getMemberPath().add((SMember) se3);
                        scomponent.getOwnedAssignments().add(sma);
                    }
                }
            }
            if (lp1.size() == 1 && lp0.size() == 2)
            {
                SElement se1 = refs.get(getKey(lp1.get(0)));
                SElement se2 = refs.get(getKey(lp0.get(0)));
                SElement se3 = refs.get(getKey(lp0.get(1)));
                if (se2 instanceof SComponentReference)
                {
                    if (se1 instanceof SParameter && se3 instanceof SParameter)
                    {
                        SMemberAssignment sma = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                        sma.getAssignedReference().add((SMember) se1);
                        sma.getMemberPath().add((SMember) se2);
                        sma.getMemberPath().add((SMember) se3);
                        scomponent.getOwnedAssignments().add(sma);
                    }
                }
            }
            // binding nested variables
            
            boolean lp0v = lp0p.isStereotypeApplied(sysmlutil.getPhSVariable());
            boolean lp1v = lp1p.isStereotypeApplied(sysmlutil.getPhSVariable());
            boolean lp0c = lp0p.isStereotypeApplied(sysmlutil.getPhSConstant());
            boolean lp1c = lp1p.isStereotypeApplied(sysmlutil.getPhSConstant());
            if ((lp0v && (lp1v || lp1c)) || (lp1v && (lp0v || lp0c)))
            {
                SEquation seq = SimscapeFactory.eINSTANCE.createSEquation();
                seq.setExpression(sysmlutil.joinProperties(lp0, ".") + "==" + sysmlutil.joinProperties(lp1, "."));
                scomponent.getOwnedEquations().add(seq);
            }
        }
        
        // in the second pass because all members need to be fully defined
        List<Constraint> uconstraints = inh ? ublock.getOwnedRules() : SysMLUtil.getAllConstraints(ublock);
        for (Constraint uconstraint : uconstraints)
        {
            el2m.prepareNextEquationParsing(scomponent, null);
            String value = getEquation(uconstraint.getSpecification());
            if (value != null)
            {
                SEquation sequation = SimscapeFactory.eINSTANCE.createSEquation();
                sequation.setExpression(value);
                scomponent.getOwnedEquations().add(sequation);
            }
        }
        
        List<Property> uproperties = inh ? SysMLUtil.getOwnedAttributes(ublock)
                : SysMLUtil.getAllCorrectedAttributes(ublock);
        
        for (Property uproperty : uproperties)
        {
            if (uproperty.getType().isStereotypeApplied(sysmlutil.getConstraintBlock()))
            {
                Hashtable<String, String> subs = new Hashtable<String, String>();
                for (Connector uconnector : SysMLUtil.getAllConnectors(ublock))
                {
                    if (!uconnector.isStereotypeApplied(sysmlutil.getBindingConnector()))
                        continue;
                    if (uconnector.getEnds().size() != 2)
                        continue;
                    
                    List<Property> lp0 = sysmlutil.getPropertyPath(uconnector.getEnds().get(0));
                    List<Property> lp1 = sysmlutil.getPropertyPath(uconnector.getEnds().get(1));
                    
                    if (lp0.size() == 2 && lp0.get(0) == uproperty)
                    {
                        String from = lp0.get(1).getName();
                        StringBuilder to = new StringBuilder();
                        for (int i = 0; i < lp1.size(); i++)
                        {
                            Property up = lp1.get(i);
                            if (sysmlutil.isPhSProperty(up))
                                continue;
                            if (i != 0 && !(up.isStereotypeApplied(sysmlutil.getPhSVariable())
                                    && (Boolean) up.getValue(sysmlutil.getPhSVariable(), "isConserved")))
                                to.append(".");
                            to.append(up.getName());
                        }
                        subs.put(from, to.toString());
                    }
                    else if (lp1.size() == 2 && lp1.get(0) == uproperty)
                    {
                        String from = lp1.get(1).getName();
                        StringBuilder to = new StringBuilder();
                        for (int i = 0; i < lp0.size(); i++)
                        {
                            Property up = lp0.get(i);
                            if (sysmlutil.isPhSProperty(up))
                                continue;
                            if (i != 0 && !(up.isStereotypeApplied(sysmlutil.getPhSVariable())
                                    && (Boolean) up.getValue(sysmlutil.getPhSVariable(), "isConserved")))
                                to.append(".");
                            to.append(up.getName());
                        }
                        subs.put(from, to.toString());
                    }
                }
                
                for (Constraint uconstraint : SysMLUtil.getAllConstraints((Classifier) uproperty.getType()))
                {
                    el2m.prepareNextEquationParsing(scomponent, subs);
                    String value = getEquation(uconstraint.getSpecification());
                    if (value != null)
                    {
                        SEquation sequation = SimscapeFactory.eINSTANCE.createSEquation();
                        sequation.setExpression(value);
                        scomponent.getOwnedEquations().add(sequation);
                    }
                }
                continue;
            }
            
        }
        return scomponent;
    }
    
    /*
     * public static String replace(String original, String pattern,
     * HashMap<String, String> replacements) { if (original == null ||
     * replacements == null) return original;
     * 
     * Pattern pat = Pattern.compile(pattern); Matcher mat =
     * pat.matcher(original); StringBuffer sb = new
     * StringBuffer(original.length()); while (mat.find()) {
     * mat.appendReplacement(sb, ""); String rep =
     * replacements.get(mat.group()); sb.append(rep == null ? mat.group() :
     * rep); } mat.appendTail(sb); return sb.toString(); }
     */
    
    @Override
    protected String getOpaqueLanguage()
    {
        return "matlab";
    }
    
    // TODO: merge with something else?
    private void addInitialValue(SComponent scomponent, LinkedList<SMember> ll, Slot slot, Property definingfeature)
            throws UMLModelErrorException
    {
        if (slot.getValues().size() == 0)
            log.warn("Slot " + print(slot) + " has no values");
        else if (slot.getValues().size() > 1)
            log.warn("Slot " + print(slot) + " has several values");
        else
        {
            StructuralFeature sf = definingfeature;
            if (!(sf instanceof Property))
            {
                log.warn("The feature of slot " + print(slot) + " should be a property");
                return;
            }
            SMember lastmember = createMember((Property) sf);
            
            if (slot.isStereotypeApplied(sysmlutil.getInitialValueReference())
                    && slot.getValue(sysmlutil.getInitialValueReference(), "propertyPath") instanceof List<?>)
            {
                @SuppressWarnings("unchecked")
                List<Property> references = (List<Property>) slot.getValue(sysmlutil.getInitialValueReference(),
                        "propertyPath");
                List<SMember> members = new LinkedList<SMember>();
                for (Property prop : references)
                {
                    SMember member = createMember(prop);
                    if (member != null)
                        members.add(member);
                }
                SMemberAssignment sassignment = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                sassignment.getMemberPath().addAll(ll);
                sassignment.getMemberPath().add(lastmember);
                sassignment.getAssignedReference().addAll(members);
                scomponent.getOwnedAssignments().add(sassignment);
            }
            else
            {
                ValueSpecification vs = slot.getValues().get(0);
                SDataValue sv = getValue(vs);
                if (sv != null)
                {
                    SMemberAssignment sassignment = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                    sassignment.getMemberPath().addAll(ll);
                    sassignment.getMemberPath().add(lastmember);
                    sassignment.setAssignedValue(sv);
                    scomponent.getOwnedAssignments().add(sassignment);
                }
                else if (vs instanceof InstanceValue)
                {
                    for (Slot subslot : ((InstanceValue) vs).getInstance().getSlots())
                    {
                        LinkedList<SMember> newlist = new LinkedList<SMember>(ll);
                        if (sysmlutil.isPhSProperty((Property) subslot.getDefiningFeature()))
                            addInitialValue(scomponent, newlist, subslot, (Property) sf);
                        else
                        {
                            newlist.add(lastmember);
                            addInitialValue(scomponent, newlist, subslot, (Property) subslot.getDefiningFeature());
                        }
                    }
                }
                else
                    log.warn("Unable to process the value of " + print(vs)
                            + ". Use String, Boolean, Integer, Real or SysML initial values");
            }
        }
    }
    
    private SDataValue getValue(ValueSpecification vs) throws UMLModelErrorException
    {
        if (vs instanceof LiteralReal)
        {
            SDoubleValue sdouble = SimulinkFactory.eINSTANCE.createSDoubleValue();
            sdouble.setValue(((LiteralReal) vs).getValue());
            return sdouble;
        }
        if (vs instanceof OpaqueExpression)
        {
            String val = getEquation(vs);
            if (val != null)
            {
                SExpressionValue sexpression = SimulinkFactory.eINSTANCE.createSExpressionValue();
                sexpression.setValue(val);
                return sexpression;
            }
        }
        return null;
    }
    
    private SDataValue getValue(Slot slot) throws UMLModelErrorException
    {
        if (slot.getValues().size() == 1)
        {
            return getValue(slot.getValues().get(0));
        }
        else if (slot.getValues().size() > 1)
        {
            StringBuilder sb = new StringBuilder();
            
            List<Integer> dimensions = new LinkedList<Integer>();
            
            if (slot.isStereotypeApplied(sysmlutil.getMultidimensionalElement())
                    && slot.getValue(sysmlutil.getMultidimensionalElement(), "dimension") instanceof List<?>)
            {
                @SuppressWarnings("unchecked")
                List<Object> dims = (List<Object>) slot.getValue(sysmlutil.getMultidimensionalElement(), "dimension");
                for (int i = 0; i < dims.size(); i++)
                {
                    int cumul = ((Integer) dims.get(i)).intValue();
                    for (int j = i + 1; j < dims.size(); j++)
                        cumul = cumul * ((Integer) dims.get(j)).intValue();
                    dimensions.add(Integer.valueOf(cumul));
                }
            }
            else if (slot.getDefiningFeature().isStereotypeApplied(sysmlutil.getMultidimensionalElement())
                    && slot.getDefiningFeature().getValue(sysmlutil.getMultidimensionalElement(),
                            "dimension") instanceof List<?>)
            {
                @SuppressWarnings("unchecked")
                List<Object> dims = (List<Object>) slot.getDefiningFeature()
                        .getValue(sysmlutil.getMultidimensionalElement(), "dimension");
                for (int i = 0; i < dims.size(); i++)
                {
                    int cumul = ((Integer) dims.get(i)).intValue();
                    for (int j = i + 1; j < dims.size(); j++)
                        cumul = cumul * ((Integer) dims.get(j)).intValue();
                    dimensions.add(Integer.valueOf(cumul));
                }
            }
            else
                dimensions.add(Integer.valueOf(slot.getValues().size()));
            
            // not actually checking the sizes
            if (dimensions.size() == 1)
            {
                sb.append("[");
                for (int i = 0; i < slot.getValues().size(); i++)
                {
                    if (i != 0)
                        sb.append(" ");
                    sb.append(getValue(slot.getValues().get(i)).getRawValue());
                }
                sb.append("]");
            }
            else if (dimensions.size() == 2)
            {
                @SuppressWarnings("boxing")
                int mod = dimensions.get(1);
                sb.append("[");
                for (int i = 0; i < slot.getValues().size(); i++)
                {
                    if (i != 0)
                    {
                        if (i % mod == 0)
                            sb.append(";");
                        else
                            sb.append(" ");
                    }
                    sb.append(getValue(slot.getValues().get(i)).getRawValue());
                }
                sb.append("]");
            }
            
            SExpressionValue sexv = SimulinkFactory.eINSTANCE.createSExpressionValue();
            sexv.setValue(sb.toString());
            return sexv;
        }
        return null;
    }
    
    private static String getValue(SFVariable svar)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < svar.getValue().size(); i++)
        {
            SDataValue sdv = svar.getValue().get(i);
            if (i != 0)
                sb.append(";");
            if (sdv instanceof SExpressionValue)
                sb.append(((SExpressionValue) sdv).getValue());
            else if (sdv instanceof SDoubleValue)
                sb.append(((SDoubleValue) sdv).getValue());
        }
        return sb.toString();
    }
    
    private String getUnitFromType(Type utype)
    {
        // Could store the result in an HT
        if (!(utype instanceof Classifier))
            return null;
        Classifier uclassifier = (Classifier) utype;
        List<Classifier> lclassifiers = new ArrayList<Classifier>();
        lclassifiers.add(uclassifier);
        lclassifiers.addAll(uclassifier.allParents());
        for (Classifier cl : lclassifiers)
        {
            if (cl.isStereotypeApplied(sysmlutil.getValueType()))
            {
                InstanceSpecification unit = (InstanceSpecification) cl.getValue(sysmlutil.getValueType(), "unit");
                if (unit != null)
                {
                    for (Slot unitslot : unit.getSlots())
                    {
                        if (unitslot.getDefiningFeature() != null
                                && unitslot.getDefiningFeature().getName().equals("symbol"))
                            for (ValueSpecification symbol : unitslot.getValues())
                            {
                                String val = symbol.stringValue();
                                if (Configuration.getSimulinkUnitMapping().get(val) != null)
                                    return Configuration.getSimulinkUnitMapping().get(val);
                                return val;
                            }
                    }
                }
            }
        }
        return null;
    }
    
    private boolean useInheritance(Class uclass)
    {
        // inherited only when Simscape component with no redefined property AND
        // only single inheritence
        if (getTranslatedType(uclass) != TranslatedType.COMPONENT)
            return false;
        for (Property uproperty : SysMLUtil.getAllAttributes(uclass))
            if (uproperty.getRedefinedProperties().size() != 0)
                return false;
            
        List<Class> generals = new ArrayList<Class>(uclass.getGeneralizations().size());
        for (Generalization generalization : uclass.getGeneralizations())
        {
            Classifier gen = generalization.getGeneral();
            Classifier spe = generalization.getSpecific();
            if (spe == uclass && getTranslatedType(gen) == TranslatedType.COMPONENT)
            {
                Class cl = (Class) gen;
                if (!useInheritance(cl))
                    return false;
                generals.add(cl);
            }
        }
        if (generals.size() > 1)
            return false;
        return true;
    }
}
