package com.engisis.sysphs.translation.simulink;

import java.util.Hashtable;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Property;

import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SConnectionPortBlock;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SLocation;
import com.engisis.sysphs.language.simscape.SNode;
import com.engisis.sysphs.language.simscape.SPackage;
import com.engisis.sysphs.language.simscape.SVariable;
import com.engisis.sysphs.language.simscape.SimscapeFactory;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SimulinkFactory;
import com.engisis.sysphs.util.SysMLToSimulationTranslator;
import com.engisis.sysphs.util.SysMLToSimulationTranslator.ReferenceKey;

/**
 * Simscape utility class
 * 
 * @author barbau
 *
 */
public class SimscapeUtil
{
    private static final Logger log = Logger.getLogger(SimscapeUtil.class);
    
    /**
     * Utility package
     */
    private static SPackage spackage;
    /**
     * Solver configuration component
     */
    private static SComponent ssolverconfiguration;
    /**
     * Physical signal to Simulink component
     */
    private static SComponent sps2sim;
    /**
     * Simulink to physical signal component
     */
    private static SComponent ssim2ps;
    
    /**
     * Left connection port
     */
    private static SComponent lcpc;
    
    /**
     * Right connection port
     */
    private static SComponent rcpc;
    
    /**
     * Returns the Utility package
     * 
     * @return the Utility package
     */
    public synchronized static SPackage getUtilityPackage()
    {
        if (spackage == null)
        {
            spackage = SimscapeFactory.eINSTANCE.createSPackage();
            spackage.setName("nesl_utility");
            spackage.setGenerated(false);
        }
        return spackage;
    }
    
    /**
     * Returns the Solver Configuration component
     * 
     * @return the Solver Configuration component
     */
    public synchronized static SComponent getSolverConfiguration()
    {
        if (ssolverconfiguration == null)
        {
            ssolverconfiguration = SimscapeFactory.eINSTANCE.createSComponent();
            ssolverconfiguration.setName("Solver\nConfiguration");
            ssolverconfiguration.setPhysicalDomain("network_engine_domain");
            ssolverconfiguration.setSubClass("solver");
            ssolverconfiguration.setOwningPackage(getUtilityPackage());
            SNode snode = SimscapeFactory.eINSTANCE.createSNode();
            snode.setLocation(SLocation.RIGHT);
            SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            scpb.setLocation(SLocation.RIGHT);
            scpb.setMember(snode);
            scpb.setOwningSystem(ssolverconfiguration);
            ssolverconfiguration.getOwnedMembers().add(snode);
        }
        return ssolverconfiguration;
    }
    
    /**
     * Returns the Simulink to Physical Signal component
     * 
     * @return the Simulink to Physical Signal component
     */
    public synchronized static SComponent getSimulink2PS()
    {
        if (ssim2ps == null)
        {
            ssim2ps = SimscapeFactory.eINSTANCE.createSComponent();
            ssim2ps.setName("Simulink-PS\nConverter");
            ssim2ps.setPhysicalDomain("network_engine_domain");
            ssim2ps.setSubClass("ps_output");
            ssim2ps.setOwningPackage(getUtilityPackage());
            
            SInport sinport = SimulinkFactory.eINSTANCE.createSInport();
            sinport.setOwningSystem(ssim2ps);
            
            SNode snode = SimscapeFactory.eINSTANCE.createSNode();
            snode.setLocation(SLocation.RIGHT);
            ssim2ps.getOwnedMembers().add(snode);
            
            SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            scpb.setMember(snode);
            scpb.setLocation(SLocation.RIGHT);
            scpb.setOwningSystem(ssim2ps);
        }
        return ssim2ps;
    }
    
    /**
     * Returns the Physical Signal to Simulink component
     * 
     * @return the Physical Signal to Simulink component
     */
    public synchronized static SComponent getPS2Simulink()
    {
        if (sps2sim == null)
        {
            sps2sim = SimscapeFactory.eINSTANCE.createSComponent();
            sps2sim.setName("PS-Simulink\nConverter");
            sps2sim.setPhysicalDomain("network_engine_domain");
            sps2sim.setSubClass("ps_input");
            sps2sim.setOwningPackage(getUtilityPackage());
            
            SOutport soutport = SimulinkFactory.eINSTANCE.createSOutport();
            soutport.setOwningSystem(sps2sim);
            
            SNode snode = SimscapeFactory.eINSTANCE.createSNode();
            snode.setLocation(SLocation.LEFT);
            sps2sim.getOwnedMembers().add(snode);
            
            SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            scpb.setMember(snode);
            scpb.setLocation(SLocation.LEFT);
            scpb.setOwningSystem(sps2sim);
        }
        return sps2sim;
    }
    
    public synchronized static SComponent getConnectionPortComponent()
    {
        if (rcpc == null)
        {
            rcpc = SimscapeFactory.eINSTANCE.createSComponent();
            SNode sn = SimscapeFactory.eINSTANCE.createSNode();
            sn.setLocation(SLocation.RIGHT);
            rcpc.getOwnedMembers().add(sn);

            SConnectionPortBlock scpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            scpb.setMember(sn);
            scpb.setLocation(SLocation.RIGHT);
            scpb.setOwningSystem(rcpc);
            
        }
        return rcpc;
    }
    
    /**
     * Creates the predefined Simscape domains. Maps the blocks/properties from
     * the UML resource to their equivalent Simscape domains/variables
     * 
     * @param r
     *            resource from which SimBlocks and SimVariables are looked for
     * @param refs
     *            reference table
     */
    public static void mapDomains(Resource r, Hashtable<ReferenceKey, SElement> refs)
    {
        SDomain sdelectrical = SimscapeFactory.eINSTANCE.createSDomain();
        {
            sdelectrical.setName("foundation.electrical.electrical");
            SVariable svi = SimscapeFactory.eINSTANCE.createSVariable();
            svi.setName("i");
            svi.setUnit("A");
            svi.setBalancing(true);
            sdelectrical.getVariables().add(svi);
            SVariable svv = SimscapeFactory.eINSTANCE.createSVariable();
            svv.setName("v");
            svv.setUnit("V");
            sdelectrical.getVariables().add(svv);
        }
        SDomain sdhydraulic = SimscapeFactory.eINSTANCE.createSDomain();
        {
            sdhydraulic.setName("foundation.hydraulic.hydraulic");
            SVariable svq = SimscapeFactory.eINSTANCE.createSVariable();
            svq.setName("q");
            svq.setUnit("m^3/s");
            svq.setBalancing(true);
            sdhydraulic.getVariables().add(svq);
            SVariable svp = SimscapeFactory.eINSTANCE.createSVariable();
            svp.setName("p");
            svp.setUnit("Pa");
            sdhydraulic.getVariables().add(svp);
        }
        SDomain sdmagnetic = SimscapeFactory.eINSTANCE.createSDomain();
        {
            sdmagnetic.setName("foundation.magnetic.magnetic");
            SVariable phi = SimscapeFactory.eINSTANCE.createSVariable();
            phi.setName("phi");
            phi.setUnit("Wb");
            phi.setBalancing(true);
            sdmagnetic.getVariables().add(phi);
            SVariable mmf = SimscapeFactory.eINSTANCE.createSVariable();
            mmf.setName("mmf");
            mmf.setUnit("A");
            sdmagnetic.getVariables().add(mmf);
        }
        SDomain sdrotational = SimscapeFactory.eINSTANCE.createSDomain();
        {
            sdrotational.setName("foundation.mechanical.rotational");
            SVariable t = SimscapeFactory.eINSTANCE.createSVariable();
            t.setName("t");
            t.setUnit("N*m");
            t.setBalancing(true);
            sdrotational.getVariables().add(t);
            SVariable w = SimscapeFactory.eINSTANCE.createSVariable();
            w.setName("w");
            w.setUnit("rad/s");
            sdrotational.getVariables().add(w);
        }
        SDomain sdtranslational = SimscapeFactory.eINSTANCE.createSDomain();
        {
            sdtranslational.setName("foundation.mechanical.translational");
            SVariable f = SimscapeFactory.eINSTANCE.createSVariable();
            f.setName("f");
            f.setUnit("N");
            f.setBalancing(true);
            sdtranslational.getVariables().add(f);
            SVariable v = SimscapeFactory.eINSTANCE.createSVariable();
            v.setName("v");
            v.setUnit("m/s");
            sdtranslational.getVariables().add(v);
        }
        SDomain sdpneumatic = SimscapeFactory.eINSTANCE.createSDomain();
        {
            sdpneumatic.setName("foundation.pneumatic.pneumatic");
            SVariable g = SimscapeFactory.eINSTANCE.createSVariable();
            g.setName("G");
            g.setUnit("kg/s");
            g.setBalancing(true);
            sdpneumatic.getVariables().add(g);
            SVariable q = SimscapeFactory.eINSTANCE.createSVariable();
            q.setName("Q");
            q.setUnit("J/s");
            q.setBalancing(true);
            sdpneumatic.getVariables().add(q);
            SVariable p = SimscapeFactory.eINSTANCE.createSVariable();
            p.setName("p");
            p.setUnit("Pa");
            sdpneumatic.getVariables().add(p);
            SVariable t = SimscapeFactory.eINSTANCE.createSVariable();
            t.setName("t");
            t.setUnit("K");
            sdpneumatic.getVariables().add(t);
        }
        SDomain sdthermal = SimscapeFactory.eINSTANCE.createSDomain();
        {
            sdthermal.setName("foundation.thermal.thermal");
            SVariable q = SimscapeFactory.eINSTANCE.createSVariable();
            q.setName("Q");
            q.setUnit("J/s");
            q.setBalancing(true);
            sdthermal.getVariables().add(q);
            SVariable t = SimscapeFactory.eINSTANCE.createSVariable();
            t.setName("t");
            q.setUnit("J/s");
            sdthermal.getVariables().add(t);
        }
        
        TreeIterator<EObject> iter = r.getAllContents();
        while (iter.hasNext())
        {
            EObject eo = iter.next();
            if (eo instanceof NamedElement)
            {
                if (eo instanceof Class)
                {
                    Class c = (Class) eo;
                    if (match(c, sdelectrical))
                    {
                        refs.put(SysMLToSimulationTranslator.getKey(c), sdelectrical);
                        log.info("Adding " + c.getQualifiedName() + " as " + sdelectrical.getName());
                    }
                    if (match(c, sdhydraulic))
                    {
                        refs.put(SysMLToSimulationTranslator.getKey(c), sdhydraulic);
                        log.info("Adding " + c.getQualifiedName() + " as " + sdhydraulic.getName());
                    }
                    if (match(c, sdmagnetic))
                    {
                        refs.put(SysMLToSimulationTranslator.getKey(c), sdmagnetic);
                        log.info("Adding " + c.getQualifiedName() + " as " + sdmagnetic.getName());
                    }
                    if (match(c, sdrotational))
                    {
                        refs.put(SysMLToSimulationTranslator.getKey(c), sdrotational);
                        log.info("Adding " + c.getQualifiedName() + " as " + sdrotational.getName());
                    }
                    if (match(c, sdtranslational))
                    {
                        refs.put(SysMLToSimulationTranslator.getKey(c), sdtranslational);
                        log.info("Adding " + c.getQualifiedName() + " as " + sdtranslational.getName());
                    }
                    if (match(c, sdpneumatic))
                    {
                        refs.put(SysMLToSimulationTranslator.getKey(c), sdpneumatic);
                        log.info("Adding " + c.getQualifiedName() + " as " + sdpneumatic.getName());
                    }
                    if (match(c, sdthermal))
                    {
                        refs.put(SysMLToSimulationTranslator.getKey(c), sdthermal);
                        log.info("Adding " + c.getQualifiedName() + " as " + sdthermal.getName());
                    }
                }
            }
            else
                iter.prune();
        }
    }
    
    /**
     * Returns true if the properties of the class match the variables of the domain
     * @param c
     * @param sd
     * @return
     */
    private static boolean match(Class c, SDomain sd)
    {
        if (c == null || sd == null)
            return false;
        if (c.getOwnedAttributes().size() != sd.getVariables().size())
            return false;
        for(Property p : c.getOwnedAttributes())
        {
            boolean ok = false;
            for(SVariable sv : sd.getVariables())
                if (p.getName().equals(sv.getName()))
                {
                    ok = true;
                    break;
                }
            if (!ok)
                return false;
        }
        return true;
    }
    
}
