package com.engisis.sysphs.translation.modelica;

import java.util.ArrayList;
import java.util.List;

import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MComponent;
import com.engisis.sysphs.language.modelica.MModel;
import com.engisis.sysphs.language.modelica.MNamedElement;
import com.engisis.sysphs.language.modelica.MPrimitiveType;
import com.engisis.sysphs.language.modelica.MType;
import com.engisis.sysphs.language.modelica.ModelicaFactory;

/**
 * Class regrouping some basic Modelica classes
 * 
 * @author barbau
 *
 */
public class ModelicaUtil
{
    private static MPrimitiveType mstringtype;
    private static MPrimitiveType mbooleantype;
    private static MPrimitiveType mintegertype;
    private static MPrimitiveType mrealtype;
    private static MType mstring;
    private static MType mboolean;
    private static MType minteger;
    private static MType mreal;
    private static MModel minitialstep;
    private static MModel mstep;
    private static MModel mtransition;
    
    /**
     * Returns the Modelica String type
     * 
     * @return the Modelica String type
     */
    private static MPrimitiveType getStringType()
    {
        if (mstringtype == null)
        {
            mstringtype = ModelicaFactory.eINSTANCE.createMPrimitiveType();
            mstringtype.setName("StringType");
        }
        return mstringtype;
    }
    
    /**
     * Returns the Modelica Boolean type
     * 
     * @return the Modelica Boolean type
     */
    private static MPrimitiveType getBooleanType()
    {
        if (mbooleantype == null)
        {
            mbooleantype = ModelicaFactory.eINSTANCE.createMPrimitiveType();
            mbooleantype.setName("BooleanType");
        }
        return mbooleantype;
    }
    
    /**
     * Returns the Modelica Integer type
     * 
     * @return the Modelica Integer type
     */
    private static MPrimitiveType getIntegerType()
    {
        if (mintegertype == null)
        {
            mintegertype = ModelicaFactory.eINSTANCE.createMPrimitiveType();
            mintegertype.setName("IntegerType");
        }
        return mintegertype;
    }
    
    /**
     * Returns the Modelica Real type
     * 
     * @return the Modelica Real type
     */
    private static MPrimitiveType getRealType()
    {
        if (mrealtype == null)
        {
            mrealtype = ModelicaFactory.eINSTANCE.createMPrimitiveType();
            mrealtype.setName("RealType");
        }
        return mrealtype;
    }
    
    /**
     * Returns Modelica Real
     * 
     * @return Modelica Real
     */
    public static MType getReal()
    {
        if (mreal == null)
        {
            mreal = ModelicaFactory.eINSTANCE.createMType();
            mreal.setName("Real");
            addComponentToType(mreal, "quantity", getStringType());
            addComponentToType(mreal, "unit", getStringType());
            addComponentToType(mreal, "displayUnit", getStringType());
            addComponentToType(mreal, "start", getRealType());
            addComponentToType(mreal, "fixed", getBooleanType());
        }
        return mreal;
    }
    
    /**
     * Returns Modelica Integer
     * 
     * @return Modelica Integer
     */
    public static MType getInteger()
    {
        if (minteger == null)
        {
            minteger = ModelicaFactory.eINSTANCE.createMType();
            minteger.setName("Integer");
            addComponentToType(minteger, "quantity", getStringType());
            addComponentToType(minteger, "start", getIntegerType());
            addComponentToType(minteger, "fixed", getBooleanType());
        }
        return minteger;
    }
    
    /**
     * Returns Modelica Boolean
     * 
     * @return Modelica Boolean
     */
    public static MType getBoolean()
    {
        if (mboolean == null)
        {
            mboolean = ModelicaFactory.eINSTANCE.createMType();
            mboolean.setName("Boolean");
            addComponentToType(mboolean, "quantity", getStringType());
            addComponentToType(mboolean, "start", getBooleanType());
            addComponentToType(mboolean, "fixed", getBooleanType());
        }
        return mboolean;
    }
    
    /**
     * Returns Modelica String
     * 
     * @return Modelica String
     */
    public static MType getString()
    {
        if (mstring == null)
        {
            mstring = ModelicaFactory.eINSTANCE.createMType();
            mstring.setName("String");
            addComponentToType(mstring, "quantity", getStringType());
            addComponentToType(mstring, "start", getStringType());
        }
        return mstring;
    }
    
    /**
     * Returns the Modelica Initial Step class
     * 
     * @return the Modelica Initial Step class
     */
    public static MModel getInitialStep()
    {
        if (minitialstep == null)
        {
            minitialstep = ModelicaFactory.eINSTANCE.createMModel();
            minitialstep.setName("Modelica.StateGraph.InitialStep");
            
            MComponent nIn = ModelicaFactory.eINSTANCE.createMComponent();
            nIn.setName("nIn");
            minitialstep.getComponents().add(nIn);
            
            MComponent nOut = ModelicaFactory.eINSTANCE.createMComponent();
            nOut.setName("nOut");
            minitialstep.getComponents().add(nOut);
            
            MComponent active = ModelicaFactory.eINSTANCE.createMComponent();
            active.setName("active");
            minitialstep.getComponents().add(active);
        }
        return minitialstep;
    }
    
    /**
     * Return the Modelica Step class
     * 
     * @return the Modelica Step class
     */
    public static MModel getStep()
    {
        if (mstep == null)
        {
            mstep = ModelicaFactory.eINSTANCE.createMModel();
            mstep.setName("Modelica.StateGraph.Step");
            
            MComponent nIn = ModelicaFactory.eINSTANCE.createMComponent();
            nIn.setName("nIn");
            mstep.getComponents().add(nIn);
            
            MComponent nOut = ModelicaFactory.eINSTANCE.createMComponent();
            nOut.setName("nOut");
            mstep.getComponents().add(nOut);
            
            MComponent active = ModelicaFactory.eINSTANCE.createMComponent();
            active.setName("active");
            mstep.getComponents().add(active);
        }
        return mstep;
    }
    
    /**
     * Return the Modelica Transition class
     * 
     * @return the Modelica Transition class
     */
    public static MModel getTransition()
    {
        if (mtransition == null)
        {
            mtransition = ModelicaFactory.eINSTANCE.createMModel();
            mtransition.setName("Modelica.StateGraph.Transition");
            
            MComponent condition = ModelicaFactory.eINSTANCE.createMComponent();
            condition.setName("condition");
            mtransition.getComponents().add(condition);
            
            MComponent enabletimer = ModelicaFactory.eINSTANCE.createMComponent();
            enabletimer.setName("enableTimer");
            mtransition.getComponents().add(enabletimer);
            
            MComponent waittime = ModelicaFactory.eINSTANCE.createMComponent();
            waittime.setName("waitTime");
            mtransition.getComponents().add(waittime);
            
        }
        return mtransition;
    }
    
    /**
     * Helper to add components to types
     * 
     * @param mtype
     *            type to which the component is added
     * @param name
     *            name of the component
     * @param type
     *            type of the component
     */
    private static void addComponentToType(MType mtype, String name, MClass type)
    {
        if (mtype == null)
            throw new IllegalArgumentException("You must provide a modelica type");
        if (name == null)
            throw new IllegalArgumentException("You must provide a name");
        if (type == null)
            throw new IllegalArgumentException("You must provide a modelica");
        MComponent mcomponent = ModelicaFactory.eINSTANCE.createMComponent();
        mcomponent.setName(name);
        mcomponent.setType(type);
        mtype.getComponents().add(mcomponent);
    }
    
    public static void setName(MNamedElement mne, String name)
    {
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        if (name != null)
            for (char c : name.toCharArray())
            {
                if (first)
                {
                    if (Character.isLetter(c) || c == '_')
                    {
                        sb.append(c);
                        first = false;
                    }
                }
                else if (Character.isLetter(c) || c == '_' || Character.isDigit(c))
                {
                    sb.append(c);
                }
            }
        String fname = sb.length() == 0 ? mne.getClass().getName() : sb.toString();
        
        List<MNamedElement> mnes = new ArrayList<MNamedElement>();
        if (mne instanceof MClass && ((MClass) mne).getOwningClass() != null)
        {
            mnes.addAll(((MClass) mne).getOwningClass().getOwnedClasses());
            mnes.addAll(((MClass) mne).getOwningClass().getComponents());
        }
        else if (mne instanceof MComponent && ((MComponent) mne).getOwningClass() != null)
        {
            mnes.addAll(((MComponent) mne).getOwningClass().getOwnedClasses());
            mnes.addAll(((MComponent) mne).getOwningClass().getComponents());
        }
        
        if (!isPresent(fname, mnes))
        {
            mne.setName(fname);
            return;
        }
        int i = 0;
        while (true)
        {
            String nname = fname + (i++);
            if (!isPresent(nname, mnes))
            {
                mne.setName(nname);
                return;
            }
        }
    }
    
    private static boolean isPresent(String name, List<MNamedElement> mnes)
    {
        if (name != null)
            for (MNamedElement mne : mnes)
                if (name.equals(mne.getName()))
                    return true;
        return false;
    }
}
