package com.engisis.sysphs.deserialization.simulink;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import com.engisis.sysphs.language.simscape.SMember;
import com.engisis.sysphs.language.simscape.SMemberAssignment;
import com.engisis.sysphs.language.simscape.SPhysicalBlock;
import com.engisis.sysphs.language.simscape.SPhysicalConnectionPoint;
import com.engisis.sysphs.language.simscape.SPhysicalLine;
import com.engisis.sysphs.language.simscape.SimscapeFactory;
import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SConnectionPoint;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFunction1Block;
import com.engisis.sysphs.language.simulink.SFunction2;
import com.engisis.sysphs.language.simulink.SFunction2Block;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SInterface;
import com.engisis.sysphs.language.simulink.SLine;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SReference;
import com.engisis.sysphs.language.simulink.SSubsystem;
import com.engisis.sysphs.language.simulink.SSystem;
import com.engisis.sysphs.language.simulink.SimulinkFactory;
import com.engisis.sysphs.translation.simulink.SimulinkUtil;
import com.engisis.sysphs.util.Util;
import com.engisis.sysphs.util.XMLLocation;
import com.engisis.sysphs.util.XMLReader;

/**
 * Second-pass parser of a Simulink XML file
 * 
 * @author barbau
 *
 */
public class SimulinkXMLReaderPass2 extends XMLReader implements SimulinkReader
{
    private static final Logger log = Logger.getLogger(SimulinkXMLReaderPass2.class);
    /**
     * Root model
     */
    private SModel smodel;
    /**
     * Map with XML locations as keys, and Simulink objects as values
     */
    private Hashtable<XMLLocation, SElement> objects;
    
    /**
     * Collection of Simulink references
     */
    private Collection<SimulinkReference> references;
    
    /**
     * Input XML stream
     */
    private InputStream is;
    
    /**
     * Constructs a second-pass parser of Simulink file
     * 
     * @param sxmlr
     *            First pass reader
     * @param inputStream
     *            Input stream
     */
    public SimulinkXMLReaderPass2(SimulinkXMLReaderPass1 sxmlr, InputStream inputStream)
    {
        this.objects = sxmlr.getObjects();
        this.references = sxmlr.getReferences();
        is = inputStream;
    }
    
    @Override
    public void process()
    {
        if (objects == null || references == null)
            throw new RuntimeException("Objects and references must be provided to the reader");
        
        // open xml
        try
        {
            XmlPullParserFactory xmlppf = XmlPullParserFactory.newInstance();
            xmlpp = xmlppf.newPullParser();
            xmlpp.setInput(is, null);
            lookForElement("Model", "Library");
            smodel = (SModel) objects.get(getLocation());
            processModelOrLibrary();
        }
        catch (XmlPullParserException e)
        {
            log.error(e);
        }
        catch (IOException e)
        {
            log.error(e);
        }
        finally
        {
            try
            {
                is.close();
            }
            catch (IOException e)
            {
            }
        }
    }
    
    /**
     * Processes a Model or Library element
     * 
     * @return a Model or Library object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SModel processModelOrLibrary() throws XmlPullParserException, IOException
    {
        int depth = xmlpp.getDepth();
        int et = xmlpp.getEventType();
        String ln2;
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                ln2 = xmlpp.getName();
                if (ln2.equals("System"))
                {
                    processSystem();
                }
                else
                    skipToEnd();
            }
        }
        return smodel;
    }
    
    /**
     * Process a System element
     * 
     * @return a System object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SSystem processSystem() throws XmlPullParserException, IOException
    {
        log.info("Parsing system for lines");
        SElement selement = objects.get(getLocation());
        if (!(selement instanceof SSystem))
        {
            log.error("System badly associated");
            return null;
        }
        SSystem ssystem = (SSystem) selement;
        log.info("System " + ssystem);
        
        if (ssystem.getOwningModel() != null)
            SimulinkUtil.setName(ssystem, ssystem.getOwningModel().getName());
        else if (ssystem.getOwningSubsystem() != null)
            SimulinkUtil.setName(ssystem, ssystem.getOwningSubsystem().getName());
        
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        String ln2;
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                ln2 = xmlpp.getName();
                if (ln2.equals("Block"))
                {
                    processBlock();
                }
                else if (ln2.equals("Line"))
                {
                    SLine sline = processLine();
                    if (sline != null)
                    {
                        log.debug("Adding line to system " + ssystem.getName());
                        ssystem.getOwnedLines().add(sline);
                    }
                }
                else
                    skipToEnd();
            }
        }
        
        List<SFunction1Block> sf1s = new LinkedList<SFunction1Block>();
        List<SFunction2Block> sf2s = new LinkedList<SFunction2Block>();
        List<SInport> sis = new LinkedList<SInport>();
        List<SOutport> sos = new LinkedList<SOutport>();
        for (SBlock sb : ssystem.getOwnedBlocks())
        {
            if (sb instanceof SFunction1Block)
                sf1s.add((SFunction1Block) sb);
            else if (sb instanceof SFunction2Block)
                sf2s.add((SFunction2Block) sb);
            else if (sb instanceof SInport)
                sis.add((SInport) sb);
            else if (sb instanceof SOutport)
                sos.add((SOutport) sb);
        }
        
        // only care about SF2 for now
        for (SLine sl : ssystem.getOwnedLines())
        {
            SConnectionPoint src = sl.getSource();
            if (src != null)
                for (SConnectionPoint dst : sl.getDestinations())
                {
                    if (src.getBlock() instanceof SInport && dst.getBlock() instanceof SFunction2Block)
                    {
                        SFunction2 sf2 = ((SFunction2Block) dst.getBlock()).getSfunction();
                        int i = sf2.getInports().indexOf(dst.getPort());
                        if (i != -1)
                        {
                            SFInputVariable sfiv = sf2.getInputs().get(i);
                            sfiv.setInport((SInport) src.getBlock());
                        }
                        else
                            log.error("Couldn't find the block in the SFunction: ");
                        break;
                    }
                    else if (src.getBlock() instanceof SFunction2Block && dst.getBlock() instanceof SOutport)
                    {
                        SFunction2 sf2 = ((SFunction2Block) src.getBlock()).getSfunction();
                        int i = sf2.getOutports().indexOf(src.getPort());
                        if (i != -1)
                        {
                            SFOutputVariable sfov = sf2.getOutputs().get(i);
                            sfov.setOutport((SOutport) dst.getBlock());
                        }
                        else
                            log.error("Couldn't find the block in the SFunction");
                        break;
                    }
                }
        }
        
        return ssystem;
    }
    
    /**
     * Process a Line element
     * 
     * @return a Line object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SLine processLine() throws XmlPullParserException, IOException
    {
        log.info("Parsing line");
        if ("Connection".equals(xmlpp.getAttributeValue(null, "LineType")))
        {
            SPhysicalLine spline = SimscapeFactory.eINSTANCE.createSPhysicalLine();
            int et = xmlpp.getEventType();
            int depth = xmlpp.getDepth();
            while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
            {
                et = xmlpp.next();
                String name = xmlpp.getName();
                if (et == XmlPullParser.START_TAG)
                {
                    if (name.equals("P"))
                    {
                        String name2 = xmlpp.getAttributeValue(null, "Name");
                        if ("Src".equals(name2) || "Dst".equals(name2))
                        {
                            SPhysicalConnectionPoint spcp = getPhysicalConnectionPoint(xmlpp.nextText());
                            if (spcp != null)
                            {
                                log.info("Adding physical point");
                                spline.getPoints().add(spcp);
                            }
                            else
                                log.error("Null physical line point");
                        }
                    }
                }
            }
            return spline;
        }
        SLine sline = SimulinkFactory.eINSTANCE.createSLine();
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            String name = xmlpp.getName();
            if (et == XmlPullParser.START_TAG)
            {
                if (name.equals("P"))
                {
                    String name2 = xmlpp.getAttributeValue(null, "Name");
                    if ("Src".equals(name2))
                    {
                        SConnectionPoint scp = getConnectionPoint(xmlpp.nextText());
                        if (scp != null)
                        {
                            log.info("Adding source");
                            sline.setSource(scp);
                        }
                        else
                            log.error("Null line source");
                    }
                    else if ("Dst".equals(name2))
                    {
                        SConnectionPoint scp = getConnectionPoint(xmlpp.nextText());
                        if (scp != null)
                        {
                            log.info("Adding destination");
                            sline.getDestinations().add(scp);
                        }
                        else
                            log.error("Null line destination");
                    }
                }
                /*
                 * else if (name.equals("Branch")) { int depth2 =
                 * xmlpp.getDepth(); while (et != XmlPullParser.END_TAG ||
                 * depth2 != xmlpp.getDepth()) { et = xmlpp.next(); if (et ==
                 * XmlPullParser.START_TAG && xmlpp.getName().equals("P") &&
                 * "Dst".equals(xmlpp.getAttributeValue(null, "Name"))) {
                 * SConnectionPoint scp = getConnectionPoint(xmlpp.nextText());
                 * sline.getDestinations().add(scp); } } } else skipToEnd();
                 */
            }
        }
        return sline;
    }
    
    /**
     * Process a Block element
     * 
     * @throws XmlPullParserException
     * @throws IOException
     */
    private void processBlock() throws XmlPullParserException, IOException
    {
        log.info("Parsing block");
        SElement selement = objects.get(getLocation());
        if (selement instanceof SPhysicalBlock)
        {
            SPhysicalBlock sblock = (SPhysicalBlock) selement;
            
            int et = xmlpp.getEventType();
            int depth = xmlpp.getDepth();
            while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
            {
                et = xmlpp.next();
                if (et == XmlPullParser.START_TAG)
                {
                    if (xmlpp.getName().equals("P"))
                    {
                        
                        String name = xmlpp.getAttributeValue(null, "Name");
                        SMember sm = sblock.getComponent().getMember(name);
                        if (sm != null)
                        {
                            try
                            {
                                SMemberAssignment sma = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                                sma.getMemberPath().add(sm);
                                double d = Double.parseDouble(xmlpp.nextText().trim());
                                SDoubleValue sdv = SimulinkFactory.eINSTANCE.createSDoubleValue();
                                sdv.setValue(d);
                                sma.setAssignedValue(sdv);
                                sblock.getAssignments().add(sma);
                                log.info("Adding assignment for " + sblock.getName() + "." + sm.getName());
                            }
                            catch (NumberFormatException e)
                            {
                                
                            }
                        }
                        else
                            log.warn("Skipped property " + name);
                    }
                    else
                        skipToEnd();
                }
            }
        }
        else if (selement != null)
        {
            int et = xmlpp.getEventType();
            int depth = xmlpp.getDepth();
            while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
            {
                et = xmlpp.next();
                if (et == XmlPullParser.START_TAG && xmlpp.getName().equals("System"))
                {
                    processSystem();
                }
            }
        }
        
    }
    
    /**
     * Returns a connection point from a string
     * 
     * @param str
     *            string corresponding to the connection point
     * @return a connection point
     */
    private SConnectionPoint getConnectionPoint(String str)
    {
        SConnectionPoint scp = SimulinkFactory.eINSTANCE.createSConnectionPoint();
        String[] spl = str.split("#");
        if (spl.length != 2)
        {
            log.warn("Bad line string: " + str);
            return null;
        }
        SBlock sbl = smodel.getBlockBySID(spl[0]);
        if (sbl == null)
        {
            log.warn("Can't find block " + spl[0]);
            return null;
        }
        scp.setBlock(sbl);
        if (sbl instanceof SInport || sbl instanceof SOutport)
            return scp;
        
        SSystem ss = null;
        if (sbl instanceof SSubsystem)
            ss = ((SSubsystem) sbl).getSystem();
        else if (sbl instanceof SReference)
            ss = ((SReference) sbl).getSystem();
        else if (sbl instanceof SInterface)
            ss = ((SInterface) sbl).getSystem();
        else if (sbl instanceof SFunction2Block)
            ss = ((SFunction2Block) sbl).getSfunction();
        if (ss == null)
        {
            log.debug("The block has no system: " + sbl.getName());
            return scp;
        }
        String[] spl2 = spl[1].split(":");
        if (spl2.length != 2)
        {
            log.warn("Bad line string: " + spl[1]);
            return null;
        }
        try
        {
            int p = Util.toInt(spl2[1], 0);
            if (spl2[0].equals("in"))
            {
                if (p < 1 || p > ss.getInports().size())
                {
                    log.error("Inport index out of bound in block " + sbl.getName() + "/" + ss.getName());
                    return null;
                }
                scp.setPort(ss.getInports().get(p - 1));
            }
            else if (spl2[0].equals("out"))
            {
                if (p < 1 || p > ss.getOutports().size())
                {
                    log.error("Outport index out of bound in block " + sbl.getName() + "/" + ss.getName());
                    return null;
                }
                scp.setPort(ss.getOutports().get(p - 1));
            }
            else
            {
                log.error("Unsupported port " + spl2[0]);
                return null;
            }
        }
        catch (NumberFormatException e)
        {
            log.warn("Bad port number string: " + spl[1]);
            return null;
        }
        return scp;
    }
    
    /**
     * Returns a physical connection point from a string
     * 
     * @param str
     *            a string representing a physical connection point
     * @return a connection point
     */
    private SPhysicalConnectionPoint getPhysicalConnectionPoint(String str)
    {
        SPhysicalConnectionPoint scp = SimscapeFactory.eINSTANCE.createSPhysicalConnectionPoint();
        String[] spl = str.split("#");
        if (spl.length != 2)
        {
            log.warn("Bad line string: " + str);
            return null;
        }
        SBlock sbl = smodel.getBlockBySID(spl[0]);
        SSystem sblocksys = null;
        if (sbl instanceof SPhysicalBlock)
            sblocksys = ((SPhysicalBlock) sbl).getComponent();
        else if (sbl instanceof SSubsystem)
            sblocksys = ((SSubsystem)sbl).getSystem();
        else if (sbl instanceof SReference)
            sblocksys = ((SReference)sbl).getSystem();
        else
        {
            log.warn("Not a physical, subsystem, or reference block: " + sbl.getName());
            return null;
        }
        if (sblocksys == null)
        {
            log.debug("The block has no component or system: " + sbl.getName());
            return null;
        }
        
        scp.setBlock(sbl);
        String[] spl2 = spl[1].split(":");
        if (spl2.length != 2)
        {
            log.warn("Bad line string: " + spl[1]);
            return null;
        }
        
        try
        {
            int p = Integer.parseInt(spl2[1]);
            if (spl2[0].equals("lconn"))
            {
                if (p < 1 || p > sblocksys.getLConnectionPorts().size())
                {
                    log.error("Left connection port index out of bound in component " + sblocksys.getName());
                    return null;
                }
                scp.setPort(sblocksys.getLConnectionPorts().get(p - 1));
            }
            else if (spl2[0].equals("rconn"))
            {
                if (p < 1 || p > sblocksys.getRConnectionPorts().size())
                {
                    log.error("R connection port index out of bound in component " + sblocksys.getName());
                    return null;
                }
                scp.setPort(sblocksys.getRConnectionPorts().get(p - 1));
            }
            else
            {
                log.error("Unsupported port " + spl2[0]);
                return null;
            }
        }
        catch (NumberFormatException e)
        {
            log.warn("Bad port number string: " + spl[1]);
            return null;
        }
        return scp;
    }
    
    @Override
    public Collection<SimulinkReference> getReferences()
    {
        return references;
    }
    
    @Override
    public int getPriority()
    {
        return 11;
    }
    
    @Override
    public SElement getRootElement()
    {
        return smodel;
    }
    
}
