// Adapted from https://www.cqse.eu/en/products/simulink-library-for-java/overview/
grammar MDL;

LBRACE	: '{';
RBRACE	: '}';
LBRACK	: '[';
RBRACK	: ']';
COMMA   : ',';
SEMICOLON   : ';';

file:
    section+
    ;

section :
    IDENTIFIER LBRACE NEWLINE? (section | parameter)* RBRACE NEWLINE?
    ;

parameter :
    IDENTIFIER value NEWLINE|
    IDENTIFIER IDENTIFIER* NEWLINE
    ;

value :
    literal |
    array |
    STRING_LITERAL +
    ;

literal :
    INT_LITERAL |
    FLOAT_LITERAL |
    BOOLEAN_LITERAL
    ;

separator :
    (COMMA | SEMICOLON)
    ;

array :
    LBRACK (literal (separator? literal)*)? RBRACK
    ;

NEWLINE:
    [\r\n]+
    ;

COMMENT:
    '#' ~[\r\n]* [\r\n]+ -> skip
       ;

WS  
    :  [ \t\u000C]+ -> skip
    ;

fragment
IDENTIFIERSTART 
    : [a-zA-Z_#$@]
    ;

IDENTIFIER 
    : IDENTIFIERSTART (IDENTIFIERSTART | DIGIT | [-.])*
    ;

INT_LITERAL:
    DECINTEGER | DECLONG | HEXINTEGER | HEXLONG | OCTINTEGER | OCTLONG
    ;

FLOAT_LITERAL:
    FLOAT | DOUBLE
    ;

BOOLEAN_LITERAL:
    'on' | 'off'
    ;

STRING_LITERAL:
    '"' STRING? '"'
    ;

fragment DIGIT: 
    [0-9]
    ;

INTEGER: 
    ('-'? '0' |'-'? [1-9][0-9]*)
    ;
    
DECINTEGER: 
    INTEGER [uU]?
    ;

DECLONG: 
    INTEGER [lL]?
    ;

HEX: 
    '0' [xX] [0-9a-fA-F]+
    ;

HEXINTEGER:  
    HEX [uU]?
    ;

HEXLONG:  
    HEX [lL]?
    ;

OCT: 
    [0-7]
    ;

OCTINTEGER: 
    [0]+ [1-3]? OCT+ [uU]?
    ;

OCTLONG:
    [0]+ [1]? OCT+ [lL]
    ;

NUMBER:
    ([-]? [0-9]+ '.'? [0-9]+)|
    ([-]? '.'? [0-9]+)
    ;
    
DOUBLE:
    NUMBER ([eE] [+-]? [0-9]+)?
    ;

FLOAT:
    DOUBLE [fF]
    ;

/* string and character literals */
fragment
STRING:
    ~["\\\n]+
    ;