package com.engisis.xmiutil;

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;

/**
 * A Class used to find external references in a given XMI file
 * 
 * @author barbau
 *
 */
public class XMIReferenceFinder implements Runnable
{
    static final Logger log = Logger.getLogger(XMIReferenceFinder.class);
    private HashSet<URI> references;
    // private OutputStreamWriter osw2;
    private XMLStreamReader xmlr;
    private Map<String, String> objectmap;
    private Map<URI, URI> locationmap;
    private List<URI> baseuris;
    
    // static int i = 0;
    /**
     * Main constructor for reference finder
     * 
     * @param inputstream
     *            Given input stream, to read as XMI file
     * @param objectmap
     *            Map between object IDs
     * @param locationmap
     *            Map of URI
     * @param baseuri
     *            Absolute URI to resolve relative URIs
     * @param paths
     *            List of directories to look for references
     * @throws XMLStreamException
     *             if the XML input stream cannot be created
     * @throws FactoryConfigurationError
     *             if the XML factory cannot be created
     */
    public XMIReferenceFinder(InputStream inputstream, Map<String, String> objectmap, Map<URI, URI> locationmap,
            URI baseuri, List<URI> paths) throws XMLStreamException, FactoryConfigurationError
    {
        references = new HashSet<URI>();
        this.objectmap = objectmap;
        this.locationmap = locationmap;
        baseuris = new ArrayList<URI>(paths != null ? paths.size() : 1);
        baseuris.add(baseuri);
        if (paths != null)
            baseuris.addAll(paths);
        
        XMLInputFactory xmlif = XMLInputFactory.newFactory();
        xmlif.setProperty(XMLInputFactory.IS_REPLACING_ENTITY_REFERENCES, Boolean.valueOf(false));
        xmlr = xmlif.createXMLStreamReader(inputstream);
        
    }
    
    @Override
    public void run()
    {
        
        int eventType;
        
        // check if there are more events
        // in the input stream
        try
        {
            while (xmlr.hasNext())
            {
                eventType = xmlr.next();
                switch (eventType)
                {
                case XMLStreamConstants.START_ELEMENT:
                    for (int i = 0; i < xmlr.getAttributeCount(); i++)
                    {
                        String name = xmlr.getAttributeLocalName(i);
                        String value = xmlr.getAttributeValue(i);
                        if ("href".equals(name) && value != null)
                        {
                            // hrefs will be replaced in the filter
                            if (objectmap.containsKey(value))
                                value = objectmap.get(value);
                            // get the file part
                            URI uri = URI.createURI(value).trimFragment().trimQuery();
                            // need to be a last segment
                            if (uri.lastSegment() == null)
                                continue;
                            // if relative URI, try to resolve it to local file
                            if (uri.hasRelativePath())
                                for (URI baseuri : baseuris)
                                {
                                    URI resuri = uri.resolve(baseuri);
                                    if (resuri.isFile() && new File(resuri.toFileString()).exists())
                                    {
                                        String suri = resuri.toString();
                                        if (suri.contains("!") && !suri.startsWith("jar:"))
                                            suri = "jar:" + suri;
                                        references.add(URI.createURI(suri));
                                        break;
                                    }
                                }
                            // if absolute URI, try to find a mapping
                            else
                            {
                                URI urimap = locationmap.get(uri);
                                if (urimap != null)
                                    references.add(urimap);
                                else
                                    log.debug("No location mapping found for " + uri);
                            }
                        }
                    }
                    break;
                default:
                    break;
                }
            }
        }
        catch (XMLStreamException e)
        {
            log.error("Error while parsing the file", e);
        }
    }
    
    /**
     * Returns all references found in the stream
     * 
     * @return All references found in the stream
     */
    public HashSet<URI> getReferences()
    {
        return references;
    }
}
