package com.engisis.xmiutil;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.Location;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;

/**
 * Class to pre-process XMI deserialization, to make sure XMI can be read by
 * EMF. Notably, removes the XMI:Extension tags, adds xmi:type to elements when
 * missing, and performs ID/namespace replacements
 * 
 * @author barbau
 *
 */
public class XMIPreReader implements Runnable
{
    private static final Logger log = Logger.getLogger(XMIPreReader.class);
    
    /**
     * resource et
     */
    private ResourceSet rs;
    /**
     * main stream writer
     */
    private OutputStreamWriter osw;
    /**
     * secondary stream writer (into a file)
     */
    private OutputStreamWriter osw2;
    /**
     * XML reader
     */
    private XMLStreamReader xmlr;
    /**
     * locations where dependencies are looked for
     */
    private List<URI> baseuris;
    /**
     * ID replacement map
     */
    private Hashtable<String, String> idmaps;
    /**
     * namespace replacement map
     */
    private Hashtable<String, String> nsmaps;
    
    /**
     * whether secondary writer is enabled
     */
    private boolean WRITE = false;
    
    /**
     * Create the XMI pre-processor
     * 
     * @param rs
     *            the resource set to rely on
     * @param inputstream
     *            the stream from which the XML file is read
     * @param os
     *            the stream to which the XML file is written into
     * @param baseuri
     *            the URI from which the reference URLs are resolved
     * @param idmaps
     *            ID mappings
     * @param nsmaps
     *            namspace mappings
     * @param paths
     *            paths
     * @throws FileNotFoundException
     * @throws XMLStreamException
     * @throws FactoryConfigurationError
     */
    public XMIPreReader(ResourceSet rs, InputStream inputstream, OutputStream os, URI baseuri,
            Hashtable<String, String> idmaps, Hashtable<String, String> nsmaps, List<URI> paths)
            throws FileNotFoundException, XMLStreamException, FactoryConfigurationError
    {
        this.rs = rs;
        baseuris = new ArrayList<URI>(paths != null ? paths.size() : 1);
        baseuris.add(baseuri);
        if (paths != null)
            baseuris.addAll(paths);
        osw = new OutputStreamWriter(os, Charset.forName("UTF-8").newEncoder());
        if (WRITE)
            osw2 = new OutputStreamWriter(
                    new FileOutputStream(new File(baseuri.appendFileExtension("filtered").lastSegment())));
        
        this.idmaps = idmaps;
        this.nsmaps = nsmaps;
        XMLInputFactory xmlif = XMLInputFactory.newFactory();
        xmlif.setProperty(XMLInputFactory.IS_REPLACING_ENTITY_REFERENCES, Boolean.valueOf(false));
        xmlr = xmlif.createXMLStreamReader(inputstream);
        
    }
    
    /**
     * Executes the reader/writer thread
     */
    @Override
    public void run()
    {
        
        int eventType;
        
        // check if there are more events
        // in the input stream
        int skip = 0;
        try
        {
            while (xmlr.hasNext())
            {
                eventType = xmlr.next();
                switch (eventType)
                {
                case XMLStreamConstants.START_ELEMENT:
                    // manage skip
                    if (skip == 0)
                    {
                        if ("xmi".equals(xmlr.getPrefix()) && "Extension".equals(xmlr.getLocalName()))
                            skip++;
                    }
                    else
                        skip++;
                    if (skip > 0)
                        continue;
                    
                    write("<");
                    String prefix = xmlr.getPrefix();
                    if (prefix != null && !prefix.isEmpty())
                        write(prefix + ":");
                    write(xmlr.getLocalName());
                    
                    String href = "";
                    for (int i = 0; i < xmlr.getAttributeCount(); i++)
                    {
                        write(" ");
                        prefix = xmlr.getAttributePrefix(i);
                        String name = xmlr.getAttributeLocalName(i);
                        String value = xmlr.getAttributeValue(i);
                        
                        // fix for MD: only write the fragment if local
                        // authority
                        if ("href".equals(name))
                        {
                            if (idmaps != null && idmaps.containsKey(value))
                            {
                                // log.debug("Replacing " + value + " by " +
                                // objmappings.get(value));
                                value = idmaps.get(value);
                            }
                            URI uriref = URI.createURI(value);
                            if ("local".equals(uriref.scheme()))
                                value = "#" + uriref.fragment();
                            // replace all relative URIs by absolute
                            if (uriref.isRelative() && !uriref.hasEmptyPath())
                            {
                                boolean changed = false;
                                for (URI baseuri : baseuris)
                                {
                                    URI resuriref = uriref.resolve(baseuri);
                                    if (resuriref.isFile() && new File(resuriref.toFileString()).exists())
                                    {
                                        value = resuriref.toString();
                                        log.debug("Changing " + uriref + " to " + resuriref);
                                        changed = true;
                                        break;
                                    }
                                }
                                if (!changed)
                                    log.warn("Could not revolve relative uri " + uriref);
                            }
                        }
                        
                        // write attribute
                        if (prefix != null && !prefix.isEmpty())
                            write(prefix + ":");
                        value = escape(value);
                        write(name + "=\"" + value + "\"");
                        
                        // href==null if xmi:type already found
                        if ("href".equals(name) && href != null)
                        {
                            // store href to be looked for xmi:type
                            href = value;
                        }
                        // if type indicated, no need to store
                        if ("type".equals(name) && "xmi".equals(prefix))
                            href = null;
                    }
                    // add xmi:type corresponding to href
                    if (href != null && !href.isEmpty())
                    {
                        boolean written = false;
                        // add xmi:type if possible
                        URI urihref = URI.createURI(href);
                        // first approach, used for local references
                        if (!urihref.hasEmptyPath())
                        {
                            if (urihref.isFile())
                            {
                                EObject eo = rs.getEObject(urihref, true);
                                if (eo != null)
                                {
                                    EClass ec = eo.eClass();
                                    // TODO: find a better way
                                    write(" xmi:type=\"uml:" + ec.getName() + "\"");
                                    written = true;
                                }
                            }
                            // second approach, for mapped references
                            else
                                for (URI mapped : rs.getURIConverter().getURIMap().keySet())
                                {
                                    if (urihref.toString().startsWith(mapped.toString()))
                                    {
                                        EObject eo = rs.getEObject(urihref, true);
                                        if (eo == null)
                                        {
                                            log.error("Unable to get " + urihref);
                                            continue;
                                        }
                                        EClass ec = eo.eClass();
                                        // TODO: find a better way
                                        write(" xmi:type=\"uml:" + ec.getName() + "\"");
                                        written = true;
                                        break;
                                    }
                                }
                        }
                        if (!written)
                        {
                            Location loc = xmlr.getLocation();
                            log.warn("Couldn't find the xmi:type for element at " + loc.getLineNumber() + ":"
                                    + loc.getColumnNumber());
                        }
                    }
                    for (int i = 0; i < xmlr.getNamespaceCount(); i++)
                    {
                        // just because of the wrong URI used by MD
                        String namespace = xmlr.getNamespaceURI(i);
                        if (nsmaps != null)
                        {
                            String rep = nsmaps.get(namespace);
                            // if
                            // (namespace.equals("http://www.omg.org/spec/SysML/20120401/SysML"))
                            // namespace =
                            // "http://www.omg.org/spec/SysML/20120322/SysML";
                            if (rep != null)
                                namespace = rep;
                        }
                        write(" xmlns:" + xmlr.getNamespacePrefix(i) + "=\"" + namespace + "\"");
                    }
                    
                    write(">");
                    break;
                case XMLStreamConstants.END_ELEMENT:
                    if (skip > 0)
                    {
                        skip--;
                        continue;
                    }
                    write("</");
                    if (xmlr.getPrefix() != null && !xmlr.getPrefix().isEmpty())
                        write(xmlr.getPrefix() + ":");
                    write(xmlr.getLocalName() + ">");
                    break;
                
                case XMLStreamConstants.PROCESSING_INSTRUCTION:
                    if (skip > 0)
                        continue;
                    write("<?" + xmlr.getText() + "?>");
                    break;
                case XMLStreamConstants.CHARACTERS:
                    if (skip > 0)
                        continue;
                    write(escape(new String(xmlr.getTextCharacters(), xmlr.getTextStart(), xmlr.getTextLength())));
                    break;
                case XMLStreamConstants.COMMENT:
                    break;
                case XMLStreamConstants.START_DOCUMENT:
                    write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
                    break;
                case XMLStreamConstants.END_DOCUMENT:
                    osw.close();
                    if (WRITE)
                        osw2.close();
                    break;
                case XMLStreamConstants.ENTITY_REFERENCE:
                    if (skip > 0)
                        continue;
                    write("&" + xmlr.getLocalName() + ";");
                    break;
                case XMLStreamConstants.ATTRIBUTE:
                    break;
                case XMLStreamConstants.DTD:
                    break;
                case XMLStreamConstants.CDATA:
                    if (skip > 0)
                        continue;
                    write("<![CDATA[" + new String(xmlr.getTextCharacters(), xmlr.getTextStart(), xmlr.getTextLength())
                            + "]]>");
                    break;
                case XMLStreamConstants.SPACE:
                    break;
                default:
                    break;
                }
            }
        }
        catch (XMLStreamException e)
        {
            log.error("Error while parsing the file", e);
        }
        catch (IOException e)
        {
            log.error("Error while parsing the file", e);
        }
        catch (Exception e)
        {
            log.error("Error while parsing the file", e);
        }
        
    }
    
    /**
     * Writes the specified string into the writer
     * 
     * @param str
     *            string to write
     * @throws IOException
     */
    private void write(String str) throws IOException
    {
        osw.write(str);
        osw.flush();
        if (WRITE)
        {
            osw2.write(str);
            osw2.flush();
        }
    }
    
    /**
     * Escapes the specified string
     * 
     * @param str
     *            string to escape
     * @return escaped string
     */
    private static String escape(String str)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < str.length(); i++)
        {
            char ch = str.charAt(i);
            switch (ch)
            {
            case '<':
                sb.append("&lt;");
                break;
            case '>':
                sb.append("&gt;");
                break;
            case '&':
                sb.append("&amp;");
                break;
            case '"':
                sb.append("&quot;");
                break;
            default:
                sb.append(ch);
            }
        }
        return sb.toString();
    }
}
