package com.engisis.xmiutil;

import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * SysML utility class
 * 
 * @author barbau
 *
 */
public class SysMLUtil extends UMLUtil
{
    private static final Logger log = Logger.getLogger(SysMLUtil.class);
    
    private Stereotype sblock;
    private Stereotype sconstraintblock;
    private Stereotype sbindingconnector;
    private Stereotype snestedconnectorend;
    private Stereotype svaluetype;
    private Class sunit;
    private Class squantitykind;
    private Stereotype spropertyspecifictype;
    private Stereotype sflowproperty;
    private Stereotype sinterfaceblock;
    private Stereotype sdirectedfeature;
    private Stereotype sconnectorproperty;
    private Stereotype sparticipantproperty;
    private Stereotype sadjunctproperty;
    
    private DataType sstring;
    private DataType sboolean;
    private DataType sinteger;
    private DataType sdouble;
    
    /**
     * Namespace of latest SysML version
     */
    public static final URI nsSysML = URI.createURI("http://www.omg.org/spec/SysML/20181001/SysML");
    /**
     * URL of latest SysML version
     */
    public static final URI urlSysML = URI.createURI("http://www.omg.org/spec/SysML/20181001/SysML.xmi");
    
    /**
     * Constructs an utility object for the specified resource set
     * 
     * @param rs
     *            resource set in which the models are loaded
     * @throws UMLModelErrorException
     */
    public SysMLUtil(ResourceSet rs) throws UMLModelErrorException
    {
        super(rs);
        loadSysML();
    }
    
    /**
     * Loads the SysML stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysML() throws UMLModelErrorException
    {
        // get and check all the stereotypes
        sblock = loadStereotype(nsSysML, "Block");
        sconstraintblock = loadStereotype(nsSysML, "ConstraintBlock");
        sbindingconnector = loadStereotype(nsSysML, "BindingConnector");
        snestedconnectorend = loadStereotype(nsSysML, "NestedConnectorEnd");
        svaluetype = loadStereotype(nsSysML, "ValueType");
        sunit = loadClass(nsSysML, "Unit");
        squantitykind = loadClass(nsSysML, "QuantityKind");
        spropertyspecifictype = loadStereotype(nsSysML, "PropertySpecificType");
        sflowproperty = loadStereotype(nsSysML, "FlowProperty");
        sinterfaceblock = loadStereotype(nsSysML, "InterfaceBlock");
        sdirectedfeature = loadStereotype(nsSysML, "DirectedFeature");
        sconnectorproperty = loadStereotype(nsSysML, "ConnectorProperty");
        sparticipantproperty = loadStereotype(nsSysML, "ParticipantProperty");
        sadjunctproperty = loadStereotype(nsSysML, "AdjunctProperty");
        
        sstring = loadDataType(urlSysML, "SysML_dataType.String");
        sboolean = loadDataType(urlSysML, "SysML_dataType.Boolean");
        sinteger = loadDataType(urlSysML, "SysML_dataType.Integer");
        sdouble = loadDataType(urlSysML, "SysML_dataType.Real");
        
    }
    
    /**
     * Loads the stereotype with the specified namespace and name into the
     * resource set
     * 
     * @param rs
     *            resource set in which to load the stereotype
     * @param namespace
     *            namespace of the stereotype
     * @param name
     *            name of the stereotype
     * @return loaded stereotype
     * @throws UMLModelErrorException
     */
    protected Stereotype loadStereotype(URI namespace, String name) throws UMLModelErrorException
    {
        NamedElement namedelement = loadFromNamespaceAndName(namespace, name);
        if (!(namedelement instanceof Stereotype))
            throw new ClassCastException("The element named " + name + " in " + namespace + " is not a stereotype");
        return (Stereotype) namedelement;
    }
    
    /**
     * Loads the class with the specified namespace and name into the resource
     * set
     * 
     * @param rs
     *            resource set in which to load the class
     * @param namespace
     *            namespace of the class
     * @param name
     *            name of the class
     * @return loaded class
     * @throws UMLModelErrorException
     */
    private Class loadClass(URI namespace, String name) throws UMLModelErrorException
    {
        Element element = loadFromNamespaceAndName(namespace, name);
        if (!(element instanceof Class))
            throw new ClassCastException("The element named " + name + " in " + namespace + " is not a class");
        return (Class) element;
    }
    
    /**
     * Returns SysML Block
     * 
     * @return SysML Block
     */
    public Stereotype getBlock()
    {
        return sblock;
    }
    
    /**
     * Returns SysML ConstraintBlock
     * 
     * @return SysML ConstraintBlock
     */
    public Stereotype getConstraintBlock()
    {
        return sconstraintblock;
    }
    
    /**
     * Returns SysML BindingConnector
     * 
     * @return SysML BindingConnector
     */
    public Stereotype getBindingConnector()
    {
        return sbindingconnector;
    }
    
    /**
     * Returns SysML NestedConnectorEnd
     * 
     * @return SysML NestedConnectorEnd
     */
    public Stereotype getNestedConnectorEnd()
    {
        return snestedconnectorend;
    }
    
    /**
     * Returns SysML ValueType
     * 
     * @return SysML ValueType
     */
    public Stereotype getValueType()
    {
        return svaluetype;
    }
    
    /**
     * Returns SysML Unit
     * 
     * @return SysML Unit
     */
    public Class getUnit()
    {
        return sunit;
    }
    
    /**
     * Returns SysML QuantityKind
     * 
     * @return SysML QuantityKind
     */
    public Class getQuantityKind()
    {
        return squantitykind;
    }
    
    /**
     * Returns SysML PropertySpecificType
     * 
     * @return SysML PropertySpecificType
     */
    public Stereotype getPropertySpecificType()
    {
        return spropertyspecifictype;
    }
    
    /**
     * Returns SysML FlowProperty
     * 
     * @return SysML FlowProperty
     */
    public Stereotype getFlowProperty()
    {
        return sflowproperty;
    }
    
    /**
     * Returns SysML FlowProperty in direction
     * 
     * @return SysML FlowProperty in direction
     */
    public EnumerationLiteral getFlowPropertyIn()
    {
        return ((Enumeration) getFlowProperty().getOwnedAttribute("direction", null).getType()).getOwnedLiteral("in");
    }
    
    /**
     * Returns SysML FlowProperty out direction
     * 
     * @return SysML FlowProperty out direction
     */
    public EnumerationLiteral getFlowPropertyOut()
    {
        return ((Enumeration) getFlowProperty().getOwnedAttribute("direction", null).getType()).getOwnedLiteral("out");
    }
    
    /**
     * Returns SysML FlowProperty inout direction
     * 
     * @return SysML FlowProperty inout direction
     */
    public EnumerationLiteral getFlowPropertyInout()
    {
        return ((Enumeration) getFlowProperty().getOwnedAttribute("direction", null).getType())
                .getOwnedLiteral("inout");
    }
    
    /**
     * Returns SysML InterfaceBlock
     * 
     * @return SysML InterfaceBlock
     */
    public Stereotype getInterfaceBlock()
    {
        return sinterfaceblock;
    }
    
    /**
     * Returns SysML DirectedFeature
     * 
     * @return SysML DirectedFeature
     */
    public Stereotype getDirectedFeature()
    {
        return sdirectedfeature;
    }
    
    /**
     * Returns SysML ConnectorProperty
     * 
     * @return SysML ConnectorProperty
     */
    public Stereotype getConnectorProperty()
    {
        return sconnectorproperty;
    }
    
    /**
     * Returns SysML ParticipantProperty
     * 
     * @return SysML ParticipantProperty
     */
    public Stereotype getParticipantProperty()
    {
        return sparticipantproperty;
    }
    
    /**
     * Returns SysML AdjunctProperty
     * 
     * @return SysML AdjunctProperty
     */
    public Stereotype getAdjunctProperty()
    {
        return sadjunctproperty;
    }
    
    /**
     * Returns SysML String
     * 
     * @return SysML String
     */
    public DataType getSysMLString()
    {
        return sstring;
    }
    
    /**
     * Returns SysML Boolean
     * 
     * @return SysML Boolean
     */
    public DataType getSysMLBoolean()
    {
        return sboolean;
    }
    
    /**
     * Returns SysML Integer
     * 
     * @return SysML Integer
     */
    public DataType getSysMLInteger()
    {
        return sinteger;
    }
    
    /**
     * Returns SysML Double
     * 
     * @return SysML Double
     */
    public DataType getSysMLDouble()
    {
        return sdouble;
    }
    
    /**
     * Returns the path of a connector end
     * 
     * @param end
     *            connector end
     * @return list of property (path)
     */
    public List<Property> getPropertyPath(ConnectorEnd end)
    {
        List<Property> result = new ArrayList<Property>();
        if (end.isStereotypeApplied(snestedconnectorend))
        {
            @SuppressWarnings("unchecked")
            List<Object> properties = (List<Object>) (end.getValue(snestedconnectorend, "propertyPath"));
            for (Object property : properties)
                result.add((Property) property);
        }
        else if (end.getPartWithPort() != null)
            result.add(end.getPartWithPort());
        if (!(end.getRole() instanceof Property))
            throw new IllegalArgumentException("The connector role " + end.getRole() + " should be a property");
        result.add((Property) end.getRole());
        return result;
    }
    
    /**
     * Updates the connector end with the given path
     * 
     * @param end
     *            connector end to update
     * @param lp
     *            path for the connector end
     */
    public void updatePropertyPath(ConnectorEnd end, List<Property> lp)
    {
        if (lp.size() > 0)
            end.setRole(lp.get(lp.size() - 1));
        if (lp.size() > 1)
        {
            if (!end.isStereotypeApplied(getNestedConnectorEnd()))
                end.applyStereotype(getNestedConnectorEnd());
            @SuppressWarnings("unchecked")
            List<Property> pp = (List<Property>) end.getValue(getNestedConnectorEnd(), "propertyPath");
            if (pp == null)
            {
                pp = new ArrayList<Property>(lp.size() - 1);
                end.setValue(getNestedConnectorEnd(), "propertyPath", pp);
            }
            pp.clear();
            for (int i = 0; i < lp.size() - 1; i++)
                pp.add(lp.get(i));
        }
    }
    
    /**
     * Returns all the owned flow properties.
     * 
     * @param uclassifier
     *            classifier in which the flow properties are looked for
     * @return list of flow properties
     */
    public Set<Property> getOwnedFlowProperties(Classifier uclassifier)
    {
        Set<Property> lp = getOwnedAttributes(uclassifier);
        LinkedHashSet<Property> properties = new LinkedHashSet<Property>();
        for (Property uproperty : lp)
        {
            if (uproperty.isStereotypeApplied(sflowproperty) && uproperty.getType() != null)
                properties.add(uproperty);
        }
        return properties;
    }
    
    /**
     * Returns all the flow properties, even inherited and accounting for
     * redefinitions.
     * 
     * @param uclassifier
     *            classifier in which the flow properties are looked for
     * @return list of flow properties
     */
    public Set<Property> getAllFlowProperties(Classifier uclassifier)
    {
        Set<Property> lp = getAllCorrectedAttributes(uclassifier);
        
        LinkedHashSet<Property> ret = new LinkedHashSet<Property>();
        for (Property uproperty : lp)
        {
            if (uproperty.isStereotypeApplied(sflowproperty) && uproperty.getType() != null)
                ret.add(uproperty);
        }
        return ret;
    }
    
    /**
     * Returns the property path corrected by redefinitions
     * 
     * @param ce
     *            Connector end with nested properties
     * @param context
     *            Classifier context from which the property path is corrected
     * @return List of properties corresponding to the connector end path
     */
    public List<Property> getCorrectedPropertyPath(ConnectorEnd ce, Classifier context)
    {
        if (ce == null)
            throw new IllegalArgumentException("The connector end can't be null");
        if (context == null)
            throw new IllegalArgumentException("The context end can't be null");
        List<Property> lp = getPropertyPath(ce);
        List<Property> ret = new ArrayList<Property>(lp.size());
        Classifier c = context;
        for (Property p : lp)
        {
            Property p2 = correctProperty(p, c);
            ret.add(p2);
            if (ret.size() != lp.size())
            {
                if (p2.getType() instanceof Classifier)
                    c = (Classifier) p2.getType();
                else
                {
                    log.error("The property has no classifier type: " + p2.getQualifiedName());
                    break;
                }
            }
        }
        return ret;
    }
    
    /**
     * Creates a basic SysML model
     * 
     * @param rs
     *            resource set in which the resource is created
     * @param file
     *            location of the model
     * @param paths
     *            List of directories
     * @return resource representing the model
     */
    public static Resource createSysMLModel(ResourceSet rs, URI file, List<URI> paths)
    {
        Resource r = UMLUtil.createUMLModel(rs, file, paths);
        
        Model model = (Model) r.getContents().stream().filter(e -> e instanceof Model).findFirst().orElse(null);
        if (model != null)
        {
            URL fsysml = SysMLUtil.class.getResource("/SysML.xmi");
            if (fsysml != null)
            {
                Resource sysml = EMFUtil.loadResource(rs, URI.createURI(fsysml.toString()), paths);
                Profile sysmlprofile = (Profile) EcoreUtil.getObjectByType(sysml.getContents(),
                        UMLPackage.Literals.PROFILE);
                model.applyProfile(sysmlprofile);
            }
            else
                log.error("Unable to locate the SysML profile file in the path");
        }
        return r;
    }
    
}
