package com.engisis.xmiutil;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Stack;

import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLStreamException;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EPackage.Registry;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.impl.DynamicEObjectImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xml.type.impl.AnyTypeImpl;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.UMLPlugin;
import org.eclipse.uml2.uml.profile.standard.StandardPackage;
import org.eclipse.uml2.uml.resource.UMLResource;
import org.eclipse.uml2.uml.resources.util.UMLResourcesUtil;

/**
 * Utility class for EMF
 * 
 * @author barbau
 *
 */
public class EMFUtil
{
    private static final Logger log = Logger.getLogger(EMFUtil.class);
    
    /**
     * Create a basic ResourceSet, without URL mappings
     * 
     * @return resource set
     * @throws FileNotFoundException
     *             if the UML2 Resource library is not in a jar
     */
    public static ResourceSet createBasicResourceSet() throws FileNotFoundException
    {
        // setup the ResourceSet environment
        ResourceSet rs = new ResourceSetImpl();
        rs.getLoadOptions().put(XMLResource.OPTION_RECORD_UNKNOWN_FEATURE, Boolean.TRUE);
        rs.getLoadOptions().put(XMLResource.OPTION_RECORD_ANY_TYPE_NAMESPACE_DECLARATIONS, Boolean.TRUE);
        rs.getLoadOptions().put(XMLResource.OPTION_DEFER_IDREF_RESOLUTION, Boolean.valueOf(false));
        Registry registry = rs.getPackageRegistry();
        
        // Namespace used by Papyrus
        registry.put(UMLPackage.eNS_URI, UMLPackage.eINSTANCE);
        
        Map<URI, URI> urimap = rs.getURIConverter().getURIMap();
        
        // Where to get the resources from (from UMLResourceUtil, JAR assumed)
        java.lang.Class<UMLResourcesUtil> umlre = UMLResourcesUtil.class;
        String url = umlre.getResource("/" + umlre.getName().replace('.', '/') + ".class").toExternalForm();
        if (url.indexOf("!/") == -1)
            throw new FileNotFoundException("The UML2 resource library must be in a JAR: " + url);
        
        URI umljar = URI.createURI(url.substring(0, url.lastIndexOf('!') + 2));
        
        UMLPlugin.getEPackageNsURIToProfileLocationMap().put(StandardPackage.eNS_URI,
                umljar.appendSegment("profiles").appendSegment("Standard.profile.uml").appendFragment("_0"));
        
        // map locations used in UML profiles and libraries
        urimap.put(URI.createURI(UMLResource.UML_METAMODEL_URI),
                umljar.appendSegment("metamodels").appendSegment("UML.metamodel.uml"));
        urimap.put(URI.createURI(UMLResource.UML_PRIMITIVE_TYPES_LIBRARY_URI),
                umljar.appendSegment("libraries").appendSegment("UMLPrimitiveTypes.library.uml"));
        
        // uris of MD17 and MD18
        String[] umluris = new String[] { "http://www.omg.org/spec/UML/20110701",
                "http://www.omg.org/spec/UML/20131001", "http://www.omg.org/spec/UML/20161101" };
        for (String umluri : umluris)
        {
            registry.put(umluri, UMLPackage.eINSTANCE);
            EPackage.Registry.INSTANCE.put(umluri, UMLPackage.eINSTANCE);
            
            // register standard profile namespace
            registry.put(umluri + "/StandardProfile", StandardPackage.eINSTANCE);
            
            // register standard profile location
            UMLPlugin.getEPackageNsURIToProfileLocationMap().put(umluri + "/StandardProfile",
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml").appendFragment("_0"));
            
            // register metamodel and library locations
            urimap.put(URI.createURI(umluri + "/UML.xmi"),
                    umljar.appendSegment("metamodels").appendSegment("UML.metamodel.uml"));
            urimap.put(URI.createURI(umluri + "/PrimitiveTypes.xmi"),
                    umljar.appendSegment("libraries").appendSegment("UMLPrimitiveTypes.library.uml"));
            urimap.put(URI.createURI(umluri + "/StandardProfile.xmi"),
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml"));
            urimap.put(URI.createURI(umluri + "/StandardProfileL2.xmi"),
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml"));
            urimap.put(URI.createURI(umluri + "/StandardProfileL3.xmi"),
                    umljar.appendSegment("profiles").appendSegment("Standard.profile.uml"));
        }
        
        Map<String, Object> extension2factory = rs.getResourceFactoryRegistry().getExtensionToFactoryMap();
        
        // Extension used by Papyrus
        extension2factory.put(UMLResource.FILE_EXTENSION, UMLResource.Factory.INSTANCE);
        // Extension used by MD
        extension2factory.put("xmi", UMLResource.Factory.INSTANCE);
        extension2factory.put("xml", UMLResource.Factory.INSTANCE);
        
        // Old way
        
        // Mapping between the pathmaps used in Papyrus and the actual models
        urimap.put(URI.createURI(UMLResource.LIBRARIES_PATHMAP), umljar.appendSegment("libraries").appendSegment(""));
        urimap.put(URI.createURI(UMLResource.METAMODELS_PATHMAP), umljar.appendSegment("metamodels").appendSegment(""));
        urimap.put(URI.createURI(UMLResource.PROFILES_PATHMAP), umljar.appendSegment("profiles").appendSegment(""));
        
        return rs;
    }
    
    /**
     * Creates an EMF resource set with URL mappings
     * 
     * @return a ResourceSet able to read OMG UML files
     * @throws FileNotFoundException
     */
    public static ResourceSet createResourceSet() throws FileNotFoundException
    {
        ResourceSet rs = createBasicResourceSet();
        Map<URI, URI> urimap = rs.getURIConverter().getURIMap();
        for (Entry<URI, URI> localmapentry : EMFConfiguration.getLocationMappings().entrySet())
            urimap.put(localmapentry.getKey(), localmapentry.getValue());
        return rs;
    }
    
    /**
     * 
     * @param rs
     *            Resource set in which all resources are loaded
     * @param modeluri
     *            URI of the main model being loaded
     * @param paths
     *            List of directories where relative references are looked for
     * @return Resource corresponding to the main model
     */
    public static Resource loadResourceWithDependencies(ResourceSet rs, URI modeluri, List<URI> paths)
    {
        // The files need to be processed in order because of the stereotypes
        // This part determines the order (simply put dependencies at the end)
        
        Resource resource = null;
        
        // stack of URIs to parse
        Stack<URI> toprepare = new Stack<URI>();
        
        // add main model
        toprepare.push(modeluri);
        
        // build map model -> dependencies
        Hashtable<URI, HashSet<URI>> deps = new Hashtable<URI, HashSet<URI>>();
        
        // get all dependencies
        while (toprepare.size() != 0)
        {
            URI input = toprepare.pop();
            log.info("Analyzing " + input);
            try (InputStream is = rs.getURIConverter().createInputStream(input))
            {
                XMIReferenceFinder xmir = new XMIReferenceFinder(is, EMFConfiguration.getReaderIDMappings(),
                        EMFConfiguration.getLocationMappings(), modeluri, paths);
                xmir.run();
                is.close();
                for (URI uriref : xmir.getReferences())
                    log.debug("Found reference " + uriref);
                deps.put(input, xmir.getReferences());
                // get the external references, and make sure they are loaded
                // before
                for (URI ref : xmir.getReferences())
                {
                    log.debug("Dependency between " + input.lastSegment() + " and " + ref.lastSegment());
                    if (!deps.containsKey(ref))
                        toprepare.push(ref);
                }
            }
            catch (IOException e)
            {
                log.error("Couldn't parse the XMI for dependencies", e);
            }
            catch (XMLStreamException e)
            {
                log.error("Couldn't parse the XMI for dependencies", e);
            }
            catch (FactoryConfigurationError e)
            {
                log.error("Couldn't parse the XMI for dependencies", e);
            }
        }
        
        // List of URIs to load
        List<URI> toload = new ArrayList<URI>();
        // create loading list in the right order
        // put the sources first
        while (toload.size() != deps.keySet().size())
        {
            boolean action = false;
            // inefficient
            loop: for (URI uri : deps.keySet())
            {
                if (!toload.contains(uri))
                {
                    // URI not loaded yet
                    // skip if one of the dependencies is not loaded
                    for (URI ref : deps.get(uri))
                        if (!toload.contains(ref) && !ref.equals(uri))
                            continue loop;
                    log.debug("Added " + uri.lastSegment() + " to load list");
                    toload.add(uri);
                    action = true;
                }
            }
            // break if no action was taken and loop not finished
            if (!action)
            {
                log.error("Dependency resolution problem");
                break;
            }
        }
        // This part filters and loads the files in the order
        for (int i = 0; i < toload.size(); i++)
        {
            resource = EMFUtil.loadResource(rs, toload.get(i), paths);
        }
        
        // To load the indirect profiles, may be unnecessary
        EcoreUtil.resolveAll(rs);
        
        return resource;
    }
    
    /**
     * Filters and loads a file. Also, Defines UML profiles.
     * 
     * @param rs
     *            the ResourceSet in which the UML file should be loaded
     * @param fileuri
     *            the location of the file
     * @param paths
     *            List of directories to look for relative references
     * @return loaded resource
     */
    public static Resource loadResource(final ResourceSet rs, URI fileuri, List<URI> paths)
    {
        log.info("Loading file " + fileuri);
        Resource r = rs.createResource(fileuri);
        try
        {
            // is->pos->pis->loading
            PipedInputStream pis = new PipedInputStream(2048);
            PipedOutputStream pos = new PipedOutputStream(pis);
            InputStream is = rs.getURIConverter().createInputStream(fileuri);
            XMIPreReader xmif = new XMIPreReader(rs, is, pos, fileuri, EMFConfiguration.getReaderIDMappings(),
                    EMFConfiguration.getReaderNSMappings(), paths);
            
            Thread th = new Thread(xmif);
            // feeds pos
            th.start();
            // consumes pos
            try
            {
                r.load(pis, null);
            }
            catch (Exception e)
            {
                log.error("Error while loading " + fileuri, e);
            }
            
            pis.close();
            pos.close();
            is.close();
            
            log.debug("Resource " + r + " loaded");
            
            // profiles to load
            LinkedList<Profile> profiles = new LinkedList<Profile>();
            
            // profiles to apply
            Hashtable<Package, LinkedList<Profile>> applications = new Hashtable<Package, LinkedList<Profile>>();
            
            // parse content
            TreeIterator<EObject> allcontent = r.getAllContents();
            while (allcontent.hasNext())
            {
                EObject eo = allcontent.next();
                if (!(eo instanceof Package))
                {
                    allcontent.prune();
                    continue;
                }
                Package p = (Package) eo;
                
                LinkedList<Profile> lpa = new LinkedList<Profile>();
                applications.put(p, lpa);
                
                for (Profile prof : p.getAppliedProfiles())
                {
                    lpa.add(prof);
                }
                if (p instanceof Profile)
                {
                    Profile prof = (Profile) p;
                    profiles.add(prof);
                }
            }
            
            // flatten and define
            for (Profile prof : profiles)
            {
                flatten(prof);
                String profURI = prof.getURI();
                if (!prof.isDefined())
                {
                    EPackage definition = prof.define();
                    log.debug("Definition: " + definition.getName() + "  with " + definition.getNsPrefix() + " :"
                            + definition.getNsURI() + " at " + EcoreUtil.getURI(definition) + "--"
                            + EcoreUtil.getURI(prof));
                    
                    log.debug("Registering " + profURI + " for " + definition.getNsURI());
                    UMLPlugin.getEPackageNsURIToProfileLocationMap().put(profURI, EcoreUtil.getURI(definition));
                    rs.getPackageRegistry().put(profURI, definition);
                    String alt = EMFConfiguration.getReaderNSMappings().get(profURI);
                    if (alt != null && !alt.equals(profURI))
                    {
                        log.debug("Registering alternative " + alt + " for " + definition.getNsURI());
                        UMLPlugin.getEPackageNsURIToProfileLocationMap().put(alt, EcoreUtil.getURI(definition));
                        rs.getPackageRegistry().put(alt, definition);
                    }
                }
                else
                    log.debug("Profile " + profURI + " defined");
            }
            // attempt to recreate stereotype applications
            List<DynamicEObjectImpl> ldeos = new LinkedList<DynamicEObjectImpl>();
            for (EObject eo : r.getContents())
            {
                if (eo instanceof AnyTypeImpl)
                {
                    AnyTypeImpl at = (AnyTypeImpl) eo;
                    String name = at.eClass().getName();
                    for (Profile profile : profiles)
                    {
                        for (EClassifier ec : profile.getDefinition().getEClassifiers())
                        {
                            if (ec instanceof EClass && ec.getName().equals(name))
                            {
                                DynamicEObjectImpl deo = new DynamicEObjectImpl((EClass) ec);
                                ldeos.add(deo);
                                for (FeatureMap.Entry fm : at.getAnyAttribute())
                                {
                                    String attname = fm.getEStructuralFeature().getName();
                                    EStructuralFeature esf = ((EClass) ec).getEStructuralFeature(attname);
                                    if (esf != null)
                                    {
                                        if (esf instanceof EAttribute)
                                        {
                                            String etypename = ((EAttribute) esf).getEType().getInstanceClassName();
                                            try
                                            {
                                                if (etypename.equals("boolean"))
                                                {
                                                    deo.eSet(esf, Util.toBoolean(fm.getValue().toString(), false));
                                                }
                                                else if (etypename.equals("int"))
                                                {
                                                    deo.eSet(esf, Util.toInt(fm.getValue().toString(), 0));
                                                }
                                                else if (etypename.equals("double"))
                                                {
                                                    deo.eSet(esf, Util.toDouble(fm.getValue().toString(), 0.0));
                                                }
                                                else if (etypename.equals("java.lang.String"))
                                                {
                                                    deo.eSet(esf, fm.getValue().toString());
                                                }
                                                else
                                                    log.warn("Unsupported type: " + etypename);
                                            }
                                            catch (Exception e)
                                            {
                                                log.warn("Can't assign value " + fm.getValue() + " to " + esf);
                                            }
                                            
                                            continue;
                                        }
                                        else if (esf instanceof EReference)
                                        {
                                            deo.eSet(esf, r.getEObject(fm.getValue().toString()));
                                            continue;
                                        }
                                    }
                                    log.error("Coundn't set value of " + ec + "::" + esf);
                                }
                            }
                        }
                    }
                }
            }
            r.getContents().addAll(ldeos);
            /*
             * for (Profile prof : profiles) { flatten(prof); String profURI =
             * prof.getURI(); if (!prof.isDefined()) { EPackage definition =
             * prof.define(); log.debug("Definition: " + definition.getName() +
             * "  with " + definition.getNsPrefix() + " :" +
             * definition.getNsURI() + " at " + EcoreUtil.getURI(definition) +
             * "--" + EcoreUtil.getURI(prof));
             * 
             * log.debug("Registering " + profURI + " for " +
             * definition.getNsURI());
             * UMLPlugin.getEPackageNsURIToProfileLocationMap().put(profURI,
             * EcoreUtil.getURI(definition));
             * rs.getPackageRegistry().put(profURI, definition); if
             * (getNamespaceMappings().get(profURI) != null) for (String s :
             * getNamespaceMappings().get(profURI)) { log.debug("Registering " +
             * s + " for " + definition.getNsURI());
             * UMLPlugin.getEPackageNsURIToProfileLocationMap().put(s,
             * EcoreUtil.getURI(definition)); rs.getPackageRegistry().put(s,
             * definition); }
             * 
             * 
             * } else log.debug("Profile " + profURI + " defined"); }
             */
            for (Entry<Package, LinkedList<Profile>> app : applications.entrySet())
            {
                for (Profile prof : app.getValue())
                {
                    log.info("Applying profile " + prof.getName() + " defined in " + EcoreUtil.getURI(prof) + ": "
                            + prof.getDefinition());
                    app.getKey().applyProfile(prof);
                }
            }
            
            // r.save(new
            // FileOutputStream(r.getURI().appendFileExtension("defined").toFileString()),
            // null);
            
        }
        catch (IOException e)
        {
            log.error("Couldn't read the file", e);
        }
        catch (XMLStreamException e)
        {
            log.error("Error during the XML parsing", e);
        }
        catch (FactoryConfigurationError e)
        {
            log.error("Error during the StAX configuration", e);
        }
        catch (Exception e)
        {
            log.error("Reading error", e);
        }
        EcoreUtil.resolveAll(r);
        return r;
    }
    
    /**
     * Saves the resource using the specified XMI serialization target
     * 
     * @param r
     *            resource to save
     * @param target
     *            XMI serialization target
     * @param outputfile
     *            Name of the output file
     * @throws IOException
     * @throws InterruptedException
     */
    public static void saveResource(final Resource r, String target, String outputfile) throws IOException
    {
        // saving->pos->pis
        try (PipedOutputStream pos = new PipedOutputStream(); PipedInputStream pis = new PipedInputStream();)
        {
            pos.connect(pis);
            XMIPostWriter cleaner = new XMIPostWriter(pis, outputfile,
                    EMFConfiguration.getWriterLocationMappings(target), EMFConfiguration.getWriterIDMappings(target),
                    EMFConfiguration.getWriterNSMappings(target), EMFConfiguration.getNamespaceBlacklist(target),
                    EMFConfiguration.getNamespaceWhitelist(target), EMFConfiguration.getElementBlacklist(target),
                    EMFConfiguration.getElementWhitelist(target));
            
            Thread th = new Thread(cleaner);
            th.start();
            log.info("Starting EMF saving");
            r.save(pos, Collections.EMPTY_MAP);
            log.info("Finished EMF saving");
            log.info("Closing output stream");
            pos.close();
            th.join();
        }
        catch (IOException e)
        {
            log.error("Can't save " + r.getURI(), e);
        }
        catch (InterruptedException e)
        {
            log.error("Can't save " + r.getURI(), e);
        }
        EMFUtil.log.info("Saved " + r.getURI());
    }
    
    /**
     * Saves the resource using the specified XMI serialization target
     * 
     * @param r
     *            resource to save
     * @param target
     *            XMI serialization target
     * @throws IOException
     * @throws InterruptedException
     */
    public static void saveResource(final Resource r, String target) throws IOException
    {
        saveResource(r, target, r.getURI().toFileString());
    }
    
    /**
     * Flattens the profile, due to UML2 not being able to access stereotypes in
     * nested profiles
     * 
     * @param profile
     *            profile to flatten
     */
    private static void flatten(Profile profile)
    {
        move(profile, profile);
    }
    
    /**
     * Moves the content of a package to specified profile
     * 
     * @param from
     *            package from which the content is moved
     * @param to
     *            destination of the moved content
     */
    private static void move(Package from, Profile to)
    {
        int i = 0;
        while (i != from.getPackagedElements().size())
        {
            PackageableElement pe = from.getPackagedElements().get(i);
            if (pe instanceof Package)
            {
                move((Package) pe, to);
                from.getPackagedElements().remove(i);
            }
            else if (from != to)
            {
                from.getPackagedElements().remove(i);
                to.getPackagedElements().add(pe);
            }
            else
                i++;
        }
    }
    
    /**
     * 
     * @param paths
     *            List of directories in which the file is looked for
     * @param name
     *            Name of the file to look for
     * @return File located in one of the directory, or null if none is found
     */
    public static File locateFile(List<URI> paths, String name)
    {
        for (URI path : paths)
        {
            File f = new File(path.toFileString(), name);
            if (f.exists())
                return f;
        }
        return null;
    }
    
    /**
     * 
     * @param r
     *            Resource in which the named element is looked for
     * @param fullname
     *            Fully-qualified name of the element
     * @return Named element of the given full name in the resource
     */
    public static NamedElement getNamedElement(Resource r, String fullname)
    {
        if (fullname == null)
            return null;
        String[] spl = fullname.split("::");
        Namespace ns = null;
        for (int i = 0; i < spl.length; i++)
        {
            String name = spl[i];
            if (ns == null)
            {
                for (EObject eo : r.getContents())
                    if (eo instanceof Namespace && ((Namespace) eo).getName().equals(name))
                        ns = (Namespace) eo;
            }
            else
            {
                for (NamedElement ne : ns.getOwnedMembers())
                {
                    if (name.equals(ne.getName()))
                    {
                        if (i == spl.length - 1)
                            return ne;
                        else if (ne instanceof Namespace)
                            ns = (Namespace) ne;
                        else
                            log.warn("NamedElement " + name + " in position " + i + " in " + fullname
                                    + " is not a namespace");
                    }
                }
            }
            if (ns == null)
            {
                log.warn("NamedElement " + name + " in position " + i + " not found in " + fullname);
                return null;
            }
        }
        return null;
    }
}
