package com.engisis.xmiutil;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.PackageImport;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.ProfileApplication;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Utility class for debugging EMF resources
 * 
 * @author barbau
 *
 */
public class DebugUtil
{
    
    static final Logger log = Logger.getLogger(DebugUtil.class);
    
    /**
     * Prints information on the package imports and applied profiles
     * 
     * @param r
     *            the Resource to be checked
     */
    public static void checkResource(Resource r)
    {
        TreeIterator<EObject> allcontent = r.getAllContents();
        log.debug("Checking " + r.getURI());
        while (allcontent.hasNext())
        {
            EObject eo = allcontent.next();
            if (eo instanceof PackageImport)
            {
                PackageImport pi = (PackageImport) eo;
                log.debug("imported: " + pi.getImportedPackage().getQualifiedName() + "- uri: "
                        + pi.getImportedPackage().getURI() + "- proxy: " + pi.getImportedPackage().eIsProxy()
                        + "- content: " + pi.getImportedPackage().eContents().size());
            }
            else if (eo instanceof ProfileApplication)
            {
                ProfileApplication pa = (ProfileApplication) eo;
                Profile prof = pa.getAppliedProfile();
                EPackage definition = pa.getAppliedDefinition();
                if (definition == null)
                {
                    log.debug("Can't retrieve the definition for " + prof + ", size: "
                            + prof.getPackagedElements().size());
                    EAnnotation ea = pa.createEAnnotation(UMLUtil.UML2_UML_PACKAGE_2_0_NS_URI);
                    ea.getReferences().add((EPackage) r.getResourceSet().getPackageRegistry().get(prof.getURI()));
                    log.debug(pa.getAppliedDefinition());
                }
                else
                    log.debug("applied: " + pa.getAppliedProfile().getQualifiedName() + "- prefix: "
                            + definition.getNsPrefix() + "- uri: " + definition.getNsURI() + "- proxy: "
                            + definition.eIsProxy() + "- info: " + definition + "- content: "
                            + definition.eContents().size());
            }
        }
    }
    
    /**
     * Lists the resources loaded in the ResourceSet
     * 
     * @param rs
     *            the ResourseSet to check
     */
    public static void printLoadedResources(ResourceSet rs)
    {
        // TODO: use SB
        EList<Resource> resources = rs.getResources();
        log.debug(resources.size() + " resources loaded:");
        for (Resource r : resources)
            log.debug(" " + r.getURI() + "-" + r.isLoaded() + "-" + r.getContents().size() + " elements");
    }
    
    /**
     * Prints applied stereotype
     * 
     * @param ne
     *            the NamedElement for which stereotypes are printed
     */
    public static void printAppliedStereotypes(NamedElement ne)
    {
        StringBuilder sb = new StringBuilder();
        sb.append("Applied to " + ne + ":");
        for (EObject s : ne.getAppliedStereotypes())
            sb.append(" " + s);
        sb.append("\n");
        log.info(sb);
    }
    
    /**
     * Prints the Stereotypes in the Resource
     * 
     * @param r2
     *            the checked Resource
     */
    public static void printStereotypes(Resource r2)
    {
        // list stereotypes
        TreeIterator<EObject> allcontent2 = r2.getAllContents();
        while (allcontent2.hasNext())
        {
            EObject eo = allcontent2.next();
            if (eo instanceof Stereotype)
            {
                Stereotype st = (Stereotype) eo;
                log.debug("Stereotype found: " + st.getQualifiedName());
                
            }
        }
        
    }
    
    /**
     * Prints stereotype applications
     * 
     * @param r
     *            resource in which applications are looked for
     */
    public static void printStereotypedElements(Resource r)
    {
        log.debug("Stereotyped elements:");
        TreeIterator<EObject> allcontent = r.getAllContents();
        while (allcontent.hasNext())
        {
            EObject eo = allcontent.next();
            if (eo instanceof Element)
            {
                Element el = (Element) eo;
                if (!el.getAppliedStereotypes().isEmpty())
                {
                    StringBuilder sb = new StringBuilder();
                    sb.append(el + " has: ");
                    for (Stereotype st : el.getAppliedStereotypes())
                    {
                        sb.append(" " + st.getName() + "-" + el.getStereotypeApplication(st));
                    }
                    log.debug(sb);
                }
            }
        }
    }
    
}
