package com.engisis.sysphs.deserialization.simulink;

import com.engisis.sysphs.language.simscape.SBranch;
import com.engisis.sysphs.language.simscape.SComponent;
import com.engisis.sysphs.language.simscape.SComponentReference;
import com.engisis.sysphs.language.simscape.SConnection;
import com.engisis.sysphs.language.simscape.SDomain;
import com.engisis.sysphs.language.simscape.SMember;
import com.engisis.sysphs.language.simscape.SMemberAssignment;
import com.engisis.sysphs.language.simscape.SMemberPath;
import com.engisis.sysphs.language.simscape.SNode;
import com.engisis.sysphs.language.simscape.SVariable;
import com.engisis.sysphs.language.simscape.SimscapeFactory;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.generation.simulink.SimscapeBaseListener;
import com.engisis.sysphs.generation.simulink.SimscapeParser.AssignmentContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.BranchContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.BranchesContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ComponentContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ConnectionContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.DomainContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.ReferenceContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.SetupContext;
import com.engisis.sysphs.generation.simulink.SimscapeParser.StatementContext;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTreeProperty;
import org.antlr.v4.runtime.tree.ParseTreeWalker;
import org.apache.log4j.Logger;

/**
 * Second-pass parser of a Simscape file
 * 
 * @author barbau
 *
 */
public class SimscapeReaderPass2 extends SimscapeBaseListener implements SimulinkReader
{
    private static final Logger log = Logger.getLogger(SimscapeReaderPass2.class);
    
    /**
     * Collection of references
     */
    private Collection<SimulinkReference> references;
    /**
     * Simscape objects associated with syntax tree nodes
     */
    private ParseTreeProperty<SElement> objects;
    
    /**
     * Current component
     */
    private SComponent scomponent = null;
    /**
     * Current domain
     */
    private SDomain sdomain = null;
    
    /**
     * Root of the syntax tree
     */
    private ParserRuleContext prc;
    
    /**
     * First-pass parser
     */
    private SimscapeReaderPass1 srd;
    
    /**
     * Constructs a second-pass parser of a Simscape file
     * 
     * @param srd
     *            first-pass parser
     * @param prc
     *            root of the syntax tree
     */
    public SimscapeReaderPass2(SimscapeReaderPass1 srd, ParserRuleContext prc)
    {
        this.prc = prc;
        this.srd = srd;
        references = srd.getReferences();
        objects = srd.getObjects();
    }
    
    @Override
    public void process()
    {
        new ParseTreeWalker().walk(this, prc);
    }
    
    @Override
    public void enterComponent(ComponentContext ctx)
    {
        scomponent = (SComponent) objects.get(ctx);
        if (scomponent != null)
            log.info("Retrieving component " + scomponent.getName());
    }
    
    @Override
    public void enterDomain(DomainContext ctx)
    {
        sdomain = (SDomain) objects.get(ctx);
        if (sdomain != null)
            log.info("Retrieving domain " + sdomain.getName());
    }
    
    @Override
    public void enterBranches(BranchesContext ctx)
    {
        for (BranchContext bc : ctx.branch())
        {
            log.info("Resolving branch");
            SBranch sbranch = (SBranch) objects.get(bc);
            SMember smember = scomponent.getMember(bc.ID().getText());
            if (smember instanceof SVariable)
                sbranch.setVariable((SVariable) smember);
            
            ReferenceContext rc0 = bc.branchpoint(0).reference();
            ReferenceContext rc1 = bc.branchpoint(1).reference();
            
            if (rc0 != null)
            {
                SMemberPath smp = SimscapeFactory.eINSTANCE.createSMemberPath();
                smp.getPath().addAll(getListMembers(rc0));
                sbranch.setFrom(smp);
            }
            else if (rc1 != null)
            {
                SMemberPath smp = SimscapeFactory.eINSTANCE.createSMemberPath();
                smp.getPath().addAll(getListMembers(rc1));
                sbranch.setTo(smp);
            }
        }
    }
    
    @Override
    public void enterSetup(SetupContext ctx)
    {
        for (StatementContext sc : ctx.statement())
        {
            AssignmentContext ac = sc.assignment();
            if (ac != null)
            {
                SMemberAssignment sma = SimscapeFactory.eINSTANCE.createSMemberAssignment();
                sma.getMemberPath().addAll(getListMembers(ac.reference()));
                sma.setAssignedValue(srd.getValue(ac.expr()));
                scomponent.getOwnedAssignments().add(sma);
                log.info("Adding member assignment for " + sc.getText());
            }
            else
                log.info("Skipping statement " + sc.getText());
        }
    }

    @Override
    public void enterConnection(ConnectionContext ctx)
    {
        SConnection sc = (SConnection) objects.get(ctx);
        if (sc != null)
        {
            log.info("Resolving connection " + ctx.getText());
            for(ReferenceContext rc : ctx.reference())
            {
                SMemberPath smp = SimscapeFactory.eINSTANCE.createSMemberPath();
                smp.getPath().addAll(getListMembers(rc));
                sc.getPoints().add(smp);
                log.info("Added point for " + rc.getText());
            }
        }
        else
            log.error("Can't resolve connection " + ctx.getText());
    }

    private List<SMember> getListMembers(ReferenceContext rc)
    {
        List<SMember> ret = new LinkedList<SMember>();
        SMember sm = null;
        if (scomponent != null)
        {
            sm = scomponent.getMember(rc.id_arg(0).getText());
            ret.add(sm);
        }
        for (int i = 1; i < rc.id_arg().size(); i++)
        {
            if (sm == null)
                break;
            String id = rc.id_arg(i).getText();
            if (sm instanceof SComponentReference)
            {
                if (((SComponentReference) sm).getComponent() == null)
                {
                    log.error("The member has no component: " + id + " of " + rc.getText());
                    break;
                }
                sm = ((SComponentReference) sm).getComponent().getMember(id);
                ret.add(sm);
                if (sm == null)
                {
                    log.error("can't find reference " + id + " of " + rc.getText());
                    break;
                }
            }
            else if (sm instanceof SNode)
            {
                if (((SNode) sm).getDomain() == null)
                {
                    log.error("The member has no domain: " + id + " of " + rc.getText());
                    break;
                }
                sm = ((SNode) sm).getDomain().getMember(id);
                ret.add(sm);
                if (sm == null)
                {
                    log.error("can't find reference " + id + " of " + rc.getText());
                    break;
                }
            }
            else
            {
                sm = null;
            }
        }
        return ret;
        
    }
    
    @Override
    public Collection<SimulinkReference> getReferences()
    {
        return references;
    }
    
    @Override
    public int getPriority()
    {
        return 5;
    }
    
    @Override
    public SElement getRootElement()
    {
        return scomponent != null ? scomponent : sdomain;
    }
    
}
