package com.engisis.sysphs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.eclipse.emf.ecore.resource.ResourceSet;

import com.engisis.sysphs.util.SimulationToSysMLTranslator;
import com.engisis.xmiutil.EMFUtil;
import com.engisis.xmiutil.UMLModelErrorException;

/**
 * Class used to manage the translation of simulation models into SysML.
 * 
 * @author barbau
 *
 */
public class SimulationToSysMLTranslationManager
{
    /**
     * Path to the simulation file
     */
    private String model;
    
    /**
     * List of paths where XMI files are located
     */
    private String[] paths;
    
    /**
     * Translator used
     */
    private SimulationToSysMLTranslator translator;
    
    /**
     * Main resource set for the UML models
     */
    private ResourceSet rs;
    
    /**
     * XMI serialization target
     */
    private String target;
    
    /**
     * Output directory
     */
    private File outputdirectory;
    
    /**
     * Constructs a translation manager object for a simulation model
     * 
     * @param model
     *            path to the simulation model
     * @param paths
     *            list of paths to XMI files
     * @throws FileNotFoundException
     */
    public SimulationToSysMLTranslationManager(String model, String[] paths) throws FileNotFoundException
    {
        if (model == null)
            throw new IllegalArgumentException("The model can't be null");
        this.model = model;
        this.paths = paths;
        rs = EMFUtil.createResourceSet();
        
    }
    
    /**
     * Sets the translator to be used
     * 
     * @param translator
     *            used translator (Modelica or Simulink)
     */
    public void setTranslator(SimulationToSysMLTranslator translator)
    {
        this.translator = translator;
    }
    
    /**
     * Sets the directory in which the SysML file should be generated
     * 
     * @param outputdirectory
     *            file representing the output directory
     */
    public void setOutputDirectory(File outputdirectory)
    {
        this.outputdirectory = outputdirectory;
    }
    
    /**
     * Sets the target XMI, to determine the ID and namespace mappings. Valid
     * values are found in the properties.config file.
     * 
     * @param target
     *            target XMI. Call
     *            {@link com.engisis.sysphs.util.EMFUtil#getTargets()} to get
     *            acceptable values.
     */
    public void setTarget(String target)
    {
        this.target = target;
    }
    
    /**
     * Executes the translation.
     * 
     * @throws IOException
     * @throws UMLModelErrorException
     * @throws InterruptedException
     */
    public void execute() throws IOException, UMLModelErrorException
    {
        translator.execute(model, rs, outputdirectory, target, paths);
    }
}
