package com.engisis.xmiutil;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Hashtable;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;

/**
 * Class to post-process the XMI serialization.
 * 
 * @author barbau
 *
 */
public class XMIPostWriter implements Runnable
{
    private static final Logger log = Logger.getLogger(XMIPostWriter.class);
    
    private static final String INDENT = "  ";
    /**
     * stream from which the content is read
     */
    private InputStream is;
    /**
     * File in which the content is written into
     */
    private String file;
    /**
     * location replacement map
     */
    private Hashtable<String, String> locationsmap;
    /**
     * ID replacement map
     */
    private Hashtable<String, String> idmaps;
    /**
     * namespace replacement map
     */
    private Hashtable<String, String> nsmaps;
    
    private Set<String> nsfilter_bl;
    private Set<String> nsfilter_wl;
    private Set<QName> elfilter_bl;
    private Set<QName> elfilter_wl;
    
    /**
     * Creates a post-processor
     * 
     * @param is
     *            input stream
     * @param file
     *            output stream
     * @param idmaps
     *            ID map
     * @param nsmaps
     *            namespace map
     * @throws FileNotFoundException
     */
    public XMIPostWriter(InputStream is, String file, Hashtable<String, String> locationsmap,
            Hashtable<String, String> idmaps, Hashtable<String, String> nsmaps, Set<String> nsfilter_bl,
            Set<String> nsfilter_wl, Set<QName> elfilter_bl, Set<QName> elfilter_wl) throws FileNotFoundException
    {
        this.is = is;
        this.file = file;
        this.locationsmap = locationsmap;
        this.idmaps = idmaps;
        this.nsmaps = nsmaps;
        this.nsfilter_bl = nsfilter_bl;
        this.nsfilter_wl = nsfilter_wl;
        this.elfilter_bl = elfilter_bl;
        this.elfilter_wl = elfilter_wl;
        log.debug("Namespaces mappings");
        if (nsmaps != null)
            for (Entry<String, String> nsent : nsmaps.entrySet())
                log.debug(nsent.getKey() + "->" + nsent.getValue());
        log.debug("Locations mappings");
        if (locationsmap != null)
            for (Entry<String, String> locent : locationsmap.entrySet())
                log.debug(locent.getKey() + "->" + locent.getValue());
    }
    
    /**
     * Essentially rewrites the input stream into the file stream, but performs
     * ID/namespace replacements
     */
    @Override
    public void run()
    {
        log.info("Starting postwriter");
        int eventType;
        
        // number of elements that should currently be skipped
        int skip = 0;
        log.info("Opening file stream to " + file);
        
        StringBuilder sb = new StringBuilder();
        String str = null;
        
        XMLOutputFactory xmlof = XMLOutputFactory.newFactory();
        int indent = 0;
        try (FileOutputStream fos = new FileOutputStream(file))
        {
            XMLStreamWriter xmlw = xmlof.createXMLStreamWriter(fos);
            XMLInputFactory xmlif = XMLInputFactory.newFactory();
            xmlif.setProperty(XMLInputFactory.IS_REPLACING_ENTITY_REFERENCES, Boolean.valueOf(false));
            XMLStreamReader xmlr = xmlif.createXMLStreamReader(is);
            while (xmlr.hasNext())
            {
                eventType = xmlr.next();
                switch (eventType)
                {
                case XMLStreamConstants.START_ELEMENT:
                    // manage skip
                    String ns = xmlr.getNamespaceURI();
                    String pr = xmlr.getPrefix();
                    QName el = xmlr.getName();
                    String ln = xmlr.getLocalName();
                    if (ns != null && nsmaps.containsKey(ns))
                    {
                        ns = nsmaps.get(ns);
                        el = new QName(ns, ln);
                    }
                    if (skip == 0)
                    {
                        // not skipping, see if skip current
                        
                        boolean bskip = false;
                        if (ns != null)
                        {
                            String ns_ = ns;
                            if (nsfilter_wl != null && nsfilter_wl.stream().noneMatch(s -> ns_.matches(s)))
                            {
                                log.debug("Namespace " + ns_ + " not in whitelist "
                                        + nsfilter_wl.stream().collect(Collectors.joining(" ")));
                                bskip = true;
                            }
                            if (nsfilter_bl != null && nsfilter_bl.stream().anyMatch(s -> ns_.matches(s)))
                            {
                                log.debug("Namespace " + ns_ + " in blacklist "
                                        + nsfilter_bl.stream().collect(Collectors.joining(" ")));
                                bskip = true;
                            }
                        }
                        QName el_ = el;
                        if (elfilter_wl != null && elfilter_wl.stream().noneMatch(qn -> el_.equals(qn)))
                        {
                            log.debug("Element " + el_ + " not in whitelist "
                                    + elfilter_wl.stream().map(qn -> qn.toString()).collect(Collectors.joining(" ")));
                            bskip = true;
                        }
                        if (elfilter_bl != null && elfilter_bl.stream().anyMatch(qn -> el_.equals(qn)))
                        {
                            log.debug("Element " + el_ + " in blacklist "
                                    + elfilter_bl.stream().map(qn -> qn.toString()).collect(Collectors.joining(" ")));
                            bskip = true;
                        }
                        // if ("eAnnotations".equals(xmlr.getLocalName()))
                        // skip++;
                        if (bskip)
                        {
                            skip++;
                        }
                    }
                    else
                        skip++;
                    if (skip > 0)
                        continue;
                    
                    str = sb.toString().trim();
                    if (!str.isEmpty())
                    {
                        for (int i = 0; i < indent; i++)
                            xmlw.writeCharacters(INDENT);
                        xmlw.writeCharacters(str + "\n");
                        sb.setLength(0);
                    }
                    
                    for (int i = 0; i < indent; i++)
                        xmlw.writeCharacters(INDENT);
                    
                    if (ns != null)
                        xmlw.writeStartElement(pr, ln, ns);
                    else
                        xmlw.writeStartElement(ln);
                    
                    for (int i = 0; i < xmlr.getNamespaceCount(); i++)
                    {
                        // just because of the wrong URI used by MD
                        // TODO: define that in EMFUtil
                        String ans = xmlr.getNamespaceURI(i);
                        if (nsmaps != null)
                        {
                            String rep = nsmaps.get(ans);
                            if (rep != null)
                                ans = rep;
                        }
                        
                        boolean bskip = false;
                        String ans_ = ans;
                        if (nsfilter_wl != null && nsfilter_wl.stream().noneMatch(s -> ans_.matches(s)))
                        {
                            log.debug("Namespace " + ans + " not in whitelist "
                                    + nsfilter_wl.stream().collect(Collectors.joining(" ")));
                            bskip = true;
                        }
                        if (nsfilter_bl != null && nsfilter_bl.stream().anyMatch(s -> ans_.matches(s)))
                        {
                            log.debug("Namespace " + ans + " in blacklist "
                                    + nsfilter_bl.stream().collect(Collectors.joining(" ")));
                            bskip = true;
                        }
                        if (bskip)
                            continue;
                        String apr = xmlr.getNamespacePrefix(i);
                        xmlw.writeNamespace(apr, ans);
                    }
                    for (int i = 0; i < xmlr.getAttributeCount(); i++)
                    {
                        String ans = xmlr.getAttributeNamespace(i);
                        boolean bskip = false;
                        if (ans != null)
                        {
                            if (nsmaps != null)
                            {
                                String rep = nsmaps.get(ans);
                                if (rep != null)
                                    ans = rep;
                            }
                            String ans_ = ans;
                            if (nsfilter_wl != null && nsfilter_wl.stream().noneMatch(s -> ans_.matches(s)))
                            {
                                log.debug("Namespace " + ans + " not in whitelist "
                                        + nsfilter_wl.stream().collect(Collectors.joining(" ")));
                                bskip = true;
                            }
                            if (nsfilter_bl != null && nsfilter_bl.stream().anyMatch(s -> ans_.matches(s)))
                            {
                                log.debug("Namespace " + ans + " in blacklist "
                                        + nsfilter_bl.stream().collect(Collectors.joining(" ")));
                                bskip = true;
                            }
                        }
                        if (bskip)
                            continue;
                        String aln = xmlr.getAttributeLocalName(i);
                        String av = xmlr.getAttributeValue(i);
                        if ("href".equals(aln))
                        {
                            if (idmaps != null)
                            {
                                String rep = idmaps.get(av);
                                if (rep != null)
                                    av = rep;
                            }
                            if (locationsmap != null)
                            {
                                int idx = av.indexOf("#");
                                if (idx > 0 && av.length() > idx)
                                {
                                    String loc = av.substring(0, idx);
                                    if (locationsmap.containsKey(loc))
                                        av = locationsmap.get(loc) + "#" + av.substring(idx + 1);
                                    else
                                    {
                                        String fn = URI.createURI(loc).lastSegment();
                                        if (locationsmap.containsKey(fn))
                                            av = locationsmap.get(fn) + "#" + av.substring(idx + 1);
                                    }
                                }
                            }
                        }
                        if (ans != null)
                            xmlw.writeAttribute(ans, aln, av);
                        else
                            xmlw.writeAttribute(aln, av);
                    }
                    xmlw.writeCharacters("\n");
                    indent++;
                    break;
                case XMLStreamConstants.END_ELEMENT:
                    if (skip > 0)
                    {
                        skip--;
                        continue;
                    }
                    
                    str = sb.toString().trim();
                    if (!str.isEmpty())
                    {
                        for (int i = 0; i < indent; i++)
                            xmlw.writeCharacters(INDENT);
                        xmlw.writeCharacters(str + "\n");
                        sb.setLength(0);
                    }
                    
                    indent--;
                    for (int i = 0; i < indent; i++)
                        xmlw.writeCharacters(INDENT);
                    xmlw.writeEndElement();
                    xmlw.writeCharacters("\n");
                    break;
                
                case XMLStreamConstants.PROCESSING_INSTRUCTION:
                    if (skip > 0)
                        continue;
                    xmlw.writeProcessingInstruction(xmlr.getPITarget(), xmlr.getPIData());
                    break;
                case XMLStreamConstants.CHARACTERS:
                    if (skip > 0)
                        continue;
                    String txt = xmlr.getText().trim();
                    if (txt.length() != 0)
                    {
                        sb.append(txt);
                    }
                    break;
                case XMLStreamConstants.COMMENT:
                    // purposely avoid comments
                    break;
                case XMLStreamConstants.START_DOCUMENT:
                    xmlw.writeStartDocument();
                    break;
                case XMLStreamConstants.END_DOCUMENT:
                    break;
                case XMLStreamConstants.ENTITY_REFERENCE:
                    if (skip > 0)
                        continue;
                    sb.append(xmlr.getText());
                    break;
                case XMLStreamConstants.ATTRIBUTE:
                    break;
                case XMLStreamConstants.DTD:
                    break;
                case XMLStreamConstants.CDATA:
                    if (skip > 0)
                        continue;
                    xmlw.writeCData(xmlr.getText());
                    break;
                case XMLStreamConstants.SPACE:
                    break;
                default:
                    break;
                }
                
            }
        }
        catch (XMLStreamException e)
        {
            log.error("Error while parsing the file", e);
        }
        catch (IOException e)
        {
            log.error("Error while parsing the file", e);
        }
        
        log.info("Finished postwriter");
    }
    
}
