package com.engisis.xmiutil;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;

/**
 * Contains the configuration properties extracted from the configuration file.
 * 
 * @author barbau
 *
 */
public class EMFConfiguration extends CoreConfiguration
{
    private static final Logger log = Logger.getLogger(EMFConfiguration.class);
    
    /**
     * Indicator of a resource mapping in the property file
     */
    public static final String RESOURCE_MAP = "resourcemap";
    /**
     * Indicator of a location mapping in the property file
     */
    public static final String LOCATION_MAP = "locmap";
    /**
     * Indicator of a namespace mapping in the property file
     */
    public static final String NAMESPACE_MAP = "nsmap";
    /**
     * Indicator of an ID mapping in the property file
     */
    public static final String ID_MAP = "idmap";
    /**
     * Indicator of a namespace filter in the property file
     */
    public static final String NS_FILTER = "nsfilter";
    /**
     * Indicator of an element filter in the property file
     */
    public static final String EL_FILTER = "elfilter";
    
    /**
     * a map with keys representing network URLs, and values representing local
     * URLs
     */
    private static Hashtable<URI, URI> locmaps_r = null;
    
    /**
     * a map with keys representing local URLs, and values representing network
     * URLs
     */
    private static Hashtable<String, Hashtable<String, String>> locmaps_w = null;
    
    /**
     * a map of namespace mappings
     */
    private static Hashtable<String, String> nsmaps_r = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing namespace mappings
     */
    private static Hashtable<String, Hashtable<String, String>> nsmaps_w = null;
    
    /**
     * a map of ID mappings
     */
    private static Hashtable<String, String> idmaps_r = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing ID mappings
     */
    private static Hashtable<String, Hashtable<String, String>> idmaps_w = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing blacklisted namespaces
     */
    private static Hashtable<String, Set<String>> nsfilter_bl = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing whitelisted namespaces
     */
    private static Hashtable<String, Set<String>> nsfilter_wl = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing blacklisted elements
     */
    private static Hashtable<String, Set<QName>> elfilter_bl = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing whitelisted elements
     */
    private static Hashtable<String, Set<QName>> elfilter_wl = null;
    
    /**
     * Loads the properties, and populate the location, namespace, and ID
     * mappings
     */
    public synchronized static void loadProperties()
    {
        if (locmaps_r != null)
            return;
        
        locmaps_r = new Hashtable<URI, URI>();
        locmaps_w = new Hashtable<String, Hashtable<String, String>>();
        nsmaps_r = new Hashtable<String, String>();
        nsmaps_w = new Hashtable<String, Hashtable<String, String>>();
        idmaps_r = new Hashtable<String, String>();
        idmaps_w = new Hashtable<String, Hashtable<String, String>>();
        nsfilter_bl = new Hashtable<>();
        nsfilter_wl = new Hashtable<>();
        elfilter_bl = new Hashtable<>();
        elfilter_wl = new Hashtable<>();
        
        for (String[] spl : getValues())
        {
            if (spl.length > 1)
            {
                if (spl[0].equals(NAMESPACE_MAP + "_r"))
                {
                    if (spl.length < 3)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    String from = spl[1];
                    String to = spl[2];
                    nsmaps_r.put(from, to);
                    log.info("add ns mapping from " + from + " to " + to);
                    continue;
                }
                if (spl[0].equals(NAMESPACE_MAP + "_w"))
                {
                    if (spl.length < 4)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    String target = spl[1];
                    String from = spl[2];
                    String to = spl[3];
                    
                    Hashtable<String, String> ht = nsmaps_w.get(target);
                    if (ht == null)
                    {
                        ht = new Hashtable<String, String>();
                        nsmaps_w.put(target, ht);
                    }
                    ht.put(from, to);
                    log.info("add ns mapping from " + from + " to " + to + " in target " + target);
                    continue;
                }
                else if (spl[0].equals(ID_MAP) || spl[0].equals(ID_MAP + "_r"))
                {
                    if (spl.length < 3)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    String from = spl[1];
                    String to = spl[2];
                    idmaps_r.put(from, to);
                    log.info("add id mapping from " + from + " to " + to);
                    continue;
                }
                else if (spl[0].equals(ID_MAP + "_w"))
                {
                    if (spl.length < 4)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    
                    String target = spl[1];
                    String from = spl[2];
                    String to = spl[3];
                    
                    Hashtable<String, String> ht = idmaps_w.get(target);
                    if (ht == null)
                    {
                        ht = new Hashtable<String, String>();
                        idmaps_w.put(target, ht);
                    }
                    ht.put(from, to);
                    
                    log.info("add id mapping from " + from + " to " + to + " in target " + target);
                    continue;
                }
                else if (spl[0].equals(RESOURCE_MAP) || spl[0].equals(LOCATION_MAP + "_r"))
                {
                    if (spl.length < 3)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    URI from = URI.createURI(spl[1]);
                    URI to = URI.createURI(transformResource(EMFConfiguration.class.getResource(spl[2]).toString()));
                    locmaps_r.put(from, to);
                    log.info("add resource mapping from " + from + " to " + to);
                    continue;
                }
                else if (spl[0].equals(LOCATION_MAP + "_w"))
                {
                    if (spl.length < 4)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    String target = spl[1];
                    String from = spl[2];
                    String to = spl[3];
                    
                    Hashtable<String, String> ht = locmaps_w.get(target);
                    if (ht == null)
                    {
                        ht = new Hashtable<String, String>();
                        locmaps_w.put(target, ht);
                    }
                    ht.put(from, to);
                    
                    log.info("add location mapping from " + from + " to " + to);
                    continue;
                }
                else if (spl[0].equals(NS_FILTER + "_bl"))
                {
                    if (spl.length < 3)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    
                    String target = spl[1];
                    String ns = spl[2];
                    
                    Set<String> s = nsfilter_bl.get(target);
                    if (s == null)
                    {
                        s = new HashSet<>();
                        nsfilter_bl.put(target, s);
                    }
                    s.add(ns);
                    
                    log.info("add namespace " + s + " to blacklist in target " + target);
                }
                else if (spl[0].equals(NS_FILTER + "_wl"))
                {
                    if (spl.length < 3)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    
                    String target = spl[1];
                    String ns = spl[2];
                    
                    Set<String> s = nsfilter_wl.get(target);
                    if (s == null)
                    {
                        s = new HashSet<>();
                        nsfilter_wl.put(target, s);
                    }
                    s.add(ns);
                    
                    log.info("add namespace " + s + " to whitelist in target " + target);
                }
                else if (spl[0].equals(EL_FILTER + "_bl"))
                {
                    if (spl.length < 4)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    
                    String target = spl[1];
                    String ns = spl[2];
                    String el = spl[3];
                    
                    Set<QName> s = elfilter_bl.get(target);
                    if (s == null)
                    {
                        s = new HashSet<>();
                        elfilter_bl.put(target, s);
                    }
                    s.add(new QName(ns, el));
                    
                    log.info("add element " + s + " to blacklist in target " + target);
                }
                else if (spl[0].equals(NS_FILTER + "_wl"))
                {
                    if (spl.length < 4)
                    {
                        log.error("Line in the wrong format: " + String.join(" ", spl));
                        continue;
                    }
                    
                    String target = spl[1];
                    String ns = spl[2];
                    String el = spl[3];
                    
                    Set<QName> s = elfilter_wl.get(target);
                    if (s == null)
                    {
                        s = new HashSet<>();
                        elfilter_wl.put(target, s);
                    }
                    s.add(new QName(ns, el));
                    
                    log.info("add element " + s + " to whitelist in target " + target);
                }
            }
        }
    }
    
    /**
     * Adds a jar prefix if the URI has jar content
     * 
     * @param suri
     * @return
     */
    private static String transformResource(String suri)
    {
        if (suri.contains("!") && !suri.startsWith("jar:"))
            return "jar:" + suri;
        return suri;
    }
    
    /**
     * Returns the mappings of locations
     * 
     * @return mappings of locations (URLs)
     */
    public static Hashtable<URI, URI> getLocationMappings()
    {
        if (locmaps_r == null)
            loadProperties();
        return locmaps_r;
    }
    
    /**
     * Returns the location mappings to perform during the XMI writing
     * 
     * @return namespace mappings
     */
    public static Hashtable<String, String> getWriterLocationMappings(String target)
    {
        if (locmaps_w == null)
            loadProperties();
        return locmaps_w.get(target);
    }
    
    /**
     * Returns the namespace mappings to perform during the XMI reading
     * 
     * @return namespace mappings
     */
    public static Hashtable<String, String> getReaderNSMappings()
    {
        if (nsmaps_r == null)
            loadProperties();
        return nsmaps_r;
    }
    
    /**
     * Returns the namespace mappings to perform during the XMI writing
     * 
     * @return namespace mappings
     */
    public static Hashtable<String, String> getWriterNSMappings(String target)
    {
        if (nsmaps_w == null)
            loadProperties();
        return nsmaps_w.get(target);
    }
    
    /**
     * Returns the ID mappings to perform during the XMI reading
     * 
     * @return ID mappings
     */
    public static Hashtable<String, String> getReaderIDMappings()
    {
        if (idmaps_r == null)
            loadProperties();
        return idmaps_r;
    }
    
    /**
     * Returns the ID mappings to perform during the XMI writing
     * 
     * @return ID mappings
     */
    public static Hashtable<String, String> getWriterIDMappings(String target)
    {
        if (idmaps_w == null)
            loadProperties();
        return idmaps_w.get(target);
    }
    
    public static Set<QName> getElementBlacklist(String target)
    {
        if (elfilter_bl == null)
            loadProperties();
        return elfilter_bl.get(target);
    }
    
    public static Set<QName> getElementWhitelist(String target)
    {
        if (elfilter_wl == null)
            loadProperties();
        return elfilter_wl.get(target);
    }
    
    public static Set<String> getNamespaceBlacklist(String target)
    {
        if (nsfilter_bl == null)
            loadProperties();
        return nsfilter_bl.get(target);
    }
    
    public static Set<String> getNamespaceWhitelist(String target)
    {
        if (nsfilter_wl == null)
            loadProperties();
        return nsfilter_wl.get(target);
    }
    
    /**
     * Returns the list of acceptable targets for the XMI writing
     * 
     * @return list of targets
     */
    public static Collection<String> getTargets()
    {
        if (idmaps_w == null)
            loadProperties();
        return Collections.unmodifiableCollection(idmaps_w.keySet());
    }
    
}
