package com.engisis.sysphs.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;

import com.engisis.xmiutil.EMFUtil;
import com.engisis.xmiutil.SysMLUtil;
import com.engisis.xmiutil.UMLModelErrorException;

public class SysPhSUtil extends SysMLUtil
{
    private static final Logger log = Logger.getLogger(SysPhSUtil.class);
    public static final String SYSPHS_LANGUAGE = "sysphs";
    
    private Stereotype sinitialvaluesspec;
    
    private Class sconservedsubstance;
    private Class srealsignalin;
    private Class srealsignalout;
    
    private Stereotype sphsvariable;
    private Stereotype sphsconstant;
    private Stereotype smultidimelement;
    private Stereotype sinitialvalueref;
    private Stereotype ssimulationvertex;
    
    private Stereotype smodelicablock;
    private Stereotype smodelicaport;
    private Stereotype smodelicaparameter;
    private Stereotype ssimulinkblock;
    private Stereotype ssimulinkport;
    private Stereotype ssimulinkparameter;
    
    private List<LibraryComponent> lscomponents;
    private List<LibraryComponent> lmcomponents;
    
    public static final URI nsSysPhS = URI.createURI("https://www.omg.org/spec/SysPhS/20200925");
    private static final URI nsSysPhSExtended = URI.createURI(nsSysPhS + "/SysPhSExtended");
    private static final URI nsSysPhSPlatformProfile = URI.createURI(nsSysPhS + "/SysPhSPlatformProfile");
    
    public static final String nameSysPhSProfile = "SysPhSProfile.xmi";
    public static final String nameSysPhSLibrary = "SysPhSLibrary.xmi";
    
    public SysPhSUtil(ResourceSet rs) throws UMLModelErrorException
    {
        super(rs);
        loadSysPhS();
        loadSysPhSExtended();
        loadSysPhSPlatform();
    }
    
    /**
     * Loads the SysPhS stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysPhS() throws UMLModelErrorException
    {
        sphsvariable = loadStereotype(nsSysPhS, "PhSVariable");
        sphsconstant = loadStereotype(nsSysPhS, "PhSConstant");
    }
    
    /**
     * Loads the extended SysPhS stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysPhSExtended() throws UMLModelErrorException
    {
        if (getResourceSet().getResource(nsSysPhSExtended, false) != null)
        {
            sinitialvalueref = loadStereotype(nsSysPhSExtended, "InitialValueReference");
            ssimulationvertex = loadStereotype(nsSysPhSExtended, "SimulationVertex");
            sinitialvaluesspec = loadStereotype(nsSysPhSExtended, "InitialValuesSpecification");
        }
    }
    
    /**
     * Loads the SysPhS platform stereotypes
     * 
     * @throws UMLModelErrorException
     */
    public void loadSysPhSPlatform() throws UMLModelErrorException
    {
        Resource r = getResourceSet().getResources().stream()
                .filter(_r -> _r.getURI().lastSegment().equals(nameSysPhSLibrary)).findFirst().orElse(null);
        if (r != null)
        {
            smultidimelement = loadStereotype(nsSysPhSPlatformProfile, "MultidimensionalElement");
            smodelicablock = loadStereotype(nsSysPhSPlatformProfile, "ModelicaBlock");
            smodelicaparameter = loadStereotype(nsSysPhSPlatformProfile, "ModelicaParameter");
            smodelicaport = loadStereotype(nsSysPhSPlatformProfile, "ModelicaPort");
            
            ssimulinkblock = loadStereotype(nsSysPhSPlatformProfile, "SimulinkBlock");
            ssimulinkparameter = loadStereotype(nsSysPhSPlatformProfile, "SimulinkParameter");
            ssimulinkport = loadStereotype(nsSysPhSPlatformProfile, "SimulinkPort");
            
            lscomponents = new LinkedList<LibraryComponent>();
            lmcomponents = new LinkedList<LibraryComponent>();
            
            TreeIterator<EObject> ti = r.getAllContents();
            while (ti.hasNext())
            {
                EObject eo = ti.next();
                if (!(eo instanceof Namespace))
                {
                    ti.prune();
                    continue;
                }
                if (eo instanceof Class)
                {
                    Class uclass = (Class) eo;
                    processLibraryComponent(lmcomponents, uclass, smodelicablock, smodelicaparameter, smodelicaport);
                    processLibraryComponent(lscomponents, uclass, ssimulinkblock, ssimulinkparameter, ssimulinkport);
                    if ("ConservedSubstance".equals(uclass.getName())
                            || "ConservedQuantityKind".equals(uclass.getName()))
                        sconservedsubstance = uclass;
                    else if ("RealSignalInElement".equals(uclass.getName()))
                        srealsignalin = uclass;
                    else if ("RealSignalOutElement".equals(uclass.getName()))
                        srealsignalout = uclass;
                }
            }
        }
        else
            log.error("Could not locate the library in loaded files");
        
    }
    
    /**
     * Adds the given library component to the list
     * 
     * @param list
     *            list in which the library components should be added
     * @param uclass
     *            library component to parse
     * @param stclass
     *            library block stereotype
     * @param stparameter
     *            library parameter stereotype
     * @param stport
     *            library port stereotype
     */
    private void processLibraryComponent(List<LibraryComponent> list, Class uclass, Stereotype stclass,
            Stereotype stparameter, Stereotype stport)
    {
        if (uclass.isStereotypeApplied(stclass))
        {
            String name = (String) uclass.getValue(stclass, "name");
            if (name == null)
                name = uclass.getName();
            LibraryComponent lc = new LibraryComponent(uclass, name);
            for (Property uproperty : getAllAttributes(uclass))
            {
                if (uproperty.isStereotypeApplied(stparameter))
                {
                    name = (String) uproperty.getValue(stparameter, "name");
                    if (name == null)
                        name = uproperty.getName();
                    String value = (String) uproperty.getValue(stparameter, "value");
                    LibraryParameter lp = new LibraryParameter(uproperty, name, value);
                    lc.parameters.add(lp);
                }
                else if (uproperty.isStereotypeApplied(stport))
                {
                    name = (String) uproperty.getValue(stport, "name");
                    if (name == null)
                        name = uproperty.getName();
                    LibraryPort lp = new LibraryPort((Port) uproperty, name);
                    lc.ports.add(lp);
                }
                
            }
            list.add(lc);
        }
    }
    
    public Stereotype getInitialValuesSpecification()
    {
        return sinitialvaluesspec;
    }
    
    public Stereotype getPhSVariable()
    {
        return sphsvariable;
    }
    
    public Stereotype getPhSConstant()
    {
        return sphsconstant;
    }
    
    public Stereotype getMultidimensionalElement()
    {
        return smultidimelement;
    }
    
    public Stereotype getInitialValueReference()
    {
        return sinitialvalueref;
    }
    
    public Stereotype getSimulationVertex()
    {
        return ssimulationvertex;
    }
    
    public Class getConservedSubstance()
    {
        return sconservedsubstance;
    }
    
    public Class getRealSignalIn()
    {
        return srealsignalin;
    }
    
    public Class getRealSignalOut()
    {
        return srealsignalout;
    }
    
    public Stereotype getModelicaBlock()
    {
        return smodelicablock;
    }
    
    public Stereotype getModelicaParameter()
    {
        return smodelicaparameter;
    }
    
    public Stereotype getModelicaPort()
    {
        return smodelicaport;
    }
    
    public Stereotype getSimulinkBlock()
    {
        return ssimulinkblock;
    }
    
    public Stereotype getSimulinkParameter()
    {
        return ssimulinkparameter;
    }
    
    public Stereotype getSimulinkPort()
    {
        return ssimulinkport;
    }
    
    /**
     * Returns a list of Simulink components
     * 
     * @return list of Simulink components
     */
    public List<LibraryComponent> getSimulinkComponents()
    {
        return Collections.unmodifiableList(lscomponents);
    }
    
    /**
     * Returns a list of Modelica components
     * 
     * @return list of Modelica components
     */
    public List<LibraryComponent> getModelicaComponents()
    {
        return Collections.unmodifiableList(lmcomponents);
    }
    
    /**
     * Returns all the PhS flow properties
     * 
     * @param uclass
     *            classifier in which the sim properties are looked for
     * @return list of sim properties
     */
    public List<Property> getOwnedPhSProperties(Classifier uclassifier)
    {
        Set<Property> lp = getOwnedAttributes(uclassifier);
        ArrayList<Property> properties = new ArrayList<Property>();
        for (Property uproperty : lp)
        {
            if (!uproperty.isStereotypeApplied(getFlowProperty()))
                continue;
            if (uproperty.isStereotypeApplied(sphsvariable) || isSimBlock((Classifier) uproperty.getType()))
                properties.add(uproperty);
        }
        return properties;
    }
    
    public boolean isPhSProperty(Property uproperty)
    {
        if (uproperty == null)
            return false;
        if (!uproperty.isStereotypeApplied(getFlowProperty()))
            return false;
        if (uproperty.getType() instanceof DataType && uproperty.isStereotypeApplied(sphsvariable))
            return true;
        else if (uproperty.getType() instanceof Classifier && isSimBlock((Classifier) uproperty.getType()))
            return true;
        return false;
    }
    
    public boolean isSimBlock(Classifier uclassifier)
    {
        if (uclassifier == null)
            return false;
        for (Classifier ugeneral : uclassifier.getGenerals())
            if (ugeneral == sconservedsubstance)
                return true;
            else if (isSimBlock(ugeneral))
                return true;
        return false;
        
    }
    
    /**
     * Returns all the sim properties, even inherited and accounting for
     * redefinitions.
     * 
     * @param uclass
     *            classifier in which the sim properties are looked for
     * @return list of sim properties
     */
    public List<Property> getAllPhSProperties(Classifier uclassifier)
    {
        Set<Property> lp = getAllCorrectedAttributes(uclassifier);
        
        ArrayList<Property> ret = new ArrayList<Property>();
        for (Property uproperty : lp)
        {
            if (isPhSProperty(uproperty))
                ret.add(uproperty);
        }
        return ret;
    }
    
    /**
     * Returns all the sim variables in the given property's type
     * 
     * @param uproperty
     *            property whose type is used to look for sim variables
     * @return list of sim variable in the property's type
     */
    public List<Property> getSimVariables(Property uproperty)
    {
        ArrayList<Property> properties = new ArrayList<Property>();
        if (uproperty.getType() instanceof Class)
        {
            for (Property property : getAllAttributes((Class) uproperty.getType()))
                if (property.isStereotypeApplied(getPhSVariable()))
                    properties.add(property);
        }
        return properties;
    }
    
    public static List<Classifier> getAllCompatibleTypes(Classifier uclassifier)
    {
        LinkedList<Classifier> ret = new LinkedList<Classifier>();
        if (uclassifier == null)
            return ret;
        ret.add(uclassifier);
        for (Classifier general : uclassifier.getGenerals())
        {
            ret.addAll(getAllCompatibleTypes(general));
        }
        return ret;
    }
    
    /*
     * Removes any sim property in the specified path
     *
     * @param properties list in which sim properties should be removed
     */
    public void removeSimProperty(List<Property> properties)
    {
        int i = 0;
        while (i != properties.size())
        {
            if (properties.get(i).isStereotypeApplied(getFlowProperty()))
            {
                properties.remove(i);
                continue;
            }
            i++;
        }
    }
    
    public static void setName(NamedElement ne, String name)
    {
        if (name == null)
        {
            ne.setName(null);
            return;
        }
        // raw name, not filtered since everything is allowed
        String rname = name.length() == 0 ? "_" + ne.getClass().getName() : name;
        
        List<NamedElement> nes = new LinkedList<NamedElement>();
        Namespace ns = ne.getNamespace();
        if (ns != null)
            nes.addAll(ns.getOwnedMembers());
        
        if (!isPresent(rname, nes))
        {
            ne.setName(rname);
            return;
        }
        
        int i = 0;
        while (true)
        {
            String nname = rname + (i++);
            if (!isPresent(nname, nes))
            {
                ne.setName(nname);
                return;
            }
        }
    }
    
    private static boolean isPresent(String name, List<NamedElement> nes)
    {
        if (name != null)
            for (NamedElement ne : nes)
                if (name.equals(ne.getName()))
                    return true;
        return false;
    }
    
    /**
     * Returns a list of property joined by the specified separator
     * 
     * @param properties
     *            path to join
     * @param separator
     *            string to add between the elements
     * @return joined properties
     */
    public String joinProperties(List<Property> properties, String separator)
    {
        if (properties == null || separator == null)
            return "";
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < properties.size(); i++)
        {
            Property p = properties.get(i);
            if (i != 0)
                sb.append(separator);
            sb.append(getName(p));
            if (p.isStereotypeApplied(smultidimelement)
                    && p.getValue(smultidimelement, "dimensions") instanceof List<?>)
            {
                @SuppressWarnings("unchecked")
                List<Integer> dims = (List<Integer>) p.getValue(smultidimelement, "dimensions");
                if (dims.size() == 1 && dims.get(0).equals(Integer.valueOf(1)))
                    sb.append("[1]");
            }
        }
        return sb.toString();
    }
    
    /**
     * Library block class
     * 
     * @author barbau
     *
     */
    public class LibraryComponent
    {
        private String name;
        private Class clas;
        
        private List<LibraryParameter> parameters = new LinkedList<LibraryParameter>();
        private List<LibraryPort> ports = new LinkedList<LibraryPort>();
        
        private LibraryComponent(Class clas, String name)
        {
            this.clas = clas;
            this.name = name;
        }
        
        public String getName()
        {
            return name;
        }
        
        public Class getClas()
        {
            return clas;
        }
        
        public List<LibraryParameter> getParameters()
        {
            return Collections.unmodifiableList(parameters);
        }
        
        public List<LibraryPort> getPorts()
        {
            return Collections.unmodifiableList(ports);
        }
        
        public List<LibraryPort> getInputPorts()
        {
            List<LibraryPort> ret = new LinkedList<LibraryPort>();
            for (LibraryPort lp : ports)
            {
                if (!(lp.getPort().getType() instanceof Classifier))
                    continue;
                Set<Property> lfp = getAllFlowProperties((Classifier) lp.getPort().getType());
                if (lfp.size() == 1)
                {
                    Object o = lfp.iterator().next().getValue(getFlowProperty(), "direction");
                    if (o instanceof EnumerationLiteral)
                    {
                        EnumerationLiteral el = ((EnumerationLiteral) o);
                        if ((el.getName().equals("in") && !lp.getPort().isConjugated())
                                || (el.getName().equals("out") && lp.getPort().isConjugated()))
                            
                            ret.add(lp);
                    }
                }
            }
            return Collections.unmodifiableList(ret);
        }
        
        public List<LibraryPort> getOutputPorts()
        {
            List<LibraryPort> ret = new LinkedList<LibraryPort>();
            for (LibraryPort lp : ports)
            {
                if (!(lp.getPort().getType() instanceof Classifier))
                    continue;
                Set<Property> lfp = getAllFlowProperties((Classifier) lp.getPort().getType());
                if (lfp.size() == 1)
                {
                    Object o = lfp.iterator().next().getValue(getFlowProperty(), "direction");
                    if (o instanceof EnumerationLiteral)
                    {
                        EnumerationLiteral el = ((EnumerationLiteral) o);
                        if ((el.getName().equals("out") && !lp.getPort().isConjugated())
                                || (el.getName().equals("in") && lp.getPort().isConjugated()))
                            ret.add(lp);
                    }
                }
            }
            return Collections.unmodifiableList(ret);
        }
    }
    
    /**
     * Library parameter class
     * 
     * @author barbau
     *
     */
    public class LibraryParameter
    {
        private Property property;
        private String name;
        private String value;
        
        private LibraryParameter(Property property, String name, String value)
        {
            this.property = property;
            this.name = name;
            this.value = value;
        }
        
        public Property getProperty()
        {
            return property;
        }
        
        public String getName()
        {
            return name;
        }
        
        public String getValue()
        {
            return value;
        }
    }
    
    /**
     * Library port class
     * 
     * @author barbau
     *
     */
    public class LibraryPort
    {
        private Port port;
        private String name;
        
        private LibraryPort(Port port, String name)
        {
            this.port = port;
            this.name = name;
        }
        
        public Port getPort()
        {
            return port;
        }
        
        public String getName()
        {
            return name;
        }
    }
    
    public static Resource createSysPhSModel(ResourceSet rs, URI uri, List<URI> lpath)
    {
        Resource r = SysMLUtil.createSysMLModel(rs, uri, lpath);
        
        Model model = (Model) r.getContents().stream().filter(e -> e instanceof Model).findFirst().orElse(null);
        if (model != null)
        {
            Resource sysphs = EMFUtil.loadResource(rs, nsSysPhS.appendSegment(nameSysPhSProfile), lpath);
            if (sysphs != null)
            {
                TreeIterator<EObject> ti = sysphs.getAllContents();
                while (ti.hasNext())
                {
                    EObject eo = ti.next();
                    if (eo instanceof Profile && ((Profile) eo).getURI().equals(nsSysPhS.toString()))
                    {
                        model.applyProfile((Profile) eo);
                        log.info("Applying SysPhS profile");
                        break;
                    }
                    if (!(eo instanceof Package))
                        ti.prune();
                }
            }
            else
                log.error("Unable to locate the SysPhS profile");
            
            Resource sysphslibrary = EMFUtil.loadResource(rs, nsSysPhS.appendSegment(nameSysPhSLibrary), lpath);
            if (sysphslibrary != null)
            {
                TreeIterator<EObject> ti = sysphslibrary.getAllContents();
                while (ti.hasNext())
                {
                    EObject eo = ti.next();
                    if (eo instanceof Profile && (((Profile) eo).getURI().equals(nsSysPhSPlatformProfile.toString())
                            || ((Profile) eo).getURI().equals(nsSysPhSExtended.toString())))
                    {
                        model.applyProfile((Profile) eo);
                        log.info("Applying SysPhS platform profile");
                        break;
                    }
                    if (!(eo instanceof Package))
                        ti.prune();
                }
            }
            if (sysphslibrary == null)
                log.error("Unable to locate the SysPhS library");
            
        }
        return r;
    }
    
    /**
     * Returns a resource for the simulation library
     * 
     * @param rs
     *            resource set in which the library is loaded
     * @param file
     *            location from which the library is loaded
     * @param paths
     *            List of directories
     * @return the simulation library
     */
    public static Resource getSimulationLibrary(ResourceSet rs, URI file, List<URI> paths)
    {
        return EMFUtil.loadResourceWithDependencies(rs, nsSysPhS.appendSegment(nameSysPhSLibrary), paths);
    }
    
}
