package com.engisis.sysphs.util;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;

import org.antlr.v4.runtime.ANTLRInputStream;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTreeWalker;
import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.ValueSpecification;

import com.engisis.xmiutil.UMLModelErrorException;

/**
 * Generic SysML to simulation translator
 * 
 * @author barbau
 *
 */
public abstract class SysMLToSimulationTranslator
{
    private static final Logger log = Logger.getLogger(SysMLToSimulationTranslator.class);
    
    /**
     * map with value specifications as keys, and syntax tree nodes as values
     */
    private Hashtable<ValueSpecification, ParserRuleContext> htVSs = new Hashtable<ValueSpecification, ParserRuleContext>();
    /**
     * map with opaque behaviors as keys, and syntax tree nodes as values
     */
    private Hashtable<OpaqueBehavior, ParserRuleContext> htOBs = new Hashtable<OpaqueBehavior, ParserRuleContext>();
    
    /**
     * Suffix of the file name
     */
    private String filenamesuffix;
    /**
     * Suffix of the model name
     */
    private String modelnamesuffix;
    /**
     * SysML resource
     */
    protected Resource r;
    /**
     * Output file name
     */
    protected String outputfilename;
    
    public SysMLToSimulationTranslator()
    {
    }
    
    /**
     * Sets the file name suffix
     * 
     * @param suffix
     *            file name suffix
     */
    public void setFileNameSuffix(String suffix)
    {
        this.filenamesuffix = suffix;
    }
    
    /**
     * Gets the file name suffix
     * 
     * @return file name suffix
     */
    public String getFileNameSuffix()
    {
        return filenamesuffix == null ? "" : filenamesuffix;
    }
    
    /**
     * Sets the model name suffix
     * 
     * @param suffix
     *            model name suffix
     */
    public void setModelNameSuffix(String suffix)
    {
        this.modelnamesuffix = suffix;
    }
    
    /**
     * Gets the model name suffix
     * 
     * @return model name suffix
     */
    public String getModelNameSuffix()
    {
        return modelnamesuffix == null ? "" : modelnamesuffix;
    }
    
    /**
     * Loads the given set of options
     * 
     * @param options
     */
    public void loadOptions(Set<Object> options)
    {
        if (options == null)
            return;
        Set<Object> values = getOptionValues();
        for (Object o : options)
            if (!values.contains(o))
                throw new IllegalArgumentException("The option is not valid: " + o);
    }
    
    /**
     * Returns the options
     * 
     * @return set of options
     */
    public abstract Set<java.lang.Class<?>> getOptions();
    
    /**
     * Returns the acceptable option values
     * 
     * @return srt of option values
     */
    public final Set<Object> getOptionValues()
    {
        HashSet<Object> res = new HashSet<Object>();
        Set<java.lang.Class<?>> options = getOptions();
        if (options != null)
            for (java.lang.Class<?> en : options)
                if (en.isEnum())
                    for (Object o : en.getEnumConstants())
                        res.add(o);
        return res;
    }
    
    /**
     * Executes the transformation.
     * 
     * @param rootblock
     *            root SysML block name
     * @param r
     *            SysML resource
     * @param outputdirectory
     *            output directory
     * @throws UMLModelErrorException
     * @throws IOException
     */
    public abstract void execute(Class rootblock, Resource r, File outputdirectory)
            throws UMLModelErrorException, IOException;
    
    /**
     * Resets the object for another translation
     */
    public void reset()
    {
        htVSs.clear();
        htOBs.clear();
    }
    
    /**
     * Returns the generated simulation file name
     * 
     * @return generated simulation file name
     */
    public String getOutputFileName()
    {
        return outputfilename;
    }
    
    /**
     * Key to use to identify the source objects in the translation
     * 
     * @author barbau
     *
     */
    public static class ReferenceKey
    {
        private Element[] key;
        
        public ReferenceKey(Element[] key)
        {
            this.key = key.clone();
        }
        
        public Element[] getKey()
        {
            return key.clone();
        }
        
        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof ReferenceKey)
            {
                ReferenceKey rk = (ReferenceKey) obj;
                if (rk.key.length != key.length)
                    return false;
                for (int i = 0; i < key.length; i++)
                    if (!EcoreUtil.getURI(rk.key[i]).equals(EcoreUtil.getURI(key[i])))
                        return false;
                return true;
            }
            return false;
        }
        
        @Override
        public int hashCode()
        {
            return toString().hashCode();
        }
        
        @Override
        public String toString()
        {
            StringBuilder sb = new StringBuilder();
            for (Element elem : key)
                sb.append(EcoreUtil.getURI(elem) + " ");
            return sb.toString();
        }
    }
    
    /**
     * Returns a key for the specified objects
     * 
     * @param elements
     *            objects for which the key corresponds
     * @return key
     */
    public static ReferenceKey getKey(Element... elements)
    {
        return new ReferenceKey(elements);
    }
    
    /**
     * Prints the given SysML objects
     * 
     * @param element
     *            SysML object
     * @return string representing the object
     */
    public static String print(Element element)
    {
        StringBuilder sb = new StringBuilder();
        if (element instanceof Connector)
        {
            Connector c = (Connector) element;
            for (int i = 0; i < c.getEnds().size(); i++)
            {
                if (i != 0)
                    sb.append("-");
                sb.append(print(c.getEnds().get(i)));
            }
        }
        if (element instanceof NamedElement)
        {
            NamedElement ne = ((NamedElement) element);
            if (ne.getNamespace() != null)
                sb.append(ne.getNamespace().getQualifiedName() + "::");
            sb.append(ne.getName());
        }
        else
            sb.append(EcoreUtil.getURI(element).fragment());
        return sb.toString();
    }
    
    /**
     * Prints the reference key
     * 
     * @param rk
     *            reference key
     * @return string representing the reference key
     */
    public static String print(ReferenceKey rk)
    {
        StringBuilder sb = new StringBuilder();
        for (Element elem : rk.getKey())
        {
            if (elem != rk.getKey()[0])
                sb.append("/");
            sb.append(print(elem));
        }
        return sb.toString();
    }
    
    /**
     * Returns a string corresponding to the language used by the simulation
     * platform
     * 
     * @return language string
     */
    protected abstract String getOpaqueLanguage();
    
    /**
     * Returns the expression translator
     * 
     * @return expression translator
     */
    protected abstract ExpressionLanguageToSimulation getExpressionTranslator();
    
    /**
     * Returns a string corresponding to the equation in the value specification
     * 
     * @param vs
     *            value specification
     * @return string
     * @throws UMLModelErrorException
     */
    public String getEquation(ValueSpecification vs) throws UMLModelErrorException
    {
        ParserRuleContext prc = htVSs.get(vs);
        if (prc == null)
        {
            // if language explicitly specified, return associated body
            // if not, parse body without language and treat it as Modelica
            
            String val = vs.stringValue();
            if (vs instanceof OpaqueExpression)
            {
                OpaqueExpression oe = (OpaqueExpression) vs;
                List<String> languages = oe.getLanguages();
                if (languages.size() > 0)
                {
                    for (int i = 0; i < languages.size(); i++)
                    {
                        String lng = languages.get(i).trim();
                        if (getOpaqueLanguage().equals(lng))
                            return oe.getBodies().get(i);
                        else if (lng == null || lng.equals("") || lng.equals(SysPhSUtil.SYSPHS_LANGUAGE))
                            val = oe.getBodies().get(i).trim();
                        else
                            log.warn("Unrecognized language " + lng);
                    }
                }
            }
            if (val != null)
            {
                ExpressionLanguageLexer ell = new ExpressionLanguageLexer(new ANTLRInputStream(val));
                CommonTokenStream cts = new CommonTokenStream(ell);
                ExpressionLanguageParser elp = new ExpressionLanguageParser(cts);
                elp.removeErrorListeners();
                elp.addErrorListener(new ANTLRErrorListener(log));
                prc = elp.equation();
                htVSs.put(vs, prc);
            }
            else
                throw new UMLModelErrorException(r, "No body found with: " + print(vs));
        }
        ParseTreeWalker ptw = new ParseTreeWalker();
        ExpressionLanguageToSimulation translator = getExpressionTranslator();
        ptw.walk(translator, prc);
        return translator.getValue(prc);
    }
    
    /**
     * Returns a string corresponding to the equation in the value specification
     * 
     * @param vs
     *            value specification
     * @return string
     * @throws UMLModelErrorException
     */
    public String getExpression(ValueSpecification vs) throws UMLModelErrorException
    {
        ParserRuleContext prc = htVSs.get(vs);
        if (prc == null)
        {
            // if language explicitly specified, return associated body
            // if not, parse body without language and treat it as Modelica
            
            String val = vs.stringValue();
            if (vs instanceof OpaqueExpression)
            {
                OpaqueExpression oe = (OpaqueExpression) vs;
                List<String> languages = oe.getLanguages();
                if (languages.size() > 0)
                {
                    for (int i = 0; i < languages.size(); i++)
                    {
                        String lng = languages.get(i).trim();
                        if (getOpaqueLanguage().equals(lng))
                            return oe.getBodies().get(i);
                        else if (lng == null || lng.equals("") || lng.equals(SysPhSUtil.SYSPHS_LANGUAGE))
                            val = oe.getBodies().get(i).trim();
                        else
                            log.warn("Unrecognized language " + lng);
                    }
                }
            }
            if (val != null)
            {
                ExpressionLanguageLexer ell = new ExpressionLanguageLexer(new ANTLRInputStream(val));
                CommonTokenStream cts = new CommonTokenStream(ell);
                ExpressionLanguageParser elp = new ExpressionLanguageParser(cts);
                elp.removeErrorListeners();
                elp.addErrorListener(new ANTLRErrorListener(log));
                prc = elp.expression();
                htVSs.put(vs, prc);
            }
            else
                throw new UMLModelErrorException(r, "No body found with: " + print(vs));
        }
        ParseTreeWalker ptw = new ParseTreeWalker();
        ExpressionLanguageToSimulation translator = getExpressionTranslator();
        ptw.walk(translator, prc);
        return translator.getValue(prc);
    }
    
    /**
     * Returns a string corresponding to the assignments in the opaque behavior
     * 
     * @param ob
     *            opaque behavior
     * @return string
     * @throws UMLModelErrorException
     */
    public String getAssignments(OpaqueBehavior ob) throws UMLModelErrorException
    {
        ParserRuleContext prc = htOBs.get(ob);
        if (prc == null)
        {
            
            // if language explicitly specified, return associated body
            // if not, parse body without language and treat it as Modelica
            String val = null;
            List<String> languages = ob.getLanguages();
            if (languages.size() > 0)
            {
                for (int i = 0; i < languages.size(); i++)
                {
                    String lng = languages.get(i).trim();
                    if (getOpaqueLanguage().equals(lng))
                        return ob.getBodies().get(i);
                    else if (lng == null || lng.isEmpty() || lng.equals(SysPhSUtil.SYSPHS_LANGUAGE))
                        val = ob.getBodies().get(i).trim();
                    else
                        log.warn("Unrecognized language " + lng);
                }
            }
            else if (ob.getBodies().size() != 0)
                val = ob.getBodies().get(0);
            
            if (val == null)
                throw new UMLModelErrorException(r, "No body found with: " + print(ob));
            
            ExpressionLanguageLexer ell = new ExpressionLanguageLexer(new ANTLRInputStream(val));
            CommonTokenStream cts = new CommonTokenStream(ell);
            ExpressionLanguageParser elp = new ExpressionLanguageParser(cts);
            elp.removeErrorListeners();
            elp.addErrorListener(new ANTLRErrorListener(log));
            prc = elp.statements();
            htOBs.put(ob, prc);
        }
        ParseTreeWalker ptw = new ParseTreeWalker();
        ptw.walk(getExpressionTranslator(), prc);
        return getExpressionTranslator().getValue(prc);
    }
    
}
