package com.engisis.sysphs.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EStructuralFeature.Setting;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.uml2.uml.AssociationClass;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.InstanceSpecification;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.StructuredClassifier;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLFactory;

import com.engisis.xmiutil.UMLModelErrorException;

/**
 * Preprocessor for SysML models. The preprocessor performs the following tasks:
 * 1. Replace connector properties by the content of their association blocks.
 * 2. Replace non-sim ports by parts 3. Add sim ports and associated constraints
 * on blocks that specialize sim port types 4. Replace connectors between parts
 * by connector between matching parts' sim ports 5. Split connectors involving
 * nested connector ends (not encapsulated) 6. Replace "long hand" signal flow
 * modeling by "short hand" 7. Split sim ports/sim port types with multiple flow
 * properties, to have one flow property per sim port
 * 
 * @author barbau
 *
 */
public class SysMLPreProcessor
{
    private static final Logger log = Logger.getLogger(SysMLPreProcessor.class);
    
    private SysPhSUtil sysphsutil;
    private Resource r;
    private List<Class> classes;
    private List<AssociationClass> associationclasses;
    private List<Property> properties;
    private List<Port> ports;
    private List<Connector> connectors;
    private List<InstanceSpecification> instances;
    // maps of ports created for sim property
    private Hashtable<Classifier, List<Port>> htports1;
    private Hashtable<Port, List<Port>> htports2;
    
    /**
     * Constructs a preprocessor for a specified SysML model
     * 
     * @param r
     *            Resource on which the preprocessing transformations are
     *            performed
     * @throws UMLModelErrorException
     */
    public SysMLPreProcessor(Resource r) throws UMLModelErrorException
    {
        sysphsutil = new SysPhSUtil(r.getResourceSet());
        this.r = r;
        classes = new LinkedList<Class>();
        associationclasses = new LinkedList<AssociationClass>();
        properties = new LinkedList<Property>();
        ports = new LinkedList<Port>();
        connectors = new LinkedList<Connector>();
        instances = new LinkedList<InstanceSpecification>();
        htports1 = new Hashtable<Classifier, List<Port>>();
        htports2 = new Hashtable<Port, List<Port>>();
    }
    
    /**
     * Executes the preprocessing
     * 
     * @throws UMLModelErrorException
     */
    public void execute() throws UMLModelErrorException
    {
        // populate the various lists
        TreeIterator<EObject> ti = r.getAllContents();
        while (ti.hasNext())
        {
            EObject eo = ti.next();
            if (eo instanceof AssociationClass)
            {
                associationclasses.add((AssociationClass) eo);
                for (Connector uconnector : ((Class) eo).getOwnedConnectors())
                {
                    connectors.add(uconnector);
                }
            }
            else if (eo instanceof Class)
            {
                classes.add((Class) eo);
                for (Connector uconnector : ((Class) eo).getOwnedConnectors())
                {
                    connectors.add(uconnector);
                }
            }
            else if (eo instanceof Port)
                ports.add((Port) eo);
            else if (eo instanceof Property)
                properties.add((Property) eo);
            else if (eo instanceof InstanceSpecification)
                instances.add((InstanceSpecification) eo);
            
            if (!(eo instanceof Namespace))
                ti.prune();
        }
        
        int cit;
        int poit;
        int pit;
        int coit;
        int iit;
        
        // 0. Remove binding connectors between parts
        for (coit = 0; coit < connectors.size(); coit++)
        {
            Connector c = connectors.get(coit);
            if (c.isStereotypeApplied(sysphsutil.getBindingConnector()))
            {
                List<Property> lp0 = sysphsutil.getPropertyPath(c.getEnds().get(0));
                List<Property> lp1 = sysphsutil.getPropertyPath(c.getEnds().get(1));
                Property p0 = lp0.get(lp0.size() - 1);
                Type t0 = p0.getType();
                Property p1 = lp1.get(lp1.size() - 1);
                Type t1 = p1.getType();
                
                Property proxy = null;
                if (t0 instanceof Class && t1 instanceof Class)
                {
                    log.info("Taking care of binding " + print(c));
                    List<Property> fr = null;
                    List<Property> to = null;
                    if (lp0.size() > lp1.size())
                    {
                        fr = lp1;
                        to = lp0;
                        proxy = p1;
                    }
                    else if (lp1.size() > lp0.size())
                    {
                        fr = lp0;
                        to = lp1;
                        proxy = p0;
                    }
                    else
                    {
                        log.error("Binding connector between two elements at the same level: " + print(c));
                        continue;
                    }
                    if (fr.size() > 0 && to.size() > 0)
                        for (int coit2 = 0; coit2 < connectors.size(); coit2++)
                        {
                            Connector c2 = connectors.get(coit2);
                            loop1: for (ConnectorEnd ce : c2.getEnds())
                            {
                                List<Property> lp = sysphsutil.getPropertyPath(ce);
                                loop2: for (int i = 0; i < lp.size(); i++)
                                {
                                    for (int j = 0; j < fr.size(); j++)
                                    {
                                        if (lp.get(i + j) != fr.get(j))
                                            continue loop2;
                                    }
                                    // success!
                                    for (int j = 0; j < fr.size(); j++)
                                        lp.remove(i);
                                    lp.addAll(i, to);
                                    sysphsutil.updatePropertyPath(ce, lp);
                                    log.info("Changed connector to " + print(c));
                                    continue loop1;
                                }
                            }
                        }
                    
                    log.info("Removing binding " + print(c));
                    connectors.remove(coit--);
                    EcoreUtil.delete(c, true);
                    // not sure if proxy can be removed
                    log.info("Removing proxy " + print(proxy));
                    EcoreUtil.delete(proxy, true);
                    properties.remove(proxy);
                    ports.remove(proxy);
                }
            }
        }
        
        // 1. Replace connector properties by the content of their association
        // blocks.
        // for (acit = 0; acit < associationclasses.size(); acit++)
        // {
        // AssociationClass ac = associationclasses.get(acit);
        // transformAssociationBlocks(ac);
        // }
        for (pit = 0; pit < properties.size(); pit++)
        {
            Property p = properties.get(pit);
            if (p.isStereotypeApplied(sysphsutil.getConnectorProperty())
                    && p.getValue(sysphsutil.getConnectorProperty(), "connector") != null)
            {
                
                transformConnectorProperty(p);
                properties.remove(pit--);
                log.info("Removing connector property " + print(p));
                EcoreUtil.delete(p, true);
            }
        }
        
        // 2. Replace non-sim ports by parts
        
        for (poit = 0; poit > ports.size(); poit++)
        {
            Port uport = ports.get(poit);
            Type utype = uport.getType();
            if (utype instanceof Classifier && !isPortType((Classifier) uport.getType()))
            {
                
                log.info("Replacing port " + print(uport));
                Property uproperty = UMLFactory.eINSTANCE.createProperty();
                ((Class) uport.getOwner()).getOwnedAttributes().add(uproperty);
                replace(uport, uproperty);
                ports.remove(poit--);
                EcoreUtil.delete(uport, true);
                properties.add(uproperty);
            }
        }
        
        // List classes that own sim properties (will have generalizations
        // removed later)
        
        List<Classifier> spclasses = new LinkedList<Classifier>();
        
        // 3. Have one sim property per class, move corresponding flow property
        // in that class
        for (cit = 0; cit < classes.size(); cit++)
        {
            Class uclass = classes.get(cit);
            List<Property> sps = sysphsutil.getOwnedPhSProperties(uclass);
            
            // port type, 1 sim property: use class as is
            // port type, 2+ sim property: extract additional sim properties
            // into generalizations
            // non port type, extract sim properties into
            // generalizations
            
            if (sps.size() > 0)
            {
                if (isPortType(uclass))
                {
                    // move flow property to same class
                    Property fp = sps.get(0);
                    moveProperty(fp, uclass);
                    spclasses.add(uclass);
                    
                    for (int j = 1; j < sps.size(); j++)
                    {
                        fp = sps.get(j);
                        
                        Classifier spcontainer = createPhSPropertyContainer(fp);
                        if (spcontainer instanceof Class)
                            classes.add((Class) spcontainer);
                        spclasses.add(spcontainer);
                        
                        moveProperty(fp, spcontainer);
                        addGeneralization(uclass, spcontainer);
                    }
                }
                else
                {
                    // always extract sim properties
                    
                    for (int j = 0; j < sps.size(); j++)
                    {
                        Property fp = sps.get(j);
                        
                        Classifier spcontainer = createPhSPropertyContainer(fp);
                        if (spcontainer instanceof Class)
                            classes.add((Class) spcontainer);
                        spclasses.add(spcontainer);
                        
                        moveProperty(fp, spcontainer);
                        
                        addGeneralization(uclass, spcontainer);
                    }
                }
            }
        }
        
        // 4. Create port for each inherited sim property in non-sim port types,
        // redirect connectors
        // split port, have one per sim property port, redirect connectors
        cloop: for (cit = 0; cit < classes.size(); cit++)
        {
            Class uclass = classes.get(cit);
            
            // skip if port type
            if (isPortType(uclass))
                continue cloop;
            
            // take care of sim properties inherited from port types
            List<Property> sps = new LinkedList<Property>();
            
            // make sure the ports will be created as early as possible
            for (Classifier ugeneral : uclass.getGenerals())
                if (isPortType(ugeneral))
                    sps.addAll(sysphsutil.getAllPhSProperties(ugeneral));
                
            List<Port> lports = new LinkedList<Port>();
            for (Property sp : sps)
            {
                // create port
                Port p = UMLFactory.eINSTANCE.createPort();
                uclass.getOwnedPorts().add(p);
                setUniqueName(p, lowerFirst(uclass.getName()) + upperFirst(sp.getName()));
                log.info("Adding port " + p.getName() + " for inherited " + print(sp) + " in " + print(uclass));
                ports.add(p);
                
                Classifier sptype = findBetterPortType(uclass, sp);
                log.info("Setting type " + sptype.getQualifiedName());
                p.setType(sptype);
                lports.add(p);
                
                loop: for (coit = 0; coit < connectors.size(); coit++)
                {
                    Connector uconnector = connectors.get(coit);
                    
                    // connectors pointing to the inherited sim property must
                    // point to the sim property in the newly created ports
                    for (ConnectorEnd uconnectorend : uconnector.getEnds())
                    {
                        Type ut = (Type) uconnector.getOwner();
                        if (ut == null)
                        {
                            log.error("Owner of " + print(uconnector) + " null");
                            continue;
                        }
                        Property up = null;
                        if (!(uconnectorend.getRole() instanceof Property))
                            continue loop;
                        
                        List<Property> lp = sysphsutil.getPropertyPath(uconnectorend);
                        boolean mod = false;
                        for (int k = 0; k < lp.size(); k++)
                        {
                            up = lp.get(k);
                            if (ut instanceof Classifier)
                                up = sysphsutil.correctProperty(up, (Classifier) ut);
                            if (ut.conformsTo(uclass) && up == sp)
                            {
                                // insert port
                                up = p;
                                lp.add(k++, p);
                                mod = true;
                            }
                            ut = up.getType();
                        }
                        if (mod)
                        {
                            sysphsutil.updatePropertyPath(uconnectorend, lp);
                            log.info("Updating " + print(uconnectorend));
                        }
                    }
                    
                    // instance specs pointing at the inherited sim property
                    // must point at the sim property in the newly created port
                    for (iit = 0; iit < instances.size(); iit++)
                    {
                        InstanceSpecification is0 = instances.get(iit);
                        for (Slot s0 : is0.getSlots())
                        {
                            if (s0.getDefiningFeature() == sp)
                            {
                                InstanceSpecification is1 = UMLFactory.eINSTANCE.createInstanceSpecification();
                                is0.getNearestPackage().getPackagedElements().add(is1);
                                is1.getClassifiers().add((Classifier) p.getType());
                                Slot s1 = UMLFactory.eINSTANCE.createSlot();
                                is1.getSlots().add(s1);
                                s1.setDefiningFeature(sp);
                                s1.getValues().addAll(s0.getValues());
                                
                                s0.setDefiningFeature(p);
                                s0.getValues().clear();
                                InstanceValue iv0 = UMLFactory.eINSTANCE.createInstanceValue();
                                iv0.setInstance(is1);
                                
                                s0.getValues().add(iv0);
                                continue;
                            }
                        }
                    }
                }
            }
            if (!lports.isEmpty())
                htports1.put(uclass, lports);
        }
        
        // when a port has several sim properties, create additional ports
        for (poit = 0; poit < ports.size(); poit++)
        {
            Port p = ports.get(poit);
            if (p.getType() instanceof Classifier && isPortType((Classifier) p.getType()))
            {
                List<Property> sps = sysphsutil.getAllPhSProperties((Classifier) p.getType());
                if (sps.size() == 0)
                    continue;
                List<Port> lports = new LinkedList<Port>();
                Classifier utype = (Classifier) p.getType();
                p.setType(findBetterPortType(utype, sps.get(0)));
                log.info("Not changing port " + p.getName() + " for sp " + sysphsutil.getName(sps.get(0)));
                for (int i = 1; i < sps.size(); i++)
                {
                    // create port
                    Port p2 = UMLFactory.eINSTANCE.createPort();
                    Class uclass = (Class) p.getOwner();
                    uclass.getOwnedPorts().add(p2);
                    ports.add(poit++, p2);
                    Property sp = sps.get(i);
                    p2.setType(findBetterPortType(utype, sp));
                    setUniqueName(p2, lowerFirst(uclass.getName()) + upperFirst(sp.getName()));
                    log.info("Adding port " + p2.getName() + " for additional sp " + sysphsutil.getName(sp));
                    lports.add(p2);
                    
                    // redirect connectors involving simproperty, p.sp ->
                    // p2.sp
                    loop: for (coit = 0; coit < connectors.size(); coit++)
                    {
                        Connector uconnector = connectors.get(coit);
                        
                        for (ConnectorEnd uconnectorend : uconnector.getEnds())
                        {
                            if (!(uconnectorend.getRole() instanceof Property))
                                continue loop;
                            
                            List<Property> lp = sysphsutil.getPropertyPath(uconnectorend);
                            boolean mod = false;
                            for (int k = 0; k < lp.size() - 1; k++)
                            {
                                if (lp.get(k) == p && lp.get(k + 1) == sp)
                                {
                                    lp.remove(k);
                                    lp.add(k, p2);
                                    mod = true;
                                }
                            }
                            if (mod)
                            {
                                sysphsutil.updatePropertyPath(uconnectorend, lp);
                                log.info("Updating " + print(uconnectorend));
                            }
                        }
                    }
                    
                    // redirect instance specs
                    for (iit = 0; iit < instances.size(); iit++)
                    {
                        InstanceSpecification is0 = instances.get(iit);
                        loop: for (Slot s0 : is0.getSlots())
                        {
                            if (s0.getDefiningFeature() == p && s0.getValues().size() == 1)
                            {
                                InstanceSpecification is1 = ((InstanceValue) s0.getValues().get(0)).getInstance();
                                for (Slot s1 : is1.getSlots())
                                {
                                    if (s1.getDefiningFeature() == sp)
                                    {
                                        s0.getValues().clear();
                                        s0.setDefiningFeature(p2);
                                        continue loop;
                                    }
                                }
                            }
                        }
                    }
                    
                }
                if (!lports.isEmpty())
                    htports2.put(p, lports);
            }
        }
        
        List<Connector> newconnectors = new LinkedList<Connector>();
        // Change all connectors
        for (coit = 0; coit < connectors.size(); coit++)
        {
            Connector uconnector = connectors.get(coit);
            if (uconnector.isStereotypeApplied(sysphsutil.getBindingConnector()))
                continue;
            ConnectorEnd uconnectorend0 = uconnector.getEnds().get(0);
            Type e0t = uconnectorend0.getRole().getType();
            ConnectorEnd uconnectorend1 = uconnector.getEnds().get(1);
            Type e1t = uconnectorend1.getRole().getType();
            if (e0t instanceof Classifier && e1t instanceof Classifier)
            {
                List<Property> lsp0 = sysphsutil.getAllPhSProperties((Classifier) e0t);
                List<Property> lsp1 = sysphsutil.getAllPhSProperties((Classifier) e1t);
                if (lsp0 == null || lsp1 == null)
                    continue;
                loop: for (Property sp0 : lsp0)
                {
                    for (Property sp1 : lsp1)
                    {
                        if (sp0.getType() != null && sp0.getType() == sp1.getType())
                        {
                            List<Property> path0 = correctPath(uconnectorend0, sp0);
                            List<Property> path1 = correctPath(uconnectorend1, sp1);
                            
                            if (path0 != null && path1 != null)
                            {
                                Connector nuconnector = UMLFactory.eINSTANCE.createConnector();
                                ((Class) uconnector.getOwner()).getOwnedConnectors().add(nuconnector);
                                newconnectors.add(nuconnector);
                                
                                ConnectorEnd nuconnectorend0 = UMLFactory.eINSTANCE.createConnectorEnd();
                                nuconnector.getEnds().add(nuconnectorend0);
                                sysphsutil.updatePropertyPath(nuconnectorend0, path0);
                                
                                ConnectorEnd nuconnectorend1 = UMLFactory.eINSTANCE.createConnectorEnd();
                                nuconnector.getEnds().add(nuconnectorend1);
                                sysphsutil.updatePropertyPath(nuconnectorend1, path1);
                                
                                log.info("Creating connector " + print(nuconnector) + " for " + print(uconnector));
                            }
                            else
                            {
                                log.error("Can't create connector for " + print(uconnector) + ": " + print(path0) + "//"
                                        + print(path1));
                            }
                            continue loop;
                        }
                    }
                }
            }
            log.info("Removing " + print(uconnector));
            EcoreUtil.delete(uconnector, true);
            connectors.remove(coit--);
        }
        connectors.addAll(newconnectors);
        
        // 5. remove generalization from sim property containers
        for (Classifier uclassifier : spclasses)
        {
            log.info("Removing generalizations from " + sysphsutil.getName(uclassifier));
            uclassifier.getGeneralizations().clear();
        }
        
        // remove generalizations between port types and non-port types
        for (Class uclass : classes)
        {
            if (isPortType(uclass))
                continue;
            Iterator<Generalization> itgen = uclass.getGeneralizations().listIterator();
            while (itgen.hasNext())
            {
                Generalization ugeneralization = itgen.next();
                if (ugeneralization.getSpecific() == uclass && isPortType(ugeneralization.getGeneral()))
                {
                    itgen.remove();
                    log.info("Removing generalization between " + uclass.getQualifiedName() + " and "
                            + ugeneralization.getGeneral().getQualifiedName());
                }
            }
        }
        
        // 6. Replace "long hand" signal flow modeling by "short hand"
        
        for (Class uclass : classes)
        {
            List<Property> simproperties = sysphsutil.getAllPhSProperties(uclass);
            for (Property simproperty : simproperties)
            {
                // change type
                Type simpropertytype = simproperty.getType();
                if (simpropertytype instanceof Class)
                {
                    Set<Property> attrs = sysphsutil.getAllAttributes((Class) simpropertytype);
                    if (attrs.size() == 1)
                    {
                        Property simvariable = attrs.iterator().next();
                        if (simvariable.isStereotypeApplied(sysphsutil.getPhSVariable())
                                && !((Boolean) simvariable.getValue(sysphsutil.getPhSVariable(), "isConserved"))
                                        .booleanValue())
                        {
                            simproperty.setType(simvariable.getType());
                            for (Connector uconnector : connectors)
                            {
                                for (ConnectorEnd uconnectorend : uconnector.getEnds())
                                {
                                    List<Property> lp = sysphsutil.getPropertyPath(uconnectorend);
                                    for (int k = 0; k < lp.size() - 1; k++)
                                        if (lp.get(k) == simproperty && lp.get(k + 1) == simvariable)
                                        {
                                            lp.remove(k + 1);
                                            sysphsutil.updatePropertyPath(uconnectorend, lp);
                                            break;
                                        }
                                }
                            }
                        }
                    }
                }
                
            }
        }
        
        // 7. Split connectors involving nested connector ends (not
        // encapsulated)
        
        loop: for (coit = 0; coit < connectors.size(); coit++)
        {
            Connector uconnector = connectors.get(coit);
            if (uconnector.isStereotypeApplied(sysphsutil.getBindingConnector()))
                continue;
            
            for (int i = 0; i < uconnector.getEnds().size(); i++)
            {
                ConnectorEnd ce = uconnector.getEnds().get(i);
                List<Property> lp = sysphsutil.getPropertyPath(ce);
                // antepenu.penu.last -> antepenu.last - penu.last
                if (lp.size() > 2)
                {
                    log.info("Splitting connector " + print(uconnector));
                    Property last = lp.get(lp.size() - 1);
                    while (lp.size() != 2)
                    {
                        Property penu = lp.get(lp.size() - 2);
                        Property antepenu = lp.get(lp.size() - 3);
                        Type tantepenu = antepenu.getType();
                        
                        if (!(tantepenu instanceof Class))
                        {
                            log.error("Element must be a class: " + tantepenu.getQualifiedName());
                            continue loop;
                            
                        }
                        // create antepenu.last
                        Port proxy = UMLFactory.eINSTANCE.createPort();
                        ((Class) tantepenu).getOwnedPorts().add(proxy);
                        setUniqueName(proxy, last.getName());
                        ports.add(proxy);
                        proxy.setType(last.getType());
                        
                        // connect antepenu.last and penu.last
                        Connector nuconnector = UMLFactory.eINSTANCE.createConnector();
                        ((Class) tantepenu).getOwnedConnectors().add(nuconnector);
                        connectors.add(nuconnector);
                        List<Property> nlp = new ArrayList<Property>(2);
                        nlp.add(penu);
                        nlp.add(last);
                        
                        ConnectorEnd nce0 = UMLFactory.eINSTANCE.createConnectorEnd();
                        nuconnector.getEnds().add(nce0);
                        sysphsutil.updatePropertyPath(nce0, nlp);
                        ConnectorEnd nce1 = UMLFactory.eINSTANCE.createConnectorEnd();
                        nuconnector.getEnds().add(nce1);
                        nce1.setRole(proxy);
                        
                        log.info("Adding connector " + print(nuconnector));
                        
                        // remove penu+last, add proxy
                        lp.remove(lp.size() - 1);
                        lp.remove(lp.size() - 1);
                        lp.add(proxy);
                    }
                    sysphsutil.updatePropertyPath(uconnector.getEnds().get(i), lp);
                    log.info("Changed connector to " + print(uconnector));
                }
            }
        }
        
        // add proxy
        
    }
    
    /**
     * This method replaces associationblocks by blocks, copies participants as
     * ports, and replaces connectors
     */
    private void transformAssociationBlocks(AssociationClass ac)
    {
        log.debug("Transforming association class " + ac.getQualifiedName());
        // create ports
        EList<Property> memberends = new BasicEList<Property>(ac.getMemberEnds());
        for (int i = 0; i < memberends.size(); i++)
        {
            Property member = memberends.get(i);
            Port port = UMLFactory.eINSTANCE.createPort();
            ac.getOwnedPorts().add(port);
            log.info("Replacing member end " + sysphsutil.getName(member));
            replace(member, port);
            
            // set port ownership
        }
        
        // look for connector properties
        ListIterator<Property> pit = properties.listIterator();
        while (pit.hasNext())
        {
            Property uproperty = pit.next();
            if (!uproperty.isStereotypeApplied(sysphsutil.getConnectorProperty()))
                continue;
            Connector conn = (Connector) uproperty.getValue(sysphsutil.getConnectorProperty(), "connector");
            if (conn != null && conn.getType() == ac)
            {
                // two ends, connected to two memberends
                StructuredClassifier owner = ((StructuredClassifier) conn.getOwner());
                for (int i = 0; i < conn.getEnds().size(); i++)
                {
                    Connector nconn = UMLFactory.eINSTANCE.createConnector();
                    ConnectorEnd ce = conn.getEnds().get(i);
                    
                    ConnectorEnd nce0 = UMLFactory.eINSTANCE.createConnectorEnd();
                    nce0.setPartWithPort(ce.getPartWithPort());
                    nce0.setRole(ce.getRole());
                    
                    ConnectorEnd nce1 = UMLFactory.eINSTANCE.createConnectorEnd();
                    nce1.setPartWithPort(uproperty);
                    nce1.setRole(ac.getOwnedPorts().get(i));
                    
                    nconn.getEnds().add(nce0);
                    nconn.getEnds().add(nce1);
                    owner.getOwnedConnectors().add(nconn);
                    connectors.add(nconn);
                }
                EcoreUtil.delete(conn, true);
                connectors.remove(conn);
            }
        }
        
        // replace associationblock by block
        Class c = UMLFactory.eINSTANCE.createClass();
        Package owner = ac.getNearestPackage();
        owner.getPackagedElements().add(c);
        // print(ac);
        replace(ac, c);
        if (!c.isStereotypeApplied(sysphsutil.getBlock()))
            c.applyStereotype(sysphsutil.getBlock());
        // print(c);
        classes.add(c);
    }
    
    /**
     * This method replaces connector properties by the content of the
     * association blocks.
     * 
     * @throws UMLModelErrorException
     */
    private void transformConnectorProperty(Property cp) throws UMLModelErrorException
    {
        Class owner = (Class) cp.getOwner();
        Type t = cp.getType();
        Connector c = (Connector) cp.getValue(sysphsutil.getConnectorProperty(), "connector");
        
        if (t == null || t != c.getType())
        {
            throw new UMLModelErrorException(r, "The connector property and its connector must have the same type");
        }
        AssociationClass ac = (AssociationClass) t;
        Hashtable<Property, Property> map = new Hashtable<Property, Property>();
        for (Property ap : sysphsutil.getAllAttributes(ac))
        {
            // copy, except participants
            if (ap.isStereotypeApplied(sysphsutil.getParticipantProperty()) || ac.getMemberEnds().contains(ap))
                continue;
            Property ap2 = null;
            if (ap instanceof Port)
                ap2 = UMLFactory.eINSTANCE.createPort();
            else
                ap2 = UMLFactory.eINSTANCE.createProperty();
            owner.getOwnedAttributes().add(ap2);
            copy(ap, ap2);
            
            String ap2name = ap2.getName();
            ap2.setName(null);
            setUniqueName(ap2, ap2name);
            
            log.info("Adding " + print(ap2) + " to " + print(owner) + " for " + print(cp) + "." + print(ap));
            map.put(ap, ap2);
            
            // connector going to the cp.ap should go to ap2
            // owner's connector, or all?
            for (Connector c2 : owner.getOwnedConnectors())
            {
                loop: for (ConnectorEnd ce : c2.getEnds())
                {
                    List<Property> lp = sysphsutil.getPropertyPath(ce);
                    for (int i = 0; i < lp.size() - 1; i++)
                        if (lp.get(i) == cp && lp.get(i + 1) == ap)
                        {
                            lp.remove(i);
                            lp.remove(i);
                            lp.add(i, ap2);
                            String from = print(c2);
                            sysphsutil.updatePropertyPath(ce, lp);
                            log.info("Changing connector from " + from + " to " + print(c2));
                            continue loop;
                        }
                }
            }
            // slots giving value to cp.ap should go to ap2
            for (int iit = 0; iit < instances.size(); iit++)
            {
                InstanceSpecification is0 = instances.get(iit);
                loop: for (Slot s0 : is0.getSlots())
                {
                    if (s0.getDefiningFeature() == cp && s0.getValues().size() == 1)
                    {
                        InstanceSpecification is1 = ((InstanceValue) s0.getValues().get(0)).getInstance();
                        for (Slot s1 : is1.getSlots())
                        {
                            if (s1.getDefiningFeature() == ap2)
                            {
                                s0.getValues().clear();
                                s0.setDefiningFeature(ap2);
                                s0.getValues().addAll(s1.getValues());
                                continue loop;
                            }
                        }
                        
                    }
                }
            }
        }
        for (Connector oc : ac.getOwnedConnectors())
        {
            // copy, except that the connectors going to participants go to c's
            // ends
            Connector oc2 = UMLFactory.eINSTANCE.createConnector();
            owner.getOwnedConnectors().add(oc2);
            if (oc.isStereotypeApplied(sysphsutil.getBindingConnector()))
                oc2.applyStereotype(sysphsutil.getBindingConnector());
            connectors.add(oc2);
            
            List<Property> lp0 = sysphsutil.getPropertyPath(oc.getEnds().get(0));
            List<Property> lp1 = sysphsutil.getPropertyPath(oc.getEnds().get(1));
            // Note: getting the corresponding connector end from the member
            // end's position does NOT work due to a bug in EMF 2.11.1
            // use "bad" matching type as workaround
            
            Property lp0p = lp0.get(0);
            if (lp0p.isStereotypeApplied(sysphsutil.getParticipantProperty()))
            {
                lp0.remove(0);
                // Property memberend = (Property)
                // lp0p.getValue(sysmlutil.getParticipantProperty(), "end");
                // lp0.addAll(0,
                // sysmlutil.getPropertyPath(c.getEnds().get(ac.getMemberEnds().indexOf(memberend))));
                for (ConnectorEnd ce : c.getEnds())
                {
                    if (ce.getRole().getType() != null && lp0p.getType() != null
                            && ce.getRole().getType().conformsTo(lp0p.getType()))
                    {
                        lp0.addAll(0, sysphsutil.getPropertyPath(ce));
                        break;
                    }
                }
            }
            else if (map.containsKey(lp0p))
            {
                lp0.remove(0);
                lp0.add(0, map.get(lp0p));
            }
            else
                log.warn("Property " + print(lp0p) + " not a participant nor mapped");
            Property lp1p = lp1.get(0);
            if (lp1p.isStereotypeApplied(sysphsutil.getParticipantProperty()))
            {
                lp1.remove(0);
                // Property memberend = (Property)
                // lp1p.getValue(sysmlutil.getParticipantProperty(), "end");
                // lp1.addAll(0,
                // sysmlutil.getPropertyPath(c.getEnds().get(ac.getMemberEnds().indexOf(memberend))));
                for (ConnectorEnd ce : c.getEnds())
                {
                    if (ce.getRole().getType() != null && lp1p.getType() != null
                            && ce.getRole().getType().conformsTo(lp1p.getType()))
                    {
                        lp1.addAll(0, sysphsutil.getPropertyPath(ce));
                        break;
                    }
                }
            }
            else if (map.containsKey(lp1p))
            {
                lp1.remove(0);
                lp1.add(0, map.get(lp1p));
            }
            else
                log.warn("Property " + print(lp1p) + " not a participant nor mapped");
            
            ConnectorEnd ce0 = UMLFactory.eINSTANCE.createConnectorEnd();
            oc2.getEnds().add(ce0);
            sysphsutil.updatePropertyPath(ce0, lp0);
            
            ConnectorEnd ce1 = UMLFactory.eINSTANCE.createConnectorEnd();
            oc2.getEnds().add(ce1);
            sysphsutil.updatePropertyPath(ce1, lp1);
            log.info("Creating " + print(oc2) + " for " + print(oc));
        }
        log.info("Removing " + print(c));
        EcoreUtil.delete(c, true);
        connectors.remove(c);
    }
    
    /**
     * Returns true if the given classifier owns at least a sim property and/or
     * a flow property, but nothing else
     * 
     * @param uclassifier
     * @return
     */
    private boolean isPortType(Classifier uclassifier)
    {
        // if attributes && all attributes flow property or sim property
        List<Property> attrs = null;
        if (uclassifier instanceof Class)
            attrs = ((Class) uclassifier).getOwnedAttributes();
        else if (uclassifier instanceof Interface)
            attrs = ((Interface) uclassifier).getOwnedAttributes();
        else
            return false;
        if (attrs == null || attrs.size() == 0)
            return false;
        for (Property prop : attrs)
            if (!sysphsutil.isPhSProperty(prop))
                return false;
        return true;
    }
    
    /**
     * Returns the most specialized classifier than can be used as port type for
     * one sp.
     * 
     * @param uclassifier
     * @param sp
     * @return
     */
    private Classifier findBetterPortType(Classifier uclassifier, Property sp)
    {
        Property cursp = getPhSPropertyOfPortType(uclassifier);
        if (cursp == sp)
            return uclassifier;
        for (Classifier ugeneral : uclassifier.getGenerals())
        {
            Classifier ret = findBetterPortType(ugeneral, sp);
            if (ret != null)
                return ret;
        }
        return null;
    }
    
    private Property getPhSPropertyOfPortType(Classifier uclassifier)
    {
        if (!isPortType(uclassifier))
            return null;
        List<Property> lp = sysphsutil.getOwnedPhSProperties(uclassifier);
        if (lp.size() > 0)
            return lp.get(0);
        
        // if no owned sp, look at generic. Return null if more than one choice
        Property ret = null;
        for (Classifier ugeneral : uclassifier.getGenerals())
        {
            Property sp = getPhSPropertyOfPortType(ugeneral);
            if (sp != null)
            {
                if (ret == null)
                    ret = sp;
                else
                    return null;
            }
        }
        return ret;
    }
    
    private static void moveProperty(Property uproperty, Classifier target)
    {
        List<Property> lsp = null;
        Classifier source = (Classifier) uproperty.getOwner();
        if (source instanceof Class)
            lsp = ((Class) source).getOwnedAttributes();
        else if (source instanceof Interface)
            lsp = ((Interface) source).getOwnedAttributes();
        else
        {
            log.warn("Can't move property " + uproperty.getQualifiedName());
            return;
        }
        
        List<Property> ltp = null;
        if (target instanceof Class)
            ltp = ((Class) target).getOwnedAttributes();
        else if (target instanceof Interface)
            ltp = ((Interface) target).getOwnedAttributes();
        else
        {
            log.warn("Can't move property " + uproperty.getQualifiedName());
            return;
        }
        
        lsp.remove(uproperty);
        ltp.add(uproperty);
        
    }
    
    private Classifier createPhSPropertyContainer(Property sp)
    {
        Classifier owner = (Classifier) sp.getOwner();
        if (owner instanceof Class)
        {
            Class uclass = (Class) sp.getOwner();
            Class uclass2 = UMLFactory.eINSTANCE.createClass();
            uclass.getNearestPackage().getPackagedElements().add(uclass2);
            uclass2.applyStereotype(sysphsutil.getBlock());
            uclass.getOwnedAttributes().remove(sp);
            uclass2.getOwnedAttributes().add(sp);
            uclass2.setName(sysphsutil.getName(sp.getType()) + "Sim");
            
            log.info("Moving sp " + sysphsutil.getName(sp) + " from " + sysphsutil.getName(uclass) + " to "
                    + sysphsutil.getName(uclass2));
            return uclass2;
        }
        else if (owner instanceof Interface)
        {
            Interface uinterface = (Interface) sp.getOwner();
            Interface uinterface2 = UMLFactory.eINSTANCE.createInterface();
            uinterface.getNearestPackage().getPackagedElements().add(uinterface2);
            uinterface.getOwnedAttributes().remove(sp);
            uinterface2.getOwnedAttributes().add(sp);
            uinterface2.setName(sysphsutil.getName(sp.getType()) + "Sim");
            
            log.info("Moving sp " + sysphsutil.getName(sp) + " from " + sysphsutil.getName(uinterface) + " to "
                    + sysphsutil.getName(uinterface2));
            return uinterface2;
        }
        return null;
    }
    
    /**
     * Add a sim port to a path if needed
     * 
     * @param uconnectorend
     *            connector on which the sim port may be added
     * @param sp
     *            sim property for which the port may be added
     * @return
     */
    private List<Property> correctPath(ConnectorEnd uconnectorend, Property sp)
    {
        if (uconnectorend == null || sp == null)
            return null;
        
        List<Property> path = sysphsutil.getPropertyPath(uconnectorend);
        if (path == null)
            return null;
        
        int size = path.size();
        if (size == 0)
            return null;
        
        Property last = path.get(size - 1);
        Type lasttype = last.getType();
        if (lasttype instanceof Classifier)
        {
            Classifier clasttype = (Classifier) lasttype;
            
            Classifier spcontainer = findBetterPortType(clasttype, sp);
            // when not a sim port type, add the port associated with the sim
            // property
            if (!isPortType(clasttype))
            {
                List<Port> lports = getPortList(clasttype);
                for (Port uport : lports)
                    if (uport.getType() == spcontainer)
                    {
                        path.add(uport);
                        return path;
                    }
                log.error("Couldn't get the port for " + print(sp) + " in " + clasttype.getQualifiedName() + ", found "
                        + lports.size() + " potentially");
            }
            // when a sim port type, redirect to another port if the type is not
            // the one corresponding to the sim property
            else
            {
                if (clasttype == spcontainer)
                    return path;
                Class ucontainer = null;
                if (size == 1)
                    ucontainer = (Class) ((Connector) uconnectorend.getOwner()).getOwner();
                else if (path.get(size - 2).getType() instanceof Class)
                    ucontainer = (Class) path.get(size - 2).getType();
                
                if (ucontainer != null)
                {
                    List<Port> lports = htports2.get(last);
                    if (lports != null)
                    {
                        for (Port uport : lports)
                            if (uport.getType() == spcontainer)
                            {
                                path.remove(size - 1);
                                path.add(uport);
                                return path;
                            }
                    }
                }
                log.error("Couldn't get the port for " + print(sp) + " in " + clasttype.getQualifiedName());
            }
        }
        return null;
    }
    
    private List<Port> getPortList(Classifier uclassifier)
    {
        List<Port> ret = new LinkedList<Port>();
        List<Port> lp = htports1.get(uclassifier);
        if (lp != null)
            ret.addAll(lp);
        for (Classifier ugeneral : uclassifier.getGenerals())
            ret.addAll(getPortList(ugeneral));
        return ret;
    }
    
    /**
     * Adds a generalization between the two classifiers
     * 
     * @param specific
     *            specific classifier
     * @param general
     *            general classifier
     */
    private void addGeneralization(Classifier specific, Classifier general)
    {
        if (!specific.getGenerals().contains(general))
        {
            Generalization ugeneralization = UMLFactory.eINSTANCE.createGeneralization();
            ugeneralization.setGeneral(general);
            ugeneralization.setSpecific(specific);
            specific.getGeneralizations().add(ugeneralization);
            log.info("Adding generalization between " + sysphsutil.getName(specific) + " and "
                    + sysphsutil.getName(general));
        }
    }
    
    private void setUniqueName(Property uproperty, String name)
    {
        Classifier owner = (Classifier) uproperty.getOwner();
        
        Set<Property> attrs = sysphsutil.getAllAttributes(owner);
        List<String> strs = new ArrayList<String>(attrs.size());
        for (Property up : attrs)
        {
            strs.add(up.getName());
        }
        
        if (!strs.contains(name))
        {
            uproperty.setName(name);
            return;
        }
        String name2;
        int i = 2;
        while (true)
        {
            name2 = name + (i++);
            if (!strs.contains(name2))
            {
                uproperty.setName(name2);
                return;
            }
        }
    }
    
    private static String lowerFirst(String s)
    {
        if (s == null || s.length() < 1)
            return s;
        return s.substring(0, 1).toLowerCase() + s.substring(1);
    }
    
    private static String upperFirst(String s)
    {
        if (s == null || s.length() < 1)
            return s;
        return s.substring(0, 1).toUpperCase() + s.substring(1);
    }
    
    /**
     * Copy the content of an object to another
     * 
     * @param efrom
     *            copied object
     * @param eto
     *            destination of the copy
     */
    private void copy(EObject efrom, EObject eto)
    {
        if (efrom instanceof Element && eto instanceof Element)
        {
            Element elfrom = (Element) efrom;
            Element elto = (Element) eto;
            for (Stereotype st : elfrom.getAppliedStereotypes())
            {
                if (elto.isStereotypeApplicable(st))
                {
                    elto.applyStereotype(st);
                    for (Property p : sysphsutil.getAllAttributes(st))
                    {
                        String pname = p.getName();
                        if (pname.startsWith("base_"))
                            continue;
                        Object v = elfrom.getValue(st, pname);
                        if (v != null)
                            elto.setValue(st, pname, v);
                    }
                }
            }
        }
        
        // copy features from old to new
        for (EAttribute ea : efrom.eClass().getEAllAttributes())
        {
            if (!ea.isChangeable() || ea.isDerived() || !efrom.eIsSet(ea))
                continue;
            if (eto.eClass().getEAllAttributes().contains(ea))
            {
                eto.eSet(ea, efrom.eGet(ea));
            }
            else
                log.warn("Can't copy attribute " + efrom.eClass().getName() + "." + ea.getName() + ", not contained in "
                        + eto.eClass().getName());
        }
        for (EReference er : efrom.eClass().getEAllReferences())
        {
            if (!er.isChangeable() || er.isDerived() || !efrom.eIsSet(er))
                continue;
            if (eto.eClass().getEAllReferences().contains(er))
            {
                eto.eSet(er, efrom.eGet(er));
            }
            else
                log.warn("Can't copy reference " + efrom.eClass().getName() + "." + er.getName() + ", not contained in "
                        + eto.eClass().getName());
        }
        
    }
    
    /**
     * Replace an object by another
     * 
     * @param eold
     *            object to be replaced
     * @param enew
     *            replacement object
     */
    @SuppressWarnings("unchecked")
    private void replace(EObject eold, EObject enew)
    {
        if (eold instanceof Element && enew instanceof Element)
        {
            Element elold = (Element) eold;
            Element elnew = (Element) enew;
            while (elold.getAppliedStereotypes().size() != 0)
            {
                Stereotype st = elold.getAppliedStereotypes().get(0);
                if (elnew.isStereotypeApplicable(st))
                {
                    elnew.applyStereotype(st);
                    for (Property p : sysphsutil.getAllAttributes(st))
                    {
                        String pname = p.getName();
                        if (pname.startsWith("base_"))
                            continue;
                        Object v = elold.getValue(st, pname);
                        if (v != null)
                            elnew.setValue(st, pname, v);
                    }
                }
                elold.unapplyStereotype(st);
            }
        }
        ResourceSet rs = eold.eResource().getResourceSet();
        if (!eold.eContainmentFeature().getEReferenceType().isSuperTypeOf(enew.eClass()))
        {
            log.error("The old object can't be replaced by the new one: incompatible containment feature");
            return;
        }
        // move contained features from old to new
        for (EReference er : eold.eClass().getEAllContainments())
        {
            replace(eold, enew, er);
        }
        // copy features from old to new
        for (EAttribute ea : eold.eClass().getEAllAttributes())
        {
            if (!ea.isChangeable() || ea.isDerived() || !eold.eIsSet(ea))
                continue;
            if (enew.eClass().getEAllAttributes().contains(ea))
            {
                enew.eSet(ea, eold.eGet(ea));
            }
            else
                log.warn("Can't translate attribute " + eold.eClass().getName() + "." + ea.getName()
                        + ", not contained in " + enew.eClass().getName());
        }
        for (EReference er : eold.eClass().getEAllReferences())
        {
            replace(eold, enew, er);
        }
        // copy cross referenced features from old to new
        
        Collection<Setting> sets = EcoreUtil.UsageCrossReferencer.find(eold, rs);
        
        for (Setting set : sets)
        {
            EReference er = (EReference) set.getEStructuralFeature();
            EObject eo = set.getEObject();
            if (!er.isChangeable() || er.isDerived())
                continue;
            if (er.isMany())
            {
                EList<EObject> el = (EList<EObject>) eo.eGet(er);
                int i = el.indexOf(eold);
                while (i != -1)
                {
                    el.remove(i);
                    if (er.getEReferenceType().isSuperTypeOf(enew.eClass()))
                    {
                        if (er.isUnique() && !el.contains(enew))
                            el.add(i, enew);
                        // eo.eSet(er, el);
                    }
                    else
                    {
                        log.warn("Can't translate reference list " + eo.eClass().getName() + "." + er.getName() + ", "
                                + enew.eClass().getName() + " incompatible with " + er.getEReferenceType().getName());
                    }
                    i = el.indexOf(eold);
                }
            }
            else
            {
                
                eo.eUnset(er);
                if (er.getEReferenceType().isSuperTypeOf(enew.eClass()))
                {
                    eo.eSet(er, enew);
                }
                else
                {
                    log.warn("Can't translate reference " + eo.eClass().getName() + "." + er.getName() + ", "
                            + enew.eClass().getName() + " incompatible with " + er.getEReferenceType().getName());
                }
            }
        }
        // display left usages
        
        // replace old by new in the container
        EObject eo = eold.eContainer();
        EReference er = eold.eContainmentFeature();
        if (eo != null && er != null)
            if (er.isMany())
            {
                EList<EObject> el = (EList<EObject>) eo.eGet(er);
                int i = el.indexOf(eold);
                el.remove(i);
                if (!er.isUnique() && el.contains(enew))
                    el.add(i, enew);
                // eo.eSet(er, el);
            }
            else
            {
                eo.eSet(er, enew);
                log.info("set " + eo.eClass().getName() + "." + er.getName());
            }
        
        sets = EcoreUtil.UsageCrossReferencer.find(eold, rs);
        for (Setting set : sets)
        {
            er = (EReference) set.getEStructuralFeature();
            eo = set.getEObject();
            log.error("Usage not treated: " + eo.eClass().getName() + "." + er.getName());
        }
    }
    
    /**
     * Replaces a reference from an object by a reference from a new object
     * 
     * @param eold
     *            object in which the reference was
     * @param enew
     *            object in which the reference will be
     * @param er
     *            reference
     */
    @SuppressWarnings("unchecked")
    private static void replace(EObject eold, EObject enew, EReference er)
    {
        if (!er.isChangeable() || er.isDerived() || !eold.eIsSet(er))
            return;
        
        EReference er2 = er.getEOpposite();
        
        String sref = eold.eClass().getName() + "." + er.getName();
        
        if (er2 != null)
            sref += "(" + er.getEType().getName() + "." + er2.getName() + ")";
        
        // determine if reference is moved
        boolean moved = true;
        if (!enew.eClass().getEAllReferences().contains(er) && !enew.eClass().getEAllContainments().contains(er))
        {
            log.warn("Can't translate reference " + sref + ", not contained in " + enew.eClass().getName());
            moved = false;
        }
        if (er.isMany())
        {
            // for each object of the list, get index in the opposite
            EList<EObject> el = (EList<EObject>) eold.eGet(er);
            while (el.size() != 0)
            {
                int i = -1;
                EObject eo = el.get(0);
                
                // get position in the opposite reference
                if (er2 != null && er2.isChangeable() && !er2.isDerived() && er2.isMany())
                    i = ((EList<EObject>) eo.eGet(er2)).indexOf(eold);
                
                // move eo to the new object
                el.remove(0);
                if (moved)
                {
                    EList<EObject> el2 = (EList<EObject>) enew.eGet(er);
                    el2.add(eo);
                }
                
                // restore position
                if (i != -1 && moved)
                {
                    ((EList<EObject>) eo.eGet(er2)).move(i, enew);
                }
                else
                    log.info("set " + sref);
            }
        }
        else
        {
            EObject eo = (EObject) eold.eGet(er);
            int i = -1;
            // get position in opposite reference
            if (er2 != null && er2.isChangeable() && !er2.isDerived() && er2.isMany())
                i = ((EList<EObject>) eo.eGet(er2)).indexOf(eold);
            
            eold.eUnset(er);
            if (moved)
                enew.eSet(er, eo);
            
            // restore position
            if (i != -1 && moved)
            {
                ((EList<EObject>) eo.eGet(er2)).move(i, enew);
            }
            else
                log.info("set " + sref);
        }
    }
    
    /**
     * Prints information about the specified object
     * 
     * @param eo
     *            object to print
     */
    @SuppressWarnings("boxing")
    private static void print(EObject eo)
    {
        log.debug("Printing " + Integer.toHexString(eo.hashCode()));
        for (EStructuralFeature esf : eo.eClass().getEAllStructuralFeatures())
        {
            Object v = eo.eGet(esf);
            if (v instanceof EList<?>)
            {
                log.debug("Attr " + esf.getName() + ":");
                for (Object o : (EList<?>) v)
                    if (o instanceof EObject)
                        log.debug(" " + Integer.toHexString(o.hashCode()));
                    else
                        log.debug(" " + o);
            }
            else if (v != null)
            {
                log.debug("Attr " + esf.getName() + ":");
                if (v instanceof EObject)
                    log.debug(v.hashCode());
                else
                    log.debug(v);
            }
        }
    }
    
    /**
     * Prints information about the specified connector
     * 
     * @param c
     *            connector to print
     * @return connector information
     */
    private String print(Connector c)
    {
        StringBuilder sb = new StringBuilder();
        if (c.getOwner() instanceof Class)
            sb.append(((Class) c.getOwner()).getName() + "::");
        for (int i = 0; i < c.getEnds().size(); i++)
        {
            if (i != 0)
                sb.append("-");
            sb.append(print(c.getEnds().get(i)));
        }
        return sb.toString();
    }
    
    /**
     * Prints information about the connector end
     * 
     * @param ce
     *            connector end to print
     * @return connector end information
     */
    private String print(ConnectorEnd ce)
    {
        return print(sysphsutil.getPropertyPath(ce));
    }
    
    /**
     * Prints a list of property as path
     * 
     * @param lp
     *            list of property
     * @return path
     */
    private String print(List<Property> lp)
    {
        if (lp == null)
            return null;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < lp.size(); i++)
        {
            if (i != 0)
                sb.append(".");
            sb.append(print(lp.get(i)));
        }
        return sb.toString();
    }
    
    /**
     * Prints a property
     * 
     * @param p
     *            property to print
     * @return string
     */
    private String print(Property p)
    {
        return p.getName() == null ? p.toString() : p.getQualifiedName();
    }
    
    /**
     * Prints a class
     * 
     * @param c
     *            class to print
     * @return string
     */
    private String print(Class c)
    {
        return c.getQualifiedName();
    }
}
