package com.engisis.sysphs.util;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * Intermediate class between Visitor and Visitable.
 * 
 * @author barbau
 *
 */
public class Dispatcher
{
    private static final Logger log = Logger.getLogger(Dispatcher.class);
    
    /**
     * list of visitors
     */
    private List<Visitor> visitors = new ArrayList<Visitor>();
    
    /**
     * Adds a visitor to the list
     * 
     * @param visitor
     *            visitor to add
     */
    public void addVisitor(Visitor visitor)
    {
        if (visitor.dispatcher != null)
            visitor.dispatcher.removeVisitor(visitor);
        
        if (!visitors.contains(visitor))
            visitors.add(visitor);
        
        visitor.dispatcher = this;
    }
    
    /**
     * Removes a visitor from the list
     * 
     * @param visitor
     *            visitor to remove
     */
    public void removeVisitor(Visitor visitor)
    {
        visitors.remove(visitor);
        visitor.dispatcher = null;
    }
    
    /**
     * Returns the list of visitors
     * 
     * @return list of visitors
     */
    List<Visitor> getVisitors()
    {
        return visitors;
    }
    
    /**
     * Calls the appropriate visit methods for the given object
     * 
     * @param o
     *            object for which visit methods are called
     */
    public void dispatch(Object o)
    {
        // c is the current type being checked among the visitors
        Class<?> c = o.getClass();
        do
        {
            // we look at the visitors to see if a method matches
            for (Visitor visitor : visitors)
            {
                Method[] methods = visitor.getClass().getMethods();
                for (int i = 0; i < methods.length; i++)
                {
                    Method m = methods[i];
                    if (m.getName().equals("visit"))
                    {
                        if (m.getParameterTypes().length == 1 && m.getParameterTypes()[0] == c)
                        {
                            try
                            {
                                m.invoke(visitor, o);
                            }
                            catch (Exception e)
                            {
                                log.error("Failed to execute visit on " + o.getClass().getName(), e);
                            }
                            return;
                        }
                    }
                }
            }
            
            c = c.getSuperclass();
        }
        while (c != Object.class);
        
        throw new UnsupportedOperationException("A visitor method couldn't be found for the class "
                + o.getClass().getName());
    }
}
