package com.engisis.sysphs.serialization.simulink;

import java.io.IOException;
import java.io.Writer;
import java.util.LinkedList;
import java.util.List;

import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.stateflow.SChart;
import com.engisis.sysphs.language.stateflow.SData;
import com.engisis.sysphs.language.stateflow.SDataScope;
import com.engisis.sysphs.language.stateflow.SEvent;
import com.engisis.sysphs.language.stateflow.SIdentifiedElement;
import com.engisis.sysphs.language.stateflow.SInstance;
import com.engisis.sysphs.language.stateflow.SJunction;
import com.engisis.sysphs.language.stateflow.SLinkNode;
import com.engisis.sysphs.language.stateflow.SMachine;
import com.engisis.sysphs.language.stateflow.SState;
import com.engisis.sysphs.language.stateflow.SStateflow;
import com.engisis.sysphs.language.stateflow.STarget;
import com.engisis.sysphs.language.stateflow.STransition;
import com.engisis.sysphs.language.stateflow.STreeNode;
import com.engisis.sysphs.util.TextWriter;

/**
 * 
 * Writes the Stateflow model in MDL
 * 
 * @author barbau
 *
 */
public class StateflowMDLWriter extends TextWriter
{
    
    public StateflowMDLWriter(Writer w)
    {
        super(w);
    }
    
    public void positionState(SState sstate) throws IOException
    {
        if (sstate.getOwningTreeNode() != null)
        {
            int pos = sstate.getOwningTreeNode().getTreeNode().indexOf(sstate);
            int w = 100;
            int h = 40;
            int m = 20;
            int x1 = (m + m * pos) + (pos * w);
            int x2 = w;
            int y1 = m;
            int y2 = h;
            String position = "[" + x1 + " " + y1 + " " + x2 + " " + y2 + "]";
            append("position " + position, TL);
        }
    }
    
    public void visit(SChart object) throws IOException
    {
        append("chart", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        
        if (object.getName() != null)
            append("name \"" + object.getName() + "\"", TL);
        if (object.getOwningMachine() != null)
            append("machine " + object.getOwningMachine().getId(), TL);
        
        // [container, first contained, previous sibling, next sibling]
        append("treeNode " + object.getTreeNodeString(), TL);
        for (SLinkNode sln : object.getLinkNode())
        {
            if (sln instanceof SData)
            {
                append("firstData " + sln.getId(), TL);
                break;
            }
        }
        append("chartFileNumber " + String.valueOf(object.getOwningMachine().getChart().indexOf(object) + 1), TL);
        append("saturateOnIntegerOverflow 1", TL);
        append("userSpecifiedStateTransitionExecutionOrder 1", TL);
        append("disableImplicitCasting 1", TL);
        append("actionLanguage 2", TL);
        unindent();
        append("}", TL);
        
        for (STreeNode streenode : object.getTreeNode())
            streenode.accept(getDispatcher());
        
        for (SLinkNode slinknode : object.getLinkNode())
            slinknode.accept(getDispatcher());
        
    }
    
    public void visit(SData object) throws IOException
    {
        append("data", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        if (object.getName() != null)
            append("name \"" + object.getName() + "\"", TL);
        if (object.getMachine() != null)
            append("machine " + object.getMachine().getId(), TL);
        append("linkNode " + getLinkNodeString(object), TL);
        if (object.getScope() == SDataScope.INPUT)
            append("scope INPUT_DATA", TL);
        else if (object.getScope() == SDataScope.OUTPUT)
            append("scope OUTPUT_DATA", TL);
        else if (object.getScope() == SDataScope.CONSTANT)
            append("scope CONSTANT_DATA", TL);
        if (object.getValue() != null)
        {
            append("props", TL);
            append("{", TL);
            append("initialValue \"", T);
            object.getValue().accept(getDispatcher());
            append("\"", L);
            append("}", TL);
        }
        unindent();
        append("}", TL);
    }
    
    public void visit(SEvent object) throws IOException
    {
        append("event", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        if (object.getName() != null)
            append("name " + object.getName(), TL);
        if (object.getMachine() != null)
            append("machine " + object.getMachine().getId(), TL);
        append("linkNode " + object.getLinkNodeString(), TL);
        unindent();
        append("}", TL);
    }
    
    public void visit(SExpressionValue object) throws IOException
    {
        append(object.getValue());
    }
    
    public void visit(SInstance object) throws IOException
    {
        append("instance", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        if (object.getName() != null)
            append("name \"" + object.getName() + "\"", TL);
        if (object.getMachine() != null)
            append("machine " + object.getMachine().getId(), TL);
        if (object.getChart() != null)
            append("chart " + object.getChart().getId(), TL);
        unindent();
        append("}", TL);
    }
    
    public void visit(SJunction object) throws IOException
    {
        append("junction", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        if (object.getChart() != null)
            append("chart " + object.getChart().getId(), TL);
        append("linkNode " + getLinkNodeString(object), TL);
        unindent();
        append("}");
    }
    
    public static String getLinkNodeString(SLinkNode sln)
    {
        // owner, next, previous
        int owner = 0, pre = 0, nex = 0;
        
        if (sln.getOwningTreeNode() != null)
        {
            owner = sln.getOwningTreeNode().getId();
            // get list of node of the same type
            List<SLinkNode> list = new LinkedList<SLinkNode>();
            for (SLinkNode slinknode : sln.getOwningTreeNode().getLinkNode())
                if (slinknode.getClass() == sln.getClass())
                    list.add(slinknode);
            
            for (int i = 0; i < list.size(); i++)
            {
                if (list.get(i) == sln)
                {
                    if (i - 1 >= 0)
                        pre = list.get(i - 1).getId();
                    if (i + 1 < list.size())
                        nex = list.get(i + 1).getId();
                    break;
                }
            }
        }
        
        return "[" + owner + " " + pre + " " + nex + "]";
    }
    
    public void visit(SMachine object) throws IOException
    {
        append("machine", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        if (object.getName() != null)
            append("name \"" + object.getName() + "\"", TL);
        append("isLibrary 0", TL);
        if (object.getTarget() != null)
            append("firstTarget " + object.getTarget().getId(), TL);
        unindent();
        append("}", TL);
        for (SChart schart : object.getChart())
        {
            schart.accept(getDispatcher());
        }
        if (object.getTarget() != null)
            object.getTarget().accept(getDispatcher());
    }
    
    public void visit(SDoubleValue object) throws IOException
    {
        append(String.valueOf(object.getValue()));
    }
    
    public void visit(SState object) throws IOException
    {
        append("state", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        String label = object.getLabel() == null ? object.getName() : object.getName() + "\n" + object.getLabel();
        append("labelString \"" + label + "\"", TL);
        positionState(object);
        if (object.getChart() != null)
            append("chart " + object.getChart().getId(), TL);
        append("treeNode " + getTreeNodeString(object), TL);
        unindent();
        append("}", TL);
        for (STreeNode streenode : object.getTreeNode())
            streenode.accept(getDispatcher());
        for (SLinkNode slinknode : object.getLinkNode())
            slinknode.accept(getDispatcher());
    }
    
    public void visit(SStateflow object) throws IOException
    {
        append("Stateflow", TL);
        append("{", TL);
        indent();
        if (object.getMachine() != null)
            object.getMachine().accept(getDispatcher());
        for (SInstance sinstance : object.getInstance())
            sinstance.accept(getDispatcher());
        unindent();
        append("}", TL);
    }
    
    public void visit(STarget object) throws IOException
    {
        append("target", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        append("name \"sfun\"", TL);
        if (object.getOwningMachine() != null)
        {
            append("machine " + object.getOwningMachine().getId(), TL);
            append("linkNode [" + object.getOwningMachine().getId() + " 0 0]", TL);
        }
        unindent();
        append("}", TL);
    }
    
    public void visit(STransition object) throws IOException
    {
        append("transition", TL);
        append("{", TL);
        indent();
        append("id " + object.getId(), TL);
        if (object.getLabel() != null)
            append("labelString \"" + object.getLabel() + "\"", TL);
        if (object.getChart() != null)
            append("chart " + object.getChart().getId(), TL);
        append("linkNode " + getLinkNodeString(object), TL);
        append("src", TL);
        append("{", TL);
        indent();
        if (object.getSrc() != null)
            append("id " + ((SIdentifiedElement) object.getSrc()).getId(), TL);
        unindent();
        append("}", TL);
        append("dst", TL);
        append("{", TL);
        indent();
        append("id " + ((SIdentifiedElement) object.getDst()).getId(), TL);
        unindent();
        append("}", TL);
        if (object.getSrc() != null)
            append("executionOrder " + (object.getSrc().getOutgoing().indexOf(object) + 1), TL);
        else
            append("executionOrder 1", TL);
        unindent();
        append("}", TL);
    }
    
    public static String getTreeNodeString(STreeNode stn)
    {
        // owner, first owned, next, previous
        int owner = 0, owned = 0, pre = 0, nex = 0;
        
        if (stn.getOwningTreeNode() != null)
        {
            owner = stn.getOwningTreeNode().getId();
            // get list of node of the same type
            List<STreeNode> list = new LinkedList<STreeNode>();
            for (STreeNode streenode : stn.getOwningTreeNode().getTreeNode())
                if (streenode.getClass() == stn.getClass())
                    list.add(streenode);
            
            for (int i = 0; i < list.size(); i++)
            {
                if (list.get(i) == stn)
                {
                    if (i - 1 >= 0)
                        pre = list.get(i - 1).getId();
                    if (i + 1 < list.size())
                        nex = list.get(i + 1).getId();
                    break;
                }
            }
        }
        
        if (stn.getTreeNode().size() != 0)
            owned = stn.getTreeNode().get(0).getId();
        
        return "[" + owner + " " + owned + " " + pre + " " + nex + "]";
    }
}
