package com.engisis.sysphs.serialization.modelica;

import java.io.IOException;
import java.io.Writer;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.common.util.EList;

import com.engisis.sysphs.language.modelica.MAccessControl;
import com.engisis.sysphs.language.modelica.MAlgorithm;
import com.engisis.sysphs.language.modelica.MBlock;
import com.engisis.sysphs.language.modelica.MBooleanValue;
import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MComponent;
import com.engisis.sysphs.language.modelica.MConnect;
import com.engisis.sysphs.language.modelica.MConnector;
import com.engisis.sysphs.language.modelica.MDataFlow;
import com.engisis.sysphs.language.modelica.MDirection;
import com.engisis.sysphs.language.modelica.MEquation;
import com.engisis.sysphs.language.modelica.MExpressionValue;
import com.engisis.sysphs.language.modelica.MExtension;
import com.engisis.sysphs.language.modelica.MHierarchy;
import com.engisis.sysphs.language.modelica.MIntegerValue;
import com.engisis.sysphs.language.modelica.MModel;
import com.engisis.sysphs.language.modelica.MModification;
import com.engisis.sysphs.language.modelica.MRealValue;
import com.engisis.sysphs.language.modelica.MStringValue;
import com.engisis.sysphs.language.modelica.MType;
import com.engisis.sysphs.language.modelica.MVariability;
import com.engisis.sysphs.util.TextWriter;

/**
 * Class in charge of writing Modelica models
 * 
 * @author barbau
 *
 */
public class ModelicaWriter extends TextWriter
{
    /**
     * Constructs a writer
     * 
     * @param w
     *            the object to write the model into
     */
    public ModelicaWriter(Writer w)
    {
        super(w);
    }
    
    public void visit(MAlgorithm object) throws IOException
    {
        
        append(object.getExpression(), TL);
    }
    
    public void visit(MBlock object) throws IOException
    {
        printClass(object, "block");
    }
    
    public void visit(MBooleanValue object) throws IOException
    {
        append(object.isValue() ? "true" : "false");
    }
    
    public void visit(MClass object) throws IOException
    {
        printClass(object, "class");
    }
    
    public void printClass(MClass object, String prefix) throws IOException
    {
        // non-redefining protected
        LinkedList<MComponent> mcs_pro = new LinkedList<MComponent>();
        // non-redefining public
        LinkedList<MComponent> mcs_pub = new LinkedList<MComponent>();
        
        for (MComponent mc : object.getComponents())
        {
            if (mc.getRedefinedComponent() == null)
            {
                if (mc.getAccessControl() == MAccessControl.PROTECTED)
                    mcs_pro.add(mc);
                else
                    mcs_pub.add(mc);
            }
        }
        
        // protected extensions
        LinkedList<MExtension> mes_pro = new LinkedList<MExtension>();
        // public extensions
        LinkedList<MExtension> mes_pub = new LinkedList<MExtension>();
        
        for (MExtension me : object.getExtensions())
        {
            if (me.getAccessControl() == MAccessControl.PROTECTED)
                mes_pro.add(me);
            else
                mes_pub.add(me);
        }
        
        // Short class definition
        if (mcs_pub.size() == 0 && mcs_pro.size() == 0 && object.getEquations().size() == 0
                && object.getExtensions().size() == 1 && object.getOwnedClasses().size() == 0
                && object.getAlgorithms().size() == 0 && (object instanceof MType||object instanceof MConnector))
        {
            MExtension extension = object.getExtensions().get(0);
            String direction = "";
            if (extension.getDirection() == MDirection.INPUT)
                direction = "input ";
            else if (extension.getDirection() == MDirection.OUTPUT)
                direction = "output ";
            append(prefix + " " + object.getName() + "=" + direction + extension.getExtendedClass().getName(), T);
            List<MComponent> mcs_red = new LinkedList<MComponent>();
            for (MComponent mc : object.getComponents())
            {
                MComponent red = mc.getRedefinedComponent();
                if (red != null && extension.getExtendedClass().getAllComponents().contains(red))
                {
                    mcs_red.add(mc);
                }
            }
            serializeModificationsAndRedefinitions(extension.getModifications(), mcs_red);
            append(";", L);
            return;
        }
        
        append(prefix + " " + object.getName(), TL);
        indent();
        
        // TODO: add extends
        for (MExtension e : mes_pub)
        {
            e.accept(getDispatcher());
        }
        for (MComponent c : mcs_pub)
        {
            append("", T);
            c.accept(getDispatcher());
            append(";", L);
        }
        for (MClass c : object.getOwnedClasses())
        {
            c.accept(getDispatcher());
        }
        boolean prot = false;
        for (MExtension e : mes_pro)
        {
            if (!prot)
            {
                unindent();
                append("protected", TL);
                indent();
                prot = true;
            }
            e.accept(getDispatcher());
        }
        for (MComponent c : mcs_pro)
        {
            if (!prot)
            {
                unindent();
                append("protected", TL);
                indent();
                prot = true;
            }
            
            append("", T);
            c.accept(getDispatcher());
            append(";", L);
            
        }
        if (object.getEquations().size() != 0)
        {
            unindent();
            append("equation", TL);
            indent();
            for (MEquation mequation : object.getEquations())
                mequation.accept(getDispatcher());
        }
        if (object.getAlgorithms().size() != 0)
        {
            unindent();
            append("algorithm", TL);
            indent();
            for (MAlgorithm malgorithm : object.getAlgorithms())
                malgorithm.accept(getDispatcher());
        }
        unindent();
        append("end " + object.getName() + ";", TL);
    }
    
    public void visit(MComponent object) throws IOException
    {
        if (object.getName() == null)
            throw new NullPointerException("The component has no name: " + object + ", type " + object.getType()
                    + ", owner " + object.getOwningClass());
        if (object.getType() == null)
            throw new NullPointerException("The type of " + object.getName() + " is null");
        if (object.getRedefiningComponent().size() > 0)
            append("replaceable ");
        
        MComponent red = object.getRedefinedComponent();
        
        if (red == null || object.getHierarchy() != red.getHierarchy())
        {
            if (object.getHierarchy() == MHierarchy.INNER)
                append("inner ");
            else if (object.getHierarchy() == MHierarchy.OUTER)
                append("outer ");
        }
        
        if (red == null || object.getDataFlow() != red.getDataFlow())
        {
            if (object.getDataFlow() == MDataFlow.FLOW)
                append("flow ");
        }
        if (red == null || object.getVariability() != red.getVariability())
        {
            if (object.getVariability() == MVariability.CONSTANT)
                append("constant ");
            else if (object.getVariability() == MVariability.PARAMETER)
                append("parameter ");
            else if (object.getVariability() == MVariability.DISCRETE)
                append("discrete ");
        }
        
        if (red == null || object.getDirection() != red.getDirection())
        {
            if (object.getDirection() == MDirection.INPUT)
                append("input ");
            else if (object.getDirection() == MDirection.OUTPUT)
                append("output ");
        }
        
        append(object.getType().getName());
        
        if (object.getDimensions().size() > 0)
        {
            append("[");
            for (int i = 0; i < object.getDimensions().size(); i++)
            {
                if (i != 0)
                    append(",");
                append(String.valueOf(object.getDimensions().get(i)));
            }
            append("] ");
        }
        
        append(" ");
        append(object.getName());
        serializeModificationsAndRedefinitions(object.getModifications(), null);
        
        // value
        if (object.getValue() != null)
        {
            append("=");
            object.getValue().accept(getDispatcher());
        }
    }
    
    @SuppressWarnings("boxing")
    public void visit(MConnect object) throws IOException
    {
        append("connect(", T);
        // need to be deal with scalar
        EList<MComponent> ref1 = object.getRef1();
        EList<MComponent> ref2 = object.getRef2();
        if (ref1.size() != 0 && ref2.size() != 0)
        {
            MComponent mc1 = ref1.get(ref1.size() - 1);
            MComponent mc2 = ref2.get(ref2.size() - 1);
            for (int i = 0; i < ref1.size(); i++)
            {
                if (i != 0)
                    append(".");
                MComponent mc = ref1.get(i);
                append(mc.getName());
            }
            if (mc1.getDimensions().size() == 1 && mc1.getDimensions().get(0) == 1 && mc2.getDimensions().size() == 0)
                append("[1]");
            append(",");
            for (int i = 0; i < ref2.size(); i++)
            {
                if (i != 0)
                    append(".");
                MComponent mc = ref2.get(i);
                append(mc.getName());
            }
            if (mc2.getDimensions().size() == 1 && mc2.getDimensions().get(0) == 1 && mc1.getDimensions().size() == 0)
                append("[1]");
        }
        append(");", L);
    }
    
    public void visit(MConnector object) throws IOException
    {
        printClass(object, "connector");
    }
    
    public void visit(MEquation object) throws IOException
    {
        append(object.getExpression() + ";", TL);
    }
    
    public void visit(MExpressionValue object) throws IOException
    {
        append(object.getValue());
    }
    
    public void visit(MExtension object) throws IOException
    {
        List<MComponent> mcs_red = new LinkedList<MComponent>();
        for (MComponent mc : object.getOwningClass().getComponents())
        {
            MComponent red = mc.getRedefinedComponent();
            if (object.getExtendedClass().getAllComponents().contains(red))
                mcs_red.add(mc);
        }
        append("extends " + object.getExtendedClass().getName(), T);
        serializeModificationsAndRedefinitions(object.getModifications(), mcs_red);
        append(";", L);
    }
    
    public void visit(MIntegerValue object) throws IOException
    {
        append(String.valueOf(object.getValue()));
    }
    
    public void visit(MModel object) throws IOException
    {
        printClass(object, "model");
    }
    
    public void visit(MModification object) throws IOException
    {
        
        for (int i = 0; i < object.getComponentPath().size(); i++)
        {
            MComponent mc = object.getComponentPath().get(i);
            if (i != 0)
                append(".");
            if (mc != null)
                append(mc.getName());
        }
        append("=");
        if (object.getAssignedValue() != null)
            object.getAssignedValue().accept(getDispatcher());
        else if (object.getAssignedReference() != null)
        {
            for (int i = 0; i < object.getAssignedReference().size(); i++)
            {
                MComponent mc = object.getAssignedReference().get(i);
                if (i != 0)
                    append(".");
                if (mc != null)
                    append(mc.getName());
            }
        }
    }
    
    public void visit(MRealValue object) throws IOException
    {
        append(String.valueOf(object.getValue()));
    }
    
    public void visit(MStringValue object) throws IOException
    {
        append("\"" + object.getValue() + "\"");
    }
    
    public void visit(MType object) throws IOException
    {
        printClass(object, "type");
    }
    
    public void serializeModificationsAndRedefinitions(List<MModification> modifications, List<MComponent> redefinitions)
            throws IOException
    {
        boolean first = true;
        if ((modifications != null && modifications.size() != 0)
                || (redefinitions != null && redefinitions.size() != 0))
        {
            append("(");
            if (modifications != null)
                for (MModification modification : modifications)
                {
                    if (!first)
                        append(",");
                    modification.accept(getDispatcher());
                    first = false;
                }
            if (redefinitions != null)
                for (MComponent redefinition : redefinitions)
                {
                    
                    if (!first)
                        append(",");
                    
                    append("redeclare ");
                    redefinition.accept(getDispatcher());
                    first = false;
                }
            append(")");
        }
    }
}
