package com.engisis.sysphs.deserialization.simulink;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import com.engisis.sysphs.deserialization.simulink.SimulinkReference.ReferenceType;
import com.engisis.sysphs.language.simscape.SConnectionPortBlock;
import com.engisis.sysphs.language.simscape.SLocation;
import com.engisis.sysphs.language.simscape.SimscapeFactory;
import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SElement;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SInterface;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SSubsystem;
import com.engisis.sysphs.language.simulink.SSystem;
import com.engisis.sysphs.language.simulink.SSystemParameter;
import com.engisis.sysphs.language.simulink.SSystemParameterAssignment;
import com.engisis.sysphs.language.simulink.SimulinkFactory;
import com.engisis.sysphs.language.stateflow.StateflowFactory;
import com.engisis.sysphs.translation.simulink.SimscapeUtil;
import com.engisis.sysphs.translation.simulink.SimulinkUtil;
import com.engisis.sysphs.util.Util;
import com.engisis.sysphs.util.XMLLocation;
import com.engisis.sysphs.util.XMLReader;

/**
 * First-pass parser for a Simulink XML file.
 * 
 * @author barbau
 *
 */
public class SimulinkXMLReaderPass1 extends XMLReader implements SimulinkReader
{
    private static final Logger log = Logger.getLogger(SimulinkXMLReaderPass1.class);
    
    /**
     * Root Simulink model
     */
    private SModel smodel;
    /**
     * Map with XML locations ad keys, and their corresponding Simulink objects
     * as values
     */
    private Hashtable<XMLLocation, SElement> objects;
    
    /**
     * Collection of Simulink references
     */
    private Collection<SimulinkReference> references;
    
    /**
     * Input stream of the input XML file
     */
    private InputStream is;
    
    /**
     * Constructs a first-pass Simulink XML parser
     * 
     * @param inputStream
     *            input stream of the XML file
     */
    public SimulinkXMLReaderPass1(InputStream inputStream)
    {
        objects = new Hashtable<XMLLocation, SElement>();
        references = new LinkedList<SimulinkReference>();
        is = inputStream;
    }
    
    @Override
    public void process()
    {
        // open xml
        try
        {
            XmlPullParserFactory xmlppf = XmlPullParserFactory.newInstance();
            xmlpp = xmlppf.newPullParser();
            xmlpp.setInput(is, null);
            lookForElement("Model", "Library");
            smodel = processModelOrLibrary();
        }
        catch (XmlPullParserException e)
        {
            log.error(e);
        }
        catch (IOException e)
        {
            log.error(e);
        }
        finally
        {
            try
            {
                is.close();
            }
            catch (IOException e)
            {
            }
        }
    }
    
    /**
     * Processes the Model/Library element
     * 
     * @return
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SModel processModelOrLibrary() throws XmlPullParserException, IOException
    {
        String ln = xmlpp.getName();
        
        String name = xmlpp.getAttributeValue(null, "Name");
        
        if (ln.equals("Library"))
        {
            smodel = SimulinkFactory.eINSTANCE.createSModel();
            SimulinkUtil.setName(smodel, name);
            log.info("Parsing library " + name);
        }
        else if (ln.equals("Model"))
        {
            smodel = SimulinkFactory.eINSTANCE.createSLibrary();
            SimulinkUtil.setName(smodel, name);
            log.info("Parsing model " + name);
        }
        
        objects.put(getLocation(), smodel);
        
        int depth = xmlpp.getDepth();
        int et = xmlpp.getEventType();
        String ln2;
        while (et != XmlPullParser.END_TAG || xmlpp.getDepth() != depth)
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                ln2 = xmlpp.getName();
                if (ln2.equals("System"))
                {
                    SSystem ssystem = processSystem(false);
                    ssystem.setOwningModel(smodel);
                    if (ssystem.getName() == null)
                        SimulinkUtil.setName(ssystem, smodel.getName());
                }
                else
                    skipToEnd();
            }
        }
        return smodel;
    }
    
    /**
     * Processes the System element
     * 
     * @param ischart
     *            Whether the system is a chart
     * @return a System object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SSystem processSystem(boolean ischart) throws XmlPullParserException, IOException
    {
        log.info("Parsing system");
        SSystem ssystem = ischart ? StateflowFactory.eINSTANCE.createSChartSystem() : SimulinkFactory.eINSTANCE
                .createSSystem();
        
        objects.put(getLocation(), ssystem);
        
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        String ln2;
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                ln2 = xmlpp.getName();
                if (ln2.equals("Block"))
                {
                    SBlock sblock = processBlock();
                    if (sblock == null)
                        log.error("block not recognized");
                    else
                        sblock.setOwningSystem(ssystem);
                }
                else if (ln2.equals("Line"))
                {
                    
                }
                else
                    skipToEnd();
            }
        }
        return ssystem;
    }
    
    /**
     * Processes the Block element
     * 
     * @return a Block object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private SBlock processBlock() throws XmlPullParserException, IOException
    {
        SBlock sb = null;
        String type = null;
        String name = null;
        String sid = null;
        String side = null;
        // String ports = null;
        String sourceblock = null;
        // String sourcetype = null;
        String componentpath = null;
        SSystem ssystem = null;
        String functionname = null;
        String sfbt = null;
        String[] ports = new String[] { "1", "1" };
        
        String ln;
        
        name = xmlpp.getAttributeValue(null, "Name");
        type = xmlpp.getAttributeValue(null, "BlockType");
        sid = xmlpp.getAttributeValue(null, "SID");
        side = xmlpp.getAttributeValue(null, "Side");
        
        Hashtable<String, String> params = new Hashtable<String, String>();
        XMLLocation loc = getLocation();
        
        int et = xmlpp.getEventType();
        int depth = xmlpp.getDepth();
        while (et != XmlPullParser.END_TAG || depth != xmlpp.getDepth())
        {
            et = xmlpp.next();
            if (et == XmlPullParser.START_TAG)
            {
                ln = xmlpp.getName();
                if (ln.equals("P"))
                {
                    ln = xmlpp.getAttributeValue(null, "Name");
                    
                    if (ln.equals("SourceBlock"))
                    {
                        sourceblock = xmlpp.nextText();
                    }
                    // else if (ln.equals("SourceType"))
                    // {
                    // sourcetype = xmlpp.nextText();
                    // }
                    else if (ln.equals("ComponentPath"))
                    {
                        componentpath = xmlpp.nextText();
                    }
                    else if (ln.equals("FunctionName"))
                    {
                        functionname = xmlpp.nextText();
                    }
                    else if (ln.equals("SFBlockType"))
                    {
                        sfbt = xmlpp.nextText();
                    }
                    else if (ln.equals("Ports"))
                    {
                        String s = xmlpp.nextText().trim();
                        ports = s.substring(1, s.length() - 1).split(",");
                    }
                    else
                    {
                        params.put(ln, xmlpp.nextText());
                    }
                }
                else if (ln.equals("System"))
                {
                    ssystem = processSystem("Chart".equals(sfbt));
                }
                else
                    skipToEnd();
            }
        }
        
        if ("Reference".equals(type))
        {
            if (componentpath == null)
            {
                log.info("Found Reference block " + name);
                sb = SimulinkFactory.eINSTANCE.createSReference();
                SimulinkUtil.setName(sb, name);
                references.add(new SimulinkReference(sb, sourceblock, ReferenceType.BLOCK));
            }
            else
            {
                log.info("Found Physical block " + name);
                sb = SimscapeFactory.eINSTANCE.createSPhysicalBlock();
                SimulinkUtil.setName(sb, name);
                references.add(new SimulinkReference(sb, componentpath, ReferenceType.COMPONENT));
            }
            // ports
            // SourceBlock/SourceType
            // ComponentPath
        }
        else if ("SubSystem".equals(type))
        {
            log.info("Found Subsystem block " + name);
            if ("Chart".equals(sfbt))
            {
                sb = StateflowFactory.eINSTANCE.createSChartBlock();
                SimulinkUtil.setName(sb, name);
            }
            else
            {
                sb = SimulinkFactory.eINSTANCE.createSSubsystem();
                SimulinkUtil.setName(sb, name);
            }
            if (ssystem != null)
            {
                ssystem.setOwningSubsystem((SSubsystem) sb);
                if (ssystem.getName() == null)
                    SimulinkUtil.setName(ssystem, sb.getName());
            }
        }
        else if ("S-Function".equals(type))
        {
            log.info("Found S-function level 1 block " + name);
            sb = SimulinkFactory.eINSTANCE.createSFunction1Block();
            SimulinkUtil.setName(sb, name);
            if (functionname != null)
            {
                if (!functionname.equals("sf_sfun"))
                {
                    references.add(new SimulinkReference(sb, functionname, ReferenceType.SF1));
                }
            }
            else
                log.warn("S-Function without function");
        }
        else if ("M-S-Function".equals(type))
        {
            log.info("Found S-function level 2 block " + name);
            sb = SimulinkFactory.eINSTANCE.createSFunction2Block();
            SimulinkUtil.setName(sb, name);
            if (functionname != null)
            {
                references.add(new SimulinkReference(sb, functionname, ReferenceType.SF2));
            }
            else
                log.warn("S-Function without function");
        }
        else if ("Inport".equals(type))
        {
            log.info("Found Inport block " + name);
            sb = SimulinkFactory.eINSTANCE.createSInport();
            SimulinkUtil.setName(sb, name);
        }
        else if ("Outport".equals(type))
        {
            log.info("Found Outport block " + name);
            sb = SimulinkFactory.eINSTANCE.createSOutport();
            SimulinkUtil.setName(sb, name);
        }
        else if ("PMIOPort".equals(type))
        {
            log.info("Found Physical port " + type + ":" + name);
            SConnectionPortBlock sbpb = SimscapeFactory.eINSTANCE.createSConnectionPortBlock();
            sb = sbpb;
            SimulinkUtil.setName(sbpb, name);
            sbpb.setComponent(SimscapeUtil.getConnectionPortComponent());
            if (side != null){
                if ("Left".equals(side))
                    sbpb.setLocation(SLocation.LEFT);
                else if ("Right".equals(side))
                    sbpb.setLocation(SLocation.RIGHT);
                else
                    log.warn("No location for block " + name);
            }
            else
                log.warn("No location for block " + name);
        }
        else
        {
            log.info("Found Interface " + type + ":" + name);
            SInterface si = SimulinkFactory.eINSTANCE.createSInterface();
            sb = si;
            SimulinkUtil.setName(sb, name);
            
            SSystem sbsys = SimulinkFactory.eINSTANCE.createSSystem();
            SimulinkUtil.setName(sbsys, type);
            si.setSystem(sbsys);
            
            if (ports != null)
            {
                if (ports.length > 0)
                {
                    for (int i = 0; i < Util.toInt(ports[0], 0); i++)
                    {
                        SInport sinport = SimulinkFactory.eINSTANCE.createSInport();
                        sinport.setOwningSystem(sbsys);
                    }
                }
                if (ports.length > 1)
                {
                    for (int i = 0; i < Util.toInt(ports[1], 0); i++)
                    {
                        SOutport soutport = SimulinkFactory.eINSTANCE.createSOutport();
                        soutport.setOwningSystem(sbsys);
                    }
                }
            }
            
            for (Entry<String, String> param : params.entrySet())
            {
                SSystemParameter ssp = SimulinkFactory.eINSTANCE.createSSystemParameter();
                sbsys.getSystemparameters().add(ssp);
                
                SSystemParameterAssignment sspa = SimulinkFactory.eINSTANCE.createSSystemParameterAssignment();
                si.getAssignments().add(sspa);
                SimulinkUtil.setName(ssp, param.getKey());
                sspa.setParameter(ssp);
                
                try
                {
                    double d = Double.parseDouble(param.getValue());
                    SDoubleValue sdv = SimulinkFactory.eINSTANCE.createSDoubleValue();
                    sdv.setValue(d);
                    sspa.setValue(sdv);
                }
                catch (NumberFormatException e)
                {
                    SExpressionValue sev = SimulinkFactory.eINSTANCE.createSExpressionValue();
                    sev.setValue(param.getValue());
                    sspa.setValue(sev);
                }
            }
        }
        sb.setSID(sid);
        objects.put(loc, sb);
        return sb;
    }
    
    /**
     * Returns the root model
     * 
     * @return a Model object
     */
    public SModel getModel()
    {
        return smodel;
    }
    
    @Override
    public Collection<SimulinkReference> getReferences()
    {
        return references;
    }
    
    /**
     * Returns a map with XML locations as keys, and Simulink objects as values
     * 
     * @return a map with XML locations as keys, and Simulink objects as values
     */
    public Hashtable<XMLLocation, SElement> getObjects()
    {
        return objects;
    }
    
    @Override
    public int getPriority()
    {
        return 1;
    }
    
    @Override
    public SElement getRootElement()
    {
        return smodel;
    }
    
}
