package com.engisis.sysphs.deserialization.modelica;

import java.util.LinkedList;
import java.util.List;

import org.antlr.v4.runtime.tree.TerminalNode;
import org.apache.log4j.Logger;

import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MNamedElement;

/**
 * Modelica scope, used during the parsing of Modelica files.
 * 
 * @author barbau
 *
 */
public class ModelicaScope
{
    private static final Logger log = Logger.getLogger(ModelicaScope.class);
    /**
     * Name of the scope
     */
    private String name;
    /**
     * Children scopes
     */
    private List<ModelicaScope> children = new LinkedList<ModelicaScope>();
    /**
     * Sibling scopes
     */
    private List<ModelicaScope> siblings = new LinkedList<ModelicaScope>();
    /**
     * Parent scope
     */
    private ModelicaScope parent;
    /**
     * Modelica object associated with the scope
     */
    private MNamedElement mnamedelement;
    /**
     * Public status of scope
     */
    private boolean pub = true;
    
    /**
     * Constructs a Modelica scope with a given name
     * 
     * @param name
     *            name of the scope
     */
    public ModelicaScope(String name)
    {
        this.name = name;
    }
    
    /**
     * Returns the name of the scope
     * 
     * @return name of the scope
     */
    public String getName()
    {
        return name;
    }
    
    /**
     * Returns siblings
     * 
     * @return siblings of this scope
     */
    public List<ModelicaScope> getSiblings()
    {
        return siblings;
    }
    
    /**
     * Sets parent scope
     * 
     * @param parent
     *            parent scope
     */
    public void setParent(ModelicaScope parent)
    {
        if (this.parent != null)
            this.parent.children.remove(this);
        this.parent = parent;
        parent.children.add(this);
    }
    
    /**
     * Returns the parent scope
     * 
     * @return parent scope
     */
    public ModelicaScope getParent()
    {
        return parent;
    }
    
    /**
     * Sets Modelica object associated with this scope
     * 
     * @param mnamedelement
     *            Modelica object
     */
    public void setMNamedElement(MNamedElement mnamedelement)
    {
        this.mnamedelement = mnamedelement;
    }
    
    /**
     * Returns the associated Modelica object
     * 
     * @return associated Modelica object
     */
    public MNamedElement getMNamedElement()
    {
        return mnamedelement;
    }
    
    /**
     * Sets whether the scope is public
     * 
     * @param pub
     */
    public void setPublic(boolean pub)
    {
        this.pub = pub;
    }
    
    /**
     * Returns whether the scope is public
     * 
     * @param pub
     */
    public boolean isPublic()
    {
        return pub;
    }
    
    /**
     * Returns all classes in the scope
     * 
     * @return classes in the scope
     */
    public List<MClass> getAllMClasses()
    {
        LinkedList<MClass> ret = new LinkedList<MClass>();
        addMClasses(ret);
        return ret;
    }
    
    /**
     * Adds a list of classes
     * 
     * @param list
     *            list of Modelica classes
     */
    private void addMClasses(List<MClass> list)
    {
        if (mnamedelement instanceof MClass && !mnamedelement.getName().contains("."))
            list.add((MClass) mnamedelement);
        for (ModelicaScope child : children)
            child.addMClasses(list);
    }
    
    /**
     * Look for an element in the global scope with the given name
     * 
     * @param name
     *            name of the element to look for
     * @return the scope corresponding to the element
     */
    public ModelicaScope lookForElementGlobally(TerminalNode name)
    {
        ModelicaScope scope = lookForElementInScope(name);
        if (scope != null && scope.mnamedelement instanceof MClass)
            return scope;
        if (parent != null)
            return parent.lookForElementGlobally(name);
        return null;
    }
    
    /**
     * Look for the scope of a class with a given path
     * 
     * @param path
     *            path of the class, with name given as list of terminals
     * @return the scope of the class given by the path
     */
    public ModelicaScope lookForClass(List<TerminalNode> path)
    {
        
        ModelicaScope scope = lookForElementGlobally(path.get(0));
        if (scope == null)
        {
            ModelicaScope root = getRoot();
            for (ModelicaScope child : root.children)
            {
                MNamedElement mne = child.getMNamedElement();
                if (mne.getName().equals(join(path)) && mne instanceof MClass)
                    return child;
            }
            log.error("Can't find the element " + path.get(0) + " from " + path);
            return null;
        }
        for (int i = 1; i < path.size(); i++)
        {
            scope = scope.lookForElementInScope(path.get(i));
            if (scope == null)
            {
                log.error("Can't find the element " + path.get(i) + " from " + path);
                return null;
            }
        }
        if (scope.getMNamedElement() instanceof MClass)
            return scope;
        return null;
    }
    
    /**
     * Returns the scope of the element with the given name
     * 
     * @param name
     *            name of the element to look for
     * @return scope of the corresponding element
     */
    public ModelicaScope lookForElementInScope(TerminalNode name)
    {
        for (ModelicaScope scope : children)
        {
            if (scope.mnamedelement != null && name.getText().equals(scope.mnamedelement.getName()))
            {
                return scope;
            }
        }
        for (ModelicaScope sib : siblings)
        {
            ModelicaScope ret = sib.lookForElementInScope(name);
            if (ret != null)
                return ret;
        }
        return null;
    }
    
    /**
     * Returns the root scope
     * 
     * @return root scope
     */
    public ModelicaScope getRoot()
    {
        if (parent == null)
            return this;
        return parent.getRoot();
    }
    
    /**
     * Joins the terminals with a dot
     * 
     * @param path
     *            path to join with a dot
     * @return a string corresponding to the path
     */
    private static String join(List<TerminalNode> path)
    {
        
        StringBuilder name = new StringBuilder();
        for (int i = 0; i < path.size(); i++)
        {
            if (i != 0)
                name.append(".");
            name.append(path.get(i).getText());
        }
        return name.toString();
    }
    
    @Override
    public String toString()
    {
        StringBuilder sb = new StringBuilder();
        if (parent != null)
            sb.append(parent.toString() + "::");
        sb.append(name);
        return sb.toString();
    }
}
