package com.engisis.sysphs.deserialization.modelica;

import java.util.Hashtable;
import java.util.List;
import java.util.Map.Entry;

import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeProperty;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.apache.log4j.Logger;

import com.engisis.sysphs.generation.modelica.ModelicaBaseListener;
import com.engisis.sysphs.generation.modelica.ModelicaLexer;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Algorithm_sectionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.ArgumentContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Class_definitionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Class_prefixesContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Class_specifierContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Component_clause1Context;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Component_clauseContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Component_declaration1Context;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Component_declarationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Element_modificationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.EquationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Equation_sectionContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Extends_clauseContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.ModificationContext;
import com.engisis.sysphs.generation.modelica.ModelicaParser.Stored_definitionContext;
import com.engisis.sysphs.language.modelica.MAccessControl;
import com.engisis.sysphs.language.modelica.MAlgorithm;
import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MComponent;
import com.engisis.sysphs.language.modelica.MDataFlow;
import com.engisis.sysphs.language.modelica.MDirection;
import com.engisis.sysphs.language.modelica.MElement;
import com.engisis.sysphs.language.modelica.MEquation;
import com.engisis.sysphs.language.modelica.MExtension;
import com.engisis.sysphs.language.modelica.MVariability;
import com.engisis.sysphs.language.modelica.ModelicaFactory;
import com.engisis.sysphs.translation.modelica.ModelicaUtil;
import com.engisis.sysphs.util.Util;

/**
 * Class managing the first pass of the Modelica deserialization
 * 
 * @author barbau
 *
 */
public class ModelicaReaderPass1 extends ModelicaBaseListener
{
    private static final Logger log = Logger.getLogger(ModelicaReaderPass1.class);
    
    /**
     * Modelica scopes associated with syntax tree nodes
     */
    private ParseTreeProperty<ModelicaScope> scopelist;
    
    /**
     * Map with syntax tree nodes as keys, and list of terminals as values
     */
    private Hashtable<ParseTree, List<TerminalNode>> resolutionlist;
    
    /**
     * Map with Modelica objects as keys, and syntax tree nodes as values
     */
    private Hashtable<MElement, ParseTree> expressions;
    
    /**
     * Global scope
     */
    private ModelicaScope globalscope;
    
    /**
     * Current scope
     */
    private ModelicaScope scope;
    
    /**
     * Constructs a first-pass Modelica reader
     * 
     * @param globalscope
     *            Global scope to be used
     */
    public ModelicaReaderPass1(ModelicaScope globalscope)
    {
        scopelist = new ParseTreeProperty<ModelicaScope>();
        this.globalscope = globalscope;
        scope = globalscope;
        this.resolutionlist = new Hashtable<ParseTree, List<TerminalNode>>();
        expressions = new Hashtable<MElement, ParseTree>();
    }
    
    /**
     * Returns a map with Modelica objects as keys, and syntax tree nodes as
     * values
     * 
     * @return map from Modelica objects to syntax tree nodes
     */
    public Hashtable<MElement, ParseTree> getExpressions()
    {
        return expressions;
    }
    
    /**
     * Returns the associations between syntax tree nodes and Modelica scopes
     * 
     * @return map from syntax tree nodes to scope
     */
    public ParseTreeProperty<ModelicaScope> getScopeList()
    {
        return scopelist;
    }
    
    @Override
    public void enterStored_definition(Stored_definitionContext ctx)
    {
        super.enterStored_definition(ctx);
        scopelist.put(ctx, globalscope);
    }
    
    @Override
    public void enterClass_definition(Class_definitionContext ctx)
    {
        super.enterClass_definition(ctx);
        
        Class_specifierContext specifier = ctx.class_specifier();
        String name = specifier.IDENT(0).getText();
        
        ModelicaScope newscope = new ModelicaScope(name);
        
        newscope.setParent(scope);
        scope = newscope;
        scopelist.put(ctx, newscope);
        
        Class_prefixesContext prefix = ctx.class_prefixes();
        MClass mclass = null;
        if (prefix.CLASS() != null)
            mclass = ModelicaFactory.eINSTANCE.createMClass();
        else if (prefix.MODEL() != null)
            mclass = ModelicaFactory.eINSTANCE.createMModel();
        else if (prefix.BLOCK() != null)
            mclass = ModelicaFactory.eINSTANCE.createMBlock();
        else if (prefix.CONNECTOR() != null)
            mclass = ModelicaFactory.eINSTANCE.createMConnector();
        else if (prefix.TYPE() != null)
            mclass = ModelicaFactory.eINSTANCE.createMType();
        
        if (mclass != null)
        {
            ModelicaUtil.setName(mclass, name);
            log.info("Adding class " + name);
            scope.setMNamedElement(mclass);
            if (specifier.name_path() != null)
            {
                scopelist.put(specifier, scope);
                resolutionlist.put(specifier, specifier.name_path().IDENT());
            }
        }
        else
        {
            log.warn("Class type unsupported: " + name);
        }
    }
    
    @Override
    public void exitClass_definition(Class_definitionContext ctx)
    {
        super.exitClass_definition(ctx);
        scope = scope.getParent();
    }
    
    @Override
    public void enterExtends_clause(Extends_clauseContext ctx)
    {
        super.enterExtends_clause(ctx);
        if (!(scope.getMNamedElement() instanceof MClass))
        {
            log.debug("Extend clause skipped as the class is not translated: " + scope);
            return;
        }
        scopelist.put(ctx, scope);
        resolutionlist.put(ctx, ctx.name_path().IDENT());
        
    }
    
    @Override
    public void enterComponent_clause(Component_clauseContext ctx)
    {
        // non-redefined component
        super.enterComponent_clause(ctx);
        
        boolean flow = (ctx.type_prefix().FLOW() != null);
        boolean discrete = (ctx.type_prefix().DISCRETE() != null);
        boolean parameter = (ctx.type_prefix().PARAMETER() != null);
        boolean constant = (ctx.type_prefix().CONSTANT() != null);
        boolean input = (ctx.type_prefix().INPUT() != null);
        boolean output = (ctx.type_prefix().OUTPUT() != null);
        
        int[] dims = null;
        if (ctx.array_subscripts() != null)
        {
            dims = new int[ctx.array_subscripts().subscript().size()];
            for (int i = 0; i < ctx.array_subscripts().subscript().size(); i++)
                dims[i] = Util.toInt(ctx.array_subscripts().subscript().get(i).getText(), 0);
        }
        
        for (Component_declarationContext component : ctx.component_list().component_declaration())
        {
            String name = component.declaration().IDENT().getText();
            
            ModelicaScope newscope = new ModelicaScope(name);
            newscope.setParent(scope);
            scopelist.put(component, newscope);
            // don't change the scope yet: that should happen in
            // component_declaration
            
            if (!(scope.getMNamedElement() instanceof MClass))
            {
                log.warn("Component skipped because its type is skipped " + newscope);
                continue;
            }
            
            MComponent mcomponent = ModelicaFactory.eINSTANCE.createMComponent();
            mcomponent.setOwningClass((MClass) scope.getMNamedElement());
            ModelicaUtil.setName(mcomponent, name);
            
            if (scope.isPublic())
                mcomponent.setAccessControl(MAccessControl.PUBLIC);
            else
                mcomponent.setAccessControl(MAccessControl.PROTECTED);
            
            if (flow)
                mcomponent.setDataFlow(MDataFlow.FLOW);
            
            if (input)
                mcomponent.setDirection(MDirection.INPUT);
            else if (output)
                mcomponent.setDirection(MDirection.OUTPUT);
            
            if (discrete)
                mcomponent.setVariability(MVariability.DISCRETE);
            else if (parameter)
                mcomponent.setVariability(MVariability.PARAMETER);
            else if (constant)
                mcomponent.setVariability(MVariability.CONSTANT);
            else
                mcomponent.setVariability(MVariability.CONTINUOUS);
            
            resolutionlist.put(component, ctx.type_specifier().name_path().IDENT());
            
            newscope.setMNamedElement(mcomponent);
            log.info("Adding component " + newscope);
        }
        
    }
    
    @Override
    public void enterComponent_clause1(Component_clause1Context ctx)
    {
        super.enterComponent_clause1(ctx);
        
        boolean flow = (ctx.type_prefix().FLOW() != null);
        boolean discrete = (ctx.type_prefix().DISCRETE() != null);
        boolean parameter = (ctx.type_prefix().PARAMETER() != null);
        boolean constant = (ctx.type_prefix().CONSTANT() != null);
        boolean input = (ctx.type_prefix().INPUT() != null);
        boolean output = (ctx.type_prefix().OUTPUT() != null);
        
        Component_declaration1Context component = ctx.component_declaration1();
        
        String name = component.declaration().IDENT().getText();
        
        ModelicaScope newscope = new ModelicaScope(name);
        newscope.setParent(scope);
        scopelist.put(component, newscope);
        // don't change the scope yet: that should happen in
        // component_declaration
        
        if (!(scope.getMNamedElement() instanceof MClass))
        {
            log.warn("Component skipped because its type is skipped " + newscope);
            return;
        }
        
        MComponent mcomponent = ModelicaFactory.eINSTANCE.createMComponent();
        mcomponent.setOwningClass((MClass) scope.getMNamedElement());
        ModelicaUtil.setName(mcomponent, name);
        
        if (scope.isPublic())
            mcomponent.setAccessControl(MAccessControl.PUBLIC);
        else
            mcomponent.setAccessControl(MAccessControl.PROTECTED);
        
        if (flow)
            mcomponent.setDataFlow(MDataFlow.FLOW);
        
        if (input)
            mcomponent.setDirection(MDirection.INPUT);
        else if (output)
            mcomponent.setDirection(MDirection.OUTPUT);
        
        if (discrete)
            mcomponent.setVariability(MVariability.DISCRETE);
        else if (parameter)
            mcomponent.setVariability(MVariability.PARAMETER);
        else if (constant)
            mcomponent.setVariability(MVariability.CONSTANT);
        else
            mcomponent.setVariability(MVariability.CONTINUOUS);
        
        resolutionlist.put(component, ctx.type_specifier().name_path().IDENT());
        
        newscope.setMNamedElement(mcomponent);
        log.info("Adding redeclared component " + newscope);
        
    }
    
    @Override
    public void exitComponent_declaration(Component_declarationContext ctx)
    {
        super.exitComponent_declaration(ctx);
    }
    
    @Override
    public void exitEquation_section(Equation_sectionContext ctx)
    {
        super.exitEquation_section(ctx);
        if (!(scope.getMNamedElement() instanceof MClass))
        {
            log.info("Skipping equation in scope " + scope.getName());
            return;
        }
        for (EquationContext ec : ctx.equation())
        {
            if (ec.connect_clause() == null)
            {
                log.info("Added equation " + ec.getText());
                MEquation mequation = ModelicaFactory.eINSTANCE.createMEquation();
                mequation.setExpression(ec.getText());
                ((MClass) scope.getMNamedElement()).getEquations().add(mequation);
                expressions.put(mequation, ec);
            }
        }
    }
    
    @Override
    public void exitAlgorithm_section(Algorithm_sectionContext ctx)
    {
        super.exitAlgorithm_section(ctx);
        if (!(scope.getMNamedElement() instanceof MClass))
            return;
        
        MAlgorithm ma = ModelicaFactory.eINSTANCE.createMAlgorithm();
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.statement().size(); i++)
        {
            if (i != 0)
                sb.append("\n");
            sb.append(ctx.statement(i).getText());
            sb.append(';');
        }
        ma.setExpression(sb.toString());
        expressions.put(ma, ctx);
        ((MClass) scope.getMNamedElement()).getAlgorithms().add(ma);
    }
    
    @Override
    public void visitTerminal(TerminalNode node)
    {
        super.visitTerminal(node);
        
        if (scope != null)
        {
            if (node.getSymbol().getType() == ModelicaLexer.PUBLIC)
                scope.setPublic(true);
            else if (node.getSymbol().getType() == ModelicaLexer.PROTECTED)
                scope.setPublic(false);
        }
    }
    
    /**
     * Performs the object resolution
     */
    public void resolve()
    {
        for (Entry<ParseTree, List<TerminalNode>> entry : resolutionlist.entrySet())
        {
            ParseTree pt = entry.getKey();
            ModelicaScope scope = scopelist.get(pt);
            if (scope == null)
                throw new IllegalStateException("A scope should have been provided in the resolution of " + pt);
            
            if (pt instanceof Component_declarationContext)
            {
                if (scope.getMNamedElement() instanceof MComponent)
                {
                    MComponent mcomponent = (MComponent) scope.getMNamedElement();
                    ModelicaScope ms = scope.lookForClass(entry.getValue());
                    if (ms != null)
                    {
                        MClass mclass = (MClass) ms.getMNamedElement();
                        log.info("Type of component " + mcomponent.getName() + " resolved to " + mclass.getName());
                        mcomponent.setType(mclass);
                    }
                    else
                    {
                        log.warn("Type of component " + mcomponent.getName() + " couldn't be resolved");
                        
                        // Create class
                        MClass mclass = createClassAnyway(entry.getValue());
                        mcomponent.setType(mclass);
                        
                        // Create properties from modifications
                        Component_declarationContext cdc = (Component_declarationContext) pt;
                        ModificationContext mc = cdc.declaration().modification();
                        if (mc != null && mc.class_modification() != null)
                        {
                            for (ArgumentContext ac : mc.class_modification().argument_list().argument())
                            {
                                if (ac.element_modification_or_replaceable().element_modification() != null)
                                {
                                    Element_modificationContext emc = ac.element_modification_or_replaceable()
                                            .element_modification();
                                    String name = emc.component_reference().IDENT(0).getText();
                                    if (mclass.getComponentByName(name) == null)
                                    {
                                        MComponent mcomponent2 = ModelicaFactory.eINSTANCE.createMComponent();
                                        mcomponent2.setOwningClass(mclass);
                                        ModelicaUtil.setName(mcomponent2, name);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if (pt instanceof Component_declaration1Context)
            {
                if (scope.getMNamedElement() instanceof MComponent)
                {
                    MComponent mcomponent = (MComponent) scope.getMNamedElement();
                    ModelicaScope ms = scope.lookForClass(entry.getValue());
                    if (ms != null)
                    {
                        MClass mclass = (MClass) ms.getMNamedElement();
                        log.info("Type of redeclared component " + mcomponent.getName() + " resolved to "
                                + mclass.getName());
                        mcomponent.setType(mclass);
                    }
                    else
                    {
                        log.warn("Type of redeclared component " + mcomponent.getName() + " couldn't be resolved");
                        
                        // Create class
                        MClass mclass = createClassAnyway(entry.getValue());
                        mcomponent.setType(mclass);
                        
                        // Create properties from modifications
                        Component_declaration1Context cdc = (Component_declaration1Context) pt;
                        ModificationContext mc = cdc.declaration().modification();
                        if (mc != null && mc.class_modification() != null)
                        {
                            for (ArgumentContext ac : mc.class_modification().argument_list().argument())
                            {
                                if (ac.element_modification_or_replaceable().element_modification() != null)
                                {
                                    Element_modificationContext emc = ac.element_modification_or_replaceable()
                                            .element_modification();
                                    String name = emc.component_reference().IDENT(0).getText();
                                    if (mclass.getComponentByName(name) == null)
                                    {
                                        MComponent mcomponent2 = ModelicaFactory.eINSTANCE.createMComponent();
                                        mcomponent2.setOwningClass(mclass);
                                        ModelicaUtil.setName(mcomponent2, name);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if (pt instanceof Extends_clauseContext)
            {
                if (scope.getMNamedElement() instanceof MClass)
                {
                    ModelicaScope ms = scope.lookForClass(entry.getValue());
                    // add scope of inherited class
                    scope.getSiblings().add(ms);
                    MClass mclass = (MClass) ms.getMNamedElement();
                    if (mclass != null)
                    {
                        MExtension mextension = ModelicaFactory.eINSTANCE.createMExtension();
                        ((MClass) scope.getMNamedElement()).getExtensions().add(mextension);
                        mextension.setExtendedClass(mclass);
                        log.info("Base class of " + ((MClass) scope.getMNamedElement()).getName() + " resolved to "
                                + mclass.getName());
                        // change scope of extend clause
                        ModelicaScope sc = new ModelicaScope(null);
                        sc.setMNamedElement(mclass);
                        scopelist.put(pt, sc);
                    }
                    else
                    {
                        MExtension mextension = ModelicaFactory.eINSTANCE.createMExtension();
                        ((MClass) scope.getMNamedElement()).getExtensions().add(mextension);
                        mextension.setExtendedClass(createClassAnyway(entry.getValue()));
                        log.warn("Based class " + entry.getValue() + " couldn't be resolved");
                    }
                }
            }
            else if (pt instanceof Class_specifierContext)
            {
                if (scope.getMNamedElement() instanceof MClass)
                {
                    MClass mclass = (MClass) scope.lookForClass(entry.getValue()).getMNamedElement();
                    Class_specifierContext csc = (Class_specifierContext) pt;
                    MDirection direction = MDirection.NONE;
                    if (csc.base_prefix().type_prefix().INPUT() != null)
                        direction = MDirection.INPUT;
                    else if (csc.base_prefix().type_prefix().OUTPUT() != null)
                        direction = MDirection.OUTPUT;
                    if (mclass != null)
                    {
                        MExtension mextension = ModelicaFactory.eINSTANCE.createMExtension();
                        ((MClass) scope.getMNamedElement()).getExtensions().add(mextension);
                        mextension.setExtendedClass(mclass);
                        mextension.setDirection(direction);
                        log.info("Base class of " + ((MClass) scope.getMNamedElement()).getName() + " resolved to "
                                + mclass.getName());
                    }
                    else
                    {
                        MExtension mextension = ModelicaFactory.eINSTANCE.createMExtension();
                        ((MClass) scope.getMNamedElement()).getExtensions().add(mextension);
                        mextension.setExtendedClass(createClassAnyway(entry.getValue()));
                        mextension.setDirection(direction);
                        log.warn("Based class " + entry.getValue() + " couldn't be resolved");
                    }
                }
            }
        }
        
    }
    
    /**
     * Creates a Modelica class from the given path
     * 
     * @param path
     *            path of the class, given as list of terminals
     * @return
     */
    private MClass createClassAnyway(List<TerminalNode> path)
    {
        String name = join(path);
        MClass mc = ModelicaFactory.eINSTANCE.createMClass();
        mc.setName(name);
        ModelicaScope ms = new ModelicaScope(name);
        ms.setParent(globalscope);
        ms.setMNamedElement(mc);
        log.info("Creating unresolved class " + name);
        return mc;
    }
    
    /**
     * Joins the terminals with a dot
     * 
     * @param path
     *            path, given as list of terminals
     * @return
     */
    private static String join(List<TerminalNode> path)
    {
        
        StringBuilder name = new StringBuilder();
        for (int i = 0; i < path.size(); i++)
        {
            if (i != 0)
                name.append(".");
            name.append(path.get(i).getText());
        }
        return name.toString();
    }
}
