package com.engisis.sysphs.client.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashSet;
import java.util.concurrent.ExecutionException;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.GroupLayout;
import javax.swing.GroupLayout.Alignment;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.xml.stream.XMLStreamException;

import org.apache.log4j.Logger;

import com.engisis.sysphs.SysMLToSimulationTranslationManager;
import com.engisis.sysphs.client.gui.WorkerDialog.Worker;
import com.engisis.sysphs.translation.modelica.SysMLToModelicaTranslator;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator;
import com.engisis.sysphs.util.SysMLToSimulationTranslator;
import com.engisis.xmiutil.UMLModelErrorException;

/**
 * Frame used to launch the translation of SysML models into simulation models.
 * 
 * @author barbau
 *
 */
public class SysMLToSimulationFrame extends JFrame
{
    private static final long serialVersionUID = 3776110556936924804L;
    
    private static final Logger log = Logger.getLogger(SysMLToSimulationFrame.class);
    /**
     * Input SysML model
     */
    private File fmodel;
    
    /**
     * Output directory
     */
    private File foutput;
    
    /**
     * Panel containing the list of blocks
     */
    private JPanel jpBlock;
    
    /**
     * List of blocks
     */
    private JList<String> jlBlocks;
    
    /**
     * Panel containing information about input model
     */
    private JPanel jpInformation;
    
    /**
     * Label showing the input file
     */
    private JLabel jlFiles;
    /**
     * Label showing the selected block
     */
    private JLabel jlBlock;
    
    /**
     * Button group for selecting translator
     */
    private ButtonGroup bgTranslator;
    /**
     * Panel containing the translation selector
     */
    private JPanel jpTranslator;
    /**
     * Label for the translation selector
     */
    private JLabel jlTranslator;
    /**
     * Radio button for Modelica
     */
    private JRadioButton jrbModelica;
    /**
     * Radio button for Simulink
     */
    private JRadioButton jrbSimulink;
    
    /**
     * Label for the preprocessing
     */
    private JLabel jlPreproc;
    /**
     * Selection of preprocessing
     */
    private JCheckBox jcbPreproc;
    
    /**
     * Button group for selecting Simulink output format
     */
    private ButtonGroup bgFormat;
    /**
     * Label for the Simulink output format
     */
    private JLabel jlFormat;
    /**
     * Radio button for text format
     */
    private JRadioButton jrbText;
    /**
     * Radio button for XML format
     */
    private JRadioButton jrbXML;
    
    /**
     * Button group for Simulink constraint block translation
     */
    private ButtonGroup bgSF;
    /**
     * Label for Simulink constraint block translation
     */
    private JLabel jlSF;
    /**
     * Radio button for Simscape selection
     */
    private JRadioButton jrbSimscape;
    /**
     * Radio button for S-Function level 1 selection
     */
    private JRadioButton jrbSF1;
    /**
     * Radio button for S-Function level 2 selection
     */
    private JRadioButton jrbSF2;
    
    /**
     * Button group for selecting Simscape domains
     */
    private ButtonGroup bgDomain;
    /**
     * Label for Simscape domains
     */
    private JLabel jlDomain;
    /**
     * Radio button for creating new domains
     */
    private JRadioButton jrbCreate;
    /**
     * Radio button for reusing existing domains
     */
    private JRadioButton jrbReuse;
    
    /**
     * Panel for selecting the output directory
     */
    private JPanel jpOutput;
    /**
     * Button for selecting the output directory
     */
    private JButton jbOutput;
    /**
     * Label for the output directory
     */
    private JLabel jlOutput;
    
    /**
     * Button to launch the translation
     */
    private JButton jbTranslate;
    
    // private JPanel jpSimulation;
    // private JLabel jlStart;
    // private JTextField jtfStart;
    // private JLabel jlStop;
    // private JTextField jtfStop;
    // private JButton jbSimulate;
    //
    // private JPanel jpResult;
    // private JList<String> jlResult;
    // private JButton jbResult;
    //
    // private JPanel jpDisclaimer;
    // private JLabel jlDisclaimer;
    
    /**
     * Main translation launcher
     */
    private SysMLToSimulationTranslationManager sysml2simulation;
    /**
     * Current translator
     */
    private SysMLToSimulationTranslator translator;
    /**
     * Dialog to show the translation progress
     */
    private WorkerDialog swd;
    
    /**
     * Constructor of the frame
     * 
     * @param model
     *            path to the SysML file being translated
     * @param rootblock
     *            root SysML block to pre-select
     * @param paths
     *            list of directories (path form) where XMI dependencies are
     *            looked for
     */
    public SysMLToSimulationFrame(final String model, final String rootblock, final String[] paths)
    {
        super("Simulation frame");
        fmodel = new File(model);
        foutput = fmodel.getParentFile();
        if (!isWritable(foutput))
            foutput = new File(System.getProperty("user.home"));
        
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        swd = new WorkerDialog(this);
        
        addWindowListener(new WindowListener()
        {
            @Override
            public void windowOpened(WindowEvent e)
            {
                Worker<Boolean, Void> sw = swd.new Worker<Boolean, Void>()
                {
                    @Override
                    public Boolean doInBackground()
                    {
                        try
                        {
                            sysml2simulation = new SysMLToSimulationTranslationManager(model, paths);
                            jlBlocks.setListData(sysml2simulation.getClasses().toArray(new String[0]));
                            if (rootblock != null)
                                for (int i = 0; i < jlBlocks.getModel().getSize(); i++)
                                    if (jlBlocks.getModel().getElementAt(i).equals(rootblock))
                                        jlBlocks.setSelectedIndex(i);
                        }
                        catch (FileNotFoundException e)
                        {
                            log.error("File error during the initialization", e);
                            return Boolean.valueOf(false);
                        }
                        catch (UMLModelErrorException e)
                        {
                            log.error("Bad model error during the initialization", e);
                            return Boolean.valueOf(false);
                        }
                        catch (XMLStreamException e)
                        {
                            log.error("XML error during the initialization", e);
                            return Boolean.valueOf(false);
                        }
                        catch (IOException e)
                        {
                            log.error("IO error during the initialization", e);
                            return Boolean.valueOf(false);
                        }
                        return Boolean.valueOf(true);
                    }
                    
                    @Override
                    protected void done()
                    {
                        super.done();
                        Boolean res;
                        try
                        {
                            res = get();
                            if (res == null || !res.booleanValue())
                            {
                                JOptionPane.showMessageDialog(swd, "The initialization failed");
                            }
                        }
                        catch (InterruptedException e)
                        {
                            JOptionPane.showMessageDialog(swd, "The initialization failed, " + e);
                        }
                        catch (ExecutionException e)
                        {
                            JOptionPane.showMessageDialog(swd, "The initialization failed, " + e);
                        }
                    }
                };
                swd.show(sw, "Initializing");
            }
            
            @Override
            public void windowIconified(WindowEvent e)
            {
            }
            
            @Override
            public void windowDeiconified(WindowEvent e)
            {
            }
            
            @Override
            public void windowDeactivated(WindowEvent e)
            {
            }
            
            @Override
            public void windowClosing(WindowEvent e)
            {
                try
                {
                    if (sysml2simulation != null)
                        sysml2simulation.setTranslator(null);
                }
                catch (Exception ex)
                {
                }
                
            }
            
            @Override
            public void windowClosed(WindowEvent e)
            {
            }
            
            @Override
            public void windowActivated(WindowEvent e)
            {
            }
        });
        
        // Loading the translator
        
        getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.Y_AXIS));
        
        jpBlock = new JPanel();
        getContentPane().add(jpBlock);
        jpBlock.setBorder(BorderFactory.createTitledBorder("Blocks"));
        jpBlock.setLayout(new BorderLayout());
        
        jlBlocks = new JList<String>();
        JScrollPane jsp = new JScrollPane();
        jsp.setViewportView(jlBlocks);
        jpBlock.add(jsp);
        
        // Information panel
        jpInformation = new JPanel();
        getContentPane().add(jpInformation);
        GroupLayout glInformation = new GroupLayout(jpInformation);
        jpInformation.setLayout(glInformation);
        jpInformation.setBorder(BorderFactory.createTitledBorder("Information"));
        
        jlFiles = new JLabel();
        StringBuilder sbinfo = new StringBuilder();
        sbinfo.append("Files: " + model);
        jlFiles.setText(sbinfo.toString());
        
        jlBlock = new JLabel();
        jlBlock.setText("Block: " + rootblock);
        
        glInformation.setHorizontalGroup(glInformation.createParallelGroup(GroupLayout.Alignment.LEADING)
                .addComponent(jlFiles).addComponent(jlBlock));
        
        glInformation.setVerticalGroup(glInformation.createSequentialGroup().addComponent(jlFiles)
                .addComponent(jlBlock));
        
        // Translator panel
        jpTranslator = new JPanel();
        getContentPane().add(jpTranslator);
        GroupLayout glTranslator = new GroupLayout(jpTranslator);
        jpTranslator.setLayout(glTranslator);
        jpTranslator.setBorder(BorderFactory.createTitledBorder("Translator"));
        
        jlTranslator = new JLabel("Target language:");
        
        jrbModelica = new JRadioButton("Modelica");
        
        jrbSimulink = new JRadioButton("Simulink");
        
        bgTranslator = new ButtonGroup();
        bgTranslator.add(jrbModelica);
        bgTranslator.add(jrbSimulink);
        jrbModelica.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(ItemEvent e)
            {
                if (e.getStateChange() == 1)
                {
                    translator = new SysMLToModelicaTranslator();
                    
                    jrbText.setEnabled(false);
                    jrbXML.setEnabled(false);
                    jrbSF1.setEnabled(false);
                    jrbSF2.setEnabled(false);
                    jrbSimscape.setEnabled(false);
                    jrbReuse.setEnabled(false);
                    jrbCreate.setEnabled(false);
                    
                    bgFormat.clearSelection();
                    bgSF.clearSelection();
                    bgDomain.clearSelection();
                }
            }
        });
        
        jrbSimulink.addItemListener(new ItemListener()
        {
            
            @Override
            public void itemStateChanged(ItemEvent e)
            {
                translator = new SysMLToSimulinkTranslator();
                jrbText.setEnabled(true);
                jrbXML.setEnabled(true);
                jrbSF1.setEnabled(true);
                jrbSF2.setEnabled(true);
                jrbSimscape.setEnabled(true);
            }
        });
        
        jlFormat = new JLabel("Format:");
        
        jrbText = new JRadioButton("Text (.mdl)");
        jrbXML = new JRadioButton("XML (.sdl)");
        
        bgFormat = new ButtonGroup();
        bgFormat.add(jrbText);
        bgFormat.add(jrbXML);
        
        jlSF = new JLabel("S-Function or Simscape:");
        
        jrbSF1 = new JRadioButton("S-Function version 1");
        jrbSF2 = new JRadioButton("S-Function version 2");
        jrbSimscape = new JRadioButton("Simscape");
        jrbSimscape.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(ItemEvent e)
            {
                if (e.getStateChange() == 1)
                {
                    jrbCreate.setEnabled(true);
                    jrbReuse.setEnabled(true);
                }
                else
                {
                    jrbCreate.setEnabled(false);
                    jrbReuse.setEnabled(false);
                    
                    bgDomain.clearSelection();
                }
            }
        });
        
        bgSF = new ButtonGroup();
        bgSF.add(jrbSimscape);
        bgSF.add(jrbSF2);
        bgSF.add(jrbSF1);
        
        jlDomain = new JLabel("Simscape port libraries:");
        jrbCreate = new JRadioButton("Create new port types");
        jrbReuse = new JRadioButton("Reuse existing port types");
        
        bgDomain = new ButtonGroup();
        bgDomain.add(jrbCreate);
        bgDomain.add(jrbReuse);
        
        jlPreproc = new JLabel();
        jlPreproc.setText("Preprocessing transformations (Experimental)");
        
        jcbPreproc = new JCheckBox();
        
        glTranslator.setHorizontalGroup(glTranslator
                .createSequentialGroup()
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.LEADING).addComponent(jlTranslator)
                                .addComponent(jlFormat).addComponent(jlSF).addComponent(jlDomain)
                                .addComponent(jlPreproc))
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.LEADING).addComponent(jrbModelica)
                                .addComponent(jrbText).addComponent(jrbSimscape).addComponent(jrbCreate)
                                .addComponent(jcbPreproc))
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.LEADING).addComponent(jrbSimulink)
                                .addComponent(jrbXML).addComponent(jrbSF2).addComponent(jrbReuse).addGap(0))
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.LEADING).addGap(0).addGap(0).addComponent(jrbSF1)
                                .addGap(0).addGap(0)));
        
        glTranslator.setVerticalGroup(glTranslator
                .createSequentialGroup()
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.BASELINE).addComponent(jlTranslator)
                                .addComponent(jrbModelica).addComponent(jrbSimulink).addGap(0))
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.BASELINE).addComponent(jlFormat)
                                .addComponent(jrbText).addComponent(jrbXML).addGap(0))
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.BASELINE).addComponent(jlSF)
                                .addComponent(jrbSimscape).addComponent(jrbSF2).addComponent(jrbSF1))
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.BASELINE).addComponent(jlDomain)
                                .addComponent(jrbCreate).addComponent(jrbReuse).addGap(0))
                .addGroup(
                        glTranslator.createParallelGroup(Alignment.BASELINE).addComponent(jlPreproc)
                                .addComponent(jcbPreproc).addGap(0).addGap(0)));
        
        jpOutput = new JPanel();
        getContentPane().add(jpOutput);
        GroupLayout glOutput = new GroupLayout(jpOutput);
        jpOutput.setLayout(glOutput);
        jpOutput.setBorder(BorderFactory.createTitledBorder("Output directory"));
        
        jbOutput = new JButton();
        jbOutput.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent e)
            {
                JFileChooser jfc = new JFileChooser();
                jfc.setCurrentDirectory(foutput);
                jfc.setDialogTitle("Select output directory");
                jfc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
                jfc.setAcceptAllFileFilterUsed(false);
                
                if (jfc.showOpenDialog(SysMLToSimulationFrame.this) == JFileChooser.APPROVE_OPTION)
                {
                    foutput = jfc.getSelectedFile();
                    updateOutput();
                }
            }
        });
        
        jlOutput = new JLabel();
        
        glOutput.setHorizontalGroup(glOutput.createSequentialGroup().addComponent(jbOutput).addComponent(jlOutput));
        glOutput.setVerticalGroup(glOutput.createParallelGroup(Alignment.BASELINE).addComponent(jbOutput)
                .addComponent(jlOutput));
        
        jbTranslate = new JButton();
        jbTranslate.setText("Translate");
        jbTranslate.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent arg0)
            {
                try
                {
                    HashSet<Object> options = new HashSet<Object>();
                    if (jrbText.isEnabled() && jrbText.isSelected())
                        options.add(SysMLToSimulinkTranslator.Format.TEXT);
                    else if (jrbXML.isEnabled() && jrbText.isSelected())
                        options.add(SysMLToSimulinkTranslator.Format.XML);
                    
                    if (jrbSF1.isEnabled() && jrbSF1.isSelected())
                        options.add(SysMLToSimulinkTranslator.SFunction.Level1);
                    else if (jrbSF2.isEnabled() && jrbSF2.isSelected())
                        options.add(SysMLToSimulinkTranslator.SFunction.Level2);
                    else if (jrbSimscape.isEnabled() && jrbSimscape.isSelected())
                        options.add(SysMLToSimulinkTranslator.SFunction.Simscape);
                    
                    if (jrbCreate.isEnabled() && jrbCreate.isSelected())
                        options.add(SysMLToSimulinkTranslator.Domains.CREATE);
                    else if (jrbReuse.isEnabled() && jrbReuse.isSelected())
                        options.add(SysMLToSimulinkTranslator.Domains.REUSE);
                    
                    if (translator == null)
                    {
                        JOptionPane.showMessageDialog(SysMLToSimulationFrame.this, "You must select a translator");
                        return;
                    }
                    if (jlBlocks.getSelectedIndex() == -1)
                    {
                        JOptionPane.showMessageDialog(SysMLToSimulationFrame.this, "You must select a root class");
                        return;
                    }
                    translator.reset();
                    translator.loadOptions(options);
                    sysml2simulation.setInputRootName(jlBlocks.getSelectedValue());
                    sysml2simulation.setPreprocessing(jcbPreproc.isSelected());
                    sysml2simulation.setTranslator(translator);
                    sysml2simulation.setOutputDirectory(foutput);
                }
                catch (Exception e)
                {
                    log.error("Could not create the translator", e);
                }
                Worker<Void, Void> sw = swd.new Worker<Void, Void>()
                {
                    @Override
                    public Void doInBackground()
                    {
                        try
                        {
                            sysml2simulation.execute();
                        }
                        catch (UMLModelErrorException e)
                        {
                            JOptionPane.showMessageDialog(swd, "The translation failed");
                            log.error("Bad model error during the translation");
                        }
                        catch (IOException e)
                        {
                            JOptionPane.showMessageDialog(swd, "The translation failed");
                            log.error("IO error during the translation", e);
                        }
                        catch (Exception e)
                        {
                            JOptionPane.showMessageDialog(swd, "Unexpected exception");
                            log.error("Other error during the translation", e);
                        }
                        return null;
                    }
                };
                swd.show(sw, "Translating file");
            }
        });
        
        // Translator button
        getContentPane().add(jbTranslate);
        
        // Simulation panel (removed)
        
        // jpSimulation = new JPanel();
        // getContentPane().add(jpSimulation);
        // jpSimulation.setBorder(BorderFactory.createTitledBorder("Simulation"));
        // GroupLayout glSimulation = new GroupLayout(jpSimulation);
        // jpSimulation.setLayout(glSimulation);
        //
        // jlStart = new JLabel();
        // jlStart.setText("Start time: ");
        //
        // jtfStart = new JTextField();
        // jtfStart.setText("0.0");
        //
        // jlStop = new JLabel();
        // jlStop.setText("Stop time: ");
        //
        // jtfStop = new JTextField();
        // jtfStop.setText("10.0");
        //
        // jbSimulate = new JButton();
        // jbSimulate.setText("Simulate");
        // jbSimulate.addActionListener(new ActionListener()
        // {
        //
        // @Override
        // public void actionPerformed(ActionEvent arg0)
        // {
        // SimulationWorker<List<String>, Void> sw = swd.new
        // SimulationWorker<List<String>, Void>()
        // {
        //
        // @Override
        // public List<String> doInBackground()
        // {
        // sysml2simulation.simulate(Double.parseDouble(jtfStart.getText()),
        // Double.parseDouble(jtfStop.getText()));
        // return sysml2simulation.listVariables();
        // }
        //
        // @Override
        // protected void done()
        // {
        // super.done();
        // try
        // {
        // jlResult.setListData(new Vector<String>(get()));
        // log.info(get().size() + " variables added");
        // jlResult.invalidate();
        // }
        // catch (Exception e)
        // {
        // JOptionPane.showMessageDialog(swd,
        // "Unable to set the list of variables");
        // log.error("Unable to set the list of variables", e);
        // }
        // }
        // };
        // swd.show(sw, "Simulating model");
        // }
        // });
        //
        // glSimulation.setHorizontalGroup(glSimulation
        // .createSequentialGroup()
        // .addGroup(
        // glSimulation.createParallelGroup(Alignment.LEADING).addComponent(jlStart).addComponent(jlStop))
        // .addGroup(
        // glSimulation.createParallelGroup(Alignment.LEADING).addComponent(jtfStart)
        // .addComponent(jtfStop).addComponent(jbSimulate)));
        //
        // glSimulation
        // .setVerticalGroup(glSimulation
        // .createSequentialGroup()
        // .addGroup(
        // glSimulation.createParallelGroup(Alignment.BASELINE).addComponent(jlStart)
        // .addComponent(jtfStart))
        // .addGroup(
        // glSimulation.createParallelGroup(Alignment.BASELINE).addComponent(jlStop)
        // .addComponent(jtfStop)).addComponent(jbSimulate));
        
        // Results panel (removed)
        
        // jpResult = new JPanel();
        // getContentPane().add(jpResult);
        // jpResult.setBorder(BorderFactory.createTitledBorder("Results"));
        // GroupLayout glResult = new GroupLayout(jpResult);
        // jpResult.setLayout(glResult);
        //
        // jlResult = new JList<String>();
        // jlResult.setPreferredSize(null);
        //
        // JScrollPane jspResult = new JScrollPane(jlResult);
        // jspResult.setPreferredSize(null);
        //
        // jbResult = new JButton();
        // jbResult.setText("Display results");
        // jbResult.addActionListener(new ActionListener()
        // {
        //
        // @Override
        // public void actionPerformed(ActionEvent arg0)
        // {
        // if (jlResult.getSelectedValuesList().size() != 0)
        // sysml2simulation.plot(jlResult.getSelectedValuesList());
        // }
        // });
        //
        // jpResult.add(jbResult);
        //
        // glResult.setHorizontalGroup(glResult.createParallelGroup(Alignment.LEADING).addComponent(jspResult)
        // .addComponent(jbResult));
        //
        // glResult.setVerticalGroup(glResult.createSequentialGroup().addComponent(jspResult).addComponent(jbResult));
        
        pack();
        jrbModelica.setSelected(true);
        updateOutput();
    }
    
    /**
     * Update the output label
     */
    private void updateOutput()
    {
        log.debug("Changed output directory to " + foutput);
        jbOutput.setText(foutput.getAbsolutePath());
        if (isWritable(foutput))
        {
            jlOutput.setText("");
            jlOutput.setForeground(Color.BLACK);
        }
        else
        {
            jlOutput.setText("Readonly directory");
            jlOutput.setForeground(Color.RED);
        }
    }
    
    /**
     * Determines if the file is writable
     * 
     * @param dir
     * @return
     */
    private static boolean isWritable(File dir)
    {
        if (dir == null || !dir.exists() || !dir.isDirectory() || !dir.canWrite())
            return false;
        try
        {
            File tmp = File.createTempFile("sim", "tmp", dir);
            tmp.delete();
        }
        catch (IOException e)
        {
            return false;
        }
        return true;
    }
}
