package com.engisis.sysphs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.xml.stream.XMLStreamException;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;

import com.engisis.sysphs.util.SysMLPreProcessor;
import com.engisis.sysphs.util.SysMLToSimulationTranslator;
import com.engisis.sysphs.util.SysPhSUtil;
import com.engisis.xmiutil.EMFUtil;
import com.engisis.xmiutil.SysMLUtil;
import com.engisis.xmiutil.UMLModelErrorException;

/**
 * This class is used to manage the translation of SysML models into simulation
 * models
 * 
 * @author barbau
 *
 */
public class SysMLToSimulationTranslationManager
{
    private static final Logger log = Logger.getLogger(SysMLToSimulationTranslationManager.class);
    
    /**
     * UML class that will be translated
     */
    private Class rootclass;
    
    /**
     * name of the UML class that will be translated
     */
    private String rootclassname;
    
    /**
     * Resource that holds the SysML model
     */
    private Resource resource;
    
    /**
     * Translator used to create the simulation model
     */
    private SysMLToSimulationTranslator translator;
    
    /**
     * Output directory
     */
    private File outputdirectory;
    
    /**
     * Whether the user wants the preprocessing
     */
    private boolean preproc = false;
    
    /**
     * Whether the preprocessing has been done
     */
    private boolean preprocdone = false;
    
    /**
     * Constructs a SysML to simulation transformation manager. There are 3
     * phases in the translation: 1) the SysML model is loaded when the
     * constructor is called, 2) the root class is selected among the loaded
     * classes - the simulation generator is set, 3) the translation is executed
     * The SysML model is not loaded in the generator to allow different
     * generators to be applied.
     * 
     * @param model
     *            file location of a SysML model
     * @param paths
     *            paths where to look for XMI files
     * @param preproc
     *            whether SysML preprocessing is enabled
     * @throws IOException
     * @throws FileNotFoundException
     * @throws UMLModelErrorException
     * @throws XMLStreamException
     */
    public SysMLToSimulationTranslationManager(String model, String[] paths)
            throws IOException, FileNotFoundException, UMLModelErrorException, XMLStreamException
    {
        if (model == null)
            throw new IllegalArgumentException("The model can't be null");
        
        List<URI> lpaths = new ArrayList<URI>();
        if (paths != null)
            for (String path : paths)
            {
                File fpath = new File(path);
                lpaths.add(URI.createFileURI(fpath.getAbsolutePath() + File.separator));
            }
        
        ResourceSet rs = EMFUtil.createResourceSet();
        
        resource = EMFUtil.loadResourceWithDependencies(rs, URI.createFileURI(new File(model).getAbsolutePath()),
                lpaths);
        
        // return error if no SysML profile
        if (rs.getPackageRegistry().get(SysMLUtil.nsSysML.toString()) == null)
            throw new UMLModelErrorException(resource, "The model doesn't import the SysML profile");
        if (rs.getPackageRegistry().get(SysPhSUtil.nsSysPhS.toString()) == null)
            throw new UMLModelErrorException(resource, "The model doesn't import the SysPhS profile");
        // DebugUtil.checkResource(r);
    }
    
    /**
     * Returns a list of classes in the current resource
     * 
     * @return a list of qualified names
     */
    public List<String> getClasses()
    {
        LinkedList<String> ret = new LinkedList<String>();
        TreeIterator<EObject> iter = resource.getAllContents();
        while (iter.hasNext())
        {
            EObject eo = iter.next();
            if (!(eo instanceof NamedElement))
            {
                iter.prune();
            }
            else if (eo instanceof Class)
            {
                ret.add(((Class) eo).getQualifiedName());
            }
        }
        return ret;
    }
    
    /**
     * Sets the UML Class from which the translation works
     * 
     * @param rootclass
     *            a qualified name of the root class
     * @throws UMLModelErrorException
     */
    public void setInputRootName(String rootclass) throws UMLModelErrorException
    {
        if (rootclass == null)
            throw new IllegalArgumentException("The root element can't be null");
        this.rootclassname = rootclass;
        
        String[] fqname = rootclass.split("::");
        
        // look for the model (I assume there's only one)
        Model umodel = null;
        for (EObject object : resource.getContents())
        {
            if (object instanceof Model && fqname[0].equals(((Model) object).getName()))
                umodel = (Model) object;
        }
        
        // exit if none is found
        if (umodel == null)
        {
            throw new UMLModelErrorException(resource,
                    "Could not find a root model in the XMI file named " + fqname[0]);
        }
        
        // Look for the root block (more efficient than parsing the entire file)
        Namespace currentelement = umodel;
        NamedElement nerootblock = null;
        
        // Parse the block's qualified name
        for (int i = 1; i < fqname.length; i++)
        {
            // get the next name
            String nextname = fqname[i];
            
            NamedElement nextelement = currentelement.getMember(nextname);
            if (nextelement == null)
            {
                log.error("Unknown element: " + currentelement.getQualifiedName() + "::" + fqname[i]);
                break;
            }
            if (rootclass.equals(nextelement.getQualifiedName()))
            {
                nerootblock = nextelement;
                // we have to be at the end of the loop, so no break
            }
            else
            {
                if (nextelement instanceof Namespace)
                    currentelement = (Namespace) nextelement;
                else
                {
                    log.error("Not a namespace: " + currentelement.getQualifiedName() + "::" + fqname[i]);
                    break;
                }
            }
        }
        
        if (nerootblock == null)
        {
            throw new UMLModelErrorException(resource, "Could not find the root block in the model");
        }
        if (!(nerootblock instanceof Class))
        {
            throw new UMLModelErrorException(resource, "The root block is not a Class");
        }
        
        this.rootclass = ((Class) nerootblock);
    }
    
    /**
     * Returns the name of the input root class
     * 
     * @return the name of the input root class
     */
    public String getInputRootName()
    {
        return rootclassname;
    }
    
    /**
     * Enables/disables the SysML preprocessing
     * 
     * @param preproc
     *            true if preprocessing is enabled
     */
    public void setPreprocessing(boolean preproc)
    {
        this.preproc = preproc;
    }
    
    /**
     * Sets the output directory
     * 
     * @param outputdirectory
     *            file representing the output directory
     */
    public void setOutputDirectory(File outputdirectory)
    {
        this.outputdirectory = outputdirectory;
    }
    
    /**
     * Sets the translator to be used
     * 
     * @param translator
     *            simulation translator
     */
    public void setTranslator(SysMLToSimulationTranslator translator)
    {
        this.translator = translator;
    }
    
    /**
     * Executes the translation on the resource
     * 
     * @throws UMLModelErrorException
     * @throws IOException
     */
    public void execute() throws UMLModelErrorException, IOException
    {
        if (translator == null)
            throw new IllegalStateException("A generator must be provided first");
        if (rootclass == null)
            throw new IllegalStateException("A root class must be provided first");
        log.info("started");
        
        // list stereotyped elements
        // DebugUtil.printStereotypedElements(r);
        
        if (preproc && !preprocdone)
        {
            if (outputdirectory == null)
                resource.setURI(resource.getURI().appendFileExtension("bak"));
            else
                resource.setURI(URI.createFileURI(outputdirectory.getAbsolutePath())
                        .appendSegment(resource.getURI().lastSegment()).appendFileExtension("preprocessed"));
            SysMLPreProcessor sysmlt = new SysMLPreProcessor(resource);
            sysmlt.execute();
            resource.save(null);
            preprocdone = true;
        }
        
        // Execute the translation
        translator.execute(rootclass, resource, outputdirectory);
        
    }
}
