package com.engisis.xmiutil;

import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Feature;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.internal.resource.UMLResourceFactoryImpl;

/**
 * Many methods are not static to allow overriding.
 * 
 * @author barbau
 *
 */
public class UMLUtil
{
    private static final Logger log = Logger.getLogger(UMLUtil.class);
    private static final URI locationTypes = URI.createURI("http://www.omg.org/spec/UML/20161101/PrimitiveTypes.xmi#");
    private ResourceSet rs;
    
    private DataType ustring;
    private DataType uboolean;
    private DataType uinteger;
    private DataType udouble;
    
    /**
     * Main constructor for UMLUtil class
     * 
     * @param rs
     *            Resource set in which UML is loaded
     */
    public UMLUtil(ResourceSet rs)
    {
        if (rs == null)
            throw new IllegalArgumentException("You must provide a ResourceSet");
        this.rs = rs;
        loadUML();
    }
    
    protected void loadUML()
    {
        ustring = loadDataType(locationTypes, "String");
        uboolean = loadDataType(locationTypes, "Boolean");
        uinteger = loadDataType(locationTypes, "Integer");
        udouble = loadDataType(locationTypes, "Real");
    }
    
    /**
     * Loads the data type with the specified location and ID into the resource
     * set
     * 
     * @param location
     *            location of the data type
     * @param id
     *            id of the data type
     * @return loaded data type
     * @throws UMLModelErrorException
     */
    protected DataType loadDataType(URI location, String id)
    {
        Element element = loadFromLocationAndID(rs, location, id);
        if (!(element instanceof DataType))
            throw new ClassCastException("The element identified by " + id + " at " + location + " is not a data type");
        return (DataType) element;
    }
    
    /**
     * Loads an element from a location and an ID into the resource set
     * 
     * @param rs
     *            resource set in which to load the element
     * @param location
     *            location of the element
     * @param id
     *            id of the element
     * @return loaded element
     */
    protected static Element loadFromLocationAndID(ResourceSet rs, URI location, String id)
    {
        if (location == null)
            throw new NullPointerException("You must provide a location");
        if (id == null)
            throw new NullPointerException("You must provide an id");
        return (Element) rs.getEObject(location.appendFragment(id), true);
    }
    
    /**
     * Loads an element from a namespace and a name into the resource set
     * 
     * @param rs
     *            resource set in which to load the element
     * @param namespace
     *            namespace of the element
     * @param name
     *            name of the element
     * @return loaded element
     */
    protected NamedElement loadFromNamespaceAndName(URI namespace, String name) throws UMLModelErrorException
    {
        if (namespace == null)
            throw new IllegalArgumentException("You must provide a namespace");
        if (name == null)
            throw new IllegalArgumentException("You must provide a name");
        
        Resource resource = rs.getResource(namespace, true);
        TreeIterator<EObject> treeiterator = resource.getAllContents();
        while (treeiterator.hasNext())
        {
            EObject currenteobject = treeiterator.next();
            if (!(currenteobject instanceof Namespace))
                treeiterator.prune();
            else
            {
                Namespace namedelement = (Namespace) currenteobject;
                if (name.equals(namedelement.getName()))
                {
                    return namedelement;
                }
            }
        }
        throw new UMLModelErrorException(resource, "Can't retrieve an element named " + name + " in " + namespace);
    }
    
    /**
     * Returns UML String
     * 
     * @return UML String
     */
    public DataType getUMLString()
    {
        return ustring;
    }
    
    /**
     * Returns UML Boolean
     * 
     * @return UML Boolean
     */
    public DataType getUMLBoolean()
    {
        return uboolean;
    }
    
    /**
     * Returns UML Integer
     * 
     * @return UML Integer
     */
    public DataType getUMLInteger()
    {
        return uinteger;
    }
    
    /**
     * Returns UML Double
     * 
     * @return UML Double
     */
    public DataType getUMLDouble()
    {
        return udouble;
    }
    
    /**
     * Returns all general classifiers for a given classifier
     * 
     * @param uclassifier
     *            Specific classifier
     * @param recursive
     *            Whether general classifiers are looked for recursively
     * @param self
     *            Whether the specific classifier is included
     * @return General classifiers of a given classifier
     */
    public Set<Classifier> getGenerals(Classifier uclassifier, boolean recursive, boolean self)
    {
        LinkedHashSet<Classifier> ret = new LinkedHashSet<>();
        for (Generalization ugeneralization : uclassifier.getGeneralizations())
        {
            if (ugeneralization.getSpecific() == uclassifier)
            {
                Classifier ugeneral = ugeneralization.getGeneral();
                if (recursive)
                    ret.addAll(getGenerals(ugeneral, true, false));
                ret.add(ugeneral);
            }
        }
        if (self)
            ret.add(uclassifier);
        return ret;
    }
    
    /**
     * Returns the nested classes of a given class
     * 
     * @param uclass
     *            Nesting class
     * @param recursive
     *            Whether nested classes are looked for recursively
     * @param self
     *            Whether the nesting class is included
     * @return Set of nested classes
     */
    public Set<Classifier> getNestedClasses(Class uclass, boolean recursive, boolean self)
    {
        LinkedHashSet<Classifier> ret = new LinkedHashSet<>();
        for (Classifier unestedclassifier : uclass.getNestedClassifiers())
        {
            ret.add(unestedclassifier);
            if (recursive && unestedclassifier instanceof Class)
            {
                ret.addAll(getNestedClasses((Class) unestedclassifier, true, false));
            }
        }
        if (self)
            ret.add(uclass);
        return ret;
    }
    
    /**
     * Looked for the final redefining feature of the specified feature
     * 
     * @param ufeature
     *            feature for which a redefining feature is looked for
     * @param context
     *            context of the redefinitions
     * @return final redefining feature
     */
    public Feature correctFeature(Feature ufeature, Classifier context)
    {
        if (ufeature == null)
            return null;
        Feature ret = ufeature;
        Set<Feature> ufeatures = getAllFeatures(context);
        boolean cont = true;
        loop: while (cont)
        {
            cont = false;
            for (Feature ufeature2 : ufeatures)
            {
                if (ufeature2.getRedefinedElements().contains(ret))
                {
                    ret = ufeature2;
                    cont = true;
                    continue loop;
                }
            }
        }
        return ret;
    }
    
    /**
     * Looked for the final redefining property of the specified property
     * 
     * @param uproperty
     *            property for which a redefining property is looked for
     * @param context
     *            context of the redefinitions
     * @return final redefining property
     */
    public Property correctProperty(Property uproperty, Classifier context)
    {
        if (uproperty == null)
            return null;
        Property ret = uproperty;
        Set<Property> uproperties = getAllAttributes(context);
        boolean cont = true;
        loop: while (cont)
        {
            cont = false;
            for (Property uprop : uproperties)
            {
                if (uprop.getRedefinedProperties().contains(ret))
                {
                    ret = uprop;
                    cont = true;
                    continue loop;
                }
            }
        }
        return ret;
    }
    
    /**
     * Looked for the final redefining connector of the specified connector
     * 
     * @param uconnector
     *            connector for which a redefining property is looked for
     * @param context
     *            context of the redefinitions
     * @return final redefining connector
     */
    public Connector correctConnector(Connector uconnector, Class context)
    {
        if (uconnector == null)
            return null;
        Connector ret = uconnector;
        Set<Connector> uconnectors = getAllConnectors(context);
        boolean cont = true;
        loop: while (cont)
        {
            cont = false;
            for (Connector uconn : uconnectors)
            {
                if (uconn.getRedefinedConnectors().contains(ret))
                {
                    ret = uconn;
                    cont = true;
                    continue loop;
                }
            }
        }
        return ret;
    }
    
    /**
     * Returns all owned rules of the given namespace
     * 
     * @param unamespace
     *            Given namespace
     * @return Set of all rules of the given namespace
     */
    public Set<Constraint> getOwnedRules(Namespace unamespace)
    {
        return new LinkedHashSet<>(unamespace.getOwnedRules());
    }
    
    /**
     * Returns all owned and inherited rules of the given classifier
     * 
     * @param uclassifier
     *            Given classifier
     * @return Set of all owned and inherited rules of the given classifier
     */
    public Set<Constraint> getAllRules(Classifier uclassifier)
    {
        Set<Constraint> ret = new LinkedHashSet<Constraint>();
        for (Classifier ugeneral : getGenerals(uclassifier, true, true))
            ret.addAll(getOwnedRules(ugeneral));
        return ret;
    }
    
    /**
     * Returns all owned features of the given classifier
     * 
     * @param uclassifier
     *            Given classifier
     * @return Ordered set of all owned features of the given classifier
     */
    public Set<Feature> getOwnedFeatures(Classifier uclassifier)
    {
        LinkedHashSet<Feature> ret = new LinkedHashSet<Feature>(uclassifier.getFeatures());
        return ret;
    }
    
    /**
     * Returns all owned and inherited features of the given classifier
     * 
     * @param uclassifier
     *            Given classifier
     * @return Ordered set of all owned and inherited features of the given
     *         classifier
     */
    public Set<Feature> getAllFeatures(Classifier uclassifier)
    {
        Set<Feature> ret = new LinkedHashSet<Feature>();
        for (Classifier ugeneral : uclassifier.getGenerals())
            ret.addAll(getAllFeatures(ugeneral));
        ret.addAll(getOwnedFeatures(uclassifier));
        return ret;
    }
    
    /**
     * Returns all owned and inherited features of the given classifier, minus
     * redefined features
     * 
     * @param uclassifier
     *            Given classifier
     * @return Ordered set of all owned and inherited features of the given
     *         classifier, minus redefined features
     */
    public Set<Feature> getAllCorrectedFeatures(Classifier uclassifier)
    {
        Set<Feature> ret = new LinkedHashSet<>(getAllFeatures(uclassifier));
        Set<Element> rem = new LinkedHashSet<>();
        for (Feature f : ret)
            rem.addAll(f.getRedefinedElements());
        ret.removeAll(rem);
        return ret;
    }
    
    /**
     * Returns all owned connectors of the given class
     * 
     * @param uclass
     *            Given class
     * @return Set of all owned connectors of the given class
     */
    public Set<Connector> getOwnedConnectors(Class uclass)
    {
        Set<Connector> ret = new LinkedHashSet<Connector>();
        ret.addAll(uclass.getOwnedConnectors());
        return ret;
    }
    
    /**
     * Returns all owned and inherited connectors of the given class
     * 
     * @param uclass
     *            class in which connectors are looked for
     * @return Ordered set of all owned and inherited connectors of the given
     *         class
     */
    public Set<Connector> getAllConnectors(Class uclass)
    {
        Set<Connector> ret = new LinkedHashSet<Connector>();
        for (Classifier ugeneral : getGenerals(uclass, true, true))
            if (ugeneral instanceof Class)
                ret.addAll(((Class) ugeneral).getOwnedConnectors());
        return ret;
    }
    
    /**
     * Returns all owned and inherited connectors of the given class, minus
     * redefined connectors
     * 
     * @param uclass
     *            class in which connectors are looked for
     * @return Ordered set of all owned and inherited connectors of the given
     *         class, minus redefined connectors
     */
    public Set<Connector> getAllCorrectedConnectors(Class uclass)
    {
        Set<Connector> ret = new LinkedHashSet<>(getAllConnectors(uclass));
        Set<Connector> rem = new LinkedHashSet<>();
        for (Connector c : ret)
            rem.addAll(c.getRedefinedConnectors());
        ret.removeAll(rem);
        return ret;
    }
    
    /**
     * Returns all owned attributes of the given classifier
     * 
     * @param uclassifier
     *            classifier in which attributes are looked for
     * @return Ordered set of all owned attributes of the given classifier
     */
    public Set<Property> getOwnedAttributes(Classifier uclassifier)
    {
        if (uclassifier instanceof Class)
            return new LinkedHashSet<Property>(((Class) uclassifier).getOwnedAttributes());
        else if (uclassifier instanceof Interface)
            return new LinkedHashSet<Property>(((Interface) uclassifier).getOwnedAttributes());
        else if (uclassifier instanceof DataType)
            return new LinkedHashSet<Property>(((DataType) uclassifier).getOwnedAttributes());
        return new LinkedHashSet<Property>();
    }
    
    /**
     * Returns all owned and inherited attributes of the given classifier
     * 
     * @param uclassifier
     *            classifier in which attributes are looked for
     * @return Ordered set of all owned and inherited attributes of the given
     *         classifier
     */
    public Set<Property> getAllAttributes(Classifier uclassifier)
    {
        Set<Property> ret = new LinkedHashSet<Property>();
        for (Classifier ugeneral : uclassifier.getGenerals())
            ret.addAll(getAllAttributes(ugeneral));
        ret.addAll(getOwnedAttributes(uclassifier));
        return ret;
    }
    
    /**
     * Returns all owned and inherited attributes of the given classifier, minus
     * redefined attributes
     * 
     * @param uclassifier
     *            classifier in which attributes are looked for
     * @return Ordered set of all owned and inherited attributes of the given
     *         classifier, minus redefined attributes
     */
    public Set<Property> getAllCorrectedAttributes(Classifier uclassifier)
    {
        Set<Property> ret = new LinkedHashSet<Property>(getAllAttributes(uclassifier));
        Set<Property> rem = new LinkedHashSet<Property>();
        for (Property p : ret)
            rem.addAll(p.getRedefinedProperties());
        ret.removeAll(rem);
        return ret;
    }
    
    /**
     * Returns all owned operations of the given classifier
     * 
     * @param uclassifier
     *            Given classifier
     * @return Ordered set of all owned operations of the given classifier
     */
    public Set<Operation> getOwnedOperations(Classifier uclassifier)
    {
        if (uclassifier instanceof Class)
            return new LinkedHashSet<Operation>(((Class) uclassifier).getOwnedOperations());
        else if (uclassifier instanceof Interface)
            return new LinkedHashSet<Operation>(((Interface) uclassifier).getOwnedOperations());
        else if (uclassifier instanceof DataType)
            return new LinkedHashSet<Operation>(((DataType) uclassifier).getOwnedOperations());
        return new LinkedHashSet<Operation>();
    }
    
    /**
     * Returns all owned and inherited operations of the given classifier
     * 
     * @param uclassifier
     *            Given classifier
     * @return Ordered set of all owned and inherited operations of the given
     *         classifier
     */
    public Set<Operation> getAllOperations(Classifier uclassifier)
    {
        Set<Operation> ret = new LinkedHashSet<Operation>();
        for (Classifier ugeneral : uclassifier.getGenerals())
            ret.addAll(getAllOperations(ugeneral));
        ret.addAll(getOwnedOperations(uclassifier));
        return ret;
    }
    
    /**
     * Returns all owned and inherited operations of the given classifier, minus
     * redefined operations
     * 
     * @param uclassifier
     *            Given classifier
     * @return Ordered set of all owned and inherited operations of the given
     *         classifier, minus redefined operations
     */
    public Set<Operation> getAllCorrectedOperations(Classifier uclassifier)
    {
        Set<Operation> ret = new LinkedHashSet<Operation>(getAllOperations(uclassifier));
        Set<Operation> rem = new LinkedHashSet<Operation>();
        for (Operation o : ret)
            rem.addAll(o.getRedefinedOperations());
        ret.removeAll(rem);
        return ret;
    }
    
    /**
     * Returns the owned or inherited classifier behavior of the given class
     * 
     * @param uclass
     *            Given class
     * @return Classifier behavior of the given class
     */
    public Behavior getClassifierBehavior(Class uclass)
    {
        Behavior b = uclass.getClassifierBehavior();
        if (b != null)
            return b;
        for (Classifier c : uclass.getGenerals())
        {
            if (c instanceof Class)
            {
                b = getClassifierBehavior((Class) c);
                if (b != null)
                    return b;
            }
        }
        return null;
    }
    
    /**
     * Returns all corrected regions of the given state machine
     * 
     * @param ustatemachine
     *            Given state machine
     * @return all corrected regions of the given state machine
     */
    public List<Region> getAllCorrectedRegions(StateMachine ustatemachine)
    {
        List<Region> ret = new LinkedList<Region>();
        for (Classifier c : ustatemachine.getGenerals())
            if (c instanceof StateMachine)
                ret.addAll(getAllCorrectedRegions((StateMachine) c));
        for (Region r : ustatemachine.getRegions())
            ret.removeAll(r.getRedefinedElements());
        ret.addAll(ustatemachine.getRegions());
        return ret;
    }
    
    /**
     * Returns a name for the specified named elements. Generates one if null or
     * empty.
     * 
     * @param nes
     *            named elements
     * @return generated name
     */
    public String getName(NamedElement... nes)
    {
        StringBuilder ret = new StringBuilder();
        for (int i = 0; i < nes.length; i++)
        {
            if (i != 0)
                ret.append("_");
            NamedElement ne = nes[i];
            ret.append(getFilteredName(ne, null));
        }
        return ret.toString();
    }
    
    /**
     * Returns a filtered name for the given named element, fully qualified if
     * separator given
     * 
     * @param ne
     *            Given named element
     * @param qualifiedseparator
     *            Separator between qualifiers, or null if unqualified name
     * @return Filtered name of the given named element
     */
    public String getFilteredName(NamedElement ne, String qualifiedseparator)
    {
        StringBuilder ret = new StringBuilder();
        if (qualifiedseparator != null && ne.getOwner() instanceof NamedElement)
            ret.append(getFilteredName((NamedElement) ne.getOwner(), qualifiedseparator) + qualifiedseparator);
        String n = ne.getName();
        if (n != null)
        {
            // remove disallowed characters
            for (char c : n.toCharArray())
            {
                // trick...
                if (c == '\u2070')
                    c = '0';
                else if (c == '\u00B9')
                    c = '1';
                else if (c == '\u00B2')
                    c = '2';
                else if (c == '\u00B3')
                    c = '3';
                else if (c == '\u2074')
                    c = '4';
                else if (c == '\u2075')
                    c = '5';
                else if (c == '\u2076')
                    c = '6';
                else if (c == '\u2077')
                    c = '7';
                else if (c == '\u2078')
                    c = '8';
                else if (c == '\u2079')
                    c = '9';
                else if (c == '\u207A')
                    c = '+';
                else if (c == '\u207B')
                    c = '-';
                if (c == '_' || Character.isLetter(c) || (Character.isDigit(c) && ret.length() != 0))
                    ret.append(c);
            }
        }
        if (ret.length() == 0)
        {
            if (ne.getOwner() instanceof NamedElement)
                ret.append(getName((NamedElement) ne.getOwner()));
            ret.append("_");
            ret.append(ne.eClass().getName().toLowerCase());
            if (ne.getOwner() != null)
                ret.append(ne.getOwner().getOwnedElements().indexOf(ne));
            if (ne instanceof Connector)
                for (ConnectorEnd ce : ((Connector) ne).getEnds())
                {
                    if (ce.getDefiningEnd() != null)
                        ret.append("_" + ce.getDefiningEnd().getName());
                    else
                        ret.append("_" + ((Connector) ne).getEnds().indexOf(ce));
                }
        }
        return ret.toString();
    }
    
    /**
     * Returns a name for the specified named elements, using the qualified
     * name. Generates one if null or empty.
     * 
     * @param separator
     *            separator for qualified names
     * @param nes
     *            named elements
     * @return generated name
     */
    public String getQualifiedName(String separator, NamedElement... nes)
    {
        StringBuilder ret = new StringBuilder();
        for (int i = 0; i < nes.length; i++)
        {
            if (i != 0)
                ret.append("_");
            NamedElement ne = nes[i];
            String n = ne.getName();
            if (n != null)
            {
                // remove disallowed characters
                for (char c : n.toCharArray())
                {
                    if (c == '_' || Character.isLetter(c) || (Character.isDigit(c) && ret.length() != 0))
                        ret.append(c);
                }
            }
            if (ret.length() == 0)
            {
                ret.append(getName((NamedElement) ne.getOwner()));
                ret.append("_");
                ret.append(ne.eClass().getName().toLowerCase());
                if (ne.getNamespace() != null)
                    ret.append(ne.getNamespace().getOwnedMembers().indexOf(ne));
            }
        }
        return ret.toString();
    }
    
    /**
     * Returns the resource set corresponding to the utility class
     * 
     * @return resource set in which the models are loaded
     */
    public ResourceSet getResourceSet()
    {
        return rs;
    }
    
    /**
     * Creates a basic SysML model
     * 
     * @param rs
     *            resource set in which the resource is created
     * @param file
     *            location of the model
     * @param paths
     *            List of directories
     * @return resource representing the model
     */
    public static Resource createUMLModel(ResourceSet rs, URI file, List<URI> paths)
    {
        log.info("Creating " + file);
        Resource r = UMLResourceFactoryImpl.INSTANCE.createResource(file);
        rs.getResources().add(r);
        Model model = UMLFactory.eINSTANCE.createModel();
        model.setName("Data");
        r.getContents().add(model);
        
        return r;
    }
    
}
