package com.engisis.sysphs.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Collections;
import java.util.Hashtable;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;

/**
 * Contains the configuration properties extracted from the configuration file.
 * 
 * @author barbau
 *
 */
public class Configuration
{
    private static final Logger log = Logger.getLogger(Configuration.class);
    
    /**
     * Name of the property file
     */
    public static final String PROPERTYFILE = "properties.config";
    /**
     * Indicator of a resource mapping in the property file
     */
    public static final String RESOURCE_MAP = "resourcemap";
    /**
     * Indicator of a namespace mapping in the property file
     */
    public static final String NAMESPACE_MAP = "nsmap";
    /**
     * Indicator of an ID mapping in the property file
     */
    public static final String ID_MAP = "idmap";
    
    public static final String SIMULINK_UNIT = "simulinkunitmapping";
    
    /**
     * Loads the properties, and populate the location, namespace, and ID
     * mappings
     */
    public synchronized static void loadProperties()
    {
        if (locmaps != null && nsmaps_r != null && nsmaps_w != null && idmaps_r != null && idmaps_w != null)
            return;
        
        simulinkunits = new Hashtable<String, String>();
        simulinkunits.put("\u2126", "Ohm");
        simulinkunits.put("\u03A9", "Ohm");
        simulinkunits.put("\u03C9", "Ohm");
        locmaps = new Hashtable<URI, URI>();
        nsmaps_r = new Hashtable<String, String>();
        nsmaps_w = new Hashtable<String, Hashtable<String, String>>();
        idmaps_r = new Hashtable<String, String>();
        idmaps_w = new Hashtable<String, Hashtable<String, String>>();
        
        InputStream is = Configuration.class.getClassLoader().getResourceAsStream(PROPERTYFILE);
        if (is == null)
        {
            log.warn(PROPERTYFILE + " not found");
            return;
        }
        // TODO: fix stream closing
        BufferedReader br = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8));
        String line = null;
        try
        {
            int skipped = 0;
            while ((line = br.readLine()) != null)
            {
                String[] spl = line.split("\\t+");
                if (spl.length > 1)
                {
                    if (spl[0].equals(SIMULINK_UNIT))
                    {
                        if (spl.length < 3)
                        {
                            log.error("Line in the wrong format: " + line);
                            continue;
                        }
                        String from = spl[1];
                        String to = spl[2];
                        simulinkunits.put(from, to);
                        log.info("add simulink unit mapping from " + from + " to " + to);
                    }
                    if (spl[0].equals(NAMESPACE_MAP + "_r"))
                    {
                        if (spl.length < 3)
                        {
                            log.error("Line in the wrong format: " + line);
                            continue;
                        }
                        String from = spl[1];
                        String to = spl[2];
                        nsmaps_r.put(from, to);
                        log.info("add ns mapping from " + from + " to " + to);
                        continue;
                    }
                    if (spl[0].equals(NAMESPACE_MAP + "_w"))
                    {
                        if (spl.length < 4)
                        {
                            log.error("Line in the wrong format: " + line);
                            continue;
                        }
                        String target = spl[1];
                        String from = spl[2];
                        String to = spl[3];
                        
                        Hashtable<String, String> ht = nsmaps_w.get(target);
                        if (ht == null)
                        {
                            ht = new Hashtable<String, String>();
                            nsmaps_w.put(target, ht);
                        }
                        ht.put(from, to);
                        log.info("add ns mapping from " + from + " to " + to + " in target " + target);
                        continue;
                    }
                    else if (spl[0].equals(ID_MAP) || spl[0].equals(ID_MAP + "_r"))
                    {
                        if (spl.length < 3)
                        {
                            log.error("Line in the wrong format: " + line);
                            continue;
                        }
                        String from = spl[1];
                        String to = spl[2];
                        idmaps_r.put(from, to);
                        log.info("add id mapping from " + from + " to " + to);
                        continue;
                    }
                    else if (spl[0].equals(ID_MAP + "_w"))
                    {
                        if (spl.length < 4)
                        {
                            log.error("Line in the wrong format: " + line);
                            continue;
                        }
                        
                        String target = spl[1];
                        String from = spl[2];
                        String to = spl[3];
                        
                        Hashtable<String, String> ht = idmaps_w.get(target);
                        if (ht == null)
                        {
                            ht = new Hashtable<String, String>();
                            idmaps_w.put(target, ht);
                        }
                        ht.put(from, to);
                        
                        log.info("add id mapping from " + from + " to " + to + " in target " + target);
                        continue;
                    }
                    else if (spl[0].equals(RESOURCE_MAP))
                    {
                        if (spl.length < 3)
                        {
                            log.error("Line in the wrong format: " + line);
                            continue;
                        }
                        URI from = URI.createURI(spl[1]);
                        URI to = URI.createURI(transformResource(Configuration.class.getResource(spl[2]).getPath()));
                        locmaps.put(from, to);
                        log.info("add resource mapping from " + from);
                        continue;
                    }
                }
                if (!line.isEmpty())
                    skipped++;
            }
            if (skipped != 0)
                log.warn(skipped + " non-empty lines skipped");
        }
        catch (IOException e)
        {
            log.warn("Failed to read " + PROPERTYFILE);
        }
        finally
        {
            try
            {
                br.close();
            }
            catch (IOException e)
            {
            }
        }
    }
    
    /**
     * Adds a jar prefix if the URI has jar content
     * 
     * @param suri
     * @return
     */
    private static String transformResource(String suri)
    {
        if (suri.indexOf("!") == -1)
            return suri;
        return "jar:" + suri;
    }
    
    /**
     * a map of standard units to simulink units
     */
    private static Hashtable<String, String> simulinkunits = null;
    
    /**
     * a map with keys representing network URLs, and values representing local
     * URLs
     */
    private static Hashtable<URI, URI> locmaps = null;
    
    /**
     * a map of namespace mappings
     */
    private static Hashtable<String, String> nsmaps_r = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing namespace mappings
     */
    private static Hashtable<String, Hashtable<String, String>> nsmaps_w = null;
    
    /**
     * a map of ID mappings
     */
    private static Hashtable<String, String> idmaps_r = null;
    
    /**
     * a map with keys representing serialization targets, and values
     * representing ID mappings
     */
    private static Hashtable<String, Hashtable<String, String>> idmaps_w = null;
    
    public static Hashtable<String, String> getSimulinkUnitMapping()
    {
        if (simulinkunits == null)
            loadProperties();
        return simulinkunits;
    }
    
    /**
     * Returns the mappings of locations
     * 
     * @return mappings of locations (URLs)
     */
    public static Hashtable<URI, URI> getLocationMappings()
    {
        if (locmaps == null)
            loadProperties();
        return locmaps;
    }
    
    /**
     * Returns the namespace mappings to perform during the XMI reading
     * 
     * @return namespace mappings
     */
    public static Hashtable<String, String> getReaderNSMappings()
    {
        if (nsmaps_r == null)
            loadProperties();
        return nsmaps_r;
    }
    
    /**
     * Returns the namespace mappings to perform during the XMI writing
     * 
     * @return namespace mappings
     */
    public static Hashtable<String, String> getWriterNSMappings(String target)
    {
        if (nsmaps_w == null)
            loadProperties();
        return nsmaps_w.get(target);
    }
    
    /**
     * Returns the ID mappings to perform during the XMI reading
     * 
     * @return ID mappings
     */
    public static Hashtable<String, String> getReaderIDMappings()
    {
        if (idmaps_r == null)
            loadProperties();
        return idmaps_r;
    }
    
    /**
     * Returns the ID mappings to perform during the XMI writing
     * 
     * @return ID mappings
     */
    public static Hashtable<String, String> getWriterIDMappings(String target)
    {
        if (idmaps_w == null)
            loadProperties();
        return idmaps_w.get(target);
    }
    
    /**
     * Returns the list of acceptable targets for the XMI writing
     * 
     * @return list of targets
     */
    public static Collection<String> getTargets()
    {
        if (idmaps_w == null)
            loadProperties();
        return Collections.unmodifiableCollection(idmaps_w.keySet());
    }
    
}
