package com.engisis.sysphs.translation.simulink;

import com.engisis.sysphs.generation.simulink.MATLABBaseListener;
import com.engisis.sysphs.generation.simulink.MATLABParser;
import com.engisis.sysphs.generation.simulink.MATLABParser.Add_opContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Argument_listContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.ArrayContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.AssignmentContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.ConditionContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.During_actionContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Entry_actionContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Exit_actionContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr10Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr11Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr2Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr3Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr4Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr5Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr6Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr7Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr8Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.Expr9Context;
import com.engisis.sysphs.generation.simulink.MATLABParser.ExprContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Id_argContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Mul_opContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.ObjectContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Pow_opContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.ReferenceContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Relational_opContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.RowContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.Stmt_endContext;
import com.engisis.sysphs.generation.simulink.MATLABParser.UnaryContext;

import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;

import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeProperty;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.apache.log4j.Logger;

/**
 * Class to extract MATLAB expressions
 * 
 * @author barbau
 *
 */
public class MatlabExpressionExtractor extends MATLABBaseListener
{
    private static final Logger log = Logger.getLogger(MatlabExpressionExtractor.class);
    /**
     * associates some translated text to syntax tree nodes
     */
    private ParseTreeProperty<String> ptp = new ParseTreeProperty<String>();
    
    /**
     * Substitution table
     */
    private Hashtable<String, String> substitutions;
    
    /**
     * extracted condition
     */
    private String scondition;
    /**
     * extracted time
     */
    private String stime;
    /**
     * extracted entry code
     */
    private List<String> lentry;
    /**
     * extracted do code
     */
    private List<String> ldo;
    /**
     * extracted exit code
     */
    private List<String> lexit;
    
    /**
     * Initiates the parsing of a state label
     * 
     * @param substitutions
     *            substitution table
     */
    public void prepareNextStateLabelParsing(Hashtable<String, String> substitutions)
    {
        this.substitutions = substitutions;
    }
    
    /**
     * Initiates the parsing of a transition label
     * 
     * @param ht
     *            substitution table
     */
    public void prepareNextTransitionLabelParsing(Hashtable<String, String> substitutions)
    {
        this.substitutions = substitutions;
    }
    
    /**
     * Returns the extracted transition condition
     * 
     * @return the extracted transition condition
     */
    public String getTransitionCondition()
    {
        return scondition;
    }
    
    /**
     * Returns the extracted transition time
     * 
     * @return the extracted transition time
     */
    public String getTransitionTime()
    {
        return stime;
    }
    
    /**
     * Returns the extracted entry text
     * 
     * @return the extracted entry text
     */
    public List<String> getEntry()
    {
        return lentry;
    }
    
    /**
     * Returns the extracted do text
     * 
     * @return the extracted do text
     */
    public List<String> getDo()
    {
        return ldo;
    }
    
    /**
     * Returns the extracted exit text
     * 
     * @return the extracted exit text
     */
    public List<String> getExit()
    {
        return lexit;
    }
    
    public String getValue(ParseTree pt)
    {
        String v = ptp.get(pt);
        if (v == null)
            log.error("The value of " + pt.getClass() + " is null");
        return v;
    }
    
    public void setValue(ParseTree pt, String str)
    {
        ptp.put(pt, str);
    }
    
    @Override
    public void exitEntry_action(Entry_actionContext ctx)
    {
        if (lentry == null)
            lentry = new LinkedList<String>();
        for (AssignmentContext ac : ctx.assignment())
            lentry.add(getValue(ac) + ";");
    }
    
    @Override
    public void exitDuring_action(During_actionContext ctx)
    {
        if (ldo == null)
            ldo = new LinkedList<String>();
        for (AssignmentContext ac : ctx.assignment())
            ldo.add(getValue(ac) + ";");
    }
    
    @Override
    public void exitExit_action(Exit_actionContext ctx)
    {
        if (lexit == null)
            lexit = new LinkedList<String>();
        for (AssignmentContext ac : ctx.assignment())
            lexit.add(getValue(ac) + ";");
    }
    
    @Override
    public void exitCondition(ConditionContext ctx)
    {
        scondition = getValue(ctx.expr());
    }
    
    @Override
    public void exitAssignment(AssignmentContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitStmt_end(Stmt_endContext ctx)
    {
        setValue(ctx, ";");
    }
    
    @Override
    public void exitExpr(ExprContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr2(Expr2Context ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr3(Expr3Context ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr4(Expr4Context ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr5(Expr5Context ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitRelational_op(Relational_opContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr6(Expr6Context ctx)
    {
        if (ctx.expr7().size() > 1)
        {
            log.error("Colon operator not supported");
            setValue(ctx, "");
        }
        else
            setValue(ctx, getValue(ctx.expr7(0)));
    }
    
    @Override
    public void exitExpr7(Expr7Context ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.expr8().size(); i++)
        {
            String s = getValue(ctx.expr8(i));
            if (!s.equals("0"))
            {
                if (sb.length() != 0 && i > 0)
                    sb.append(getValue(ctx.add_op(i - 1)));
                sb.append(s);
            }
        }
        if (sb.length() == 0)
            sb.append("0");
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitAdd_op(Add_opContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr8(Expr8Context ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.expr9().size(); i++)
        {
            String s = getValue(ctx.expr9(i));
            if (!s.equals("1"))
            {
                if (sb.length() != 0 && i > 0)
                    sb.append(getValue(ctx.mul_op(i - 1)));
                sb.append(s);
            }
        }
        if (sb.length() == 0)
            sb.append("1");
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitMul_op(Mul_opContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr9(Expr9Context ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitUnary(UnaryContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr10(Expr10Context ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitPow_op(Pow_opContext ctx)
    {
        processStraight(ctx);
    }
    
    @Override
    public void exitExpr11(Expr11Context ctx)
    {
        if (ctx.AT() != null)
        {
            log.error("Operator @ not supported");
        }
        else
            processStraight(ctx);
    }
    
    @Override
    public void exitArray(ArrayContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        sb.append('{');
        for (RowContext rc : ctx.row())
            sb.append(getValue(rc));
        sb.append('}');
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitRow(RowContext ctx)
    {
        StringBuilder sb = new StringBuilder();
        sb.append('{');
        for (int i = 0; i < ctx.expr().size(); i++)
        {
            if (i != 0)
                sb.append(',');
            sb.append(getValue(ctx.expr(i)));
        }
        sb.append('}');
        setValue(ctx, sb.toString());
    }
    
    @Override
    public void exitReference(ReferenceContext ctx)
    {
        processStraight(ctx);
        if (substitutions != null)
        {
            String v = substitutions.get(getValue(ctx));
            if (v != null)
                setValue(ctx, v);
        }
    }
    
    @Override
    public void exitId_arg(Id_argContext ctx)
    {
        if (ctx.ID().getText().equals("after"))
        {
            List<ExprContext> exprs = ctx.argument_list().expr();
            if (exprs.size() != 2)
            {
                log.error("Unexpected number of arguments for function after");
                return;
            }
            if (!exprs.get(1).getText().equals("sec"))
            {
                log.error("Unsupported ");
            }
            stime = exprs.get(0).getText();
        }
        else
            processStraight(ctx);
    }
    
    @Override
    public void exitArgument_list(Argument_listContext ctx)
    {
        
        processStraight(ctx);
    }
    
    @Override
    public void exitObject(ObjectContext ctx)
    {
        // to translate {1, 'unit'}
        if (ctx.val().size() == 2)
        {
            String v0 = getValue(ctx.val(0).expr());
            String v1 = getValue(ctx.val(1).expr());
            if ("1".equals(v0) && v1.startsWith("'") && v1.endsWith("'"))
            {
                setValue(ctx, "1");
                return;
            }
        }
        
        log.error("Objects can't be translated");
    }
    
    @Override
    public void visitTerminal(TerminalNode node)
    {
        int t = node.getSymbol().getType();
        if (t == MATLABParser.EQ)
            setValue(node, ":=");
        else if (t == MATLABParser.NEG)
            setValue(node, " not ");
        else if (t == MATLABParser.LOG_AND)
            setValue(node, " and ");
        else if (t == MATLABParser.LOG_OR)
            setValue(node, " or ");
        else if (t == MATLABParser.CCT)
        {
            log.error("Operator ' not supported");
        }
        else if (t == MATLABParser.EL_CCT)
        {
            log.error("Operator .' not supported");
        }
        else if (t == MATLABParser.EL_EXP)
        {
            log.error("Operator .^ not supported");
        }
        else if (t == MATLABParser.EL_LEFTDIV)
        {
            log.error("Operator ./ not supported");
        }
        else if (t == MATLABParser.EL_RIGHTDIV)
        {
            log.error("Operator .\\ not supported");
        }
        else if (t == MATLABParser.EL_TIMES)
        {
            log.error("Operator .* not supported");
        }
        else
            setValue(node, node.getText());
    }
    
    protected void processStraight(ParseTree ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.getChildCount(); i++)
        {
            String v = getValue(ctx.getChild(i));
            if (v != null)
                sb.append(v);
        }
        setValue(ctx, sb.toString());
    }
    
    protected void processWithSpaces(ParseTree ctx)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < ctx.getChildCount(); i++)
        {
            if (i != 0)
                sb.append(" ");
            String v = getValue(ctx.getChild(i));
            if (v != null)
                sb.append(v);
        }
        setValue(ctx, sb.toString());
    }
    
}
