package com.engisis.sysphs.translation.modelica;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Stack;

import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeWalker;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.Artifact;
import org.eclipse.uml2.uml.ChangeEvent;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.InstanceSpecification;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.LiteralBoolean;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralReal;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.Signal;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.StructuredClassifier;
import org.eclipse.uml2.uml.TimeEvent;
import org.eclipse.uml2.uml.TimeExpression;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.Vertex;
import org.eclipse.uml2.uml.VisibilityKind;

import com.engisis.sysphs.deserialization.modelica.ModelicaDeserializer;
import com.engisis.sysphs.language.modelica.MAccessControl;
import com.engisis.sysphs.language.modelica.MAlgorithm;
import com.engisis.sysphs.language.modelica.MBooleanValue;
import com.engisis.sysphs.language.modelica.MClass;
import com.engisis.sysphs.language.modelica.MComponent;
import com.engisis.sysphs.language.modelica.MConnect;
import com.engisis.sysphs.language.modelica.MConnector;
import com.engisis.sysphs.language.modelica.MDataFlow;
import com.engisis.sysphs.language.modelica.MDataValue;
import com.engisis.sysphs.language.modelica.MDirection;
import com.engisis.sysphs.language.modelica.MElement;
import com.engisis.sysphs.language.modelica.MEquation;
import com.engisis.sysphs.language.modelica.MExpressionValue;
import com.engisis.sysphs.language.modelica.MExtension;
import com.engisis.sysphs.language.modelica.MIntegerValue;
import com.engisis.sysphs.language.modelica.MModification;
import com.engisis.sysphs.language.modelica.MNamedElement;
import com.engisis.sysphs.language.modelica.MRealValue;
import com.engisis.sysphs.language.modelica.MStringValue;
import com.engisis.sysphs.language.modelica.MType;
import com.engisis.sysphs.language.modelica.MVariability;
import com.engisis.sysphs.util.SimulationToSysMLTranslator;
import com.engisis.sysphs.util.SysPhSUtil;
import com.engisis.sysphs.util.SysPhSUtil.LibraryComponent;
import com.engisis.sysphs.util.SysPhSUtil.LibraryParameter;
import com.engisis.sysphs.util.SysPhSUtil.LibraryPort;
import com.engisis.sysphs.util.Util;
import com.engisis.xmiutil.EMFUtil;
import com.engisis.xmiutil.UMLModelErrorException;

/**
 * Class in charge of the translation from Modelica to SysML
 * 
 * @author barbau
 *
 */
public class ModelicaToSysMLTranslator extends SimulationToSysMLTranslator
{
    private static final Logger log = Logger.getLogger(ModelicaToSysMLTranslator.class);
    /**
     * SysML utility
     */
    private SysPhSUtil sysphsutil;
    
    /**
     * reference table
     */
    private Hashtable<ReferenceKey, NamedElement> refs = new Hashtable<ReferenceKey, NamedElement>();
    /**
     * first processing stack
     */
    private Stack<ReferenceKey> toProcess = new Stack<ReferenceKey>();
    /**
     * second processing stack
     */
    private Stack<ReferenceKey> toProcess2 = new Stack<ReferenceKey>();
    
    /**
     * Map with Modelica classes (connectors) as keys, and UML classes (port
     * types) as values
     */
    private Hashtable<MClass, Class> porttypes = new Hashtable<MClass, Class>();
    
    /**
     * Map with Modelica objects as keys, and syntax tree nodes as values
     */
    private Hashtable<MElement, ParseTree> expressions;
    
    /**
     * Root Simscape package
     */
    private Package root;
    
    /**
     * Key used to reference the source of the mapping
     * 
     * @author barbau
     *
     */
    protected static class ReferenceKey
    {
        private MNamedElement[] key;
        
        public ReferenceKey(MNamedElement[] key)
        {
            this.key = key.clone();
        }
        
        public MNamedElement[] getKey()
        {
            return key.clone();
        }
        
        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof ReferenceKey)
            {
                ReferenceKey rk = (ReferenceKey) obj;
                if (rk.key.length != key.length)
                    return false;
                for (int i = 0; i < key.length; i++)
                    if (rk.key[i] != key[i])
                        return false;
                return true;
            }
            return false;
        }
        
        @Override
        public int hashCode()
        {
            return toString().hashCode();
        }
        
        @Override
        public String toString()
        {
            StringBuilder sb = new StringBuilder();
            for (MNamedElement melem : key)
                sb.append(melem.getName() + " ");
            return sb.toString();
        }
    }
    
    /**
     * Generates a key from the given elements
     * 
     * @param elements
     *            elements for which the key is needed
     * @return a key
     */
    private static ReferenceKey getKey(MNamedElement... elements)
    {
        return new ReferenceKey(elements);
    }
    
    /**
     * Constructs the translator
     * 
     * @param options
     *            a set of options
     */
    public ModelicaToSysMLTranslator(Set<Object> options)
    {
        loadOptions(options);
    }
    
    @Override
    public void loadOptions(Set<Object> options)
    {
        super.loadOptions(options);
    }
    
    @Override
    public Set<java.lang.Class<?>> getOptions()
    {
        HashSet<java.lang.Class<?>> hs = new HashSet<java.lang.Class<?>>();
        return hs;
    }
    
    /**
     * Different types of objects
     * 
     * @author barbau
     *
     */
    private enum TranslatedType
    {
        BLOCK, SIMBLOCK, DATATYPE, COMPONENT, STATEMACHINE, INITIALSTATE, STATE, TRANSITION, NONE;
    }
    
    /**
     * Determines the type of a given Modelica element
     * 
     * @param mnamedelement
     *            element for which the type is to be determined
     * @return the type of the element
     */
    private static TranslatedType getTranslatedType(MNamedElement mnamedelement)
    {
        if (mnamedelement instanceof MClass)
        {
            MClass mc = ((MClass) mnamedelement);
            // regardless of type
            for (MExtension mext : mc.getExtensions())
                if (getTranslatedType(mext.getExtendedClass()) == TranslatedType.DATATYPE)
                    return TranslatedType.DATATYPE;
            if (mnamedelement instanceof MType)
                return TranslatedType.DATATYPE;
            if (mnamedelement instanceof MConnector)
                return TranslatedType.SIMBLOCK;
            for (MComponent mcomponent : mc.getComponents())
            {
                TranslatedType tt = getTranslatedType(mcomponent);
                if (tt == TranslatedType.INITIALSTATE || tt == TranslatedType.STATE || tt == TranslatedType.TRANSITION)
                    return TranslatedType.STATEMACHINE;
            }
            return TranslatedType.BLOCK;
        }
        else if (mnamedelement instanceof MComponent)
        {
            if (((MComponent) mnamedelement).getType() == ModelicaUtil.getInitialStep())
                return TranslatedType.INITIALSTATE;
            if (((MComponent) mnamedelement).getType() == ModelicaUtil.getStep())
                return TranslatedType.STATE;
            if (((MComponent) mnamedelement).getType() == ModelicaUtil.getTransition())
                return TranslatedType.TRANSITION;
            return TranslatedType.COMPONENT;
            
        }
        return TranslatedType.NONE;
    }
    
    @Override
    public void execute(String model, ResourceSet rs, File outputdirectory, String target, String[] paths)
            throws IOException, UMLModelErrorException
    {
        if (model == null)
            throw new IllegalArgumentException("The model can't be null");
        if (rs == null)
            throw new IllegalArgumentException("The ResourceSet can't be null");
        File file = new File(model);
        if (!file.isFile() || !file.canRead())
            throw new IllegalArgumentException("The file must be readable");
        
        List<URI> lpath = new ArrayList<URI>(paths != null ? paths.length + 1 : 1);
        
        if (paths != null)
            for (String path : paths)
                lpath.add(URI.createFileURI(new File(path).getAbsolutePath()));
        lpath.add(URI.createFileURI(file.getParentFile().getAbsolutePath()));
        
        URI uri = null;
        String fn = null;
        if (outputdirectory == null)
        {
            uri = URI.createFileURI(model).trimFileExtension();
            fn = uri.lastSegment() + getFileNameSuffix();
            uri = uri.trimSegments(1).appendSegment(fn).appendFileExtension("xmi");
        }
        else
        {
            uri = URI.createFileURI(outputdirectory.getAbsolutePath());
            fn = URI.createFileURI(model).trimFileExtension().lastSegment() + getFileNameSuffix();
            uri = uri.appendSegment(fn).appendFileExtension("xmi");
        }
        
        Resource r = SysPhSUtil.createSysPhSModel(rs, uri, lpath);
        
        sysphsutil = new SysPhSUtil(r.getResourceSet());
        ModelicaDeserializer md = new ModelicaDeserializer(file);
        List<MClass> list = md.getClasses();
        expressions = md.getExpressions();
        
        root = (Package) r.getContents().stream().filter(c -> c instanceof Package).findFirst()
                .orElseThrow(() -> new NullPointerException("No root package found in SysPhS model"));
        
        for (MClass mclass : list)
        {
            createClass(mclass);
        }
        while (toProcess.size() != 0)
        {
            ReferenceKey key = toProcess.pop();
            MNamedElement nme = key.getKey()[0];
            TranslatedType tt = getTranslatedType(nme);
            if (nme instanceof MClass)
            {
                if (tt == TranslatedType.STATEMACHINE)
                {
                    processStateMachine((MClass) nme);
                    toProcess2.push(key);
                }
                else
                {
                    processClass((MClass) nme);
                    toProcess2.push(key);
                }
            }
            else if (nme instanceof MComponent)
            {
                if (tt == TranslatedType.INITIALSTATE)
                    processInitialState((MComponent) nme);
                else if (tt == TranslatedType.STATE)
                    processState((MComponent) nme);
                else if (tt == TranslatedType.TRANSITION)
                    processTransition((MComponent) nme);
                else
                {
                    processProperty((MComponent) nme);
                    toProcess2.push(key);
                }
            }
        }
        while (toProcess2.size() != 0)
        {
            ReferenceKey key = toProcess2.pop();
            MNamedElement nme = key.getKey()[0];
            TranslatedType tt = getTranslatedType(nme);
            if (nme instanceof MClass)
            {
                if (tt == TranslatedType.STATEMACHINE)
                    processStateMachine2((MClass) nme);
                else
                    processClass2((MClass) nme);
            }
            else if (nme instanceof MComponent)
                processProperty2((MComponent) nme);
        }
        for (MClass mc : list)
        {
            if (mc instanceof MType)
                continue;
            // generated model
            if (mc.getComponents().size() == 1 && mc.getEquations().size() == 0)
            {
                MComponent first = mc.getComponents().get(0);
                if (first.getType() == null)
                    log.error("Type of " + first.getName() + " is null");
                else
                {
                    NamedElement ne = refs.get(getKey(first.getType()));
                    if (ne instanceof Class)
                    {
                        outputrootname = ((Class) ne).getQualifiedName();
                        log.info("Setting root element to " + outputrootname);
                        break;
                    }
                }
            }
            NamedElement ne = refs.get(getKey(mc));
            if (mc instanceof Class)
            {
                outputrootname = ((Class) ne).getQualifiedName();
                log.info("Setting root element to " + outputrootname);
                break;
            }
        }
        
        log.info("Saving " + r.getURI());
        outputfilename = r.getURI().toFileString();
        EMFUtil.saveResource(r, target);
    }
    
    private Classifier createClass(MClass mclass)
    {
        if (mclass == null)
            throw new IllegalArgumentException("The class should not be null");
        ReferenceKey key = getKey(mclass);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Classifier)
                return (Classifier) uelement;
            throw new IllegalStateException("The element is not a class: " + mclass.getName());
        }
        Classifier uclass = null;
        
        switch (getTranslatedType(mclass))
        {
        case BLOCK:
            log.info("Creating block " + mclass.getName());
            uclass = UMLFactory.eINSTANCE.createClass();
            root.getPackagedElements().add(uclass);
            uclass.applyStereotype(sysphsutil.getBlock());
            break;
        case SIMBLOCK:
            log.info("Creating port type " + mclass.getName());
            uclass = UMLFactory.eINSTANCE.createClass();
            root.getPackagedElements().add(uclass);
            uclass.applyStereotype(sysphsutil.getBlock());
            for (MComponent mc : mclass.getAllComponents())
            {
                if (mc.getDataFlow() == MDataFlow.FLOW)
                {
                    Class scs = sysphsutil.getConservedSubstance();
                    if (scs == null)
                        log.error("Couldn't find conserved substance block");
                    else
                    {
                        Generalization ugeneralization = UMLFactory.eINSTANCE.createGeneralization();
                        ugeneralization.setGeneral(scs);
                        ugeneralization.setSpecific(uclass);
                        uclass.getGeneralizations().add(ugeneralization);
                    }
                    break;
                }
            }
            break;
        case DATATYPE:
            log.info("Creating datatype " + mclass.getName());
            DataType udatatype = null;
            if (mclass == ModelicaUtil.getReal())
                udatatype = sysphsutil.getSysMLDouble();
            else if (mclass == ModelicaUtil.getInteger())
                udatatype = sysphsutil.getSysMLInteger();
            else if (mclass == ModelicaUtil.getBoolean())
                udatatype = sysphsutil.getSysMLBoolean();
            else if (mclass == ModelicaUtil.getString())
                udatatype = sysphsutil.getSysMLString();
            else
            {
                uclass = UMLFactory.eINSTANCE.createDataType();
                root.getPackagedElements().add(uclass);
            }
            if (udatatype != null)
            {
                refs.put(key, udatatype);
                return udatatype;
            }
            break;
        default:
            break;
        }
        if (uclass == null)
        {
            log.error("Couldn't translate class " + mclass.getName());
            return null;
        }
        refs.put(key, uclass);
        toProcess.push(key);
        return uclass;
    }
    
    private StateMachine createStateMachine(MClass mclass)
    {
        if (mclass == null)
            throw new IllegalArgumentException("The class should not be null");
        ReferenceKey key = getKey(mclass);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof StateMachine)
                return (StateMachine) uelement;
            throw new IllegalStateException("The element is not a state machine: " + mclass.getName());
        }
        log.info("Creating state machine " + mclass.getName());
        StateMachine ustatemachine = UMLFactory.eINSTANCE.createStateMachine();
        refs.put(key, ustatemachine);
        toProcess.add(key);
        return ustatemachine;
    }
    
    private Property createProperty(MComponent mcomponent)
    {
        if (mcomponent == null)
            throw new IllegalArgumentException("The component should not be null");
        if (getTranslatedType(mcomponent) != TranslatedType.COMPONENT)
            throw new IllegalStateException("The method can only be called on a component");
        ReferenceKey key = getKey(mcomponent);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Property)
                return (Property) uelement;
            throw new IllegalStateException("The element is not a component: " + mcomponent.getName());
        }
        Property uproperty = null;
        if (mcomponent.getDirection() == MDirection.INPUT || mcomponent.getDirection() == MDirection.OUTPUT
                || mcomponent.getType() instanceof MConnector)
        {
            log.info("Creating port " + mcomponent.getName());
            uproperty = UMLFactory.eINSTANCE.createPort();
            if (mcomponent.getDirection() == MDirection.OUTPUT)
                ((Port) uproperty).setIsConjugated(true);
        }
        else
        {
            log.info("Creating property " + mcomponent.getName() + " from " + mcomponent.getOwningClass().getName());
            uproperty = UMLFactory.eINSTANCE.createProperty();
            SysPhSUtil.setName(uproperty, mcomponent.getOwningClass().getName() + "__" + mcomponent.getName());
        }
        refs.put(key, uproperty);
        toProcess.push(key);
        return uproperty;
    }
    
    private Classifier processClass(MClass mclass)
    {
        log.info("Processing classifier " + mclass.getName());
        Classifier uclassifier = createClass(mclass);
        SysPhSUtil.setName(uclassifier, mclass.getName());
        List<Property> properties = null;
        if (uclassifier instanceof StructuredClassifier)
            properties = ((StructuredClassifier) uclassifier).getOwnedAttributes();
        else if (uclassifier instanceof DataType)
            properties = ((DataType) uclassifier).getOwnedAttributes();
        else if (uclassifier instanceof Interface)
            properties = ((Interface) uclassifier).getOwnedAttributes();
        else if (uclassifier instanceof Signal)
            properties = ((Signal) uclassifier).getOwnedAttributes();
        else if (uclassifier instanceof Artifact)
            properties = ((Artifact) uclassifier).getOwnedAttributes();
        else
            log.info("Attributes of " + mclass.getName() + " skipped");
        
        if (properties != null)
            for (MComponent mcomponent : mclass.getComponents())
            {
                if (getTranslatedType(mcomponent.getType()) == TranslatedType.STATEMACHINE)
                {
                    if (uclassifier instanceof Class)
                    {
                        StateMachine ustatemachine = createStateMachine(mcomponent.getType());
                        ((Class) uclassifier).setClassifierBehavior(ustatemachine);
                    }
                }
                else
                {
                    Property uproperty = createProperty(mcomponent);
                    properties.add(uproperty);
                }
            }
        
        for (MExtension extension : mclass.getExtensions())
        {
            Classifier general = createClass(extension.getExtendedClass());
            if (general != null)
            {
                log.info("Adding generalization between " + uclassifier.getName() + " and " + general.getName());
                Generalization generaliz = UMLFactory.eINSTANCE.createGeneralization();
                generaliz.setGeneral(general);
                uclassifier.getGeneralizations().add(generaliz);
                
                for (MModification mmodification : extension.getModifications())
                {
                    if (mmodification.getComponentPath().size() == 1
                            && mmodification.getComponentPath().get(0).getName().equals("unit")
                            && mmodification.getAssignedValue() instanceof MStringValue)
                    {
                        log.info("Unit detected " + mmodification.getAssignedValue());
                        uclassifier.applyStereotype(sysphsutil.getValueType());
                        
                        InstanceSpecification unit = UMLFactory.eINSTANCE.createInstanceSpecification();
                        uclassifier.getNearestPackage().getPackagedElements().add(unit);
                        unit.getClassifiers().add(sysphsutil.getUnit());
                        
                        Slot symbol = UMLFactory.eINSTANCE.createSlot();
                        unit.getSlots().add(symbol);
                        symbol.setDefiningFeature(sysphsutil.getUnit().getAttribute("symbol", null));
                        LiteralString ls = UMLFactory.eINSTANCE.createLiteralString();
                        ls.setValue(((MStringValue) mmodification.getAssignedValue()).getValue());
                        symbol.getValues().add(ls);
                        
                        uclassifier.setValue(sysphsutil.getValueType(), "unit", unit);
                    }
                    else
                        log.warn("Extend modification not used : " + mmodification.getComponentPath() + "---"
                                + mmodification.getAssignedValue());
                }
            }
            else
                log.error("The extension of " + mclass.getName() + " can't be translated");
        }
        return uclassifier;
    }
    
    private Classifier processClass2(MClass mclass)
    {
        log.info("Processing classifier " + mclass.getName() + " a second time");
        Classifier uclassifier = createClass(mclass);
        if (mclass.getEquations().size() > 0 && uclassifier instanceof Class)
        {
            Class constraint = UMLFactory.eINSTANCE.createClass();
            root.getPackagedElements().add(constraint);
            SysPhSUtil.setName(constraint, mclass.getName() + "Constraint");
            constraint.applyStereotype(sysphsutil.getConstraintBlock());
            StringBuilder sbproperty = new StringBuilder();
            for (char c : constraint.getName().toCharArray())
                if (Character.isUpperCase(c))
                    sbproperty.append(c);
            Property ucproperty = UMLFactory.eINSTANCE.createProperty();
            Class uclass = (Class) uclassifier;
            uclass.getOwnedAttributes().add(ucproperty);
            SysPhSUtil.setName(ucproperty, sbproperty.toString());
            ucproperty.setType(constraint);
            
            Hashtable<List<String>, String> parameters = new Hashtable<List<String>, String>();
            
            // parse equations, identify involved variables
            for (MEquation equation : mclass.getEquations())
            {
                if (equation instanceof MConnect)
                {
                    MConnect mconnect = (MConnect) equation;
                    
                    log.info("Adding connector");
                    Connector uconnector = UMLFactory.eINSTANCE.createConnector();
                    uclass.getOwnedConnectors().add(uconnector);
                    
                    ConnectorEnd ce0 = UMLFactory.eINSTANCE.createConnectorEnd();
                    uconnector.getEnds().add(ce0);
                    List<Property> lp0 = new LinkedList<Property>();
                    for (MComponent mc : mconnect.getRef1())
                    {
                        Property up = createProperty(mc);
                        if (up != null)
                            lp0.add(up);
                    }
                    sysphsutil.updatePropertyPath(ce0, lp0);
                    
                    ConnectorEnd ce1 = UMLFactory.eINSTANCE.createConnectorEnd();
                    uconnector.getEnds().add(ce1);
                    List<Property> lp1 = new LinkedList<Property>();
                    for (MComponent mc : mconnect.getRef2())
                    {
                        Property up = createProperty(mc);
                        if (up != null)
                            lp1.add(up);
                    }
                    sysphsutil.updatePropertyPath(ce1, lp1);
                }
                else
                {
                    log.info("Analyzing constraint " + equation.getExpression());
                    ParseTree pt = expressions.get(equation);
                    if (pt != null)
                    {
                        
                        ParseTreeWalker ptw = new ParseTreeWalker();
                        ModelicaExpressionExtractor mee = new ModelicaExpressionExtractor();
                        mee.prepareForConstraint();
                        ptw.walk(mee, pt);
                        
                        List<String> lb = mee.getLBinding();
                        List<String> rb = mee.getRBinding();
                        
                        if (lb != null && rb != null)
                        {
                            MComponent mcl = mclass.getComponentByName(lb.get(0));
                            if (mcl == null)
                            {
                                log.warn("Can't find component " + lb.get(0) + " in " + mclass.getName());
                                continue;
                            }
                            MComponent mcr = mclass.getComponentByName(rb.get(0));
                            if (mcr == null)
                            {
                                log.warn("Can't find component " + rb.get(0) + " in " + mclass.getName());
                                continue;
                            }
                            TranslatedType ttl = getTranslatedType(mcl.getType());
                            TranslatedType ttr = getTranslatedType(mcr.getType());
                            if (ttl != TranslatedType.STATEMACHINE && ttr != TranslatedType.STATEMACHINE)
                            {
                                Connector ubinding = UMLFactory.eINSTANCE.createConnector();
                                uclass.getOwnedConnectors().add(ubinding);
                                ubinding.applyStereotype(sysphsutil.getBindingConnector());
                                
                                List<Property> lp0 = new LinkedList<Property>();
                                lp0.add(createProperty(mcl));
                                for (int i = 1; i < lb.size(); i++)
                                {
                                    MClass type = mcl.getType();
                                    if (type == null)
                                    {
                                        log.error("The type of " + mcl.getName() + " is null, in binding "
                                                + pt.getText());
                                        break;
                                    }
                                    mcl = type.getComponentByName(lb.get(i));
                                    if (mcl == null)
                                    {
                                        log.error("No component named " + lb.get(i) + " in " + type.getName()
                                                + ", in binding " + pt.getText());
                                        break;
                                    }
                                    lp0.add(createProperty(mcl));
                                }
                                ConnectorEnd ce0 = UMLFactory.eINSTANCE.createConnectorEnd();
                                ubinding.getEnds().add(ce0);
                                sysphsutil.updatePropertyPath(ce0, lp0);
                                
                                List<Property> lp1 = new LinkedList<Property>();
                                lp1.add(createProperty(mcr));
                                for (int i = 1; i < rb.size(); i++)
                                {
                                    MClass type = mcr.getType();
                                    if (type == null)
                                    {
                                        log.error("The type of " + mcr.getName() + " is null, in binding "
                                                + pt.getText());
                                        break;
                                    }
                                    mcr = type.getComponentByName(rb.get(i));
                                    if (mcr == null)
                                    {
                                        log.error("No component named " + rb.get(i) + " in " + type.getName()
                                                + ", in binding " + pt.getText());
                                        break;
                                    }
                                    lp1.add(createProperty(mcr));
                                }
                                ConnectorEnd ce1 = UMLFactory.eINSTANCE.createConnectorEnd();
                                ubinding.getEnds().add(ce1);
                                sysphsutil.updatePropertyPath(ce1, lp1);
                                log.info("Created binding for " + pt.getText());
                            }
                            else if (ttl == TranslatedType.STATEMACHINE)
                            {
                                log.warn("Skipping binding " + pt.getText());
                                refs.put(getKey(mcl.getComponentByName(lb.get(1))), refs.get(getKey(mcr)));
                            }
                            else if (ttr == TranslatedType.STATEMACHINE)
                            {
                                log.warn("Skipping binding " + pt.getText());
                                refs.put(getKey(mcr.getComponentByName(rb.get(1))), refs.get(getKey(mcl)));
                            }
                        }
                        else
                        {
                            // create constraint
                            log.info("Creating constraint, adding " + mee.getParameters().size() + " parameters");
                            Constraint c = UMLFactory.eINSTANCE.createConstraint();
                            constraint.getOwnedRules().add(c);
                            OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
                            c.setSpecification(oe);
                            oe.getBodies().add(mee.getValue(pt));
                            oe.getLanguages().add(SysPhSUtil.SYSPHS_LANGUAGE);
                            parameters.putAll(mee.getParameters());
                        }
                    }
                    else
                        log.error("Unable to get parse tree for the equation");
                    
                }
            }
            
            loop: for (Entry<List<String>, String> entry : parameters.entrySet())
            {
                List<String> key = entry.getKey();
                String val = entry.getValue();
                
                log.info("Adding constraint parameter " + val);
                
                // parameter
                Property uparameter = UMLFactory.eINSTANCE.createProperty();
                constraint.getOwnedAttributes().add(uparameter);
                SysPhSUtil.setName(uparameter, val);
                uparameter.setType(sysphsutil.getSysMLDouble());
                
                // binding
                Connector uconnector = UMLFactory.eINSTANCE.createConnector();
                uclass.getOwnedConnectors().add(uconnector);
                
                uconnector.applyStereotype(sysphsutil.getBindingConnector());
                
                // parameter
                ConnectorEnd ce0 = UMLFactory.eINSTANCE.createConnectorEnd();
                uconnector.getEnds().add(ce0);
                List<Property> lp0 = new ArrayList<Property>(2);
                lp0.add(ucproperty);
                lp0.add(uparameter);
                sysphsutil.updatePropertyPath(ce0, lp0);
                
                // bound property
                
                ConnectorEnd ce1 = UMLFactory.eINSTANCE.createConnectorEnd();
                uconnector.getEnds().add(ce1);
                List<Property> lp1 = new ArrayList<Property>();
                MClass curclas = mclass;
                MComponent curcomp = null;
                for (int i = 0; i < key.size(); i++)
                {
                    curcomp = curclas.getComponentByName(key.get(i));
                    if (curcomp == null)
                    {
                        log.error("No component " + key.get(i) + " in " + curclas.getName());
                        continue loop;
                    }
                    Property uprop = createProperty(curcomp);
                    lp1.add(uprop);
                    curclas = curcomp.getType();
                }
                sysphsutil.updatePropertyPath(ce1, lp1);
                
            }
        }
        
        return uclassifier;
    }
    
    private StateMachine processStateMachine(MClass mclass)
    {
        log.info("Processing state machine " + mclass.getName());
        StateMachine ustatemachine = createStateMachine(mclass);
        Region uregion = UMLFactory.eINSTANCE.createRegion();
        for (MComponent mcomponent : mclass.getComponents())
        {
            TranslatedType tt = getTranslatedType(mcomponent);
            if (tt == TranslatedType.INITIALSTATE)
            {
                Pseudostate ups = createInitialState(mcomponent);
                uregion.getSubvertices().add(ups);
            }
            else if (tt == TranslatedType.STATE)
            {
                State us = createState(mcomponent);
                uregion.getSubvertices().add(us);
            }
            else if (tt == TranslatedType.TRANSITION)
            {
                Transition ut = createTransition(mcomponent);
                uregion.getTransitions().add(ut);
            }
        }
        ustatemachine.getRegions().add(uregion);
        for (MEquation equation : mclass.getEquations())
        {
            if (equation instanceof MConnect)
            {
                MConnect mconnect = (MConnect) equation;
                // transitions
                NamedElement ne1 = refs.get(getKey(mconnect.getRef1().get(0)));
                NamedElement ne2 = refs.get(getKey(mconnect.getRef2().get(0)));
                
                if (ne1 instanceof Vertex && ne2 instanceof Transition)
                {
                    String n1 = mconnect.getRef1().get(1).getName();
                    String n2 = mconnect.getRef2().get(1).getName();
                    if (n1.equals("outPort") && n2.equals("inPort"))
                    {
                        log.info("Setting source of " + ne2.getName() + " to " + ne1.getName());
                        ((Transition) ne2).setSource((Vertex) ne1);
                    }
                    else if (n1.equals("inPort") && n2.equals("outPort"))
                    {
                        log.info("Setting target of " + ne2.getName() + " to " + ne1.getName());
                        ((Transition) ne2).setTarget((Vertex) ne1);
                    }
                    continue;
                }
                if (ne1 instanceof Transition && ne2 instanceof Vertex)
                {
                    String n1 = mconnect.getRef1().get(1).getName();
                    String n2 = mconnect.getRef2().get(1).getName();
                    if (n1.equals("outPort") && n2.equals("inPort"))
                    {
                        log.info("Setting target of " + ne1.getName() + " to " + ne2.getName());
                        ((Transition) ne1).setTarget((Vertex) ne2);
                    }
                    else if (n1.equals("inPort") && n2.equals("outPort"))
                    {
                        log.info("Setting source of " + ne1.getName() + " to " + ne2.getName());
                        ((Transition) ne1).setSource((Vertex) ne2);
                    }
                    continue;
                }
            }
        }
        return ustatemachine;
    }
    
    private StateMachine processStateMachine2(MClass mclass)
    {
        log.info("Processing state machine " + mclass.getName() + " a second time");
        StateMachine ustatemachine = createStateMachine(mclass);
        
        // generate substitutions
        Hashtable<String, String> substitutions = new Hashtable<String, String>();
        for (Property uproperty : sysphsutil.getAllAttributes(ustatemachine.getContext()))
        {
            Type type = uproperty.getType();
            if (porttypes.values().contains(type))
            {
                // IS a class with sim property
                Property sp = sysphsutil.getAllPhSProperties((Class) type).get(0);
                substitutions.put(uproperty.getName(), uproperty.getName() + "." + sp.getName());
            }
        }
        for (MComponent mcomponent : mclass.getComponents())
        {
            NamedElement mne = refs.get(getKey(mcomponent));
            if (mne instanceof Transition)
            {
                for (MModification mmodification : mcomponent.getModifications())
                {
                    String modname = mmodification.getComponentPath().get(0).getName();
                    if (modname.equals("condition"))
                    {
                        ParseTree pt = expressions.get(mmodification);
                        if (pt != null)
                        {
                            ParseTreeWalker ptw = new ParseTreeWalker();
                            ModelicaExpressionExtractor mee = new ModelicaExpressionExtractor();
                            mee.setSubstitutions(substitutions);
                            ptw.walk(mee, pt);
                            Object v = mee.getValue(pt);
                            if (v != null)
                            {
                                String sval = v.toString();
                                // tricky...
                                String te0 = "time>";
                                String te1 = "time>=";
                                Trigger t = null;
                                if (sval.startsWith(te0))
                                {
                                    sval = sval.substring(te0.length());
                                    t = UMLFactory.eINSTANCE.createTrigger();
                                    TimeEvent te = UMLFactory.eINSTANCE.createTimeEvent();
                                    mne.getNearestPackage().getPackagedElements().add(te);
                                    t.setEvent(te);
                                    TimeExpression texp = UMLFactory.eINSTANCE.createTimeExpression();
                                    te.setWhen(texp);
                                    OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
                                    texp.setExpr(oe);
                                    oe.getBodies().add(sval);
                                    oe.getLanguages().add(SysPhSUtil.SYSPHS_LANGUAGE);
                                    te.setIsRelative(false);
                                }
                                else if (sval.startsWith(te1))
                                {
                                    sval = sval.substring(te1.length());
                                    t = UMLFactory.eINSTANCE.createTrigger();
                                    TimeEvent te = UMLFactory.eINSTANCE.createTimeEvent();
                                    mne.getNearestPackage().getPackagedElements().add(te);
                                    t.setEvent(te);
                                    TimeExpression texp = UMLFactory.eINSTANCE.createTimeExpression();
                                    te.setWhen(texp);
                                    OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
                                    texp.setExpr(oe);
                                    oe.getBodies().add(sval);
                                    oe.getLanguages().add(SysPhSUtil.SYSPHS_LANGUAGE);
                                    te.setIsRelative(false);
                                }
                                else if (!"true".equals(sval))
                                {
                                    t = UMLFactory.eINSTANCE.createTrigger();
                                    ChangeEvent ce = UMLFactory.eINSTANCE.createChangeEvent();
                                    mne.getNearestPackage().getPackagedElements().add(ce);
                                    t.setEvent(ce);
                                    OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
                                    ce.setChangeExpression(oe);
                                    oe.getBodies().add(sval);
                                    oe.getLanguages().add(SysPhSUtil.SYSPHS_LANGUAGE);
                                }
                                if (t != null)
                                    ((Transition) mne).getTriggers().add(t);
                            }
                        }
                    }
                    else if (modname.equals("waitTime"))
                    {
                        Trigger t = UMLFactory.eINSTANCE.createTrigger();
                        TimeEvent te = UMLFactory.eINSTANCE.createTimeEvent();
                        mne.getNearestPackage().getPackagedElements().add(te);
                        t.setEvent(te);
                        TimeExpression texp = UMLFactory.eINSTANCE.createTimeExpression();
                        te.setWhen(texp);
                        texp.setExpr(getValueSpecification(mmodification.getAssignedValue()));
                        te.setIsRelative(true);
                        ((Transition) mne).getTriggers().add(t);
                    }
                }
            }
        }
        for (MAlgorithm malgorithm : mclass.getAlgorithms())
        {
            ParseTree pt = expressions.get(malgorithm);
            if (pt != null)
            {
                ParseTreeWalker ptw = new ParseTreeWalker();
                ModelicaExpressionExtractor mee = new ModelicaExpressionExtractor();
                Hashtable<String, String> stateactions = new Hashtable<String, String>();
                mee.setStateActions(stateactions);
                mee.setSubstitutions(substitutions);
                ptw.walk(mee, pt);
                for (Entry<String, String> ent : stateactions.entrySet())
                {
                    State st = (State) refs.get(getKey(mclass.getComponentByName(ent.getKey())));
                    if (st != null)
                    {
                        OpaqueBehavior ob = UMLFactory.eINSTANCE.createOpaqueBehavior();
                        ob.getBodies().add(ent.getValue());
                        ob.getLanguages().add(SysPhSUtil.SYSPHS_LANGUAGE);
                        st.setEntry(ob);
                    }
                }
            }
            
        }
        return ustatemachine;
    }
    
    @SuppressWarnings("boxing")
    private Property processProperty(MComponent mcomponent)
    {
        Property uproperty = createProperty(mcomponent);
        log.info("Processing property " + mcomponent.getName());
        
        // name
        SysPhSUtil.setName(uproperty, mcomponent.getName());
        
        // type
        MClass type = mcomponent.getType();
        if (type == null)
        {
            log.error("Type of " + mcomponent.getName() + " not available");
            return null;
        }
        
        // library block
        if (type.getName().contains("."))
        {
            // check library blocks
            loop1: for (LibraryComponent lc : sysphsutil.getModelicaComponents())
            {
                if (type.getName().equals(lc.getName()))
                {
                    // go through without continue
                    
                    HashMap<String, String> hmlib = new HashMap<String, String>();
                    for (LibraryParameter lp : lc.getParameters())
                        hmlib.put(lp.getName(), lp.getValue());
                    
                    // modifications to delete if match
                    List<MModification> todelete = new LinkedList<MModification>();
                    
                    HashMap<String, MDataValue> hmmod = new HashMap<String, MDataValue>();
                    for (MModification mmodification : mcomponent.getModifications())
                    {
                        String nam = mmodification.getComponentPath().get(0).getName();
                        MDataValue val = mmodification.getAssignedValue();
                        
                        hmmod.put(nam, val);
                        
                        // all modifications should be in properties
                        if (!hmlib.containsKey(nam))
                            continue loop1;
                        
                        String val2 = hmlib.get(nam);
                        if (val2 != null)
                        {
                            if (val instanceof MRealValue && ((MRealValue) val).getValue() != Util.toDouble(val2, 0.0))
                                continue loop1;
                            if (val instanceof MIntegerValue && ((MIntegerValue) val).getValue() != Util.toInt(val2, 0))
                                continue loop1;
                            if (val instanceof MBooleanValue
                                    && ((MBooleanValue) val).isValue() != Util.toBoolean(val2, false))
                                continue loop1;
                            todelete.add(mmodification);
                        }
                    }
                    
                    // all properties should be in modifications
                    
                    for (Entry<String, String> ent : hmlib.entrySet())
                    {
                        if (!hmmod.containsKey(ent.getKey()))
                            continue loop1;
                    }
                    
                    log.info("Setting library block " + lc.getName() + " as type");
                    uproperty.setType(lc.getClas());
                    refs.put(getKey(type), lc.getClas());
                    mcomponent.getModifications().removeAll(todelete);
                    
                    log.info("Associating block " + type.getName() + " with " + lc.getName());
                    loop: for (MComponent mc : type.getComponents())
                    {
                        for (LibraryParameter lp : lc.getParameters())
                            if (lp.getName().equals(mc.getName()))
                            {
                                log.info("Associating property " + mc.getName());
                                refs.put(getKey(mc), lp.getProperty());
                                continue loop;
                            }
                        for (LibraryPort lp : lc.getPorts())
                            if (lp.getName().equals(mc.getName()))
                            {
                                log.info("Associating port " + mc.getName());
                                refs.put(getKey(mc), lp.getPort());
                                continue loop;
                            }
                    }
                    break loop1;
                }
            }
            if (uproperty.getType() == null)
                log.error("Couldn't find any library type for " + mcomponent.getName());
        }
        else if (mcomponent.getDirection() == MDirection.INPUT || mcomponent.getDirection() == MDirection.OUTPUT
                || type instanceof MConnector)
            uproperty.setType(getPortType(type));
        else
            uproperty.setType(createClass(type));
        
        // visibility
        if (mcomponent.getAccessControl() == MAccessControl.PROTECTED)
            uproperty.setVisibility(VisibilityKind.PROTECTED_LITERAL);
        else
            uproperty.setVisibility(VisibilityKind.PUBLIC_LITERAL);
        
        if (type instanceof MType)
        {
            if (mcomponent.getVariability() == MVariability.CONSTANT)
                uproperty.setIsReadOnly(true);
            else if (mcomponent.getVariability() == MVariability.PARAMETER)
                uproperty.applyStereotype(sysphsutil.getPhSConstant());
            else
            {
                uproperty.applyStereotype(sysphsutil.getPhSVariable());
                if (mcomponent.getVariability() == MVariability.DISCRETE)
                    uproperty.setValue(sysphsutil.getPhSVariable(), "isContinuous", false);
                else
                    uproperty.setValue(sysphsutil.getPhSVariable(), "isContinuous", true);
                
                if (mcomponent.getDataFlow() == MDataFlow.FLOW)
                    uproperty.setValue(sysphsutil.getPhSVariable(), "isConserved", true);
            }
            
        }
        
        // modifications
        
        return uproperty;
    }
    
    private Property processProperty2(MComponent mcomponent)
    {
        Property uproperty = createProperty(mcomponent);
        log.info("Processing property " + mcomponent.getName() + " a second time");
        
        if (mcomponent.getValue() != null && (mcomponent.getVariability() == MVariability.PARAMETER
                || mcomponent.getVariability() == MVariability.CONSTANT))
            uproperty.setDefaultValue(getValueSpecification(mcomponent.getValue()));
        
        if (mcomponent.getRedefinedComponent() != null)
        {
            Property redefined = createProperty(mcomponent.getRedefinedComponent());
            uproperty.getRedefinedProperties().add(redefined);
        }
        
        List<MModification> mmodifications = mcomponent.getModifications();
        if (mcomponent.getType() != null)
        {
            for (MModification mmodification : mmodifications)
            {
                if (mmodification.getComponentPath().size() == 0)
                {
                    log.warn("Skipping modification without path");
                    continue;
                }
                // assigned value = slot in default value
                // assigned reference = binding
                if (mmodification.getAssignedValue() != null)
                {
                    log.info("Assigned value");
                    MComponent previousmc = mcomponent;
                    Slot previousslot = null;
                    // go to the appropriate slot
                    for (MComponent currentmc : mmodification.getComponentPath())
                    {
                        log.info("Component " + currentmc.getName());
                        if (previousmc.getType() instanceof MType)
                        {
                            if (currentmc.getName().equals("start"))
                                break;
                        }
                        // is there already a slot corresponding to mc?
                        
                        if (!refs.containsKey(getKey(currentmc)))
                        {
                            log.warn("Ignoring modification, unknown component " + currentmc.getName());
                            mmodification.setAssignedValue(null);
                            break;
                        }
                        Property currentp = createProperty(currentmc);
                        
                        Slot currentslot = null;
                        // try to find existing slot
                        if (previousslot != null)
                        {
                            for (ValueSpecification vs : previousslot.getValues())
                                if (vs instanceof InstanceValue)
                                    for (Slot slot2 : ((InstanceValue) vs).getInstance().getSlots())
                                        if (slot2.getDefiningFeature() == currentp)
                                        {
                                            log.info("Found existing slot in previous slot " + currentmc.getName());
                                            currentslot = slot2;
                                            break;
                                        }
                        }
                        else if (uproperty.getDefaultValue() instanceof InstanceValue)
                        {
                            for (Slot slot2 : ((InstanceValue) uproperty.getDefaultValue()).getInstance().getSlots())
                                if (slot2.getDefiningFeature() == currentp)
                                {
                                    log.info("Found existing slot in default value");
                                    currentslot = slot2;
                                    break;
                                }
                        }
                        
                        if (currentslot == null)
                        {
                            log.info("Creating slot");
                            currentslot = UMLFactory.eINSTANCE.createSlot();
                            currentslot.setDefiningFeature(currentp);
                            
                            if (previousslot == null)
                            {
                                if (uproperty.getDefaultValue() instanceof InstanceValue)
                                {
                                    log.info("Assigning to existing default value");
                                    ((InstanceValue) uproperty.getDefaultValue()).getInstance().getSlots()
                                            .add(currentslot);
                                }
                                else
                                {
                                    InstanceSpecification is = UMLFactory.eINSTANCE.createInstanceSpecification();
                                    uproperty.getNearestPackage().getPackagedElements().add(is);
                                    is.getClassifiers().add(createClass(currentmc.getType()));
                                    is.getSlots().add(currentslot);
                                    
                                    InstanceValue iv = UMLFactory.eINSTANCE.createInstanceValue();
                                    iv.setInstance(is);
                                    
                                    log.info("Assigning to new default value");
                                    uproperty.setDefaultValue(iv);
                                }
                            }
                            else
                            {
                                if (previousslot.getValues().size() == 0
                                        || !(previousslot.getValues().get(0) instanceof InstanceValue))
                                {
                                    InstanceSpecification is = UMLFactory.eINSTANCE.createInstanceSpecification();
                                    uproperty.getNearestPackage().getPackagedElements().add(is);
                                    is.getClassifiers().add(createClass(currentmc.getType()));
                                    is.getSlots().add(currentslot);
                                    
                                    InstanceValue iv = UMLFactory.eINSTANCE.createInstanceValue();
                                    iv.setInstance(is);
                                    
                                    previousslot.getValues().add(iv);
                                    
                                    log.info("Assigning to previous slot with new instance spec: "
                                            + previousmc.getName());
                                }
                                else
                                {
                                    ((InstanceValue) previousslot.getValues().get(0)).getInstance().getSlots()
                                            .add(currentslot);
                                    log.info("Assigning to previous slot with existing instance spec"
                                            + previousmc.getName());
                                }
                            }
                            
                        }
                        previousslot = currentslot;
                        previousmc = currentmc;
                    }
                    
                    if (mmodification.getAssignedValue() != null)
                    {
                        ValueSpecification val = getValueSpecification(mmodification.getAssignedValue());
                        if (previousslot == null)
                        {
                            log.info("Assigning value to default value");
                            uproperty.setDefaultValue(val);
                        }
                        else
                        {
                            log.info("Assigning value to previous slot " + previousmc.getName() + "--"
                                    + previousslot.getValues().size());
                            previousslot.getValues().add(val);
                        }
                    }
                }
                else if (mmodification.getAssignedReference().size() > 0)
                {
                    log.info("Assigned reference");
                    List<Property> lp0 = new LinkedList<Property>();
                    List<Property> lp1 = new LinkedList<Property>();
                    
                    lp0.add(uproperty);
                    for (int i = 0; i < mmodification.getComponentPath().size(); i++)
                    {
                        MComponent mc = mmodification.getComponentPath().get(i);
                        if (!(mc.getName().equals("start") && i == mmodification.getComponentPath().size() - 2))
                        {
                            lp0.add(createProperty(mc));
                        }
                    }
                    for (int i = 0; i < mmodification.getAssignedReference().size(); i++)
                    {
                        MComponent mc = mmodification.getAssignedReference().get(i);
                        if (!(mc.getName().equals("start") && i == mmodification.getAssignedReference().size() - 2))
                        {
                            lp1.add(createProperty(mc));
                        }
                    }
                    log.info("Creating binding between " + lp0 + " and " + lp1);
                    Connector uconnector = UMLFactory.eINSTANCE.createConnector();
                    ((Class) uproperty.getOwner()).getOwnedConnectors().add(uconnector);
                    uconnector.applyStereotype(sysphsutil.getBindingConnector());
                    
                    ConnectorEnd ce0 = UMLFactory.eINSTANCE.createConnectorEnd();
                    uconnector.getEnds().add(ce0);
                    sysphsutil.updatePropertyPath(ce0, lp0);
                    
                    ConnectorEnd ce1 = UMLFactory.eINSTANCE.createConnectorEnd();
                    uconnector.getEnds().add(ce1);
                    sysphsutil.updatePropertyPath(ce1, lp1);
                }
                else
                    log.error("Modification without assigned value or reference");
            }
        }
        return uproperty;
        
    }
    
    private Pseudostate createInitialState(MComponent mcomponent)
    {
        if (mcomponent == null)
            throw new IllegalArgumentException("The component should not be null");
        if (getTranslatedType(mcomponent) != TranslatedType.INITIALSTATE)
            throw new IllegalStateException("The method can only be called on an initial state");
        ReferenceKey key = getKey(mcomponent);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Pseudostate)
                return (Pseudostate) uelement;
            throw new IllegalStateException("The element is not an initial state: " + mcomponent.getName());
        }
        
        log.info("Creating initial pseudostate " + mcomponent.getName());
        Pseudostate upseudostate = UMLFactory.eINSTANCE.createPseudostate();
        refs.put(key, upseudostate);
        toProcess.add(key);
        return upseudostate;
    }
    
    private Pseudostate processInitialState(MComponent mcomponent)
    {
        Pseudostate upseudostate = createInitialState(mcomponent);
        log.info("Processing initial pseudostate " + mcomponent.getName());
        SysPhSUtil.setName(upseudostate, mcomponent.getName());
        
        return upseudostate;
    }
    
    private State createState(MComponent mcomponent)
    {
        if (mcomponent == null)
            throw new IllegalArgumentException("The component should not be null");
        if (getTranslatedType(mcomponent) != TranslatedType.STATE)
            throw new IllegalStateException("The method can only be called on a state");
        
        ReferenceKey key = getKey(mcomponent);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof State)
                return (State) uelement;
            throw new IllegalStateException("The element is not a state: " + mcomponent.getName());
        }
        
        log.info("Creating state " + mcomponent.getName());
        State ustate = UMLFactory.eINSTANCE.createState();
        refs.put(key, ustate);
        toProcess.add(key);
        return ustate;
    }
    
    private State processState(MComponent mcomponent)
    {
        State ustate = createState(mcomponent);
        log.info("Processing state " + mcomponent.getName());
        SysPhSUtil.setName(ustate, mcomponent.getName());
        
        return ustate;
    }
    
    private Transition createTransition(MComponent mcomponent)
    {
        if (mcomponent == null)
            throw new IllegalArgumentException("The component should not be null");
        if (getTranslatedType(mcomponent) != TranslatedType.TRANSITION)
            throw new IllegalStateException("The method can only be called on a transition");
        
        ReferenceKey key = getKey(mcomponent);
        Element uelement = refs.get(key);
        if (uelement != null)
        {
            if (uelement instanceof Transition)
                return (Transition) uelement;
            throw new IllegalStateException("The element is not a transition: " + mcomponent.getName());
        }
        
        log.info("Creating transition " + mcomponent.getName());
        Transition utransition = UMLFactory.eINSTANCE.createTransition();
        refs.put(key, utransition);
        toProcess.add(key);
        return utransition;
    }
    
    private Transition processTransition(MComponent mcomponent)
    {
        Transition utransition = createTransition(mcomponent);
        log.info("Processing transition " + mcomponent.getName());
        
        return utransition;
    }
    
    private Class getPortType(MClass mclass)
    {
        Class ret = porttypes.get(mclass);
        if (ret != null)
            return ret;
        Classifier uclassifier = createClass(mclass);
        
        Class porttype = UMLFactory.eINSTANCE.createClass();
        root.getPackagedElements().add(porttype);
        porttype.applyStereotype(sysphsutil.getBlock());
        SysPhSUtil.setName(porttype, "PhS" + mclass.getName());
        
        Property fp = UMLFactory.eINSTANCE.createProperty();
        porttype.getOwnedAttributes().add(fp);
        SysPhSUtil.setName(fp, "phs" + mclass.getName());
        fp.applyStereotype(sysphsutil.getFlowProperty());
        
        // if (uclassifier instanceof DataType)
        // {
        // fp.setValue(sysphsutil.getFlowProperty(), "direction",
        // sysphsutil.getFlowPropertyIn());
        // fp.applyStereotype(sysphsutil.getPhSVariable());
        // }
        // else
        // fp.setValue(sysphsutil.getFlowProperty(), "direction",
        // sysphsutil.getFlowPropertyInout());
        
        for (MExtension mext : mclass.getExtensions())
        {
            if (mext.getDirection() == MDirection.INPUT)
            {
                fp.setValue(sysphsutil.getFlowProperty(), "direction", sysphsutil.getFlowPropertyIn());
                fp.applyStereotype(sysphsutil.getPhSVariable());
            }
            else if (mext.getDirection() == MDirection.OUTPUT)
            {
                fp.setValue(sysphsutil.getFlowProperty(), "direction", sysphsutil.getFlowPropertyOut());
                fp.applyStereotype(sysphsutil.getPhSVariable());
            }
        }
        
        fp.setType(uclassifier);
        porttypes.put(mclass, porttype);
        return porttype;
    }
    
    private static ValueSpecification getValueSpecification(MDataValue value)
    {
        if (value instanceof MRealValue)
        {
            LiteralReal lr = UMLFactory.eINSTANCE.createLiteralReal();
            lr.setValue(((MRealValue) value).getValue());
            return lr;
        }
        else if (value instanceof MIntegerValue)
        {
            LiteralInteger li = UMLFactory.eINSTANCE.createLiteralInteger();
            li.setValue(((LiteralInteger) value).getValue());
            return li;
        }
        else if (value instanceof MBooleanValue)
        {
            LiteralBoolean lb = UMLFactory.eINSTANCE.createLiteralBoolean();
            lb.setValue(((MBooleanValue) value).isValue());
            return lb;
        }
        else if (value instanceof MExpressionValue)
        {
            OpaqueExpression oe = UMLFactory.eINSTANCE.createOpaqueExpression();
            oe.getBodies().add(((MExpressionValue) value).getValue());
            oe.getLanguages().add(SysPhSUtil.SYSPHS_LANGUAGE);
            return oe;
        }
        return null;
    }
    
}
