package com.engisis.sysphs.serialization.simulink;

import java.util.List;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import com.engisis.sysphs.language.simscape.SConnectionPortBlock;
import com.engisis.sysphs.language.simscape.SLocation;
import com.engisis.sysphs.language.simulink.SBlock;
import com.engisis.sysphs.language.simulink.SConnectionPoint;
import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFVariable;
import com.engisis.sysphs.language.simulink.SFunction1Block;
import com.engisis.sysphs.language.simulink.SFunction2Block;
import com.engisis.sysphs.language.simulink.SInport;
import com.engisis.sysphs.language.simulink.SInterface;
import com.engisis.sysphs.language.simulink.SLibrary;
import com.engisis.sysphs.language.simulink.SLine;
import com.engisis.sysphs.language.simulink.SModel;
import com.engisis.sysphs.language.simulink.SOutport;
import com.engisis.sysphs.language.simulink.SReference;
import com.engisis.sysphs.language.simulink.SSubsystem;
import com.engisis.sysphs.language.simulink.SSystem;
import com.engisis.sysphs.language.simulink.SSystemParameter;
import com.engisis.sysphs.language.simulink.SSystemParameterAssignment;
import com.engisis.sysphs.util.XMLWriter;

/**
 * Writes the Simulink model in XML
 * 
 * @author barbau
 *
 */
public class SimulinkXMLWriter extends XMLWriter
{
    protected SimulinkXMLWriter(XMLStreamWriter xmlsw)
    {
        super(xmlsw);
    }
    
    public void introBlock() throws XMLStreamException
    {
        startElement("Block");
    }
    
    public void defaultBlock(SBlock sblock) throws XMLStreamException
    {
        if (sblock.getName() != null && !sblock.getName().isEmpty())
        {
            attribute("Name", sblock.getName());
        }
        if (sblock.getSID() != null && !sblock.getSID().isEmpty())
        {
            attribute("SID", sblock.getSID());
        }
    }
    
    public void positionBlock(SBlock sblock) throws XMLStreamException
    {
        if (sblock.getOwningSystem() != null)
        {
            int pos = sblock.getOwningSystem().getOwnedBlocks().indexOf(sblock);
            int w = 20;
            int h = 20;
            int m = 20;
            int x1 = (m + m * pos) + (pos * w);
            int x2 = x1 + w;
            int y1 = m;
            int y2 = y1 + h;
            String position = "[" + x1 + "," + y1 + "," + x2 + "," + y2 + "]";
            property("Position", position);
        }
    }
    
    public void outroBlock() throws XMLStreamException
    {
        endElement();
    }
    
    public void visit(SExpressionValue object) throws XMLStreamException
    {
        characters(object.getValue());
    }
    
    public void visit(SFunction1Block object) throws XMLStreamException
    {
        introBlock();
        attribute("BlockType", "S-Function");
        defaultBlock(object);
        if (object.getSfunction() != null && object.getSfunction().getName() != null
                && !object.getSfunction().getName().isEmpty())
        {
            property("FunctionName", object.getSfunction().getName());
        }
        if (object.getSfunction().getOwnedBlocks().size() == 0)
            property("Ports", "[1,1]");
        else
        {
            int i = 0;
            int o = 0;
            for (SBlock sbl : object.getSfunction().getOwnedBlocks())
            {
                if (sbl instanceof SInport)
                    i++;
                else if (sbl instanceof SOutport)
                    o++;
            }
            property("Ports", "[" + i + "," + o + "]");
        }
        positionBlock(object);
        outroBlock();
    }
    
    public void visit(SFunction2Block object) throws XMLStreamException
    {
        introBlock();
        attribute("BlockType", "M-S-Function");
        defaultBlock(object);
        if (object.getSfunction() != null && object.getSfunction().getName() != null
                && !object.getSfunction().getName().isEmpty())
        {
            property("FunctionName", object.getSfunction().getName());
        }
        property("Ports",
                "[" + object.getSfunction().getInputs().size() + "," + object.getSfunction().getOutputs().size() + "]");
        positionBlock(object);
        outroBlock();
    }
    
    public void visit(SInport object) throws XMLStreamException
    {
        introBlock();
        attribute("BlockType", "Inport");
        defaultBlock(object);
        int i = object.getOwningSystem().getInports().indexOf(object) + 1;
        if (i != 0)
            property("Port", String.valueOf(i));
        positionBlock(object);
        String unit = object.getUnit();
        if (unit != null)
            property("Unit", unit);
        outroBlock();
    }
    
    public void visit(SInterface object) throws XMLStreamException
    {
        introBlock();
        SSystem ssystem = object.getSystem();
        if (ssystem != null)
        {
            attribute("BlockType", ssystem.getName());
            defaultBlock(object);
            // if (ssystem.getInports().size() != 1 ||
            // ssystem.getOutports().size() != 1)
            {
                property("Ports", "[" + ssystem.getInports().size() + "," + ssystem.getOutports().size() + "]");
            }
            positionBlock(object);
            for (SSystemParameter sparameter : object.getSystem().getSystemparameters())
            {
                SDataValue val = sparameter.getValue();
                for (SSystemParameterAssignment sassignment : object.getAssignments())
                    if (sassignment.getParameter() == sparameter)
                        val = sassignment.getValue();
                    
                if (val != null)
                {
                    startElement("P");
                    attribute("Name", sparameter.getName());
                    val.accept(getDispatcher());
                    endElement();
                }
            }
        }
        outroBlock();
    }
    
    public void visit(SLibrary object) throws XMLStreamException
    {
        startElement("Library");
        property("SavedCharacterEncoding", "windows-1252");
        property("LibraryType", "BlockLibrary");
        property("SaveDefaultBlockParams", "on");
        property("ScopeRefreshTime", "0.035000");
        property("OverrideScopeRefreshTime", "on");
        property("DisableAllScopes", "off");
        property("FPTRunName", "Run 1");
        property("MaxMDLFileLineLength", "120");
        if (object.getSystem() != null)
            object.getSystem().accept(getDispatcher());
        endElement();
    }
    
    public void visit(SLine object) throws XMLStreamException
    {
        startElement("Line");
        
        SBlock sbsrcblock = object.getSource().getBlock();
        SBlock sbsrcport = object.getSource().getPort();
        
        String srcblock = sbsrcblock.getSID();
        String srcport = "out:0";
        
        if (sbsrcport == null)
            srcport = "out:1";
        else if (sbsrcblock instanceof SSubsystem)
        {
            if (sbsrcport instanceof SInport)
                srcport = "out:" + (((SSubsystem) sbsrcblock).getSystem().getInports().indexOf(sbsrcport) + 1);
            else if (sbsrcport instanceof SOutport)
                srcport = "out:" + (((SSubsystem) sbsrcblock).getSystem().getOutports().indexOf(sbsrcport) + 1);
        }
        else if (sbsrcblock instanceof SReference)
        {
            if (sbsrcport instanceof SInport)
                srcport = "out:" + (((SReference) sbsrcblock).getSystem().getInports().indexOf(sbsrcport) + 1);
            else if (sbsrcport instanceof SOutport)
            {
                srcport = "out:" + (((SReference) sbsrcblock).getSystem().getOutports().indexOf(sbsrcport) + 1);
            }
        }
        else if (sbsrcblock instanceof SInterface)
        {
            if (sbsrcport instanceof SInport)
                srcport = "out:" + (((SInterface) sbsrcblock).getSystem().getInports().indexOf(sbsrcport) + 1);
            else if (sbsrcport instanceof SOutport)
                srcport = "out:" + (((SInterface) sbsrcblock).getSystem().getOutports().indexOf(sbsrcport) + 1);
        }
        else if (sbsrcblock instanceof SFunction1Block)
        {
            if (sbsrcport instanceof SInport)
                srcport = "out:" + (((SFunction1Block) sbsrcblock).getSfunction().getInports().indexOf(sbsrcport) + 1);
            else if (sbsrcport instanceof SOutport)
                srcport = "out:" + (((SFunction1Block) sbsrcblock).getSfunction().getOutports().indexOf(sbsrcport) + 1);
        }
        else if (sbsrcblock instanceof SFunction2Block)
        {
            if (sbsrcport instanceof SInport)
                srcport = "out:" + (((SFunction2Block) sbsrcblock).getSfunction().getInports().indexOf(sbsrcport) + 1);
            else if (sbsrcport instanceof SOutport)
                srcport = "out:" + (((SFunction2Block) sbsrcblock).getSfunction().getOutports().indexOf(sbsrcport) + 1);
        }
        else
        {
            if (sbsrcport instanceof SInport)
                srcport = "out:" + (sbsrcport.getOwningSystem().getInports().indexOf(sbsrcport) + 1);
            else if (sbsrcport instanceof SOutport)
                srcport = "out:" + (sbsrcport.getOwningSystem().getOutports().indexOf(sbsrcport) + 1);
        }
        
        property("Src", srcblock + "#" + srcport);
        
        for (SConnectionPoint sdestination : object.getDestinations())
        {
            SBlock sbdstblock = sdestination.getBlock();
            SBlock sbdstport = sdestination.getPort();
            
            String dstblock = sbdstblock.getSID();
            String dstport = "in:0";
            
            if (sbdstport == null)
                dstport = "in:1";
            else if (sbdstblock instanceof SSubsystem)
            {
                if (sbdstport instanceof SInport)
                    dstport = "in:" + (((SSubsystem) sbdstblock).getSystem().getInports().indexOf(sbdstport) + 1);
                else if (sbdstport instanceof SOutport)
                    dstport = "in:" + (((SSubsystem) sbdstblock).getSystem().getOutports().indexOf(sbdstport) + 1);
            }
            else if (sbdstblock instanceof SReference)
            {
                if (sbdstport instanceof SInport)
                    dstport = "in:" + (((SReference) sbdstblock).getSystem().getInports().indexOf(sbdstport) + 1);
                else if (sbdstport instanceof SOutport)
                    dstport = "in:" + (((SReference) sbdstblock).getSystem().getOutports().indexOf(sbdstport) + 1);
            }
            else if (sbdstblock instanceof SInterface)
            {
                if (sbdstport instanceof SInport)
                    dstport = "in:" + (((SInterface) sbdstblock).getSystem().getInports().indexOf(sbdstport) + 1);
                else if (sbdstport instanceof SOutport)
                    dstport = "in:" + (((SInterface) sbdstblock).getSystem().getOutports().indexOf(sbdstport) + 1);
            }
            else if (sbdstblock instanceof SFunction1Block)
            {
                if (sbdstport instanceof SInport)
                    dstport = "in:"
                            + (((SFunction1Block) sbdstblock).getSfunction().getInports().indexOf(sbdstport) + 1);
                else if (sbdstport instanceof SOutport)
                    dstport = "in:"
                            + (((SFunction1Block) sbdstblock).getSfunction().getOutports().indexOf(sbdstport) + 1);
            }
            else if (sbdstblock instanceof SFunction2Block)
            {
                if (sbdstport instanceof SInport)
                    dstport = "in:"
                            + (((SFunction2Block) sbdstblock).getSfunction().getInports().indexOf(sbdstport) + 1);
                else if (sbdstport instanceof SOutport)
                    dstport = "in:"
                            + (((SFunction2Block) sbdstblock).getSfunction().getOutports().indexOf(sbdstport) + 1);
            }
            else
            {
                if (sbdstport instanceof SInport)
                    dstport = "in:" + (sbdstport.getOwningSystem().getInports().indexOf(sbdstport) + 1);
                else if (sbdstport instanceof SOutport)
                    dstport = "in:" + (sbdstport.getOwningSystem().getOutports().indexOf(sbdstport) + 1);
            }
            if (object.getDestinations().size() > 1)
            {
                startElement("Branch");
                property("Dst", dstblock + "#" + dstport);
                endElement();
            }
            else
            {
                property("Dst", dstblock + "#" + dstport);
            }
        }
        endElement();
    }
    
    public void visit(SModel object) throws XMLStreamException
    {
        startElement("Model");
        
        startElement("Object");
        attribute("PropName", "BdWindowsInfo");
        attribute("ObjectID", "1");
        attribute("ClassName", "Simulink.BDWindowsInfo");
        
        startElement("Object");
        attribute("PropName", "WindowsInfo");
        attribute("ObjectID", "2");
        attribute("ClassName", "Simulink.WindowInfo");
        property("IsActive", "1");
        
        startElement("Object");
        attribute("PropName", "ModelBrowserInfo");
        attribute("ObjectID", "3");
        attribute("ClassName", "Simulink.ModelBrowserInfo");
        endElement();
        
        startElement("Object");
        attribute("PropName", "ExplorerBarInfo");
        attribute("ObjectID", "4");
        attribute("ClassName", "Simulink.ExplorerBarInfo");
        endElement();
        
        startElement("Object");
        attribute("PropName", "EditorsInfo");
        attribute("ObjectID", "5");
        attribute("ClassName", "Simulink.EditorInfo");
        endElement();
        
        endElement();
        
        endElement();
        
        if (object.getSystem() != null)
            object.getSystem().accept(getDispatcher());
        endElement();
    }
    
    public void visit(SOutport object) throws XMLStreamException
    {
        introBlock();
        attribute("BlockType", "Outport");
        defaultBlock(object);
        int i = object.getOwningSystem().getOutports().indexOf(object) + 1;
        if (i != 0)
            property("Port", String.valueOf(i));
        positionBlock(object);
        String unit = object.getUnit();
        if (unit != null)
            property("Unit", unit);
        outroBlock();
    }
    
    public void visit(SDoubleValue object) throws XMLStreamException
    {
        characters(String.valueOf(object.getValue()));
    }
    
    public void visit(SReference object) throws XMLStreamException
    {
        introBlock();
        attribute("BlockType", "Reference");
        defaultBlock(object);
        if (object.getSystem() != null)
        {
            property("Ports", "[" + getSystemPorts(object.getSystem()) + "]");
            positionBlock(object);
            property("SourceBlock", object.getSystem().getPath());
            property("SourceType", "SubSystem");
        }
        outroBlock();
    }
    
    public void visit(SSubsystem object) throws XMLStreamException
    {
        introBlock();
        attribute("BlockType", "SubSystem");
        defaultBlock(object);
        if (object.getSystem() != null)
        {
            property("Ports", "[" + getSystemPorts(object.getSystem()) + "]");
            object.getSystem().accept(getDispatcher());
        }
        positionBlock(object);
        outroBlock();
    }
    
    public void visit(SSystem object) throws XMLStreamException
    {
        startElement("System");
        if (object.getOwningModel() != null)
            property("Open", "on");
        else
            property("Open", "off");
        
        if (object.getSid() != 0)
            property("SIDHighWatermark", String.valueOf(object.getSid()));
        
        printSystem(object);
        endElement();
    }
    
    protected void printSystem(SSystem ssystem) throws XMLStreamException
    {
        for (SBlock sblock : ssystem.getOwnedBlocks())
            sblock.accept(getDispatcher());
        for (SLine sline : ssystem.getOwnedLines())
            sline.accept(getDispatcher());
        
        for (SBlock sblock : ssystem.getOwnedBlocks())
            printSystemCode(sblock);
    }
    
    protected void printSystemCode(SBlock sblock) throws XMLStreamException
    {
        if (sblock instanceof SFunction2Block)
        {
            SFunction2Block sf2block = (SFunction2Block) sblock;
            // additional lines to connect inputs/outputs to sfunctions
            List<SFInputVariable> inputs = sf2block.getSfunction().getInputs();
            List<SFOutputVariable> outputs = sf2block.getSfunction().getOutputs();
            
            if (inputs != null)
            {
                for (int i = 0; i < inputs.size(); i++)
                {
                    SFInputVariable sinputvariable = inputs.get(i);
                    SInport sinport = sinputvariable.getInport();
                    if (sinport != null)
                    {
                        startElement("Line");
                        property("Src", sinport.getSID() + "#out:1");
                        
                        property("Dst", sf2block.getSID() + "#in:" + (i + 1));
                        
                        endElement();
                    }
                }
            }
            if (outputs != null)
            {
                for (int i = 0; i < outputs.size(); i++)
                {
                    SFOutputVariable soutputvariable = outputs.get(i);
                    SOutport soutport = soutputvariable.getOutport();
                    if (soutport != null)
                    {
                        startElement("Line");
                        property("Src", sf2block.getSID() + "#out:" + (i + 1));
                        
                        property("Dst", soutport.getSID() + "#in:1");
                        
                        endElement();
                    }
                }
            }
        }
    }
    
    private static String getSystemPorts(SSystem ssystem)
    {
        if (ssystem == null)
            return "";
        int ninport = 0;
        int noutport = 0;
        int nlport = 0;
        int nrport = 0;
        for (SBlock sblock : ssystem.getOwnedBlocks())
        {
            if (sblock instanceof SInport)
                ninport++;
            else if (sblock instanceof SOutport)
                noutport++;
            else if (sblock instanceof SConnectionPortBlock)
            {
                SLocation sl = ((SConnectionPortBlock) sblock).getLocation();
                if (sl == SLocation.RIGHT)
                    nrport++;
                else if (sl == SLocation.LEFT)
                    nlport++;
            }
        }
        StringBuilder sb = new StringBuilder();
        sb.append(ninport + "," + noutport);
        if (nlport != 0 || nrport != 0)
            sb.append(",0,0,0," + nlport + "," + nrport);
        return sb.toString();
    }
    
    protected void property(String name, String value) throws XMLStreamException
    {
        startElement("P");
        attribute("Name", name);
        characters(value);
        endElement();
    }
    
    public static String getValue(SFVariable svar)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < svar.getValue().size(); i++)
        {
            SDataValue sdv = svar.getValue().get(i);
            if (i != 0)
                sb.append(";");
            if (sdv instanceof SExpressionValue)
                sb.append(((SExpressionValue) sdv).getValue());
            else if (sdv instanceof SDoubleValue)
                sb.append(((SDoubleValue) sdv).getValue());
        }
        return sb.toString();
    }
    
}
