package com.engisis.sysphs.serialization.simulink;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import com.engisis.sysphs.language.simulink.SDataValue;
import com.engisis.sysphs.language.simulink.SDoubleValue;
import com.engisis.sysphs.language.simulink.SExpressionValue;
import com.engisis.sysphs.language.simulink.SFContinuousStateVariable;
import com.engisis.sysphs.language.simulink.SFDWorkAssignment;
import com.engisis.sysphs.language.simulink.SFDWorkVariable;
import com.engisis.sysphs.language.simulink.SFDerivativeAssignment;
import com.engisis.sysphs.language.simulink.SFDiscreteStateVariable;
import com.engisis.sysphs.language.simulink.SFInputVariable;
import com.engisis.sysphs.language.simulink.SFOutputAssignment;
import com.engisis.sysphs.language.simulink.SFOutputVariable;
import com.engisis.sysphs.language.simulink.SFUpdateAssignment;
import com.engisis.sysphs.language.simulink.SFVariable;
import com.engisis.sysphs.language.simulink.SFVariableAssignment;
import com.engisis.sysphs.language.simulink.SFunction2;
import com.engisis.sysphs.translation.simulink.SysMLToSimulinkTranslator;
import com.engisis.sysphs.util.TextWriter;

/**
 * Writes Level-2 S-Functions
 * 
 * @author barbau
 *
 */
public class SimulinkSF2Writer extends TextWriter
{
    
    /**
     * Constructs a Level-2 S-Functions writer
     * 
     * @param w
     *            object to write the content into
     */
    protected SimulinkSF2Writer(Writer w)
    {
        super(w);
    }
    
    public void visit(SFunction2 object) throws IOException
    {
        List<SFInputVariable> in = object.getInputs();
        List<SFContinuousStateVariable> cst = object.getContinuousStates();
        List<SFDiscreteStateVariable> dst = object.getDiscreteStates();
        List<SFDWorkVariable> dw = object.getDWorkStates();
        List<SFOutputVariable> out = object.getOutputs();
        List<SFVariable> dws = new ArrayList<SFVariable>(dst);
        dws.addAll(dw);
        
        append("function " + object.getName() + "(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ")", TL);
        indent();
        append("setup(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ");", TL);
        unindent();
        append("end", TL);
        
        append("function setup(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ")", TL);
        indent();
        append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".NumInputPorts =" + in.size() + ";", TL);
        for(int i=0;i<in.size();i++)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".InputPort(" + (i + 1) + ").DirectFeedthrough=" + (in.get(i).isDirectFeedthrough()?"1":"0") + ";", TL);
            
        append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".NumOutputPorts =" + out.size() + ";", TL);
        
        for (int i = 0; i < out.size(); i++)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".OutputPort(" + (i + 1) + ").SamplingMode = 'sample';",
                    TL);
        
        if (cst.size() > 0)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".NumContStates = " + cst.size() + ";", TL);
        if (dst.size() > 0)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".NumDiscStates = " + dst.size() + ";", TL);
        
        if (dws.size() > 0)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME
                    + ".RegBlockMethod('PostPropagationSetup',@DoPostPropSetup);", TL);
        if (dw.size() > 0)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME
                    + ".RegBlockMethod('InitializeConditions',@InitConditions);", TL);
        
        // Derivatives
        if (cst.size() != 0)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".RegBlockMethod('Derivatives',@Derivatives);", TL);
        
        // Update
        if (dst.size() != 0)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".RegBlockMethod('Update',@Update);", TL);
        
        // Outputs
        if (out.size() != 0)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".RegBlockMethod('Outputs',@Outputs);", TL);
        append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".SampleTimes = [0 0];", TL);
        if (out.size() > 1)
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".RegBlockMethod('SetInputPortSamplingMode', @InputPortSamplingMode);", TL);
        
        unindent();
        append("end", TL);
        
        if (out.size() > 1)
        {
            append("function InputPortSamplingMode(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ", port, mode)", TL);
            indent();
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".InputPort(port).SamplingMode = mode;", TL);
            unindent();
            append("end", TL);
        }
        
        if (dws.size() > 0)
        {
            append("function DoPostPropSetup(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ")", TL);
            indent();
            append(SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".NumDworks=" + dws.size() + ";", TL);
            for (int i = 0; i < dws.size(); i++)
            {
                SFVariable sw = dws.get(i);
                String pref = SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ".Dwork(" + (i + 1) + ")";
                append(pref + ".Name = '" + sw.getName() + "';", TL);
                int dim = 1;
                if (sw.getDimensions().size() > 0)
                    dim = sw.getDimensions().get(0).intValue();
                if (sw.getDimensions().size() == 2)
                    dim *= sw.getDimensions().get(1).intValue();
                
                append(pref + ".Dimensions = " + dim + ";", TL);
                append(pref + ".DatatypeID = 0;", TL);
                append(pref + ".Complexity = 'Real';", TL);
                if (dws.get(i) instanceof SFDiscreteStateVariable)
                    append(pref + ".UsedAsDiscState = true;", TL);
                else
                    append(pref + ".UsedAsDiscState = false;", TL);
            }
            unindent();
            append("end", TL);
        }
        
        List<String> lInitial = new LinkedList<String>();
        List<String> lUpdate = new LinkedList<String>();
        List<String> lDeriv = new LinkedList<String>();
        List<String> lOutput = new LinkedList<String>();
        
        for (SFVariable sv : dws)
            if (sv.getValue() != null)
                lInitial.add(sv.getName() + "=" + getValue(sv) + ";");
        
        for (SFVariableAssignment sassignment : object.getAssignments())
        {
            String expr = sassignment.getExpression() + ";";
            if (sassignment instanceof SFUpdateAssignment)
                lUpdate.add(expr);
            else if (sassignment instanceof SFDerivativeAssignment)
                lDeriv.add(expr);
            else if (sassignment instanceof SFOutputAssignment)
                lOutput.add(expr);
            else if (sassignment instanceof SFDWorkAssignment)
                lInitial.add(expr);
        }
        
        if (dw.size() > 0)
        {
            append("function InitConditions(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ")", TL);
            indent();
            for (String s : lInitial)
                append(s, TL);
            unindent();
            append("end", TL);
        }
        if (cst.size() != 0)
        {
            append("function Derivatives(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ")", TL);
            indent();
            for (String s : lDeriv)
                append(s, TL);
            unindent();
            append("end", TL);
        }
        if (dst.size() != 0)
        {
            append("function Update(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ")");
            indent();
            for (String s : lUpdate)
                append(s, TL);
            unindent();
            append("end", TL);
        }
        if (out.size() != 0)
        {
            append("function Outputs(" + SysMLToSimulinkTranslator.SF2_BLOCK_NAME + ")", TL);
            indent();
            for (String s : lOutput)
                append(s, TL);
            unindent();
            append("end", TL);
        }
    }
    
    public static String getValue(SFVariable svar)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < svar.getValue().size(); i++)
        {
            SDataValue sdv = svar.getValue().get(i);
            if (i != 0)
                sb.append(";");
            if (sdv instanceof SExpressionValue)
                sb.append(((SExpressionValue) sdv).getValue());
            else if (sdv instanceof SDoubleValue)
                sb.append(((SDoubleValue) sdv).getValue());
        }
        return sb.toString();
    }
}
